/* File: do-spell.c */

/* Purpose: Do everything for each spell */

#include "angband.h"


/*
 * Generate dice info string such as "foo 2d10"
 */
static cptr info_string_dice(cptr str, int dice, int sides, int base)
{
	/* Fix value */
	if (!dice)
		return format("%s%d", str, base);

	/* Dice only */
	else if (!base)
		return format("%s%dd%d", str, dice, sides);

	/* Dice plus base value */
	else
		return format("%s%dd%d%+d", str, dice, sides, base);
}


/*
 * Generate damage-dice info string such as "dam 2d10"
 */
static cptr info_damage(int dice, int sides, int base)
{
#ifdef JP
	return info_string_dice(":", dice, sides, base);
#else
	return info_string_dice("dam ", dice, sides, base);
#endif
}


/*
 * Generate duration info string such as "dur 20+1d20"
 */
static cptr info_duration(int base, int sides)
{
#ifdef JP
	return format(":%d+1d%d", base, sides);
#else
	return format("dur %d+1d%d", base, sides);
#endif
}


/*
 * Generate range info string such as "range 5"
 */
static cptr info_range(int range)
{
#ifdef JP
	return format("͈:%d", range);
#else
	return format("range %d", range);
#endif
}


/*
 * Generate heal info string such as "heal 2d8"
 */
static cptr info_heal(int dice, int sides, int base)
{
#ifdef JP
	return info_string_dice(":", dice, sides, base);
#else
	return info_string_dice("heal ", dice, sides, base);
#endif
}


/*
 * Generate delay info string such as "delay 15+1d15"
 */
static cptr info_delay(int base, int sides)
{
#ifdef JP
	return format("x:%d+1d%d", base, sides);
#else
	return format("delay %d+1d%d", base, sides);
#endif
}


/*
 * Generate multiple-damage info string such as "dam 25 each"
 */
static cptr info_multi_damage(int dam)
{
#ifdef JP
	return format(":e%d", dam);
#else
	return format("dam %d each", dam);
#endif
}


/*
 * Generate multiple-damage-dice info string such as "dam 5d2 each"
 */
static cptr info_multi_damage_dice(int dice, int sides)
{
#ifdef JP
	return format(":e%dd%d", dice, sides);
#else
	return format("dam %dd%d each", dice, sides);
#endif
}


/*
 * Generate power info string such as "power 100"
 */
static cptr info_power(int power)
{
#ifdef JP
	return format(":%d", power);
#else
	return format("power %d", power);
#endif
}


/*
 * Generate power info string such as "power 1d100"
 */
static cptr info_power_dice(int dice, int sides)
{
#ifdef JP
	return format(":%dd%d", dice, sides);
#else
	return format("power %dd%d", dice, sides);
#endif
}


/*
 * Generate radius info string such as "rad 100"
 */
static cptr info_radius(int rad)
{
#ifdef JP
	return format("a:%d", rad);
#else
	return format("rad %d", rad);
#endif
}


/*
 * Generate weight info string such as "max wgt 15"
 */
static cptr info_weight(int weight)
{
	char buf[30];
	format_weight(buf, weight);
#ifdef JP
	return format("őd:%s", buf);
#else
	return format("max wgt %s", buf);
#endif
}


/*
 * Prepare standard probability to become beam for fire_bolt_or_beam(creature_ptr, )
 */
static int beam_chance(creature_type *creature_ptr)
{
	if (creature_ptr->class_idx == CLASS_MAGE)
		return creature_ptr->lev;

	if (creature_ptr->class_idx == CLASS_HIGH_MAGE || creature_ptr->class_idx == CLASS_SORCERER)
		return creature_ptr->lev + 10;

	return creature_ptr->lev / 2;
}


/*
 * Handle summoning and failure of trump spells
 */
static bool trump_summoning(creature_type *creature_ptr, int num, bool pet, int y, int x, int lev, int type, u32b mode)
{
	int plev = creature_ptr->lev;

	int who;
	int i;
	bool success = FALSE;

	/* Default level */
	if (!lev) lev = plev * 2 / 3 + randint1(plev / 2);

	if (pet)
	{
		/* Become pet */
		mode |= PM_FORCE_PET;

		/* Only sometimes allow unique creature */
		if (mode & PM_ALLOW_UNIQUE)
		{
			/* Forbid often */
			if (randint1(50 + plev) >= plev / 10)
				mode &= ~PM_ALLOW_UNIQUE;
		}

		/* Player is who summons */
		who = -1;
	}
	else
	{
		/* Prevent taming, allow unique creature */
		mode |= PM_NO_PET;

		/* Behave as if they appear by themselfs */
		who = 0;
	}

	for (i = 0; i < num; i++)
	{
		if (summon_specific(creature_ptr, y, x, lev, type, mode))
			success = TRUE;
	}

	if (!success)
	{
#ifdef JP
		msg_print("NȂ̃J[ȟĂѐɓȂB");
#else
		msg_print("Nobody answers to your Trump call.");
#endif
	}

	return success;
}


/*
 * This spell should become more useful (more controlled) as the
 * player gains experience levels.  Thus, add 1/5 of the player's
 * level to the die roll.  This eliminates the worst effects later on,
 * while keeping the results quite random.  It also allows some potent
 * effects only at high level.
 */
static void cast_wonder(creature_type *creature_ptr, int dir)
{
	int plev = creature_ptr->lev;
	int die = randint1(100) + plev / 5;
	// TODO: add Karma of Fortune feature.
	int vir = 0;

	if (vir)
	{
		if (creature_ptr->karmas[vir - 1] > 0)
		{
			while (randint1(400) < creature_ptr->karmas[vir - 1]) die++;
		}
		else
		{
			while (randint1(400) < (0-creature_ptr->karmas[vir - 1])) die--;
		}
	}

	if (die > 100)
	{
#ifdef JP
		msg_print("Ȃ݂͗͂Ȃ̂I");
#else
		msg_print("You feel a surge of power!");
#endif
	}

	if (die < 8) clone_creature(creature_ptr, dir);
	else if (die < 14) speed_other_creature(creature_ptr, dir);
	else if (die < 26) heal_other_creature(creature_ptr, dir, diceroll(4, 6));
	else if (die < 31) poly_creature(creature_ptr, dir);
	else if (die < 36)
		fire_bolt_or_beam(creature_ptr, beam_chance(creature_ptr) - 10, GF_MISSILE, dir,
				  diceroll(3 + ((plev - 1) / 5), 4));
	else if (die < 41) confuse_creature(creature_ptr, dir, plev);
	else if (die < 46) fire_ball(creature_ptr, GF_POIS, dir, 20 + (plev / 2), 3);
	else if (die < 51) (void)lite_line(creature_ptr, dir);
	else if (die < 56)
		fire_bolt_or_beam(creature_ptr, beam_chance(creature_ptr) - 10, GF_ELEC, dir,
				  diceroll(3 + ((plev - 5) / 4), 8));
	else if (die < 61)
		fire_bolt_or_beam(creature_ptr, beam_chance(creature_ptr) - 10, GF_COLD, dir,
				  diceroll(5 + ((plev - 5) / 4), 8));
	else if (die < 66)
		fire_bolt_or_beam(creature_ptr, beam_chance(creature_ptr), GF_ACID, dir,
				  diceroll(6 + ((plev - 5) / 4), 8));
	else if (die < 71)
		fire_bolt_or_beam(creature_ptr, beam_chance(creature_ptr), GF_FIRE, dir,
				  diceroll(8 + ((plev - 5) / 4), 8));
	else if (die < 76) drain_life(creature_ptr, dir, 75);
	else if (die < 81) fire_ball(creature_ptr, GF_ELEC, dir, 30 + plev / 2, 2);
	else if (die < 86) fire_ball(creature_ptr, GF_ACID, dir, 40 + plev, 2);
	else if (die < 91) fire_ball(creature_ptr, GF_ICE, dir, 70 + plev, 3);
	else if (die < 96) fire_ball(creature_ptr, GF_FIRE, dir, 80 + plev, 3);
	else if (die < 101) drain_life(creature_ptr, dir, 100 + plev);
	else if (die < 104)
	{
		earthquake(creature_ptr, creature_ptr->fy, creature_ptr->fx, 12);
	}
	else if (die < 106)
	{
		(void)destroy_area(creature_ptr, creature_ptr->fy, creature_ptr->fx, 13 + randint0(5), FALSE);
	}
	else if (die < 108)
	{
		symbol_genocide(creature_ptr, plev+50, TRUE);
	}
	else if (die < 110) dispel_creatures(creature_ptr, 120);
	else // RARE
	{
		dispel_creatures(creature_ptr, 150);
		slow_creatures(creature_ptr);
		sleep_creatures(creature_ptr);
		hp_player(creature_ptr, 300);
	}
}


static void cast_invoke_spirits(creature_type *creature_ptr, int dir)
{
	floor_type *floor_ptr = get_floor_ptr(creature_ptr);
	int plev = creature_ptr->lev;
	int die = randint1(100) + plev / 5;
	// TODO: Add Karma of Fortune feature.
	int vir = 0;

	if (vir)
	{
		if (creature_ptr->karmas[vir - 1] > 0)
		{
			while (randint1(400) < creature_ptr->karmas[vir - 1]) die++;
		}
		else
		{
			while (randint1(400) < (0-creature_ptr->karmas[vir - 1])) die--;
		}
	}

#ifdef JP
	msg_print("Ȃ͎҂̗͂W...");
#else
	msg_print("You call on the power of the dead...");
#endif

	if (die > 100)
	{
#ifdef JP
		msg_print("Ȃ͂ǂ남ǂ낵͂̂˂I");
#else
		msg_print("You feel a surge of eldritch force!");
#endif
	}


	if (die < 8)
	{
#ifdef JP
		msg_print("ȂĂIȂ̎̒nʂ狀leオĂI");
#else
		msg_print("Oh no! Mouldering forms rise from the earth around you!");
#endif

		(void)summon_specific(0, creature_ptr->fy, creature_ptr->fx, floor_ptr->floor_level, SUMMON_UNDEAD, (PM_ALLOW_GROUP | PM_ALLOW_UNIQUE | PM_NO_PET));
	}
	else if (die < 14)
	{
#ifdef JP
		msg_print("󂵓׈ȑ݂Ȃ̐Sʂ߂čs...");
#else
		msg_print("An unnamable evil brushes against your mind...");
#endif

		set_afraid(creature_ptr, creature_ptr->afraid + randint1(4) + 4);
	}
	else if (die < 26)
	{
#ifdef JP
		msg_print("Ȃ̓ɑʂ̗H삽̑X񂹂Ă...");
#else
		msg_print("Your head is invaded by a horde of gibbering spectral voices...");
#endif

		set_confused(creature_ptr, creature_ptr->confused + randint1(4) + 4);
	}
	else if (die < 31)
	{
		poly_creature(creature_ptr, dir);
	}
	else if (die < 36)
	{
		fire_bolt_or_beam(creature_ptr, beam_chance(creature_ptr) - 10, GF_MISSILE, dir,
				  diceroll(3 + ((plev - 1) / 5), 4));
	}
	else if (die < 41)
	{
		confuse_creature(creature_ptr, dir, plev);
	}
	else if (die < 46)
	{
		fire_ball(creature_ptr, GF_POIS, dir, 20 + (plev / 2), 3);
	}
	else if (die < 51)
	{
		(void)lite_line(creature_ptr, dir);
	}
	else if (die < 56)
	{
		fire_bolt_or_beam(creature_ptr, beam_chance(creature_ptr) - 10, GF_ELEC, dir,
				  diceroll(3+((plev-5)/4),8));
	}
	else if (die < 61)
	{
		fire_bolt_or_beam(creature_ptr, beam_chance(creature_ptr) - 10, GF_COLD, dir,
				  diceroll(5+((plev-5)/4),8));
	}
	else if (die < 66)
	{
		fire_bolt_or_beam(creature_ptr, beam_chance(creature_ptr), GF_ACID, dir,
				  diceroll(6+((plev-5)/4),8));
	}
	else if (die < 71)
	{
		fire_bolt_or_beam(creature_ptr, beam_chance(creature_ptr), GF_FIRE, dir,
				  diceroll(8+((plev-5)/4),8));
	}
	else if (die < 76)
	{
		drain_life(creature_ptr, dir, 75);
	}
	else if (die < 81)
	{
		fire_ball(creature_ptr, GF_ELEC, dir, 30 + plev / 2, 2);
	}
	else if (die < 86)
	{
		fire_ball(creature_ptr, GF_ACID, dir, 40 + plev, 2);
	}
	else if (die < 91)
	{
		fire_ball(creature_ptr, GF_ICE, dir, 70 + plev, 3);
	}
	else if (die < 96)
	{
		fire_ball(creature_ptr, GF_FIRE, dir, 80 + plev, 3);
	}
	else if (die < 101)
	{
		drain_life(creature_ptr, dir, 100 + plev);
	}
	else if (die < 104)
	{
		earthquake(creature_ptr, creature_ptr->fy, creature_ptr->fx, 12);
	}
	else if (die < 106)
	{
		(void)destroy_area(creature_ptr, creature_ptr->fy, creature_ptr->fx, 13 + randint0(5), FALSE);
	}
	else if (die < 108)
	{
		symbol_genocide(creature_ptr, plev+50, TRUE);
	}
	else if (die < 110)
	{
		dispel_creatures(creature_ptr, 120);
	}
	else
	{ /* RARE */
		dispel_creatures(creature_ptr, 150);
		slow_creatures(creature_ptr);
		sleep_creatures(creature_ptr);
		hp_player(creature_ptr, 300);
	}

	if (die < 31)
	{
#ifdef JP
		msg_print("ATȐNXNX΂Bu܂͉X̒ԂɂȂ邾낤Bカ҂Bv");
#else
		msg_print("Sepulchral voices chuckle. 'Soon you will join us, mortal.'");
#endif
	}
}


void wild_magic(creature_type *creature_ptr, int spell)
{
	int counter = 0;
	int type = SUMMON_BIZARRE1 + randint0(6);
	floor_type *floor_ptr = get_floor_ptr(creature_ptr);

	if (type < SUMMON_BIZARRE1) type = SUMMON_BIZARRE1;
	else if (type > SUMMON_BIZARRE6) type = SUMMON_BIZARRE6;

	switch (randint1(spell) + randint1(8) + 1)
	{
	case 1:
	case 2:
	case 3:
		teleport_player(creature_ptr, 10, TELEPORT_PASSIVE);
		break;
	case 4:
	case 5:
	case 6:
		teleport_player(creature_ptr, 100, TELEPORT_PASSIVE);
		break;
	case 7:
	case 8:
		teleport_player(creature_ptr, 200, TELEPORT_PASSIVE);
		break;
	case 9:
	case 10:
	case 11:
		unlite_area(creature_ptr, 10, 3);
		break;
	case 12:
	case 13:
	case 14:
		lite_area(creature_ptr, diceroll(2, 3), 2);
		break;
	case 15:
		destroy_doors_touch(creature_ptr);
		break;
	case 16: case 17:
		wall_breaker(creature_ptr);
	case 18:
		sleep_creatures_touch(creature_ptr);
		break;
	case 19:
	case 20:
		trap_creation(creature_ptr, creature_ptr->fy, creature_ptr->fx);
		break;
	case 21:
	case 22:
		door_creation(creature_ptr);
		break;
	case 23:
	case 24:
	case 25:
		aggravate_creatures(creature_ptr);
		break;
	case 26:
		earthquake(creature_ptr, creature_ptr->fy, creature_ptr->fx, 5);
		break;
	case 27:
	case 28:
		(void)gain_trait(creature_ptr, 0, TRUE);
		break;
	case 29:
	case 30:
		apply_disenchant(creature_ptr, 1);
		break;
	case 31:
		lose_all_info(creature_ptr);
		break;
	case 32:
		fire_ball(creature_ptr, GF_CHAOS, 0, spell + 5, 1 + (spell / 10));
		break;
	case 33:
		wall_stone(creature_ptr);
		break;
	case 34:
	case 35:
		while (counter++ < 8)
		{
			(void)summon_specific(0, creature_ptr->fy, creature_ptr->fx, (floor_ptr->floor_level * 3) / 2, type, (PM_ALLOW_GROUP | PM_NO_PET));
		}
		break;
	case 36:
	case 37:
		activate_hi_summon(creature_ptr, creature_ptr->fy, creature_ptr->fx, FALSE);
		break;
	case 38:
		(void)summon_cyber(NULL, creature_ptr->fy, creature_ptr->fx);
		break;
	default:
		{
			int count = 0;
			(void)activate_ty_curse(creature_ptr, FALSE, &count);
			break;
		}
	}

	return;
}


static void cast_shuffle(creature_type *creature_ptr)
{
	floor_type *floor_ptr = get_floor_ptr(creature_ptr);
	int plev = creature_ptr->lev;
	int dir;
	int die;
	// TODO: Add Karma of Fortune feature.
	int vir = 0;
	int i;

	// Card sharks and high mages get a level bonus
	if ((creature_ptr->class_idx == CLASS_ROGUE) ||
	    (creature_ptr->class_idx == CLASS_HIGH_MAGE) ||
	    (creature_ptr->class_idx == CLASS_SORCERER))
		die = (randint1(110)) + plev / 5;
	else
		die = randint1(120);


	if (vir)
	{
		if (creature_ptr->karmas[vir - 1] > 0)
		{
			while (randint1(400) < creature_ptr->karmas[vir - 1]) die++;
		}
		else
		{
			while (randint1(400) < (0-creature_ptr->karmas[vir - 1])) die--;
		}
	}

#ifdef JP
	msg_print("Ȃ̓J[h؂Ĉꖇ...");
#else
	msg_print("You shuffle the deck and draw a card...");
#endif

	if (die < 7)
	{
#ifdef JP
		msg_print("ȂĂIstI");
#else
		msg_print("Oh no! It's Death!");
#endif

		for (i = 0; i < randint1(3); i++)
			activate_hi_summon(creature_ptr, creature_ptr->fy, creature_ptr->fx, FALSE);
	}
	else if (die < 14)
	{
#ifdef JP
		msg_print("ȂĂIstI");
#else
		msg_print("Oh no! It's the Devil!");
#endif

		summon_specific(0, creature_ptr->fy, creature_ptr->fx, floor_ptr->floor_level, SUMMON_DEMON, (PM_ALLOW_GROUP | PM_ALLOW_UNIQUE | PM_NO_PET));
	}
	else if (die < 18)
	{
		int count = 0;
#ifdef JP
		msg_print("ȂĂIs݂ꂽjtI");
#else
		msg_print("Oh no! It's the Hanged Man.");
#endif

		activate_ty_curse(creature_ptr, FALSE, &count);
	}
	else if (die < 22)
	{
#ifdef JP
		msg_print("ssǎtB");
#else
		msg_print("It's the swords of discord.");
#endif

		aggravate_creatures(creature_ptr);
	}
	else if (die < 26)
	{
#ifdef JP
		msg_print("sҁtB");
#else
		msg_print("It's the Fool.");
#endif

		do_dec_stat(creature_ptr, STAT_INT);
		do_dec_stat(creature_ptr, STAT_WIS);
	}
	else if (die < 30)
	{
#ifdef JP
		msg_print("ȃN[`[̊GB");
#else
		msg_print("It's the picture of a strange creature.");
#endif

		trump_summoning(creature_ptr, 1, FALSE, creature_ptr->fy, creature_ptr->fx, (floor_ptr->floor_level * 3 / 2), (32 + randint1(6)), PM_ALLOW_GROUP | PM_ALLOW_UNIQUE);
	}
	else if (die < 33)
	{
#ifdef JP
		msg_print("stB");
#else
		msg_print("It's the Moon.");
#endif

		unlite_area(creature_ptr, 10, 3);
	}
	else if (die < 38)
	{
#ifdef JP
		msg_print("s^̗ցtB");
#else
		msg_print("It's the Wheel of Fortune.");
#endif

		wild_magic(creature_ptr, randint0(32));
	}
	else if (die < 40)
	{
#ifdef JP
		msg_print("e|[gEJ[hB");
#else
		msg_print("It's a teleport trump card.");
#endif

		teleport_player(creature_ptr, 10, TELEPORT_PASSIVE);
	}
	else if (die < 42)
	{
#ifdef JP
		msg_print("s`tB");
#else
		msg_print("It's Justice.");
#endif

		set_blessed(creature_ptr, creature_ptr->lev, FALSE);
	}
	else if (die < 47)
	{
#ifdef JP
		msg_print("e|[gEJ[hB");
#else
		msg_print("It's a teleport trump card.");
#endif

		teleport_player(creature_ptr, 100, TELEPORT_PASSIVE);
	}
	else if (die < 52)
	{
#ifdef JP
		msg_print("e|[gEJ[hB");
#else
		msg_print("It's a teleport trump card.");
#endif

		teleport_player(creature_ptr, 200, TELEPORT_PASSIVE);
	}
	else if (die < 60)
	{
#ifdef JP
		msg_print("stB");
#else
		msg_print("It's the Tower.");
#endif

		wall_breaker(creature_ptr);
	}
	else if (die < 72)
	{
#ifdef JP
		msg_print("sߐtB");
#else
		msg_print("It's Temperance.");
#endif

		sleep_creatures_touch(creature_ptr);
	}
	else if (die < 80)
	{
#ifdef JP
		msg_print("stB");
#else
		msg_print("It's the Tower.");
#endif

		earthquake(creature_ptr, creature_ptr->fy, creature_ptr->fx, 5);
	}
	else if (die < 82)
	{
#ifdef JP
		msg_print("FDIȃN[`[̊GB");
#else
		msg_print("It's the picture of a friendly creature.");
#endif

		trump_summoning(creature_ptr, 1, TRUE, creature_ptr->fy, creature_ptr->fx, (floor_ptr->floor_level * 3 / 2), SUMMON_BIZARRE1, 0L);
	}
	else if (die < 84)
	{
#ifdef JP
		msg_print("FDIȃN[`[̊GB");
#else
		msg_print("It's the picture of a friendly creature.");
#endif

		trump_summoning(creature_ptr, 1, TRUE, creature_ptr->fy, creature_ptr->fx, (floor_ptr->floor_level * 3 / 2), SUMMON_BIZARRE2, 0L);
	}
	else if (die < 86)
	{
#ifdef JP
		msg_print("FDIȃN[`[̊GB");
#else
		msg_print("It's the picture of a friendly creature.");
#endif

		trump_summoning(creature_ptr, 1, TRUE, creature_ptr->fy, creature_ptr->fx, (floor_ptr->floor_level * 3 / 2), SUMMON_BIZARRE4, 0L);
	}
	else if (die < 88)
	{
#ifdef JP
		msg_print("FDIȃN[`[̊GB");
#else
		msg_print("It's the picture of a friendly creature.");
#endif

		trump_summoning(creature_ptr, 1, TRUE, creature_ptr->fy, creature_ptr->fx, (floor_ptr->floor_level * 3 / 2), SUMMON_BIZARRE5, 0L);
	}
	else if (die < 96)
	{
#ifdef JP
		msg_print("sltB");
#else
		msg_print("It's the Lovers.");
#endif

		if (get_aim_dir(creature_ptr, &dir))
			charm_creature(creature_ptr, dir, MIN(creature_ptr->lev, 20));
	}
	else if (die < 101)
	{
#ifdef JP
		msg_print("sBҁtB");
#else
		msg_print("It's the Hermit.");
#endif

		wall_stone(creature_ptr);
	}
	else if (die < 111)
	{
#ifdef JP
		msg_print("sRtB");
#else
		msg_print("It's the Judgement.");
#endif

		remove_all_postnatal_traits(creature_ptr);
		do_cmd_rerate(creature_ptr, FALSE);
	}
	else if (die < 120)
	{
#ifdef JP
		msg_print("sztB");
#else
		msg_print("It's the Sun.");
#endif

		wiz_lite(floor_ptr, creature_ptr, FALSE);
	}
	else
	{
#ifdef JP
		msg_print("sEtB");
#else
		msg_print("It's the World.");
#endif

		if (creature_ptr->exp < CREATURE_MAX_EXP)
		{
			s32b ee = (creature_ptr->exp / 25) + 1;
			if (ee > 5000) ee = 5000;
#ifdef JP
			msg_print("XɌoς񂾂悤ȋCB");
#else
			msg_print("You feel more experienced.");
#endif

			gain_exp(creature_ptr, ee);
		}
	}
}


// Drop 10+1d10 meteor ball at random places near the player
static void cast_meteor(creature_type *caster_ptr, int dam, int rad)
{
	floor_type *floor_ptr = get_floor_ptr(caster_ptr);
	int i;
	int b = 10 + randint1(10);

	for (i = 0; i < b; i++)
	{
		int y, x;
		int count;

		for (count = 0; count <= 20; count++)
		{
			int dy, dx, d;

			x = caster_ptr->fx - 8 + randint0(17);
			y = caster_ptr->fy - 8 + randint0(17);

			dx = (caster_ptr->fx > x) ? (caster_ptr->fx - x) : (x - caster_ptr->fx);
			dy = (caster_ptr->fy > y) ? (caster_ptr->fy - y) : (y - caster_ptr->fy);

			/* Approximate distance */
			d = (dy > dx) ? (dy + (dx >> 1)) : (dx + (dy >> 1));

			if (d >= 9) continue;

			if (!in_bounds(floor_ptr, y, x) || !projectable(floor_ptr, caster_ptr->fy, caster_ptr->fx, y, x)
			    || !cave_have_flag_bold(floor_ptr, y, x, FF_PROJECT)) continue;

			/* Valid position */
			break;
		}

		if (count > 20) continue;

		project(caster_ptr, rad, y, x, dam, GF_METEOR, PROJECT_KILL | PROJECT_JUMP | PROJECT_ITEM, -1);
	}
}


// Drop 10+1d10 disintegration ball at random places near the target
static bool cast_wrath_of_the_god(creature_type *creature_ptr, int dam, int rad)
{
	floor_type *floor_ptr = get_floor_ptr(creature_ptr);
	int x, y, tx, ty;
	int nx, ny;
	int dir, i;
	int b = 10 + randint1(10);

	if (!get_aim_dir(creature_ptr, &dir)) return FALSE;

	/* Use the given direction */
	tx = creature_ptr->fx + 99 * ddx[dir];
	ty = creature_ptr->fy + 99 * ddy[dir];

	/* Hack -- Use an actual "target" */
	if ((dir == 5) && target_okay(creature_ptr))
	{
		tx = target_col;
		ty = target_row;
	}

	x = creature_ptr->fx;
	y = creature_ptr->fy;

	while (1)
	{
		/* Hack -- Stop at the target */
		if ((y == ty) && (x == tx)) break;

		ny = y;
		nx = x;
		mmove2(&ny, &nx, creature_ptr->fy, creature_ptr->fx, ty, tx);

		/* Stop at maximum range */
		if (MAX_RANGE <= distance(creature_ptr->fy, creature_ptr->fx, ny, nx)) break;

		/* Stopped by walls/doors */
		if (!cave_have_flag_bold(floor_ptr, ny, nx, FF_PROJECT)) break;

		/* Stopped by creatures */
		if ((dir != 5) && floor_ptr->cave[ny][nx].creature_idx != 0) break;

		/* Save the new location */
		x = nx;
		y = ny;
	}
	tx = x;
	ty = y;

	for (i = 0; i < b; i++)
	{
		int count = 20, d = 0;

		while (count--)
		{
			int dx, dy;

			x = tx - 5 + randint0(11);
			y = ty - 5 + randint0(11);

			dx = (tx > x) ? (tx - x) : (x - tx);
			dy = (ty > y) ? (ty - y) : (y - ty);

			/* Approximate distance */
			d = (dy > dx) ? (dy + (dx >> 1)) : (dx + (dy >> 1));
			/* Within the radius */
			if (d < 5) break;
		}

		if (count < 0) continue;

		/* Cannot penetrate perm walls */
		if (!in_bounds(floor_ptr, y, x) ||
		    cave_stop_disintegration(floor_ptr, y, x) ||
		    !in_disintegration_range(floor_ptr, ty, tx, y, x))
			continue;

		project(creature_ptr, rad, y, x, dam, GF_DISINTEGRATE, PROJECT_JUMP | PROJECT_GRID | PROJECT_ITEM | PROJECT_KILL, -1);
	}

	return TRUE;
}


/*
 * An "item_tester_hook" for offer
 */
static bool item_tester_offer(creature_type *creature_ptr, object_type *object_ptr)
{
	/* Flasks of oil are okay */
	if (object_ptr->tval != TV_CORPSE) return (FALSE);

	if (object_ptr->sval != SV_CORPSE) return (FALSE);

	if (my_strchr("pht", species_info[object_ptr->pval].d_char)) return (TRUE);

	/* Assume not okay */
	return (FALSE);
}


/*
 * Daemon spell Summon Greater Demon
 */
static bool cast_summon_greater_demon(creature_type *creature_ptr)
{
	int plev = creature_ptr->lev;
	int item;
	cptr q, s;
	int summon_lev;
	object_type *object_ptr;

#ifdef JP
	q = "ǂ̎̂܂? ";
	s = "鎀̂ĂȂB";
#else
	q = "Sacrifice which corpse? ";
	s = "You have nothing to scrifice.";
#endif
	if (!get_item(creature_ptr, &item, q, s, (USE_INVEN | USE_FLOOR), item_tester_offer, 0)) return FALSE;

	/* Get the item (in the pack) */
	if (item >= 0)
	{
		object_ptr = &creature_ptr->inventory[item];
	}

	/* Get the item (on the floor) */
	else
	{
		object_ptr = &object_list[0 - item];
	}

	summon_lev = plev * 2 / 3 + species_info[object_ptr->pval].level;

	if (summon_specific(NULL, creature_ptr->fy, creature_ptr->fx, summon_lev, SUMMON_HI_DEMON, (PM_ALLOW_GROUP | PM_FORCE_PET)))
	{
#ifdef JP
		msg_print("̈L[B");
#else
		msg_print("The area fills with a stench of sulphur and brimstone.");
#endif


#ifdef JP
		msg_print("upł܂Allv");
#else
		msg_print("'What is thy bidding... Master?'");
#endif

		/* Decrease the item (from the pack) */
		if (item >= 0)
		{
			inven_item_increase(creature_ptr, item, -1);
			inven_item_describe(creature_ptr, item);
			inven_item_optimize(creature_ptr, item);
		}

		/* Decrease the item (from the floor) */
		else
		{
			floor_item_increase(0 - item, -1);
			floor_item_describe(creature_ptr, 0 - item);
			floor_item_optimize(0 - item);
		}
	}
	else
	{
#ifdef JP
		msg_print("͌ȂB");
#else
		msg_print("No Greater Demon arrive.");
#endif
	}

	return TRUE;
}


/*
 * Start singing if the player is a Bard 
 */
static void start_singing(creature_type *creature_ptr, int spell, int song)
{
	/* Remember the song index */
	creature_ptr->class_skills.old_skills.magic_num1[0] = song;

	/* Remember the index of the spell which activated the song */
	creature_ptr->class_skills.old_skills.magic_num2[0] = spell;


	/* Now the player is singing */
	set_action(creature_ptr, ACTION_SING);


	/* Recalculate bonuses */
	creature_ptr->creature_update |= (CRU_BONUS);

	/* Redraw status bar */
	play_redraw |= (PR_STATUS);
}


/*
 * Stop singing if the player is a Bard 
 */
void stop_singing(creature_type *creature_ptr)
{
	if (creature_ptr->class_idx != CLASS_BARD) return;

 	/* Are there interupted song? */
	if (creature_ptr->class_skills.old_skills.magic_num1[1])
	{
		/* Forget interupted song */
		creature_ptr->class_skills.old_skills.magic_num1[1] = 0;
		return;
	}

	/* The player is singing? */
	if (!creature_ptr->class_skills.old_skills.magic_num1[0]) return;

	/* Hack -- if called from set_action(creature_ptr, ), avoid recursive loop */
	if (creature_ptr->action == ACTION_SING) set_action(creature_ptr, ACTION_NONE);

	/* Message text of each song or etc. */
	do_spell(creature_ptr, REALM_MUSIC, creature_ptr->class_skills.old_skills.magic_num2[0], SPELL_STOP);

	creature_ptr->class_skills.old_skills.magic_num1[0] = MUSIC_NONE;
	creature_ptr->class_skills.old_skills.magic_num2[0] = 0;

	/* Recalculate bonuses */
	creature_ptr->creature_update |= (CRU_BONUS);

	/* Redraw status bar */
	play_redraw |= (PR_STATUS);
}


static cptr do_life_spell(creature_type *creature_ptr, int spell, int mode)
{
	floor_type *floor_ptr = get_floor_ptr(creature_ptr);

	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool info = (mode == SPELL_INFO) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;

	int dir;
	int plev = creature_ptr->lev;

	switch (spell)
	{
	case 0:
#ifdef JP
		if (name) return "y̎";
		if (desc) return "Ƒ̗͂񕜂B";
#else
		if (name) return "Cure Light Wounds";
		if (desc) return "Heals cut and HP a little.";
#endif
    
		{
			int dice = 2;
			int sides = 10;

			if (info) return info_heal(dice, sides, 0);

			if (cast)
			{
				hp_player(creature_ptr, diceroll(dice, sides));
				set_cut(creature_ptr, creature_ptr->cut - 10);
			}
		}
		break;

	case 1:
#ifdef JP
		if (name) return "j";
		if (desc) return "莞ԁAACɃ{[iX𓾂B";
#else
		if (name) return "Bless";
		if (desc) return "Gives bonus to hit and AC for a few turns.";
#endif
    
		{
			int base = 12;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_blessed(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 2:
#ifdef JP
		if (name) return "y";
		if (desc) return "1̂̃N[`[ɏ_[W^BRƖB";
#else
		if (name) return "Cause Light Wounds";
		if (desc) return "Wounds a creature a little unless resisted.";
#endif
    
		{
			int dice = 3 + (plev - 1) / 5;
			int sides = 4;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;
				fire_ball_hide(creature_ptr, GF_WOUNDS, dir, diceroll(dice, sides), 0);
			}
		}
		break;

	case 3:
#ifdef JP
		if (name) return "̏";
		if (desc) return "Ƃ炵Ă͈͂Ŝivɖ邭B";
#else
		if (name) return "Call Light";
		if (desc) return "Lights up nearby area and the inside of a room permanently.";
#endif
    
		{
			int dice = 2;
			int sides = plev / 2;
			int rad = plev / 10 + 1;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				lite_area(creature_ptr, diceroll(dice, sides), rad);
			}
		}
		break;

	case 4:
#ifdef JP
		if (name) return " & Bm";
		if (desc) return "߂̑SĂ㩂ƔƊKimB";
#else
		if (name) return "Detect Doors & Traps";
		if (desc) return "Detects traps, doors, and stairs in your vicinity.";
#endif
    
		{
			int rad = DETECT_RAD_DEFAULT;

			if (info) return info_radius(rad);

			if (cast)
			{
				detect_traps(creature_ptr, rad, TRUE);
				detect_doors(creature_ptr, rad);
				detect_stairs(creature_ptr, rad);
			}
		}
		break;

	case 5:
#ifdef JP
		if (name) return "d̎";
		if (desc) return "Ƒ̗͂𒆒x񕜂B";
#else
		if (name) return "Cure Medium Wounds";
		if (desc) return "Heals cut and HP more.";
#endif
    
		{
			int dice = 4;
			int sides = 10;

			if (info) return info_heal(dice, sides, 0);

			if (cast)
			{
				hp_player(creature_ptr, diceroll(dice, sides));
				set_cut(creature_ptr, (creature_ptr->cut / 2) - 20);
			}
		}
		break;

	case 6:
#ifdef JP
		if (name) return "";
		if (desc) return "̓̓ł菜B";
#else
		if (name) return "Cure Poison";
		if (desc) return "Cure poison status.";
#endif
    
		{
			if (cast)
			{
				set_poisoned(creature_ptr, 0);
			}
		}
		break;

	case 7:
#ifdef JP
		if (name) return "󕠏[";
		if (desc) return "ɂB";
#else
		if (name) return "Satisfy Hunger";
		if (desc) return "Satisfies hunger.";
#endif
    
		{
			if (cast)
			{
				set_food(creature_ptr, PY_FOOD_MAX - 1);
			}
		}
		break;

	case 8:
#ifdef JP
		if (name) return "";
		if (desc) return "ACeɂア􂢂B";
#else
		if (name) return "Remove Curse";
		if (desc) return "Removes normal curses from equipped items.";
#endif

		{
			if (cast)
			{
				if (remove_curse(creature_ptr))
				{
#ifdef JP
					msg_print("NɌĂ悤ȋCB");
#else
					msg_print("You feel as if someone is watching over you.");
#endif
				}
			}
		}
		break;

	case 9:
#ifdef JP
		if (name) return "d";
		if (desc) return "1̂̃N[`[ɒ_[W^BRƖB";
#else
		if (name) return "Cause Medium Wounds";
		if (desc) return "Wounds a creature unless resisted.";
#endif
    
		{
			int sides = 8 + (plev - 5) / 4;
			int dice = 8;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;
				fire_ball_hide(creature_ptr, GF_WOUNDS, dir, diceroll(sides, dice), 0);
			}
		}
		break;

	case 10:
#ifdef JP
		if (name) return "v̎";
		if (desc) return "̗͂啝ɉ񕜂AƞNOԂSB";
#else
		if (name) return "Cure Critical Wounds";
		if (desc) return "Heals cut, stun and HP greatly.";
#endif
    
		{
			int dice = 8;
			int sides = 10;

			if (info) return info_heal(dice, sides, 0);

			if (cast)
			{
				hp_player(creature_ptr, diceroll(dice, sides));
				set_stun(creature_ptr, 0);
				set_cut(creature_ptr, 0);
			}
		}
		break;

	case 11:
#ifdef JP
		if (name) return "ϔMϊ";
		if (desc) return "莞ԁAΉƗCɑ΂ϐ𓾂BɂϐɗݐςB";
#else
		if (name) return "Resist Heat and Cold";
		if (desc) return "Gives resistance to fire and cold. These resistances can be added to which from equipment for more powerful resistances.";
#endif
    
		{
			int base = 20;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_oppose_cold(creature_ptr, randint1(base) + base, FALSE);
				set_oppose_fire(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 12:
#ifdef JP
		if (name) return "ӊm";
		if (desc) return "ӂ̒n`mB";
#else
		if (name) return "Sense Surroundings";
		if (desc) return "Maps nearby area.";
#endif
    
		{
			int rad = DETECT_RAD_MAP;

			if (info) return info_radius(rad);

			if (cast)
			{
				map_area(creature_ptr, rad);
			}
		}
		break;

	case 13:
#ifdef JP
		if (name) return "pjbNEAfbh";
		if (desc) return "ẼAfbh|BRƖB";
#else
		if (name) return "Turn Undead";
		if (desc) return "Attempts to scare undead creatures in sight.";
#endif
    
		{
			if (cast)
			{
				turn_undead(creature_ptr);
			}
		}
		break;

	case 14:
#ifdef JP
		if (name) return "̗͉";
		if (desc) return "ɂ߂ċ͂ȉ񕜎ŁAƞNOԂSB";
#else
		if (name) return "Healing";
		if (desc) return "Much powerful healing magic, and heals cut and stun completely.";
#endif
    
		{
			int heal = 300;

			if (info) return info_heal(0, 0, heal);

			if (cast)
			{
				hp_player(creature_ptr, heal);
				set_stun(creature_ptr, 0);
				set_cut(creature_ptr, 0);
			}
		}
		break;

	case 15:
#ifdef JP
		if (name) return "E̖";
		if (desc) return "̂鏰̏ɁAN[`[ʂ蔲菢ꂽ肷邱ƂłȂȂ郋[`B";
#else
		if (name) return "Glyph of Warding";
		if (desc) return "Sets a glyph on the floor beneath you. Creatures cannot attack you if you are on a glyph, but can try to break glyph.";
#endif
    
		{
			if (cast)
			{
				warding_glyph(creature_ptr);
			}
		}
		break;

	case 16:
#ifdef JP
		if (name) return "**";
		if (desc) return "ACeɂ͂Ȏ􂢂B";
#else
		if (name) return "Dispel Curse";
		if (desc) return "Removes normal and heavy curse from equipped items.";
#endif
    
		{
			if (cast)
			{
				if (remove_all_curse(creature_ptr))
				{
#ifdef JP
					msg_print("NɌĂ悤ȋCB");
#else
					msg_print("You feel as if someone is watching over you.");
#endif
				}
			}
		}
		break;

	case 17:
#ifdef JP
		if (name) return "ӎ";
		if (desc) return "ACeʂB";
#else
		if (name) return "Perception";
		if (desc) return "Identifies an item.";
#endif
    
		{
			if (cast)
			{
				if (!ident_spell(creature_ptr, FALSE)) return NULL;
			}
		}
		break;

	case 18:
#ifdef JP
		if (name) return "AfbhގU";
		if (desc) return "ȆSẴAfbhɃ_[W^B";
#else
		if (name) return "Dispel Undead";
		if (desc) return "Damages all undead creatures in sight.";
#endif
    
		{
			int dice = 1;
			int sides = plev * 5;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				dispel_undead(creature_ptr, diceroll(dice, sides));
			}
		}
		break;

	case 19:
#ifdef JP
		if (name) return "̍";
		if (desc) return "ȆSẴN[`[𖣗BRƖB";
#else
		if (name) return "Day of the Dove";
		if (desc) return "Attempts to charm all creatures in sight.";
#endif
    
		{
			int power = plev * 2;

			if (info) return info_power(power);

			if (cast)
			{
				charm_creatures(creature_ptr, power);
			}
		}
		break;

	case 20:
#ifdef JP
		if (name) return "v";
		if (desc) return "1̂̃N[`[ɑ_[W^BRƖB";
#else
		if (name) return "Cause Critical Wounds";
		if (desc) return "Wounds a creature critically unless resisted.";
#endif
    
		{
			int dice = 5 + (plev - 5) / 3;
			int sides = 15;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;
				fire_ball_hide(creature_ptr, GF_WOUNDS, dir, diceroll(dice, sides), 0);
			}
		}
		break;

	case 21:
#ifdef JP
		if (name) return "A҂̏";
		if (desc) return "nɂƂ̓_W̍Ő[KցA_WɂƂ͒nւƈړB";
#else
		if (name) return "Word of Recall";
		if (desc) return "Recalls player from dungeon to town, or from town to the deepest level of dungeon.";
#endif
    
		{
			int base = 15;
			int sides = 20;

			if (info) return info_delay(base, sides);

			if (cast)
			{
				if (!word_of_recall(creature_ptr)) return NULL;
			}
		}
		break;

	case 22:
#ifdef JP
		if (name) return "^̍Ւd";
		if (desc) return "݂̊Kč\B";
#else
		if (name) return "Alter Reality";
		if (desc) return "Recreates current dungeon level.";
#endif
    
		{
			int base = 15;
			int sides = 20;

			if (info) return info_delay(base, sides);

			if (cast)
			{
				alter_reality(creature_ptr);
			}
		}
		break;

	case 23:
#ifdef JP
		if (name) return "^EE";
		if (desc) return "̂鏰Ǝ8}X̏̏ɁAN[`[ʂ蔲菢ꂽ肷邱ƂłȂȂ郋[`B";
#else
		if (name) return "Warding True";
		if (desc) return "Creates glyphs in all adjacent squares and under you.";
#endif
    
		{
			int rad = 1;

			if (info) return info_radius(rad);

			if (cast)
			{
				warding_glyph(creature_ptr);
				glyph_creation(creature_ptr);
			}
		}
		break;

	case 24:
#ifdef JP
		if (name) return "sщ";
		if (desc) return "̊K̑BN[`[BłȂȂB";
#else
		if (name) return "Sterilization";
		if (desc) return "Prevents any breeders on current level from breeding.";
#endif
    
		{
			if (cast)
			{
				floor_ptr->num_repro += MAX_REPRO;
			}
		}
		break;

	case 25:
#ifdef JP
		if (name) return "Sm";
		if (desc) return "߂̑SẴN[`[A㩁AAKiAAăACemB";
#else
		if (name) return "Detection";
		if (desc) return "Detects all creatures, traps, doors, stairs, treasures and items in your vicinity.";
#endif

		{
			int rad = DETECT_RAD_DEFAULT;

			if (info) return info_radius(rad);

			if (cast)
			{
				detect_all(creature_ptr, rad);
			}
		}
		break;

	case 26:
#ifdef JP
		if (name) return "Afbh";
		if (desc) return "̎͂ɂAfbh݂̊KBRƖB";
#else
		if (name) return "Annihilate Undead";
		if (desc) return "Eliminates all nearby undead creatures, exhausting you.  Powerful or unique creatures may be able to resist.";
#endif
    
		{
			int power = plev + 50;

			if (info) return info_power(power);

			if (cast)
			{
				mass_genocide_undead(creature_ptr, power, TRUE);
			}
		}
		break;

	case 27:
#ifdef JP
		if (name) return "痢";
		if (desc) return "̊KŜivɏƂ炵A_WׂẴACemB";
#else
		if (name) return "Clairvoyance";
		if (desc) return "Maps and lights whole dungeon level. Knows all objects location. And gives telepathy for a while.";
#endif
    
		{
			if (cast)
			{
				wiz_lite(floor_ptr, creature_ptr, FALSE);
			}
		}
		break;

	case 28:
#ifdef JP
		if (name) return "S";
		if (desc) return "ׂẴXe[^Xƌol񕜂B";
#else
		if (name) return "Restoration";
		if (desc) return "Restores all stats and experience.";
#endif
    
		{
			if (cast)
			{
				do_res_stat(creature_ptr, STAT_STR);
				do_res_stat(creature_ptr, STAT_INT);
				do_res_stat(creature_ptr, STAT_WIS);
				do_res_stat(creature_ptr, STAT_DEX);
				do_res_stat(creature_ptr, STAT_CON);
				do_res_stat(creature_ptr, STAT_CHA);
				restore_level(creature_ptr);
			}
		}
		break;

	case 29:
#ifdef JP
		if (name) return "*̗͉*";
		if (desc) return "ŋ̖̎@ŁAƞNOԂSB";
#else
		if (name) return "Healing True";
		if (desc) return "The greatest healing magic. Heals all HP, cut and stun.";
#endif
    
		{
			int heal = 2000;

			if (info) return info_heal(0, 0, heal);

			if (cast)
			{
				hp_player(creature_ptr, heal);
				set_stun(creature_ptr, 0);
				set_cut(creature_ptr, 0);
			}
		}
		break;

	case 30:
#ifdef JP
		if (name) return "ȂrW";
		if (desc) return "ACe̎\͂SɒmB";
#else
		if (name) return "Holy Vision";
		if (desc) return "*Identifies* an item.";
#endif
    
		{
			if (cast)
			{
				if (!identify_fully(creature_ptr, FALSE)) return NULL;
			}
		}
		break;

	case 31:
#ifdef JP
		if (name) return "ɂ̑ϐ";
		if (desc) return "莞ԁAϐtAACƖ@h\͂㏸B";
#else
		if (name) return "Ultimate Resistance";
		if (desc) return "Gives ultimate resistance, bonus to AC and speed.";
#endif
    
		{
			int base = plev / 2;

			if (info) return info_duration(base, base);

			if (cast)
			{
				int v = randint1(base) + base;
				set_fast(creature_ptr, v, FALSE);
				set_oppose_acid(creature_ptr, v, FALSE);
				set_oppose_elec(creature_ptr, v, FALSE);
				set_oppose_fire(creature_ptr, v, FALSE);
				set_oppose_cold(creature_ptr, v, FALSE);
				set_oppose_pois(creature_ptr, v, FALSE);
				set_ultimate_res(creature_ptr, v, FALSE);
			}
		}
		break;
	}

	return "";
}


static cptr do_sorcery_spell(creature_type *creature_ptr, int spell, int mode)
{
	floor_type *floor_ptr = get_floor_ptr(creature_ptr);
	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool info = (mode == SPELL_INFO) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;

	int dir;
	int plev = creature_ptr->lev;

	switch (spell)
	{
	case 0:
#ifdef JP
		if (name) return "N[`[m";
		if (desc) return "߂̑SĂ̌N[`[mB";
#else
		if (name) return "Detect Creatures";
		if (desc) return "Detects all creatures in your vicinity unless invisible.";
#endif
    
		{
			int rad = DETECT_RAD_DEFAULT;

			if (info) return info_radius(rad);

			if (cast)
			{
				detect_creatures_normal(creature_ptr, rad);
			}
		}
		break;

	case 1:
#ifdef JP
		if (name) return "V[gEe|[g";
		if (desc) return "ߋ̃e|[gB";
#else
		if (name) return "Phase Door";
		if (desc) return "Teleport short distance.";
#endif
    
		{
			int range = 10;

			if (info) return info_range(range);

			if (cast)
			{
				teleport_player(creature_ptr, range, 0L);
			}
		}
		break;

	case 2:
#ifdef JP
		if (name) return "㩂Ɣm";
		if (desc) return "߂̑SĂ̔㩂mB";
#else
		if (name) return "Detect Doors and Traps";
		if (desc) return "Detects traps, doors, and stairs in your vicinity.";
#endif
    
		{
			int rad = DETECT_RAD_DEFAULT;

			if (info) return info_radius(rad);

			if (cast)
			{
				detect_traps(creature_ptr, rad, TRUE);
				detect_doors(creature_ptr, rad);
				detect_stairs(creature_ptr, rad);
			}
		}
		break;

	case 3:
#ifdef JP
		if (name) return "CgEGA";
		if (desc) return "Ƃ炵Ă͈͂Ŝivɖ邭B";
#else
		if (name) return "Light Area";
		if (desc) return "Lights up nearby area and the inside of a room permanently.";
#endif
    
		{
			int dice = 2;
			int sides = plev / 2;
			int rad = plev / 10 + 1;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				lite_area(creature_ptr, diceroll(dice, sides), rad);
			}
		}
		break;

	case 4:
#ifdef JP
		if (name) return "pjbNEN[`[";
		if (desc) return "N[`[1̂BRƖB";
#else
		if (name) return "Confuse Creature";
		if (desc) return "Attempts to confuse a creature.";
#endif
    
		{
			int power = (plev * 3) / 2;

			if (info) return info_power(power);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				confuse_creature(creature_ptr, dir, power);
			}
		}
		break;

	case 5:
#ifdef JP
		if (name) return "e|[g";
		if (desc) return "̃e|[gB";
#else
		if (name) return "Teleport";
		if (desc) return "Teleport long distance.";
#endif
    
		{
			int range = plev * 5;

			if (info) return info_range(range);

			if (cast)
			{
				teleport_player(creature_ptr, range, 0L);
			}
		}
		break;

	case 6:
#ifdef JP
		if (name) return "X[vEN[`[";
		if (desc) return "N[`[1̂𖰂点BRƖB";
#else
		if (name) return "Sleep Creature";
		if (desc) return "Attempts to sleep a creature.";
#endif
    
		{
			int power = plev;

			if (info) return info_power(power);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				sleep_creature(creature_ptr, dir);
			}
		}
		break;

	case 7:
#ifdef JP
		if (name) return "͏[U";
		if (desc) return "/@_̏[U񐔂𑝂₷A[Ũbh̏[UԂ炷B";
#else
		if (name) return "Recharging";
		if (desc) return "Recharges staffs, wands or rods.";
#endif
    
		{
			int power = plev * 4;

			if (info) return info_power(power);

			if (cast)
			{
				if (!recharge(creature_ptr, power)) return NULL;
			}
		}
		break;

	case 8:
#ifdef JP
		if (name) return "@̒n}";
		if (desc) return "ӂ̒n`mB";
#else
		if (name) return "Magic Mapping";
		if (desc) return "Maps nearby area.";
#endif
    
		{
			int rad = DETECT_RAD_MAP;

			if (info) return info_radius(rad);

			if (cast)
			{
				map_area(creature_ptr, rad);
			}
		}
		break;

	case 9:
#ifdef JP
		if (name) return "Ӓ";
		if (desc) return "ACeʂB";
#else
		if (name) return "Identify";
		if (desc) return "Identifies an item.";
#endif
    
		{
			if (cast)
			{
				if (!ident_spell(creature_ptr, FALSE)) return NULL;
			}
		}
		break;

	case 10:
#ifdef JP
		if (name) return "XEEN[`[";
		if (desc) return "N[`[1̂BRƖB";
#else
		if (name) return "Slow Creature";
		if (desc) return "Attempts to slow a creature.";
#endif
    
		{
			int power = plev;

			if (info) return info_power(power);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				slow_creature(creature_ptr, dir);
			}
		}
		break;

	case 11:
#ifdef JP
		if (name) return "ӃX[v";
		if (desc) return "ȆSẴN[`[𖰂点BRƖB";
#else
		if (name) return "Mass Sleep";
		if (desc) return "Attempts to sleep all creatures in sight.";
#endif
    
		{
			int power = plev;

			if (info) return info_power(power);

			if (cast)
			{
				sleep_creatures(creature_ptr);
			}
		}
		break;

	case 12:
#ifdef JP
		if (name) return "e|[gEN[`[";
		if (desc) return "N[`[e|[gr[BRƖB";
#else
		if (name) return "Teleport Away";
		if (desc) return "Teleports all creatures on the line away unless resisted.";
#endif
    
		{
			int power = plev;

			if (info) return info_power(power);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_beam(creature_ptr, GF_AWAY_ALL, dir, power);
			}
		}
		break;

	case 13:
#ifdef JP
		if (name) return "Xs[h";
		if (desc) return "莞ԁAB";
#else
		if (name) return "Haste Self";
		if (desc) return "Hastes you for a while.";
#endif
    
		{
			int base = plev;
			int sides = 20 + plev;

			if (info) return info_duration(base, sides);

			if (cast)
			{
				set_fast(creature_ptr, randint1(sides) + base, FALSE);
			}
		}
		break;

	case 14:
#ifdef JP
		if (name) return "^Em";
		if (desc) return "߂̑SẴN[`[A㩁AAKiAAăACemB";
#else
		if (name) return "Detection True";
		if (desc) return "Detects all creatures, traps, doors, stairs, treasures and items in your vicinity.";
#endif
    
		{
			int rad = DETECT_RAD_DEFAULT;

			if (info) return info_radius(rad);

			if (cast)
			{
				detect_all(creature_ptr, rad);
			}
		}
		break;

	case 15:
#ifdef JP
		if (name) return "^EӒ";
		if (desc) return "ACe̎\͂SɒmB";
#else
		if (name) return "Identify True";
		if (desc) return "*Identifies* an item.";
#endif
    
		{
			if (cast)
			{
				if (!identify_fully(creature_ptr, FALSE)) return NULL;
			}
		}
		break;

	case 16:
#ifdef JP
		if (name) return "̂ƍ󊴒m";
		if (desc) return "߂̑SẴACeƍmB";
#else
		if (name) return "Detect items and Treasure";
		if (desc) return "Detects all treasures and items in your vicinity.";
#endif
    
		{
			int rad = DETECT_RAD_DEFAULT;

			if (info) return info_radius(rad);

			if (cast)
			{
				detect_objects_normal(creature_ptr, rad);
				detect_treasure(creature_ptr, rad);
				detect_objects_gold(creature_ptr, rad);
			}
		}
		break;

	case 17:
#ifdef JP
		if (name) return "`[EN[`[";
		if (desc) return "N[`[1̂𖣗BRƖB";
#else
		if (name) return "Charm Creature";
		if (desc) return "Attempts to charm a creature.";
#endif
    
		{
			int power = plev;

			if (info) return info_power(power);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				charm_creature(creature_ptr, dir, power);
			}
		}
		break;

	case 18:
#ifdef JP
		if (name) return "_m";
		if (desc) return "莞ԁAepV[\͂𓾂B";
#else
		if (name) return "Sense Minds";
		if (desc) return "Gives telepathy for a while.";
#endif
    
		{
			int base = 25;
			int sides = 30;

			if (info) return info_duration(base, sides);

			if (cast)
			{
				set_tim_esp(creature_ptr, randint1(sides) + base, FALSE);
			}
		}
		break;

	case 19:
#ifdef JP
		if (name) return "Xړ";
		if (desc) return "XֈړBnɂƂgȂB";
#else
		if (name) return "Teleport to town";
		if (desc) return "Teleport to a town which you choose in a moment. Can only be used outdoors.";
#endif
    
		{
			if (cast)
			{
				if (!tele_town(creature_ptr)) return NULL;
			}
		}
		break;

	case 20:
#ifdef JP
		if (name) return "ȕ";
		if (desc) return "݂̎̏ԂSɒmB";
#else
		if (name) return "Self Knowledge";
		if (desc) return "Gives you useful info regarding your current resistances, the powers of your weapon and maximum limits of your stats.";
#endif
    
		{
			if (cast)
			{
				creature_knowledge(creature_ptr);
			}
		}
		break;

	case 21:
#ifdef JP
		if (name) return "e|[gEx";
		if (desc) return "uɏォ̊KɃe|[gB";
#else
		if (name) return "Teleport Level";
		if (desc) return "Teleport to up or down stairs in a moment.";
#endif
    
		{
			if (cast)
			{
#ifdef JP
				if (!get_check("{ɑ̊KɃe|[g܂H")) return NULL;
#else
				if (!get_check("Are you sure? (Teleport Level)")) return NULL;
#endif
				teleport_level(creature_ptr, 0);
			}
		}
		break;

	case 22:
#ifdef JP
		if (name) return "A҂̎";
		if (desc) return "nɂƂ̓_W̍Ő[KցA_WɂƂ͒nւƈړB";
#else
		if (name) return "Word of Recall";
		if (desc) return "Recalls player from dungeon to town, or from town to the deepest level of dungeon.";
#endif
    
		{
			int base = 15;
			int sides = 20;

			if (info) return info_delay(base, sides);

			if (cast)
			{
				if (!word_of_recall(creature_ptr)) return NULL;
			}
		}
		break;

	case 23:
#ifdef JP
		if (name) return "̔";
		if (desc) return "Z̎w肵ꏊɃe|[gB";
#else
		if (name) return "Dimension Door";
		if (desc) return "Teleport to given location.";
#endif
    
		{
			int range = plev / 2 + 10;

			if (info) return info_range(range);

			if (cast)
			{
#ifdef JP
				msg_print("̔JBړInIŉB");
#else
				msg_print("You open a dimensional gate. Choose a destination.");
#endif

				if (!dimension_door(creature_ptr)) return NULL;
			}
		}
		break;

	case 24:
#ifdef JP
		if (name) return "";
		if (desc) return "N[`[̑Ać̗Aő̗́AXs[hÂmB";
#else
		if (name) return "Probing";
		if (desc) return "Proves all creatures' alignment, HP, speed and their true character.";
#endif
    
		{
			if (cast)
			{
				probing(get_floor_ptr(creature_ptr));
			}
		}
		break;

	case 25:
#ifdef JP
		if (name) return "̃[";
		if (desc) return "̂鏰̏ɁAN[`[ʂƔă_[W^郋[`B";
#else
		if (name) return "Explosive Rune";
		if (desc) return "Sets a glyph under you. The glyph will explode when a creature moves on it.";
#endif
    
		{
			int dice = 7;
			int sides = 7;
			int base = plev;

			if (info) return info_damage(dice, sides, base);

			if (cast)
			{
				explosive_rune(creature_ptr);
			}
		}
		break;

	case 26:
#ifdef JP
		if (name) return "O";
		if (desc) return "ACȇֈړB";
#else
		if (name) return "Telekinesis";
		if (desc) return "Pulls a distant item close to you.";
#endif
    
		{
			int weight = plev * 15;

			if (info) return info_weight(weight);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fetch(creature_ptr, dir, weight, FALSE);
			}
		}
		break;

	case 27:
#ifdef JP
		if (name) return "痢";
		if (desc) return "̊KŜivɏƂ炵A_WׂẴACemBɁA莞ԃepV[\͂𓾂B";
#else
		if (name) return "Clairvoyance";
		if (desc) return "Maps and lights whole dungeon level. Knows all objects location. And gives telepathy for a while.";
#endif
    
		{
			int base = 25;
			int sides = 30;

			if (info) return info_duration(base, sides);

			if (cast)
			{

				wiz_lite(floor_ptr, creature_ptr, FALSE);

				if (!has_trait(creature_ptr, TRAIT_ESP))
				{
					set_tim_esp(creature_ptr, randint1(sides) + base, FALSE);
				}
			}
		}
		break;

	case 28:
#ifdef JP
		if (name) return "̎";
		if (desc) return "ȆSẴN[`[𖣗BRƖB";
#else
		if (name) return "Charm creatures";
		if (desc) return "Attempts to charm all creatures in sight.";
#endif
    
		{
			int power = plev * 2;

			if (info) return info_power(power);

			if (cast)
			{
				charm_creatures(creature_ptr, power);
			}
		}
		break;

	case 29:
#ifdef JP
		if (name) return "Bp";
		if (desc) return "ACe1ɕςB";
#else
		if (name) return "Alchemy";
		if (desc) return "Turns an item into 1/3 of its value in gold.";
#endif
    
		{
			if (cast)
			{
				if (!alchemy(creature_ptr)) return NULL;
			}
		}
		break;

	case 30:
#ifdef JP
		if (name) return "Ǖ";
		if (desc) return "ȆSẴN[`[e|[gBRƖB";
#else
		if (name) return "Banishment";
		if (desc) return "Teleports all creatures in sight away unless resisted.";
#endif
    
		{
			int power = plev * 4;

			if (info) return info_power(power);

			if (cast)
			{
				banish_creatures(creature_ptr, power);
			}
		}
		break;

	case 31:
#ifdef JP
		if (name) return "̋";
		if (desc) return "莞ԁA_[W󂯂ȂȂoA𒣂B؂ꂽuԂɏ^[̂ŒӁB";
#else
		if (name) return "Globe of Invulnerability";
		if (desc) return "Generates barrier which completely protect you from almost all damages. Takes a few your turns when the barrier breaks or duration time is exceeded.";
#endif
    
		{
			int base = 4;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_invuln(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;
	}

	return "";
}


static cptr do_nature_spell(creature_type *caster_ptr, int spell, int mode)
{
	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool info = (mode == SPELL_INFO) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;

	floor_type *floor_ptr = get_floor_ptr(caster_ptr);

#ifdef JP
	static const char s_dam[] = ":";
	static const char s_rng[] = "˒";
#else
	static const char s_dam[] = "dam ";
	static const char s_rng[] = "rng ";
#endif

	int dir;
	int plev = caster_ptr->lev;

	switch (spell)
	{
	case 0:
#ifdef JP
		if (name) return "N[`[m";
		if (desc) return "߂̑SĂ̌N[`[mB";
#else
		if (name) return "Detect Creatures";
		if (desc) return "Detects all creatures in your vicinity unless invisible.";
#endif
    
		{
			int rad = DETECT_RAD_DEFAULT;

			if (info) return info_radius(rad);

			if (cast)
			{
				detect_creatures_normal(caster_ptr, rad);
			}
		}
		break;

	case 1:
#ifdef JP
		if (name) return "";
		if (desc) return "d̒Zr[B";
#else
		if (name) return "Lightning";
		if (desc) return "Fires a short beam of lightning.";
#endif
    
		{
			int dice = 3 + (plev - 1) / 5;
			int sides = 4;
			int range = plev / 6 + 2;

			if (info) return format("%s%dd%d %s%d", s_dam, dice, sides, s_rng, range);

			if (cast)
			{
				project_length = range;

				if (!get_aim_dir(caster_ptr, &dir)) return NULL;

				fire_beam(caster_ptr, GF_ELEC, dir, diceroll(dice, sides));
			}
		}
		break;

	case 2:
#ifdef JP
		if (name) return "㩂Ɣm";
		if (desc) return "߂̑SĂ㩂ƔmB";
#else
		if (name) return "Detect Doors and Traps";
		if (desc) return "Detects traps, doors, and stairs in your vicinity.";
#endif
    
		{
			int rad = DETECT_RAD_DEFAULT;

			if (info) return info_radius(rad);

			if (cast)
			{
				detect_traps(caster_ptr, rad, TRUE);
				detect_doors(caster_ptr, rad);
				detect_stairs(caster_ptr, rad);
			}
		}
		break;

	case 3:
#ifdef JP
		if (name) return "HƐ";
		if (desc) return "HoB";
#else
		if (name) return "Produce Food";
		if (desc) return "Produces a Ration of Food.";
#endif
    
		{
			if (cast)
			{
				object_type forge, *quest_ptr = &forge;

#ifdef JP
				msg_print("H𐶐B");
#else
				msg_print("A food ration is produced.");
#endif

				/* Create the food ration */
				object_prep(quest_ptr, lookup_kind(TV_FOOD, SV_FOOD_RATION), ITEM_FREE_SIZE);

				/* Drop the object from heaven */
				drop_near(floor_ptr, quest_ptr, -1, caster_ptr->fy, caster_ptr->fx);
			}
		}
		break;

	case 4:
#ifdef JP
		if (name) return "̌";
		if (desc) return "Ƃ炵Ă͈͂Ŝivɖ邭B";
#else
		if (name) return "Daylight";
		if (desc) return "Lights up nearby area and the inside of a room permanently.";
#endif
    
		{
			int dice = 2;
			int sides = plev / 2;
			int rad = (plev / 10) + 1;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				lite_area(caster_ptr, diceroll(dice, sides), rad);

				if (has_trait(caster_ptr, TRAIT_HURT_LITE) && !caster_ptr->resist_lite)
				{
#ifdef JP
					msg_print("̌Ȃ̓̂łI");
#else
					msg_print("The daylight scorches your flesh!");
#endif

#ifdef JP
					take_hit(NULL, caster_ptr, DAMAGE_NOESCAPE, diceroll(2, 2), "̌", NULL, -1);
#else
					take_hit(NULL, caster_ptr, DAMAGE_NOESCAPE, diceroll(2, 2), "daylight", NULL, -1);
#endif
				}
			}
		}
		break;

	case 5:
#ifdef JP
		if (name) return "K";
		if (desc) return "1̂𖣗BRƖB";
#else
		if (name) return "Animal Taming";
		if (desc) return "Attempts to charm an animal.";
#endif
    
		{
			int power = plev;

			if (info) return info_power(power);

			if (cast)
			{
				if (!get_aim_dir(caster_ptr, &dir)) return NULL;

				charm_animal(caster_ptr, dir, power);
			}
		}
		break;

	case 6:
#ifdef JP
		if (name) return "ւ̑ϐ";
		if (desc) return "莞ԁACAAdɑ΂ϐ𓾂BɂϐɗݐςB";
#else
		if (name) return "Resist Environment";
		if (desc) return "Gives resistance to fire, cold and electricity for a while. These resistances can be added to which from equipment for more powerful resistances.";
#endif
    
		{
			int base = 20;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_oppose_cold(caster_ptr, randint1(base) + base, FALSE);
				set_oppose_fire(caster_ptr, randint1(base) + base, FALSE);
				set_oppose_elec(caster_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 7:
#ifdef JP
		if (name) return "ƓŎ";
		if (desc) return "SAł̂犮SɎ菜A̗͂񕜂B";
#else
		if (name) return "Cure Wounds & Poison";
		if (desc) return "Heals all cut and poison status. Heals HP a little.";
#endif
    
		{
			int dice = 2;
			int sides = 8;

			if (info) return info_heal(dice, sides, 0);

			if (cast)
			{
				hp_player(caster_ptr, diceroll(dice, sides));
				set_cut(caster_ptr, 0);
				set_poisoned(caster_ptr, 0);
			}
		}
		break;

	case 8:
#ifdef JP
		if (name) return "Ηn";
		if (desc) return "ǂnďɂB";
#else
		if (name) return "Stone to Mud";
		if (desc) return "Turns one rock square to mud.";
#endif
    
		{
			int dice = 1;
			int sides = 30;
			int base = 20;

			if (info) return info_damage(dice, sides, base);

			if (cast)
			{
				if (!get_aim_dir(caster_ptr, &dir)) return NULL;

				wall_to_mud(caster_ptr, dir);
			}
		}
		break;

	case 9:
#ifdef JP
		if (name) return "ACXE{g";
		if (desc) return "C̃{g̓r[B";
#else
		if (name) return "Frost Bolt";
		if (desc) return "Fires a bolt or beam of cold.";
#endif
    
		{
			int dice = 3 + (plev - 5) / 4;
			int sides = 8;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				if (!get_aim_dir(caster_ptr, &dir)) return NULL;
				fire_bolt_or_beam(caster_ptr, beam_chance(caster_ptr) - 10, GF_COLD, dir, diceroll(dice, sides));
			}
		}
		break;

	case 10:
#ifdef JP
		if (name) return "R̊o";
		if (desc) return "ӂ̒n`mA߂㩁AAKiASẴN[`[mB";
#else
		if (name) return "Nature Awareness";
		if (desc) return "Maps nearby area. Detects all creatures, traps, doors and stairs.";
#endif
    
		{
			int rad1 = DETECT_RAD_MAP;
			int rad2 = DETECT_RAD_DEFAULT;

			if (info) return info_radius(MAX(rad1, rad2));

			if (cast)
			{
				map_area(caster_ptr, rad1);
				detect_traps(caster_ptr, rad2, TRUE);
				detect_doors(caster_ptr, rad2);
				detect_stairs(caster_ptr, rad2);
				detect_creatures_normal(caster_ptr, rad2);
			}
		}
		break;

	case 11:
#ifdef JP
		if (name) return "t@CAE{g";
		if (desc) return "Ή̃{g̓r[B";
#else
		if (name) return "Fire Bolt";
		if (desc) return "Fires a bolt or beam of fire.";
#endif
    
		{
			int dice = 5 + (plev - 5) / 4;
			int sides = 8;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				if (!get_aim_dir(caster_ptr, &dir)) return NULL;
				fire_bolt_or_beam(caster_ptr, beam_chance(caster_ptr) - 10, GF_FIRE, dir, diceroll(dice, sides));
			}
		}
		break;

	case 12:
#ifdef JP
		if (name) return "z";
		if (desc) return "BN[`[ɌʂB";
#else
		if (name) return "Ray of Sunlight";
		if (desc) return "Fires a beam of light which damages to light-sensitive creatures.";
#endif
    
		{
			int dice = 6;
			int sides = 8;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				if (!get_aim_dir(caster_ptr, &dir)) return NULL;
#ifdef JP
				msg_print("zꂽB");
#else
				msg_print("A line of sunlight appears.");
#endif

				lite_line(caster_ptr, dir);
			}
		}
		break;

	case 13:
#ifdef JP
		if (name) return "";
		if (desc) return "ȆSẴN[`[BRƖB";
#else
		if (name) return "Entangle";
		if (desc) return "Attempts to slow all creatures in sight.";
#endif
    
		{
			int power = plev;

			if (info) return info_power(power);

			if (cast)
			{
				slow_creatures(caster_ptr);
			}
		}
		break;

	case 14:
#ifdef JP
		if (name) return "";
		if (desc) return "1̏B";
#else
		if (name) return "Summon Animal";
		if (desc) return "Summons an animal.";
#endif
    
		{
			if (cast)
			{
				if (!(summon_specific(NULL, caster_ptr->fy, caster_ptr->fx, plev, SUMMON_ANIMAL_RANGER, (PM_ALLOW_GROUP | PM_FORCE_PET))))
				{
#ifdef JP
					msg_print("͌ȂB");
#else
					msg_print("No animals arrive.");
#endif
				}
				break;
			}
		}
		break;

	case 15:
#ifdef JP
		if (name) return "򑐎";
		if (desc) return "̗͂啝ɉ񕜂AANOԁAłSB";
#else
		if (name) return "Herbal Healing";
		if (desc) return "Heals HP greatly. And heals cut, stun and poison completely.";
#endif
    
		{
			int heal = 500;

			if (info) return info_heal(0, 0, heal);

			if (cast)
			{
				hp_player(caster_ptr, heal);
				set_stun(caster_ptr, 0);
				set_cut(caster_ptr, 0);
				set_poisoned(caster_ptr, 0);
			}
		}
		break;

	case 16:
#ifdef JP
		if (name) return "Ki";
		if (desc) return "̂ʒuɊKiB";
#else
		if (name) return "Stair Building";
		if (desc) return "Creates a stair which goes down or up.";
#endif
    
		{
			if (cast)
			{
				stair_creation(caster_ptr, floor_ptr);
			}
		}
		break;

	case 17:
#ifdef JP
		if (name) return "Ή";
		if (desc) return "莞ԁAAC㏸B";
#else
		if (name) return "Stone Skin";
		if (desc) return "Gives bonus to AC for a while.";
#endif
    
		{
			int base = 20;
			int sides = 30;

			if (info) return info_duration(base, sides);

			if (cast)
			{
				set_shield(caster_ptr, randint1(sides) + base, FALSE);
			}
		}
		break;

	case 18:
#ifdef JP
		if (name) return "^Eϐ";
		if (desc) return "莞ԁA_AdAACAłɑ΂ϐ𓾂BɂϐɗݐςB";
#else
		if (name) return "Resistance True";
		if (desc) return "Gives resistance to fire, cold, electricity, acid and poison for a while. These resistances can be added to which from equipment for more powerful resistances.";
#endif
    
		{
			int base = 20;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_oppose_acid(caster_ptr, randint1(base) + base, FALSE);
				set_oppose_elec(caster_ptr, randint1(base) + base, FALSE);
				set_oppose_fire(caster_ptr, randint1(base) + base, FALSE);
				set_oppose_cold(caster_ptr, randint1(base) + base, FALSE);
				set_oppose_pois(caster_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 19:
#ifdef JP
		if (name) return "Xёn";
		if (desc) return "͂ɖ؂oB";
#else
		if (name) return "Forest Creation";
		if (desc) return "Creates trees in all adjacent squares.";
#endif
    
		{
			if (cast)
			{
				tree_creation(caster_ptr);
			}
		}
		break;

	case 20:
#ifdef JP
		if (name) return "Fa";
		if (desc) return "ȆSĂ̓𖣗BRƖB";
#else
		if (name) return "Animal Friendship";
		if (desc) return "Attempts to charm all animals in sight.";
#endif
    
		{
			int power = plev * 2;

			if (info) return info_power(power);

			if (cast)
			{
				charm_animals(caster_ptr, power);
			}
		}
		break;

	case 21:
#ifdef JP
		if (name) return "";
		if (desc) return "ACe̎\͂SɒmB";
#else
		if (name) return "Stone Tell";
		if (desc) return "*Identifies* an item.";
#endif
    
		{
			if (cast)
			{
				if (!identify_fully(caster_ptr, FALSE)) return NULL;
			}
		}
		break;

	case 22:
#ifdef JP
		if (name) return "΂̕";
		if (desc) return "̎͂ɉԛ̕ǂB";
#else
		if (name) return "Wall of Stone";
		if (desc) return "Creates granite walls in all adjacent squares.";
#endif
    
		{
			if (cast)
			{
				wall_stone(caster_ptr);
			}
		}
		break;

	case 23:
#ifdef JP
		if (name) return "Hh~";
		if (desc) return "ACe_ŏȂ悤HB";
#else
		if (name) return "Protect from Corrosion";
		if (desc) return "Makes an equipment acid-proof.";
#endif
    
		{
			if (cast)
			{
				if (!rustproof(caster_ptr)) return NULL;
			}
		}
		break;

	case 24:
#ifdef JP
		if (name) return "nk";
		if (desc) return "͂̃_Wh炵AǂƏ_ɓςB";
#else
		if (name) return "Earthquake";
		if (desc) return "Shakes dungeon structure, and results in random swapping of floors and walls.";
#endif
    
		{
			int rad = 10;

			if (info) return info_radius(rad);

			if (cast)
			{
				earthquake(caster_ptr, caster_ptr->fy, caster_ptr->fx, rad);
			}
		}
		break;

	case 25:
#ifdef JP
		if (name) return "J}C^`";
		if (desc) return "SɌčUB";
#else
		if (name) return "Cyclone";
		if (desc) return "Attacks all adjacent creatures.";
#endif
    
		{
			if (cast)
			{
				int y = 0, x = 0;
				cave_type       *c_ptr;
				creature_type    *m_ptr;

				for (dir = 0; dir < 8; dir++)
				{
					y = caster_ptr->fy + ddy_ddd[dir];
					x = caster_ptr->fx + ddx_ddd[dir];
					c_ptr = &floor_ptr->cave[y][x];

					/* Get the creature */
					m_ptr = &creature_list[c_ptr->creature_idx];

					/* Hack -- attack creatures */
					if (c_ptr->creature_idx && (m_ptr->ml || cave_have_flag_bold(floor_ptr, y, x, FF_PROJECT)))
						melee_attack(caster_ptr, y, x, 0);
				}
			}
		}
		break;

	case 26:
#ifdef JP
		if (name) return "uU[h";
		if (desc) return "ȗC̋B";
#else
		if (name) return "Blizzard";
		if (desc) return "Fires a huge ball of cold.";
#endif
    
		{
			int dam = 70 + plev * 3 / 2;
			int rad = plev / 12 + 1;

			if (info) return info_damage(0, 0, dam);

			if (cast)
			{
				if (!get_aim_dir(caster_ptr, &dir)) return NULL;

				fire_ball(caster_ptr, GF_COLD, dir, dam, rad);
			}
		}
		break;

	case 27:
#ifdef JP
		if (name) return "ȗ";
		if (desc) return "ȓd̋B";
#else
		if (name) return "Lightning Storm";
		if (desc) return "Fires a huge electric ball.";
#endif
    
		{
			int dam = 90 + plev * 3 / 2;
			int rad = plev / 12 + 1;

			if (info) return info_damage(0, 0, dam);

			if (cast)
			{
				if (!get_aim_dir(caster_ptr, &dir)) return NULL;
				fire_ball(caster_ptr, GF_ELEC, dir, dam, rad);
				break;
			}
		}
		break;

	case 28:
#ifdef JP
		if (name) return "Q";
		if (desc) return "Ȑ̋B";
#else
		if (name) return "Whirlpool";
		if (desc) return "Fires a huge ball of water.";
#endif
    
		{
			int dam = 100 + plev * 3 / 2;
			int rad = plev / 12 + 1;

			if (info) return info_damage(0, 0, dam);

			if (cast)
			{
				if (!get_aim_dir(caster_ptr, &dir)) return NULL;
				fire_ball(caster_ptr, GF_WATER, dir, dam, rad);
			}
		}
		break;

	case 29:
#ifdef JP
		if (name) return "z";
		if (desc) return "𒆐SƂ̋𔭐BɁÅKŜivɏƂ炵A_WׂẴACemB";
#else
		if (name) return "Call Sunlight";
		if (desc) return "Generates ball of light centered on you. Maps and lights whole dungeon level. Knows all objects location.";
#endif
    
		{
			int dam = 150;
			int rad = 8;

			if (info) return info_damage(0, 0, dam/2);

			if (cast)
			{
				fire_ball(caster_ptr, GF_LITE, 0, dam, rad);
				wiz_lite(floor_ptr, caster_ptr, FALSE);

				if (has_trait(caster_ptr, TRAIT_HURT_LITE) && !caster_ptr->resist_lite)
				{
#ifdef JP
					msg_print("Ȃ̓̂łI");
#else
					msg_print("The sunlight scorches your flesh!");
#endif

#ifdef JP
					take_hit(NULL, caster_ptr, DAMAGE_NOESCAPE, 50, "", NULL, -1);
#else
					take_hit(NULL, caster_ptr, DAMAGE_NOESCAPE, 50, "sunlight", NULL, -1);
#endif
				}
			}
		}
		break;

	case 30:
#ifdef JP
		if (name) return "̐n";
		if (desc) return "ɉC̑B";
#else
		if (name) return "Elemental Branding";
		if (desc) return "Makes current weapon fire or frost branded.";
#endif
    
		{
			if (cast)
			{
				brand_weapon(caster_ptr, randint0(2));
			}
		}
		break;

	case 31:
#ifdef JP
		if (name) return "R̋";
		if (desc) return "߂̑SẴN[`[Ƀ_[W^AnkNA𒆐SƂ̋𔭐B";
#else
		if (name) return "Nature's Wrath";
		if (desc) return "Damages all creatures in sight. Makes quake. Generates disintegration ball centered on you.";
#endif
    
		{
			int d_dam = 4 * plev;
			int b_dam = (100 + plev) * 2;
			int b_rad = 1 + plev / 12;
			int q_rad = 20 + plev / 2;

			if (info) return format("%s%d+%d", s_dam, d_dam, b_dam/2);

			if (cast)
			{
				dispel_creatures(caster_ptr, d_dam);
				earthquake(caster_ptr, caster_ptr->fy, caster_ptr->fx, q_rad);
				project(caster_ptr, b_rad, caster_ptr->fy, caster_ptr->fx, b_dam, GF_DISINTEGRATE, PROJECT_KILL | PROJECT_ITEM, -1);
			}
		}
		break;
	}

	return "";
}


static cptr do_chaos_spell(creature_type *creature_ptr, int spell, int mode)
{
	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool info = (mode == SPELL_INFO) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;

#ifdef JP
	static const char s_dam[] = ":";
	static const char s_random[] = "_";
#else
	static const char s_dam[] = "dam ";
	static const char s_random[] = "random";
#endif

	int dir;
	int plev = creature_ptr->lev;

	switch (spell)
	{
	case 0:
#ifdef JP
		if (name) return "}WbNE~TC";
		if (desc) return "ア@̖B";
#else
		if (name) return "Magic Missile";
		if (desc) return "Fires a weak bolt of magic.";
#endif
    
		{
			int dice = 3 + ((plev - 1) / 5);
			int sides = 4;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_bolt_or_beam(creature_ptr, beam_chance(creature_ptr) - 10, GF_MISSILE, dir, diceroll(dice, sides));
			}
		}
		break;

	case 1:
#ifdef JP
		if (name) return "gbv/hAj";
		if (desc) return "אڂ㩂Ɣj󂷂B";
#else
		if (name) return "Trap / Door Destruction";
		if (desc) return "Destroys all traps in adjacent squares.";
#endif
    
		{
			int rad = 1;

			if (info) return info_radius(rad);

			if (cast)
			{
				destroy_doors_touch(creature_ptr);
			}
		}
		break;

	case 2:
#ifdef JP
		if (name) return "M";
		if (desc) return "Ƃ炵Ă͈͂Ŝivɖ邭B";
#else
		if (name) return "Flash of Light";
		if (desc) return "Lights up nearby area and the inside of a room permanently.";
#endif
    
		{
			int dice = 2;
			int sides = plev / 2;
			int rad = (plev / 10) + 1;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				lite_area(creature_ptr, diceroll(dice, sides), rad);
			}
		}
		break;

	case 3:
#ifdef JP
		if (name) return "̎";
		if (desc) return "Uł悤ɂB";
#else
		if (name) return "Touch of Confusion";
		if (desc) return "Attempts to confuse the next creature that you hit.";
#endif
    
		{
			if (cast)
			{
				if (!(creature_ptr->special_attack & ATTACK_CONFUSE))
				{
#ifdef JP
					msg_print("Ȃ̎͌n߂B");
#else
					msg_print("Your hands start glowing.");
#endif

					creature_ptr->special_attack |= ATTACK_CONFUSE;
					play_redraw |= (PR_STATUS);
				}
			}
		}
		break;

	case 4:
#ifdef JP
		if (name) return "y";
		if (desc) return "@̋B";
#else
		if (name) return "Mana Burst";
		if (desc) return "Fires a ball of magic.";
#endif
    
		{
			int dice = 3;
			int sides = 5;
			int rad = (plev < 30) ? 2 : 3;
			int base;

			if (has_trait(creature_ptr, TRAIT_MAGIC_SPECIALIST))
				base = plev + plev / 2;
			else
				base = plev + plev / 4;


			if (info) return info_damage(dice, sides, base);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_ball(creature_ptr, GF_MISSILE, dir, diceroll(dice, sides) + base, rad);

				/*
				 * Shouldn't actually use GF_MANA, as
				 * it will destroy all items on the
				 * floor
				 */
			}
		}
		break;

	case 5:
#ifdef JP
		if (name) return "t@CAE{g";
		if (desc) return "̃{g̓r[B";
#else
		if (name) return "Fire Bolt";
		if (desc) return "Fires a bolt or beam of fire.";
#endif
    
		{
			int dice = 8 + (plev - 5) / 4;
			int sides = 8;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_bolt_or_beam(creature_ptr, beam_chance(creature_ptr), GF_FIRE, dir, diceroll(dice, sides));
			}
		}
		break;

	case 6:
#ifdef JP
		if (name) return "͂̌";
		if (desc) return "ȕ̋B";
#else
		if (name) return "Fist of Force";
		if (desc) return "Fires a tiny ball of disintegration.";
#endif
    
		{
			int dice = 8 + ((plev - 5) / 4);
			int sides = 8;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_ball(creature_ptr, GF_DISINTEGRATE, dir,
					diceroll(dice, sides), 0);
			}
		}
		break;

	case 7:
#ifdef JP
		if (name) return "e|[g";
		if (desc) return "̃e|[gB";
#else
		if (name) return "Teleport Self";
		if (desc) return "Teleport long distance.";
#endif
    
		{
			int range = plev * 5;

			if (info) return info_range(range);

			if (cast)
			{
				teleport_player(creature_ptr, range, 0L);
			}
		}
		break;

	case 8:
#ifdef JP
		if (name) return "_[";
		if (desc) return "N[`[Ƀ_Ȍʂ^B";
#else
		if (name) return "Wonder";
		if (desc) return "Fires something with random effects.";
#endif
    
		{
			if (info) return s_random;

			if (cast)
			{

				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				cast_wonder(creature_ptr, dir);
			}
		}
		break;

	case 9:
#ifdef JP
		if (name) return "JIXE{g";
		if (desc) return "JIX̃{g̓r[B";
#else
		if (name) return "Chaos Bolt";
		if (desc) return "Fires a bolt or ball of chaos.";
#endif
    
		{
			int dice = 10 + (plev - 5) / 4;
			int sides = 8;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_bolt_or_beam(creature_ptr, beam_chance(creature_ptr), GF_CHAOS, dir, diceroll(dice, sides));
			}
		}
		break;

	case 10:
#ifdef JP
		if (name) return "\jbNEu[";
		if (desc) return "𒆐SƂ̋𔭐B";
#else
		if (name) return "Sonic Boom";
		if (desc) return "Generates a ball of sound centered on you.";
#endif
    
		{
			int dam = 60 + plev;
			int rad = plev / 10 + 2;

			if (info) return info_damage(0, 0, dam/2);

			if (cast)
			{
#ifdef JP
				msg_print("h[IhꂽI");
#else
				msg_print("BOOM! Shake the room!");
#endif

				project(creature_ptr, rad, creature_ptr->fy, creature_ptr->fx, dam, GF_SOUND, PROJECT_KILL | PROJECT_ITEM, -1);
			}
		}
		break;

	case 11:
#ifdef JP
		if (name) return "jł̖";
		if (desc) return "Ȗ͂̃r[B";
#else
		if (name) return "Doom Bolt";
		if (desc) return "Fires a beam of pure mana.";
#endif
    
		{
			int dice = 11 + (plev - 5) / 4;
			int sides = 8;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_beam(creature_ptr, GF_MANA, dir, diceroll(dice, sides));
			}
		}
		break;

	case 12:
#ifdef JP
		if (name) return "t@CAE{[";
		if (desc) return "̋B";
#else
		if (name) return "Fire Ball";
		if (desc) return "Fires a ball of fire.";
#endif
    
		{
			int dam = plev + 55;
			int rad = 2;

			if (info) return info_damage(0, 0, dam);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_ball(creature_ptr, GF_FIRE, dir, dam, rad);
			}
		}
		break;

	case 13:
#ifdef JP
		if (name) return "e|[gEAEFC";
		if (desc) return "N[`[e|[gr[BRƖB";
#else
		if (name) return "Teleport Other";
		if (desc) return "Teleports all creatures on the line away unless resisted.";
#endif
    
		{
			int power = plev;

			if (info) return info_power(power);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_beam(creature_ptr, GF_AWAY_ALL, dir, power);
			}
		}
		break;

	case 14:
#ifdef JP
		if (name) return "ǰt";
		if (desc) return "ӂ̃ACeAN[`[An`j󂷂B";
#else
		if (name) return "Word of Destruction";
		if (desc) return "Destroy everything in nearby area.";
#endif
    
		{
			int base = 12;
			int sides = 4;

			if (cast)
			{
				destroy_area(creature_ptr, creature_ptr->fy, creature_ptr->fx, base + randint1(sides), FALSE);
			}
		}
		break;

	case 15:
#ifdef JP
		if (name) return "OX";
		if (desc) return "ȃJIX̋B";
#else
		if (name) return "Invoke Logrus";
		if (desc) return "Fires a huge ball of chaos.";
#endif
    
		{
			int dam = plev * 2 + 99;
			int rad = plev / 5;

			if (info) return info_damage(0, 0, dam);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_ball(creature_ptr, GF_CHAOS, dir, dam, rad);
			}
		}
		break;

	case 16:
#ifdef JP
		if (name) return "ҕϗe";
		if (desc) return "N[`[1̂ϐgBRƖB";
#else
		if (name) return "Polymorph Other";
		if (desc) return "Attempts to polymorph a creature.";
#endif
    
		{
			int power = plev;

			if (info) return info_power(power);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				poly_creature(creature_ptr, dir);
			}
		}
		break;

	case 17:
#ifdef JP
		if (name) return "A";
		if (desc) return "Sɑ΂ēd̃r[B";
#else
		if (name) return "Chain Lightning";
		if (desc) return "Fires lightning beams in all directions.";
#endif
    
		{
			int dice = 5 + plev / 10;
			int sides = 8;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				for (dir = 0; dir <= 9; dir++)
					fire_beam(creature_ptr, GF_ELEC, dir, diceroll(dice, sides));
			}
		}
		break;

	case 18:
#ifdef JP
		if (name) return "͕";
		if (desc) return "/@_̏[U񐔂𑝂₷A[Ũbh̏[UԂ炷B";
#else
		if (name) return "Arcane Binding";
		if (desc) return "Recharges staffs, wands or rods.";
#endif
    
		{
			int power = 90;

			if (info) return info_power(power);

			if (cast)
			{
				if (!recharge(creature_ptr, power)) return NULL;
			}
		}
		break;

	case 19:
#ifdef JP
		if (name) return "q";
		if (desc) return "ȕ̋B";
#else
		if (name) return "Disintegrate";
		if (desc) return "Fires a huge ball of disintegration.";
#endif
    
		{
			int dam = plev + 70;
			int rad = 3 + plev / 40;

			if (info) return info_damage(0, 0, dam);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_ball(creature_ptr, GF_DISINTEGRATE, dir, dam, rad);
			}
		}
		break;

	case 20:
#ifdef JP
		if (name) return "ϗe";
		if (desc) return "݂̊Kč\B";
#else
		if (name) return "Alter Reality";
		if (desc) return "Recreates current dungeon level.";
#endif
    
		{
			int base = 15;
			int sides = 20;

			if (info) return info_delay(base, sides);

			if (cast)
			{
				alter_reality(creature_ptr);
			}
		}
		break;

	case 21:
#ifdef JP
		if (name) return "}WbNEPbg";
		if (desc) return "Pbg𔭎˂B";
#else
		if (name) return "Magic Rocket";
		if (desc) return "Fires a magic rocket.";
#endif
    
		{
			int dam = 120 + plev * 2;
			int rad = 2;

			if (info) return info_damage(0, 0, dam);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

#ifdef JP
				msg_print("PbgˁI");
#else
				msg_print("You launch a rocket!");
#endif

				fire_rocket(creature_ptr, GF_ROCKET, dir, dam, rad);
			}
		}
		break;

	case 22:
#ifdef JP
		if (name) return "ׂ̐n";
		if (desc) return "ɃJIX̑B";
#else
		if (name) return "Chaos Branding";
		if (desc) return "Makes current weapon a Chaotic weapon.";
#endif
    
		{
			if (cast)
			{
				brand_weapon(creature_ptr, 2);
			}
		}
		break;

	case 23:
#ifdef JP
		if (name) return "";
		if (desc) return "1̏B";
#else
		if (name) return "Summon Demon";
		if (desc) return "Summons a demon.";
#endif
    
		{
			if (cast)
			{
				u32b mode = 0L;
				bool pet = !one_in_(3);

				if (pet) mode |= PM_FORCE_PET;
				else mode |= PM_NO_PET;
				if (!(pet && (plev < 50))) mode |= PM_ALLOW_GROUP;

				if (summon_specific((pet ? creature_ptr : NULL), creature_ptr->fy, creature_ptr->fx, (plev * 3) / 2, SUMMON_DEMON, mode))
				{
#ifdef JP
					msg_print("̈L[B");
#else
					msg_print("The area fills with a stench of sulphur and brimstone.");
#endif

					if (pet)
					{
#ifdef JP
						msg_print("upł܂Allv");
#else
						msg_print("'What is thy bidding... Master?'");
#endif
					}
					else
					{
#ifdef JP
						msg_print("uڂ҂Ả͓lɂ炸I O̍𒸂Iv");
#else
						msg_print("'NON SERVIAM! Wretch! I shall feast on thy mortal soul!'");
#endif
					}
				}
			}
		}
		break;

	case 24:
#ifdef JP
		if (name) return "d͌";
		if (desc) return "d͂̃r[B";
#else
		if (name) return "Beam of Gravity";
		if (desc) return "Fires a beam of gravity.";
#endif
    
		{
			int dice = 9 + (plev - 5) / 4;
			int sides = 8;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_beam(creature_ptr, GF_GRAVITY, dir, diceroll(dice, sides));
			}
		}
		break;

	case 25:
#ifdef JP
		if (name) return "Q";
		if (desc) return "̎ӂ覐΂𗎂ƂB";
#else
		if (name) return "Meteor Swarm";
		if (desc) return "Makes meteor balls fall down to nearby random locations.";
#endif
    
		{
			int dam = plev * 2;
			int rad = 2;

			if (info) return info_multi_damage(dam);

			if (cast)
			{
				cast_meteor(creature_ptr, dam, rad);
			}
		}
		break;

	case 26:
#ifdef JP
		if (name) return "̈ꌂ";
		if (desc) return "𒆐SƂȉ̋𔭐B";
#else
		if (name) return "Flame Strike";
		if (desc) return "Generate a huge ball of fire centered on you.";
#endif
    
		{
			int dam = 300 + 3 * plev;
			int rad = 8;

			if (info) return info_damage(0, 0, dam/2);

			if (cast)
			{
				fire_ball(creature_ptr, GF_FIRE, 0, dam, rad);
			}
		}
		break;

	case 27:
#ifdef JP
		if (name) return "׏";
		if (desc) return "_ȑ̋r[𔭐B";
#else
		if (name) return "Call Chaos";
		if (desc) return "Generate random kind of balls or beams.";
#endif
    
		{
			if (info) return format("%s150 / 250", s_dam);

			if (cast)
			{
				call_chaos(creature_ptr);
			}
		}
		break;

	case 28:
#ifdef JP
		if (name) return "ȕϗe";
		if (desc) return "ϐg悤ƂB";
#else
		if (name) return "Polymorph Self";
		if (desc) return "Polymorphs yourself.";
#endif
    
		{
			if (cast)
			{
#ifdef JP
				if (!get_check("ϐg܂B낵łH")) return NULL;
#else
				if (!get_check("You will polymorph yourself. Are you sure? ")) return NULL;
#endif
				do_poly_self(creature_ptr);
			}
		}
		break;

	case 29:
#ifdef JP
		if (name) return "̗͂";
		if (desc) return "ɋ͂ŋȏȖ͂̋B";
#else
		if (name) return "Mana Storm";
		if (desc) return "Fires an extremely powerful huge ball of pure mana.";
#endif
    
		{
			int dam = 300 + plev * 4;
			int rad = 4;

			if (info) return info_damage(0, 0, dam);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_ball(creature_ptr, GF_MANA, dir, dam, rad);
			}
		}
		break;

	case 30:
#ifdef JP
		if (name) return "OX̃uX";
		if (desc) return "ɋ͂ȃJIX̋B";
#else
		if (name) return "Breathe Logrus";
		if (desc) return "Fires an extremely powerful ball of chaos.";
#endif
    
		{
			int dam = creature_ptr->chp;
			int rad = 2;

			if (info) return info_damage(0, 0, dam);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_ball(creature_ptr, GF_CHAOS, dir, dam, rad);
			}
		}
		break;

	case 31:
#ifdef JP
		if (name) return "";
		if (desc) return "Ɏ͂ɌāAPbgAȖ͂̋Aːp̋BAǂɗאڂĎgpƍL͈͂j󂷂B";
#else
		if (name) return "Call the Void";
		if (desc) return "Fires rockets, mana balls and nuclear waste balls in all directions each unless you are not adjacent to any walls. Otherwise *destroys* huge area.";
#endif
    
		{
			if (info) return format("%s3 * 175", s_dam);

			if (cast)
			{
				call_the_void(creature_ptr);
			}
		}
		break;
	}

	return "";
}


static cptr do_death_spell(creature_type *creature_ptr, int spell, int mode)
{
	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool info = (mode == SPELL_INFO) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;

#ifdef JP
	static const char s_dam[] = ":";
	static const char s_random[] = "_";
#else
	static const char s_dam[] = "dam ";
	static const char s_random[] = "random";
#endif

	int dir;
	int plev = creature_ptr->lev;

	switch (spell)
	{
	case 0:
#ifdef JP
		if (name) return "m";
		if (desc) return "߂̐̂ȂN[`[mB";
#else
		if (name) return "Detect Unlife";
		if (desc) return "Detects all nonliving creatures in your vicinity.";
#endif
    
		{
			int rad = DETECT_RAD_DEFAULT;

			if (info) return info_radius(rad);

			if (cast)
			{
				detect_creatures_nonliving(creature_ptr, rad);
			}
		}
		break;

	case 1:
#ifdef JP
		if (name) return "Ee";
		if (desc) return "Ȏ׈ȗ͂{[BPǂȃN[`[ɂ͑傫ȃ_[W^B";
#else
		if (name) return "Malediction";
		if (desc) return "Fires a tiny ball of evil power which hurts good creatures greatly.";
#endif
    
		{
			int dice = 3 + (plev - 1) / 5;
			int sides = 4;
			int rad = 0;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				/*
				 * A radius-0 ball may (1) be aimed at
				 * objects etc., and will affect them;
				 * (2) may be aimed at ANY visible
				 * creature, unlike a 'bolt' which must
				 * travel to the creature.
				 */

				fire_ball(creature_ptr, GF_HELL_FIRE, dir, diceroll(dice, sides), rad);

				if (one_in_(5))
				{
					/* Special effect first */
					int effect = randint1(1000);

					if (effect == 666)
						fire_ball_hide(creature_ptr, GF_DEATH_RAY, dir, plev * 200, 0);
					else if (effect < 500)
						fire_ball_hide(creature_ptr, GF_TURN_ALL, dir, plev, 0);
					else if (effect < 800)
						fire_ball_hide(creature_ptr, GF_OLD_CONF, dir, plev, 0);
					else
						fire_ball_hide(creature_ptr, GF_STUN, dir, plev, 0);
				}
			}
		}
		break;

	case 2:
#ifdef JP
		if (name) return "׈m";
		if (desc) return "߂̎׈ȃN[`[mB";
#else
		if (name) return "Detect Evil";
		if (desc) return "Detects all evil creatures in your vicinity.";
#endif
    
		{
			int rad = DETECT_RAD_DEFAULT;

			if (info) return info_radius(rad);

			if (cast)
			{
				detect_creatures_evil(creature_ptr, rad);
			}
		}
		break;

	case 3:
#ifdef JP
		if (name) return "L_";
		if (desc) return "ł̋B";
#else
		if (name) return "Stinking Cloud";
		if (desc) return "Fires a ball of poison.";
#endif
    
		{
			int dam = 10 + plev / 2;
			int rad = 2;

			if (info) return info_damage(0, 0, dam);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_ball(creature_ptr, GF_POIS, dir, dam, rad);
			}
		}
		break;

	case 4:
#ifdef JP
		if (name) return "";
		if (desc) return "1̂̃N[`[𖰂点BRƖB";
#else
		if (name) return "Black Sleep";
		if (desc) return "Attempts to sleep a creature.";
#endif
    
		{
			int power = plev;

			if (info) return info_power(power);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				sleep_creature(creature_ptr, dir);
			}
		}
		break;

	case 5:
#ifdef JP
		if (name) return "ϓ";
		if (desc) return "莞ԁAłւ̑ϐ𓾂BɂϐɗݐςB";
#else
		if (name) return "Resist Poison";
		if (desc) return "Gives resistance to poison. This resistance can be added to which from equipment for more powerful resistance.";
#endif
    
		{
			int base = 20;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_oppose_pois(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 6:
#ifdef JP
		if (name) return "Q";
		if (desc) return "N[`[1̂|ANOBRƖB";
#else
		if (name) return "Horrify";
		if (desc) return "Attempts to scare and stun a creature.";
#endif
    
		{
			int power = plev;

			if (info) return info_power(power);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fear_creature(creature_ptr, dir, power);
				stun_creature(creature_ptr, dir, power);
			}
		}
		break;

	case 7:
#ifdef JP
		if (name) return "Afbh]";
		if (desc) return "Afbh1̂𖣗BRƖB";
#else
		if (name) return "Enslave Undead";
		if (desc) return "Attempts to charm an undead creature.";
#endif
    
		{
			int power = plev;

			if (info) return info_power(power);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				control_one_undead(creature_ptr, dir, power);
			}
		}
		break;

	case 8:
#ifdef JP
		if (name) return "Ggs[̋";
		if (desc) return "̂҂Ɍʂ̂鋅B";
#else
		if (name) return "Orb of Entropy";
		if (desc) return "Fires a ball which damages living creatures.";
#endif
    
		{
			int dice = 3;
			int sides = 6;
			int rad = (plev < 30) ? 2 : 3;
			int base;

			if (has_trait(creature_ptr, TRAIT_MAGIC_SPECIALIST))
				base = plev + plev / 2;
			else
				base = plev + plev / 4;


			if (info) return info_damage(dice, sides, base);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_ball(creature_ptr, GF_OLD_DRAIN, dir, diceroll(dice, dice) + base, rad);
			}
		}
		break;

	case 9:
#ifdef JP
		if (name) return "n̖";
		if (desc) return "ñ{g̓r[B";
#else
		if (name) return "Nether Bolt";
		if (desc) return "Fires a bolt or beam of nether.";
#endif
    
		{
			int dice = 8 + (plev - 5) / 4;
			int sides = 8;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_bolt_or_beam(creature_ptr, beam_chance(creature_ptr), GF_NETHER, dir, diceroll(dice, sides));
			}
		}
		break;

	case 10:
#ifdef JP
		if (name) return "EC_";
		if (desc) return "𒆐SƂł̋𔭐B";
#else
		if (name) return "Cloud kill";
		if (desc) return "Generate a ball of poison centered on you.";
#endif
    
		{
			int dam = (30 + plev) * 2;
			int rad = plev / 10 + 2;

			if (info) return info_damage(0, 0, dam/2);

			if (cast)
			{
				project(creature_ptr, rad, creature_ptr->fy, creature_ptr->fx, dam, GF_POIS, PROJECT_KILL | PROJECT_ITEM, -1);
			}
		}
		break;

	case 11:
#ifdef JP
		if (name) return "N[`[";
		if (desc) return "N[`[1̂BolACe͎ɓȂBRƖB";
#else
		if (name) return "Genocide One";
		if (desc) return "Attempts to vanish a creature.";
#endif
    
		{
			int power = plev + 50;

			if (info) return info_power(power);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_ball_hide(creature_ptr, GF_GENOCIDE, dir, power, 0);
			}
		}
		break;

	case 12:
#ifdef JP
		if (name) return "ł̐n";
		if (desc) return "ɓł̑B";
#else
		if (name) return "Poison Branding";
		if (desc) return "Makes current weapon poison branded.";
#endif
    
		{
			if (cast)
			{
				brand_weapon(creature_ptr, 3);
			}
		}
		break;

	case 13:
#ifdef JP
		if (name) return "zhC";
		if (desc) return "N[`[1̂琶͂zƂBzƂ͂ɂĖxオB";
#else
		if (name) return "Vampiric Drain";
		if (desc) return "Absorbs some HP from a creature and gives them to you. You will also gain nutritional sustenance from this.";
#endif
    
		{
			int dice = 1;
			int sides = plev * 2;
			int base = plev * 2;

			if (info) return info_damage(dice, sides, base);

			if (cast)
			{
				int dam = base + diceroll(dice, sides);

				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				if (drain_life(creature_ptr, dir, dam))
				{
					hp_player(creature_ptr, dam);

					/*
					 * Gain nutritional sustenance:
					 * 150/hp drained
					 *
					 * A Food ration gives 5000
					 * food points (by contrast)
					 * Don't ever get more than
					 * "Full" this way But if we
					 * ARE Gorged, it won't cure
					 * us
					 */
					dam = creature_ptr->food + MIN(5000, 100 * dam);

					/* Not gorged already */
					if (creature_ptr->food < PY_FOOD_MAX)
						set_food(creature_ptr, dam >= PY_FOOD_MAX ? PY_FOOD_MAX - 1 : dam);
				}
			}
		}
		break;

	case 14:
#ifdef JP
		if (name) return "̏p";
		if (desc) return "͂̎̂⍜𐶂ԂB";
#else
		if (name) return "Animate dead";
		if (desc) return "Resurrects nearby corpse and skeletons. And makes these your pets.";
#endif
    
		{
			if (cast)
			{
				animate_dead(NULL, creature_ptr->fy, creature_ptr->fx);
			}
		}
		break;

	case 15:
#ifdef JP
		if (name) return "E";
		if (desc) return "w肵̃N[`[݂̊KBRƖB";
#else
		if (name) return "Genocide";
		if (desc) return "Eliminates an entire class of creature, exhausting you.  Powerful or unique creatures may resist.";
#endif
    
		{
			int power = plev+50;

			if (info) return info_power(power);

			if (cast)
			{
				symbol_genocide(creature_ptr, power, TRUE);
			}
		}
		break;

	case 16:
#ifdef JP
		if (name) return "m";
		if (desc) return "mA|B";
#else
		if (name) return "Berserk";
		if (desc) return "Gives bonus to hit and HP, immunity to fear for a while. But decreases AC.";
#endif
    
		{
			int base = 25;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_shero(creature_ptr, randint1(base) + base, FALSE);
				hp_player(creature_ptr, 30);
				set_afraid(creature_ptr, 0);
			}
		}
		break;

	case 17:
#ifdef JP
		if (name) return "쏢";
		if (desc) return "_ŗlXȌʂNB";
#else
		if (name) return "Invoke Spirits";
		if (desc) return "Causes random effects.";
#endif
    
		{
			if (info) return s_random;

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				cast_invoke_spirits(creature_ptr, dir);
			}
		}
		break;

	case 18:
#ifdef JP
		if (name) return "Í̖";
		if (desc) return "Í̃{g̓r[B";
#else
		if (name) return "Dark Bolt";
		if (desc) return "Fires a bolt or beam of darkness.";
#endif
    
		{
			int dice = 4 + (plev - 5) / 4;
			int sides = 8;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_bolt_or_beam(creature_ptr, beam_chance(creature_ptr), GF_DARK, dir, diceroll(dice, sides));
			}
		}
		break;

	case 19:
#ifdef JP
		if (name) return "m";
		if (desc) return "mA|AB";
#else
		if (name) return "Battle Frenzy";
		if (desc) return "Gives another bonus to hit and HP, immunity to fear for a while. Hastes you. But decreases AC.";
#endif
    
		{
			int b_base = 25;
			int sp_base = plev / 2;
			int sp_sides = 20 + plev / 2;

			if (info) return info_duration(b_base, b_base);

			if (cast)
			{
				set_shero(creature_ptr, randint1(25) + 25, FALSE);
				hp_player(creature_ptr, 30);
				set_afraid(creature_ptr, 0);
				set_fast(creature_ptr, randint1(sp_sides) + sp_base, FALSE);
			}
		}
		break;

	case 20:
#ifdef JP
		if (name) return "z̐n";
		if (desc) return "ɋz̑B";
#else
		if (name) return "Vampiric Branding";
		if (desc) return "Makes current weapon Vampiric.";
#endif
    
		{
			if (cast)
			{
				brand_weapon(creature_ptr, 4);
			}
		}
		break;

	case 21:
#ifdef JP
		if (name) return "^Ez";
		if (desc) return "N[`[1̂琶͂zƂBzƂ͂ɂđ̗͂񕜂B";
#else
		if (name) return "Vampirism True";
		if (desc) return "Fires 3 bolts. Each of the bolts absorbs some HP from a creature and gives them to you.";
#endif
    
		{
			int dam = 100;

			if (info) return format("%s3*%d", s_dam, dam);

			if (cast)
			{
				int i;

				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				for (i = 0; i < 3; i++)
				{
					if (drain_life(creature_ptr, dir, dam))
						hp_player(creature_ptr, dam);
				}
			}
		}
		break;

	case 22:
#ifdef JP
		if (name) return "̌";
		if (desc) return "E̐̂N[`[Ƀ_[W^B";
#else
		if (name) return "Nether Wave";
		if (desc) return "Damages all living creatures in sight.";
#endif
    
		{
			int sides = plev * 3;

			if (info) return info_damage(1, sides, 0);

			if (cast)
			{
				dispel_living(creature_ptr, randint1(sides));
			}
		}
		break;

	case 23:
#ifdef JP
		if (name) return "Í̗";
		if (desc) return "ȈÍ̋B";
#else
		if (name) return "Darkness Storm";
		if (desc) return "Fires a huge ball of darkness.";
#endif
    
		{
			int dam = 100 + plev * 2;
			int rad = 4;

			if (info) return info_damage(0, 0, dam);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_ball(creature_ptr, GF_DARK, dir, dam, rad);
			}
		}
		break;

	case 24:
#ifdef JP
		if (name) return "̌";
		if (desc) return "̌B";
#else
		if (name) return "Death Ray";
		if (desc) return "Fires a beam of death.";
#endif
    
		{
			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				death_ray(creature_ptr, dir);
			}
		}
		break;

	case 25:
#ifdef JP
		if (name) return "ҏ";
		if (desc) return "1̂̃AfbhB";
#else
		if (name) return "Raise the Dead";
		if (desc) return "Summons an undead creature.";
#endif
    
		{
			if (cast)
			{
				int type;
				bool pet = one_in_(3);
				u32b mode = 0L;

				type = (plev > 47 ? SUMMON_HI_UNDEAD : SUMMON_UNDEAD);

				if (!pet || (pet && (plev > 24) && one_in_(3)))
					mode |= PM_ALLOW_GROUP;

				if (pet) mode |= PM_FORCE_PET;
				else mode |= (PM_ALLOW_UNIQUE | PM_NO_PET);

				if (summon_specific((pet ? creature_ptr : NULL), creature_ptr->fy, creature_ptr->fx, (plev * 3) / 2, type, mode))
				{
#ifdef JP
					msg_print("₽Ȃ̎ɐn߂B͕sL^ł...");
#else
					msg_print("Cold winds begin to blow around you, carrying with them the stench of decay...");
#endif


					if (pet)
					{
#ifdef JP
						msg_print("Â̎ҋȂɎd邽ߓySI");
#else
						msg_print("Ancient, long-dead forms arise from the ground to serve you!");
#endif
					}
					else
					{
#ifdef JP
						msg_print("҂SBW邠Ȃ𔱂邽߂ɁI");
#else
						msg_print("'The dead arise... to punish you for disturbing them!'");
#endif
					}

				}
			}
		}
		break;

	case 26:
#ifdef JP
		if (name) return "҂̔`";
		if (desc) return "ACe1ʂBxƃACe̔\͂Sɒm邱ƂłB";
#else
		if (name) return "Esoteria";
		if (desc) return "Identifies an item. Or *identifies* an item at higher level.";
#endif
    
		{
			if (cast)
			{
				if (randint1(50) > plev)
				{
					if (!ident_spell(creature_ptr, FALSE)) return NULL;
				}
				else
				{
					if (!identify_fully(creature_ptr, FALSE)) return NULL;
				}
			}
		}
		break;

	case 27:
#ifdef JP
		if (name) return "zSω";
		if (desc) return "莞ԁAzSɕωBωĂԂ͖{̎푰̔\͂AɋzSƂĂ̔\͂𓾂B";
#else
		if (name) return "Polymorph Vampire";
		if (desc) return "Mimic a vampire for a while. Loses abilities of original race and gets abilities as a vampire.";
#endif
    
		{
			int base = 10 + plev / 2;

			if (info) return info_duration(base, base);

			if (cast)
			{
				//TODO set_mimic(creature_ptr, base + randint1(base), MIMIC_VAMPIRE, FALSE);
			}
		}
		break;

	case 28:
#ifdef JP
		if (name) return "͕";
		if (desc) return "ol񕜂B";
#else
		if (name) return "Restore Life";
		if (desc) return "Restore lost experience.";
#endif
    
		{
			if (cast)
			{
				restore_level(creature_ptr);
			}
		}
		break;

	case 29:
#ifdef JP
		if (name) return "ӖE";
		if (desc) return "̎͂ɂN[`[݂̊KBRƖB";
#else
		if (name) return "Mass Genocide";
		if (desc) return "Eliminates all nearby creatures, exhausting you.  Powerful or unique creatures may be able to resist.";
#endif
    
		{
			int power = plev + 50;

			if (info) return info_power(power);

			if (cast)
			{
				mass_genocide(creature_ptr, power, TRUE);
			}
		}
		break;

	case 30:
#ifdef JP
		if (name) return "n̍";
		if (desc) return "׈ȗ͂BPǂȃN[`[ɂ͑傫ȃ_[W^B";
#else
		if (name) return "Hellfire";
		if (desc) return "Fires a powerful ball of evil power. Hurts good creatures greatly.";
#endif
    
		{
			int dam = 666;
			int rad = 3;

			if (info) return info_damage(0, 0, dam);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_ball(creature_ptr, GF_HELL_FIRE, dir, dam, rad);
#ifdef JP
				take_hit(NULL, creature_ptr, DAMAGE_USELIFE, 20 + randint1(30), "n̍΂̎J", NULL, -1);
#else
				take_hit(NULL, creature_ptr, DAMAGE_USELIFE, 20 + randint1(30), "the strain of casting Hellfire", NULL, -1);
#endif
			}
		}
		break;

	case 31:
#ifdef JP
		if (name) return "H̉";
		if (desc) return "莞ԁAǂʂ蔲邱Ƃł󂯂_[WyĤ̏ԂɕϐgB";
#else
		if (name) return "Wraithform";
		if (desc) return "Becomes wraith form which gives ability to pass walls and makes all damages half.";
#endif
    
		{
			int base = plev / 2;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_wraith_form(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;
	}

	return "";
}


static cptr do_trump_spell(creature_type *creature_ptr, int spell, int mode)
{
	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool info = (mode == SPELL_INFO) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;
	bool fail = (mode == SPELL_FAIL) ? TRUE : FALSE;

#ifdef JP
	static const char s_random[] = "_";
#else
	static const char s_random[] = "random";
#endif

	int dir;
	int plev = creature_ptr->lev;

	switch (spell)
	{
	case 0:
#ifdef JP
		if (name) return "V[gEe|[g";
		if (desc) return "ߋ̃e|[gB";
#else
		if (name) return "Phase Door";
		if (desc) return "Teleport short distance.";
#endif
    
		{
			int range = 10;

			if (info) return info_range(range);

			if (cast)
			{
				teleport_player(creature_ptr, range, 0L);
			}
		}
		break;

	case 1:
#ifdef JP
		if (name) return "w偂̃J[h";
		if (desc) return "w偂B";
#else
		if (name) return "Trump Spiders";
		if (desc) return "Summons spiders.";
#endif
    
		{
			if (cast || fail)
			{
#ifdef JP
				msg_print("Ȃ͒w偂̃J[hɏW...");
#else
				msg_print("You concentrate on the trump of an spider...");
#endif

				if (trump_summoning(creature_ptr, 1, !fail, creature_ptr->fy, creature_ptr->fx, 0, SUMMON_SPIDER, PM_ALLOW_GROUP))
				{
					if (fail)
					{
#ifdef JP
						msg_print("ꂽw偂͓{ĂI");
#else
						msg_print("The summoned spiders get angry!");
#endif
					}
				}
			}
		}
		break;

	case 2:
#ifdef JP
		if (name) return "Vbt";
		if (desc) return "J[h̐肢B";
#else
		if (name) return "Shuffle";
		if (desc) return "Causes random effects.";
#endif
    
		{
			if (info) return s_random;

			if (cast)
			{
				cast_shuffle(creature_ptr);
			}
		}
		break;

	case 3:
#ifdef JP
		if (name) return "tAEZbg";
		if (desc) return "Ő[KύXB";
#else
		if (name) return "Reset Recall";
		if (desc) return "Resets the 'deepest' level for recall spell.";
#endif
    
		{
			if (cast)
			{
				if (!reset_recall(creature_ptr)) return NULL;
			}
		}
		break;

	case 4:
#ifdef JP
		if (name) return "e|[g";
		if (desc) return "̃e|[gB";
#else
		if (name) return "Teleport";
		if (desc) return "Teleport long distance.";
#endif
    
		{
			int range = plev * 4;

			if (info) return info_range(range);

			if (cast)
			{
				teleport_player(creature_ptr, range, 0L);
			}
		}
		break;

	case 5:
#ifdef JP
		if (name) return "m̃J[h";
		if (desc) return "莞ԁAepV[\͂𓾂B";
#else
		if (name) return "Trump Spying";
		if (desc) return "Gives telepathy for a while.";
#endif
    
		{
			int base = 25;
			int sides = 30;

			if (info) return info_duration(base, sides);

			if (cast)
			{
				set_tim_esp(creature_ptr, randint1(sides) + base, FALSE);
			}
		}
		break;

	case 6:
#ifdef JP
		if (name) return "e|[gEN[`[";
		if (desc) return "N[`[e|[gr[BRƖB";
#else
		if (name) return "Teleport Away";
		if (desc) return "Teleports all creatures on the line away unless resisted.";
#endif
    
		{
			int power = plev;

			if (info) return info_power(power);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_beam(creature_ptr, GF_AWAY_ALL, dir, power);
			}
		}
		break;

	case 7:
#ifdef JP
		if (name) return "̃J[h";
		if (desc) return "1̂̓B";
#else
		if (name) return "Trump Animals";
		if (desc) return "Summons an animal.";
#endif
    
		{
			if (cast || fail)
			{
				int type = (!fail ? SUMMON_ANIMAL_RANGER : SUMMON_ANIMAL);

#ifdef JP
				msg_print("Ȃ͓̃J[hɏW...");
#else
				msg_print("You concentrate on the trump of an animal...");
#endif

				if (trump_summoning(creature_ptr, 1, !fail, creature_ptr->fy, creature_ptr->fx, 0, type, 0L))
				{
					if (fail)
					{
#ifdef JP
						msg_print("ꂽ͓{ĂI");
#else
						msg_print("The summoned animal gets angry!");
#endif
					}
				}
			}
		}
		break;

	case 8:
#ifdef JP
		if (name) return "ړ̃J[h";
		if (desc) return "ACȇֈړB";
#else
		if (name) return "Trump Reach";
		if (desc) return "Pulls a distant item close to you.";
#endif
    
		{
			int weight = plev * 15;

			if (info) return info_weight(weight);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fetch(creature_ptr, dir, weight, FALSE);
			}
		}
		break;

	case 9:
#ifdef JP
		if (name) return "J~J[̃J[h";
		if (desc) return "̔N[`[B";
#else
		if (name) return "Trump Kamikaze";
		if (desc) return "Summons creatures which explode by itself.";
#endif
    
		{
			if (cast || fail)
			{
				int x, y;
				int type;

				if (cast)
				{
					if (!target_set(creature_ptr, TARGET_KILL)) return NULL;
					x = target_col;
					y = target_row;
				}
				else
				{
					/* Summons near player when failed */
					x = creature_ptr->fx;
					y = creature_ptr->fy;
				}

				if (creature_ptr->class_idx == CLASS_BEASTMASTER)
					type = SUMMON_KAMIKAZE_LIVING;
				else
					type = SUMMON_KAMIKAZE;

#ifdef JP
				msg_print("Ȃ̓J~J[̃J[hɏW...");
#else
				msg_print("You concentrate on several trumps at once...");
#endif

				if (trump_summoning(creature_ptr, 2 + randint0(plev / 7), !fail, y, x, 0, type, 0L))
				{
					if (fail)
					{
#ifdef JP
						msg_print("ꂽN[`[͓{ĂI");
#else
						msg_print("The summoned creatures get angry!");
#endif
					}
				}
			}
		}
		break;

	case 10:
#ifdef JP
		if (name) return "쏢";
		if (desc) return "1̗̂HB";
#else
		if (name) return "Phantasmal Servant";
		if (desc) return "Summons a ghost.";
#endif
    
		{
			/* Phantasmal Servant is not summoned as enemy when failed */
			if (cast)
			{
				int summon_lev = plev * 2 / 3 + randint1(plev / 2);

				if (trump_summoning(creature_ptr, 1, !fail, creature_ptr->fy, creature_ptr->fx, (summon_lev * 3 / 2), SUMMON_PHANTOM, 0L))
				{
#ifdef JP
					msg_print("pł܂AllH");
#else
					msg_print("'Your wish, master?'");
#endif
				}
			}
		}
		break;

	case 11:
#ifdef JP
		if (name) return "Xs[hEN[`[";
		if (desc) return "N[`[1̂B";
#else
		if (name) return "Haste Creature";
		if (desc) return "Hastes a creature.";
#endif
    
		{
			if (cast)
			{
				bool result;

				/* Temporary enable target_pet option */
				bool old_target_pet = target_pet;
				target_pet = TRUE;

				result = get_aim_dir(creature_ptr, &dir);

				/* Restore target_pet option */
				target_pet = old_target_pet;

				if (!result) return NULL;

				speed_other_creature(creature_ptr, dir);
			}
		}
		break;

	case 12:
#ifdef JP
		if (name) return "e|[gEx";
		if (desc) return "uɏォ̊KɃe|[gB";
#else
		if (name) return "Teleport Level";
		if (desc) return "Teleport to up or down stairs in a moment.";
#endif
    
		{
			if (cast)
			{
#ifdef JP
				if (!get_check("{ɑ̊KɃe|[g܂H")) return NULL;
#else
				if (!get_check("Are you sure? (Teleport Level)")) return NULL;
#endif
				teleport_level(creature_ptr, 0);
			}
		}
		break;

	case 13:
#ifdef JP
		if (name) return "̔";
		if (desc) return "Z̎w肵ꏊɃe|[gB";
#else
		if (name) return "Dimension Door";
		if (desc) return "Teleport to given location.";
#endif
    
		{
			int range = plev / 2 + 10;

			if (info) return info_range(range);

			if (cast)
			{
#ifdef JP
				msg_print("̔JBړInIŉB");
#else
				msg_print("You open a dimensional gate. Choose a destination.");
#endif

				if (!dimension_door(creature_ptr)) return NULL;
			}
		}
		break;

	case 14:
#ifdef JP
		if (name) return "A҂̎";
		if (desc) return "nɂƂ̓_W̍Ő[KցA_WɂƂ͒nւƈړB";
#else
		if (name) return "Word of Recall";
		if (desc) return "Recalls player from dungeon to town, or from town to the deepest level of dungeon.";
#endif
    
		{
			int base = 15;
			int sides = 20;

			if (info) return info_delay(base, sides);

			if (cast)
			{
				if (!word_of_recall(creature_ptr)) return NULL;
			}
		}
		break;

	case 15:
#ifdef JP
		if (name) return "Ǖ";
		if (desc) return "ȆSẴN[`[e|[gBRƖB";
#else
		if (name) return "Banish";
		if (desc) return "Teleports all creatures in sight away unless resisted.";
#endif
    
		{
			int power = plev * 4;

			if (info) return info_power(power);

			if (cast)
			{
				banish_creatures(creature_ptr, power);
			}
		}
		break;

	case 16:
#ifdef JP
		if (name) return "ʒũJ[h";
		if (desc) return "1̂̃N[`[ƈʒuB";
#else
		if (name) return "Swap Position";
		if (desc) return "Swap positions of you and a creature.";
#endif
    
		{
			if (cast)
			{
				bool result;

				/* HACK -- No range limit */
				project_length = -1;

				result = get_aim_dir(creature_ptr, &dir);

				/* Restore range to default */
				project_length = 0;

				if (!result) return NULL;

				teleport_swap(creature_ptr, dir);
			}
		}
		break;

	case 17:
#ifdef JP
		if (name) return "Afbh̃J[h";
		if (desc) return "1̂̃AfbhB";
#else
		if (name) return "Trump Undead";
		if (desc) return "Summons an undead creature.";
#endif
    
		{
			if (cast || fail)
			{
#ifdef JP
				msg_print("Ȃ̓Afbh̃J[hɏW...");
#else
				msg_print("You concentrate on the trump of an undead creature...");
#endif

				if (trump_summoning(creature_ptr, 1, !fail, creature_ptr->fy, creature_ptr->fx, 0, SUMMON_UNDEAD, 0L))
				{
					if (fail)
					{
#ifdef JP
						msg_print("ꂽAfbh͓{ĂI");
#else
						msg_print("The summoned undead creature gets angry!");
#endif
					}
				}
			}
		}
		break;

	case 18:
#ifdef JP
		if (name) return "঒ނ̃J[h";
		if (desc) return "1̂̃qhB";
#else
		if (name) return "Trump Reptiles";
		if (desc) return "Summons a hydra.";
#endif
    
		{
			if (cast || fail)
			{
#ifdef JP
				msg_print("Ȃ঒ނ̃J[hɏW...");
#else
				msg_print("You concentrate on the trump of a reptile...");
#endif

				if (trump_summoning(creature_ptr, 1, !fail, creature_ptr->fy, creature_ptr->fx, 0, SUMMON_HYDRA, 0L))
				{
					if (fail)
					{
#ifdef JP
						msg_print("ꂽ঒ނ͓{ĂI");
#else
						msg_print("The summoned reptile gets angry!");
#endif
					}
				}
			}
		}
		break;

	case 19:
#ifdef JP
		if (name) return "N[`[̃J[h";
		if (desc) return "̃N[`[B";
#else
		if (name) return "Trump Creatures";
		if (desc) return "Summons some creatures.";
#endif
    
		{
			if (cast || fail)
			{
				int type;

#ifdef JP
				msg_print("Ȃ̓N[`[̃J[hɏW...");
#else
				msg_print("You concentrate on several trumps at once...");
#endif

				if (creature_ptr->class_idx == CLASS_BEASTMASTER)
					type = SUMMON_LIVING;
				else
					type = 0;

				if (trump_summoning(creature_ptr, (1 + (plev - 15)/ 10), !fail, creature_ptr->fy, creature_ptr->fx, 0, type, 0L))
				{
					if (fail)
					{
#ifdef JP
						msg_print("ꂽN[`[͓{ĂI");
#else
						msg_print("The summoned creatures get angry!");
#endif
					}
				}

			}
		}
		break;

	case 20:
#ifdef JP
		if (name) return "nEh̃J[h";
		if (desc) return "1O[ṽnEhB";
#else
		if (name) return "Trump Hounds";
		if (desc) return "Summons a group of hounds.";
#endif
    
		{
			if (cast || fail)
			{
#ifdef JP
				msg_print("Ȃ̓nEh̃J[hɏW...");
#else
				msg_print("You concentrate on the trump of a hound...");
#endif

				if (trump_summoning(creature_ptr, 1, !fail, creature_ptr->fy, creature_ptr->fx, 0, SUMMON_HOUND, PM_ALLOW_GROUP))
				{
					if (fail)
					{
#ifdef JP
						msg_print("ꂽnEh͓{ĂI");
#else
						msg_print("The summoned hounds get angry!");
#endif
					}
				}
			}
		}
		break;

	case 21:
#ifdef JP
		if (name) return "gv̐n";
		if (desc) return "Ƀgv̑B";
#else
		if (name) return "Trump Branding";
		if (desc) return "Makes current weapon a Trump weapon.";
#endif
    
		{
			if (cast)
			{
				brand_weapon(creature_ptr, 5);
			}
		}
		break;

	case 22:
#ifdef JP
		if (name) return "lԃgv";
		if (desc) return "_Ƀe|[gˑRψقÄӎvŃe|[gˑRψقgɂB";
#else
		if (name) return "Living Trump";
		if (desc) return "Gives mutation which makes you teleport randomly or makes you able to teleport at will.";
#endif
    
		{
			if (cast)
			{
				int mutation;

				if (one_in_(7))
					/* Teleport control */
					mutation = 12;
				else
					/* Random teleportation (uncontrolled) */
					mutation = 77;

				/* Gain the mutation */
				if (gain_trait(creature_ptr, mutation, TRUE))
				{
#ifdef JP
					msg_print("Ȃ͐ĂJ[hɕςB");
#else
					msg_print("You have turned into a Living Trump.");
#endif
				}
			}
		}
		break;

	case 23:
#ifdef JP
		if (name) return "TCo[f[̃J[h";
		if (desc) return "1̂̃TCo[f[B";
#else
		if (name) return "Trump Cyberdemon";
		if (desc) return "Summons a cyber demon.";
#endif
    
		{
			if (cast || fail)
			{
#ifdef JP
				msg_print("Ȃ̓TCo[f[̃J[hɏW...");
#else
				msg_print("You concentrate on the trump of a Cyberdemon...");
#endif

				if (trump_summoning(creature_ptr, 1, !fail, creature_ptr->fy, creature_ptr->fx, 0, SUMMON_CYBER, 0L))
				{
					if (fail)
					{
#ifdef JP
						msg_print("ꂽTCo[f[͓{ĂI");
#else
						msg_print("The summoned Cyberdemon gets angry!");
#endif
					}
				}
			}
		}
		break;

	case 24:
#ifdef JP
		if (name) return "\̃J[h";
		if (desc) return "߂̑SẴN[`[A㩁AAKiAAăACemB";
#else
		if (name) return "Trump Divination";
		if (desc) return "Detects all creatures, traps, doors, stairs, treasures and items in your vicinity.";
#endif
    
		{
			int rad = DETECT_RAD_DEFAULT;

			if (info) return info_radius(rad);

			if (cast)
			{
				detect_all(creature_ptr, rad);
			}
		}
		break;

	case 25:
#ifdef JP
		if (name) return "m̃J[h";
		if (desc) return "ACe̎\͂SɒmB";
#else
		if (name) return "Trump Lore";
		if (desc) return "*Identifies* an item.";
#endif
    
		{
			if (cast)
			{
				if (!identify_fully(creature_ptr, FALSE)) return NULL;
			}
		}
		break;

	case 26:
#ifdef JP
		if (name) return "񕜃N[`[";
		if (desc) return "N[`[1̗̂̑͂񕜂B";
#else
		if (name) return "Heal Creature";
		if (desc) return "Heal a creature.";
#endif
    
		{
			int heal = plev * 10 + 200;

			if (info) return info_heal(0, 0, heal);

			if (cast)
			{
				bool result;

				/* Temporary enable target_pet option */
				bool old_target_pet = target_pet;
				target_pet = TRUE;

				result = get_aim_dir(creature_ptr, &dir);

				/* Restore target_pet option */
				target_pet = old_target_pet;

				if (!result) return NULL;

				heal_other_creature(creature_ptr, dir, heal);
			}
		}
		break;

	case 27:
#ifdef JP
		if (name) return "hS̃J[h";
		if (desc) return "1̂̃hSB";
#else
		if (name) return "Trump Dragon";
		if (desc) return "Summons a dragon.";
#endif
    
		{
			if (cast || fail)
			{
#ifdef JP
				msg_print("Ȃ̓hS̃J[hɏW...");
#else
				msg_print("You concentrate on the trump of a dragon...");
#endif

				if (trump_summoning(creature_ptr, 1, !fail, creature_ptr->fy, creature_ptr->fx, 0, SUMMON_DRAGON, 0L))
				{
					if (fail)
					{
#ifdef JP
						msg_print("ꂽhS͓{ĂI");
#else
						msg_print("The summoned dragon gets angry!");
#endif
					}
				}
			}
		}
		break;

	case 28:
#ifdef JP
		if (name) return "覐΂̃J[h";
		if (desc) return "̎ӂ覐΂𗎂ƂB";
#else
		if (name) return "Trump Meteor";
		if (desc) return "Makes meteor balls fall down to nearby random locations.";
#endif
    
		{
			int dam = plev * 2;
			int rad = 2;

			if (info) return info_multi_damage(dam);

			if (cast)
			{
				cast_meteor(creature_ptr, dam, rad);
			}
		}
		break;

	case 29:
#ifdef JP
		if (name) return "f[̃J[h";
		if (desc) return "1̂̈B";
#else
		if (name) return "Trump Demon";
		if (desc) return "Summons a demon.";
#endif
    
		{
			if (cast || fail)
			{
#ifdef JP
				msg_print("Ȃ̓f[̃J[hɏW...");
#else
				msg_print("You concentrate on the trump of a demon...");
#endif

				if (trump_summoning(creature_ptr, 1, !fail, creature_ptr->fy, creature_ptr->fx, 0, SUMMON_DEMON, 0L))
				{
					if (fail)
					{
#ifdef JP
						msg_print("ꂽf[͓{ĂI");
#else
						msg_print("The summoned demon gets angry!");
#endif
					}
				}
			}
		}
		break;

	case 30:
#ifdef JP
		if (name) return "ñJ[h";
		if (desc) return "1̂̏㋉AfbhB";
#else
		if (name) return "Trump Greater Undead";
		if (desc) return "Summons a greater undead.";
#endif
    
		{
			if (cast || fail)
			{
#ifdef JP
				msg_print("Ȃ͋͂ȃAfbh̃J[hɏW...");
#else
				msg_print("You concentrate on the trump of a greater undead being...");
#endif
				/* May allow unique depend on level and dice roll */
				if (trump_summoning(creature_ptr, 1, !fail, creature_ptr->fy, creature_ptr->fx, 0, SUMMON_HI_UNDEAD, PM_ALLOW_UNIQUE))
				{
					if (fail)
					{
#ifdef JP
						msg_print("ꂽ㋉Afbh͓{ĂI");
#else
						msg_print("The summoned greater undead creature gets angry!");
#endif
					}
				}
			}
		}
		break;

	case 31:
#ifdef JP
		if (name) return "ÑhS̃J[h";
		if (desc) return "1̂̌ÑhSB";
#else
		if (name) return "Trump Ancient Dragon";
		if (desc) return "Summons an ancient dragon.";
#endif
    
		{
			if (cast)
			{
				int type;

				if (creature_ptr->class_idx == CLASS_BEASTMASTER)
					type = SUMMON_HI_DRAGON_LIVING;
				else
					type = SUMMON_HI_DRAGON;

#ifdef JP
				msg_print("Ȃ͌ÑhS̃J[hɏW...");
#else
				msg_print("You concentrate on the trump of an ancient dragon...");
#endif

				/* May allow unique depend on level and dice roll */
				if (trump_summoning(creature_ptr, 1, !fail, creature_ptr->fy, creature_ptr->fx, 0, type, PM_ALLOW_UNIQUE))
				{
					if (fail)
					{
#ifdef JP
						msg_print("ꂽÑhS͓{ĂI");
#else
						msg_print("The summoned ancient dragon gets angry!");
#endif
					}
				}
			}
		}
		break;
	}

	return "";
}


static cptr do_arcane_spell(creature_type *creature_ptr, int spell, int mode)
{
	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool info = (mode == SPELL_INFO) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;
	floor_type *floor_ptr = get_floor_ptr(creature_ptr);

	int dir;
	int plev = creature_ptr->lev;

	switch (spell)
	{
	case 0:
#ifdef JP
		if (name) return "d";
		if (desc) return "d̃{g̓r[B";
#else
		if (name) return "Zap";
		if (desc) return "Fires a bolt or beam of lightning.";
#endif
    
		{
			int dice = 3 + (plev - 1) / 5;
			int sides = 3;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_bolt_or_beam(creature_ptr, beam_chance(creature_ptr) - 10, GF_ELEC, dir, diceroll(dice, sides));
			}
		}
		break;

	case 1:
#ifdef JP
		if (name) return "@̎{";
		if (desc) return "ɌB";
#else
		if (name) return "Wizard Lock";
		if (desc) return "Locks a door.";
#endif
    
		{
			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				wizard_lock(creature_ptr, dir);
			}
		}
		break;

	case 2:
#ifdef JP
		if (name) return "̊m";
		if (desc) return "߂̓ȃN[`[mB";
#else
		if (name) return "Detect Invisibility";
		if (desc) return "Detects all invisible creatures in your vicinity.";
#endif
    
		{
			int rad = DETECT_RAD_DEFAULT;

			if (info) return info_radius(rad);

			if (cast)
			{
				detect_creatures_invis(creature_ptr, rad);
			}
		}
		break;

	case 3:
#ifdef JP
		if (name) return "N[`[m";
		if (desc) return "߂̑SĂ̌N[`[mB";
#else
		if (name) return "Detect Creatures";
		if (desc) return "Detects all creatures in your vicinity unless invisible.";
#endif
    
		{
			int rad = DETECT_RAD_DEFAULT;

			if (info) return info_radius(rad);

			if (cast)
			{
				detect_creatures_normal(creature_ptr, rad);
			}
		}
		break;

	case 4:
#ifdef JP
		if (name) return "V[gEe|[g";
		if (desc) return "ߋ̃e|[gB";
#else
		if (name) return "Blink";
		if (desc) return "Teleport short distance.";
#endif
    
		{
			int range = 10;

			if (info) return info_range(range);

			if (cast)
			{
				teleport_player(creature_ptr, range, 0L);
			}
		}
		break;

	case 5:
#ifdef JP
		if (name) return "CgEGA";
		if (desc) return "Ƃ炵Ă͈͂Ŝivɖ邭B";
#else
		if (name) return "Light Area";
		if (desc) return "Lights up nearby area and the inside of a room permanently.";
#endif
    
		{
			int dice = 2;
			int sides = plev / 2;
			int rad = plev / 10 + 1;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				lite_area(creature_ptr, diceroll(dice, sides), rad);
			}
		}
		break;

	case 6:
#ifdef JP
		if (name) return "㩂Ɣ j";
		if (desc) return "꒼̑SĂ㩂Ɣj󂷂B";
#else
		if (name) return "Trap & Door Destruction";
		if (desc) return "Fires a beam which destroy traps and doors.";
#endif
    
		{
			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				destroy_door(creature_ptr, dir);
			}
		}
		break;

	case 7:
#ifdef JP
		if (name) return "y̎";
		if (desc) return "Ƒ̗͂񕜂B";
#else
		if (name) return "Cure Light Wounds";
		if (desc) return "Heals cut and HP a little.";
#endif
    
		{
			int dice = 2;
			int sides = 8;

			if (info) return info_heal(dice, sides, 0);

			if (cast)
			{
				hp_player(creature_ptr, diceroll(dice, sides));
				set_cut(creature_ptr, creature_ptr->cut - 10);
			}
		}
		break;

	case 8:
#ifdef JP
		if (name) return "㩂Ɣ m";
		if (desc) return "߂̑SĂ㩂ƔƊKimB";
#else
		if (name) return "Detect Doors & Traps";
		if (desc) return "Detects traps, doors, and stairs in your vicinity.";
#endif
    
		{
			int rad = DETECT_RAD_DEFAULT;

			if (info) return info_radius(rad);

			if (cast)
			{
				detect_traps(creature_ptr, rad, TRUE);
				detect_doors(creature_ptr, rad);
				detect_stairs(creature_ptr, rad);
			}
		}
		break;

	case 9:
#ifdef JP
		if (name) return "Rf";
		if (desc) return "ɔR⋋B";
#else
		if (name) return "Phlogiston";
		if (desc) return "Adds more turns of light to a lantern or torch.";
#endif
    
		{
			if (cast)
			{
				phlogiston(creature_ptr);
			}
		}
		break;

	case 10:
#ifdef JP
		if (name) return "󊴒m";
		if (desc) return "߂̍mB";
#else
		if (name) return "Detect Treasure";
		if (desc) return "Detects all treasures in your vicinity.";
#endif
    
		{
			int rad = DETECT_RAD_DEFAULT;

			if (info) return info_radius(rad);

			if (cast)
			{
				detect_treasure(creature_ptr, rad);
				detect_objects_gold(creature_ptr, rad);
			}
		}
		break;

	case 11:
#ifdef JP
		if (name) return "@ m";
		if (desc) return "߂̖@ACemB";
#else
		if (name) return "Detect Enchantment";
		if (desc) return "Detects all magical items in your vicinity.";
#endif
    
		{
			int rad = DETECT_RAD_DEFAULT;

			if (info) return info_radius(rad);

			if (cast)
			{
				detect_objects_magic(creature_ptr, rad);
			}
		}
		break;

	case 12:
#ifdef JP
		if (name) return "ACem";
		if (desc) return "߂̑SẴACemB";
#else
		if (name) return "Detect Objects";
		if (desc) return "Detects all items in your vicinity.";
#endif
    
		{
			int rad = DETECT_RAD_DEFAULT;

			if (info) return info_radius(rad);

			if (cast)
			{
				detect_objects_normal(creature_ptr, rad);
			}
		}
		break;

	case 13:
#ifdef JP
		if (name) return "";
		if (desc) return "ł̓犮SɎ菜B";
#else
		if (name) return "Cure Poison";
		if (desc) return "Cures poison status.";
#endif
    
		{
			if (cast)
			{
				set_poisoned(creature_ptr, 0);
			}
		}
		break;

	case 14:
#ifdef JP
		if (name) return "ϗ";
		if (desc) return "莞ԁACւ̑ϐ𓾂BɂϐɗݐςB";
#else
		if (name) return "Resist Cold";
		if (desc) return "Gives resistance to cold. This resistance can be added to which from equipment for more powerful resistance.";
#endif
    
		{
			int base = 20;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_oppose_cold(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 15:
#ifdef JP
		if (name) return "ω";
		if (desc) return "莞ԁAւ̑ϐ𓾂BɂϐɗݐςB";
#else
		if (name) return "Resist Fire";
		if (desc) return "Gives resistance to fire. This resistance can be added to which from equipment for more powerful resistance.";
#endif
    
		{
			int base = 20;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_oppose_fire(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 16:
#ifdef JP
		if (name) return "ϓd";
		if (desc) return "莞ԁAdւ̑ϐ𓾂BɂϐɗݐςB";
#else
		if (name) return "Resist Lightning";
		if (desc) return "Gives resistance to electricity. This resistance can be added to which from equipment for more powerful resistance.";
#endif
    
		{
			int base = 20;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_oppose_elec(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 17:
#ifdef JP
		if (name) return "ώ_";
		if (desc) return "莞ԁA_ւ̑ϐ𓾂BɂϐɗݐςB";
#else
		if (name) return "Resist Acid";
		if (desc) return "Gives resistance to acid. This resistance can be added to which from equipment for more powerful resistance.";
#endif
    
		{
			int base = 20;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_oppose_acid(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 18:
#ifdef JP
		if (name) return "d̎";
		if (desc) return "Ƒ̗͂𒆒x񕜂B";
#else
		if (name) return "Cure Medium Wounds";
		if (desc) return "Heals cut and HP more.";
#endif
    
		{
			int dice = 4;
			int sides = 8;

			if (info) return info_heal(dice, sides, 0);

			if (cast)
			{
				hp_player(creature_ptr, diceroll(dice, sides));
				set_cut(creature_ptr, (creature_ptr->cut / 2) - 50);
			}
		}
		break;

	case 19:
#ifdef JP
		if (name) return "e|[g";
		if (desc) return "̃e|[gB";
#else
		if (name) return "Teleport";
		if (desc) return "Teleport long distance.";
#endif
    
		{
			int range = plev * 5;

			if (info) return info_range(range);

			if (cast)
			{
				teleport_player(creature_ptr, range, 0L);
			}
		}
		break;

	case 20:
#ifdef JP
		if (name) return "Ӓ";
		if (desc) return "ACeʂB";
#else
		if (name) return "Identify";
		if (desc) return "Identifies an item.";
#endif
    
		{
			if (cast)
			{
				if (!ident_spell(creature_ptr, FALSE)) return NULL;
			}
		}
		break;

	case 21:
#ifdef JP
		if (name) return "Ηn";
		if (desc) return "ǂnďɂB";
#else
		if (name) return "Stone to Mud";
		if (desc) return "Turns one rock square to mud.";
#endif
    
		{
			int dice = 1;
			int sides = 30;
			int base = 20;

			if (info) return info_damage(dice, sides, base);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				wall_to_mud(creature_ptr, dir);
			}
		}
		break;

	case 22:
#ifdef JP
		if (name) return "M";
		if (desc) return "BN[`[ɌʂB";
#else
		if (name) return "Ray of Light";
		if (desc) return "Fires a beam of light which damages to light-sensitive creatures.";
#endif
    
		{
			int dice = 6;
			int sides = 8;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

#ifdef JP
				msg_print("ꂽB");
#else
				msg_print("A line of light appears.");
#endif

				lite_line(creature_ptr, dir);
			}
		}
		break;

	case 23:
#ifdef JP
		if (name) return "󕠏[";
		if (desc) return "ɂB";
#else
		if (name) return "Satisfy Hunger";
		if (desc) return "Satisfies hunger.";
#endif
    
		{
			if (cast)
			{
				set_food(creature_ptr, PY_FOOD_MAX - 1);
			}
		}
		break;

	case 24:
#ifdef JP
		if (name) return "F";
		if (desc) return "莞ԁAȂ̂悤ɂȂB";
#else
		if (name) return "See Invisible";
		if (desc) return "Gives see invisible for a while.";
#endif
    
		{
			int base = 24;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_tim_invis(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 25:
#ifdef JP
		if (name) return "G^";
		if (desc) return "1̂̃G^B";
#else
		if (name) return "Conjure Elemental";
		if (desc) return "Summons an elemental.";
#endif
    
		{
			if (cast)
			{
				if (!summon_specific(NULL, creature_ptr->fy, creature_ptr->fx, plev, SUMMON_ELEMENTAL, (PM_ALLOW_GROUP | PM_FORCE_PET)))
				{
#ifdef JP
					msg_print("G^͌ȂB");
#else
					msg_print("No Elementals arrive.");
#endif
				}
			}
		}
		break;

	case 26:
#ifdef JP
		if (name) return "e|[gEx";
		if (desc) return "uɏォ̊KɃe|[gB";
#else
		if (name) return "Teleport Level";
		if (desc) return "Teleport to up or down stairs in a moment.";
#endif
    
		{
			if (cast)
			{
#ifdef JP
				if (!get_check("{ɑ̊KɃe|[g܂H")) return NULL;
#else
				if (!get_check("Are you sure? (Teleport Level)")) return NULL;
#endif
				teleport_level(creature_ptr, 0);
			}
		}
		break;

	case 27:
#ifdef JP
		if (name) return "e|[gEN[`[";
		if (desc) return "N[`[e|[gr[BRƖB";
#else
		if (name) return "Teleport Away";
		if (desc) return "Teleports all creatures on the line away unless resisted.";
#endif
    
		{
			int power = plev;

			if (info) return info_power(power);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_beam(creature_ptr, GF_AWAY_ALL, dir, power);
			}
		}
		break;

	case 28:
#ifdef JP
		if (name) return "f̋";
		if (desc) return "AdACA_̂ǂꂩ̋B";
#else
		if (name) return "Elemental Ball";
		if (desc) return "Fires a ball of some elements.";
#endif
    
		{
			int dam = 75 + plev;
			int rad = 2;

			if (info) return info_damage(0, 0, dam);

			if (cast)
			{
				int type;

				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				switch (randint1(4))
				{
					case 1:  type = GF_FIRE;break;
					case 2:  type = GF_ELEC;break;
					case 3:  type = GF_COLD;break;
					default: type = GF_ACID;break;
				}

				fire_ball(creature_ptr, type, dir, dam, rad);
			}
		}
		break;

	case 29:
#ifdef JP
		if (name) return "Sm";
		if (desc) return "߂̑SẴN[`[A㩁AAKiAAăACemB";
#else
		if (name) return "Detection";
		if (desc) return "Detects all creatures, traps, doors, stairs, treasures and items in your vicinity.";
#endif
    
		{
			int rad = DETECT_RAD_DEFAULT;

			if (info) return info_radius(rad);

			if (cast)
			{
				detect_all(creature_ptr, rad);
			}
		}
		break;

	case 30:
#ifdef JP
		if (name) return "A҂̎";
		if (desc) return "nɂƂ̓_W̍Ő[KցA_WɂƂ͒nւƈړB";
#else
		if (name) return "Word of Recall";
		if (desc) return "Recalls player from dungeon to town, or from town to the deepest level of dungeon.";
#endif
    
		{
			int base = 15;
			int sides = 20;

			if (info) return info_delay(base, sides);

			if (cast)
			{
				if (!word_of_recall(creature_ptr)) return NULL;
			}
		}
		break;

	case 31:
#ifdef JP
		if (name) return "痢";
		if (desc) return "̊KŜivɏƂ炵A_WׂẴACemBɁA莞ԃepV[\͂𓾂B";
#else
		if (name) return "Clairvoyance";
		if (desc) return "Maps and lights whole dungeon level. Knows all objects location. And gives telepathy for a while.";
#endif
    
		{
			int base = 25;
			int sides = 30;

			if (info) return info_duration(base, sides);

			if (cast)
			{
				wiz_lite(floor_ptr, creature_ptr, FALSE);

				if (!has_trait(creature_ptr, TRAIT_ESP))
				{
					set_tim_esp(creature_ptr, randint1(sides) + base, FALSE);
				}
			}
		}
		break;
	}

	return "";
}


static cptr do_craft_spell(creature_type *creature_ptr, int spell, int mode)
{
	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool info = (mode == SPELL_INFO) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;

	int plev = creature_ptr->lev;

	switch (spell)
	{
	case 0:
#ifdef JP
		if (name) return "ԊO";
		if (desc) return "莞ԁAԊO͂B";
#else
		if (name) return "Infravision";
		if (desc) return "Gives infravision for a while.";
#endif
    
		{
			int base = 100;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_tim_infra(creature_ptr, base + randint1(base), FALSE);
			}
		}
		break;

	case 1:
#ifdef JP
		if (name) return "񕜗͋";
		if (desc) return "莞ԁA񕜗͂B";
#else
		if (name) return "Regeneration";
		if (desc) return "Gives regeneration ability for a while.";
#endif
    
		{
			int base = 80;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_tim_regen(creature_ptr, base + randint1(base), FALSE);
			}
		}
		break;

	case 2:
#ifdef JP
		if (name) return "󕠏[";
		if (desc) return "ɂȂB";
#else
		if (name) return "Satisfy Hunger";
		if (desc) return "Satisfies hunger.";
#endif
    
		{
			if (cast)
			{
				set_food(creature_ptr, PY_FOOD_MAX - 1);
			}
		}
		break;

	case 3:
#ifdef JP
		if (name) return "ϗC";
		if (desc) return "莞ԁACւ̑ϐ𓾂BɂϐɗݐςB";
#else
		if (name) return "Resist Cold";
		if (desc) return "Gives resistance to cold. This resistance can be added to which from equipment for more powerful resistance.";
#endif
    
		{
			int base = 20;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_oppose_cold(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 4:
#ifdef JP
		if (name) return "ωΉ";
		if (desc) return "莞ԁAւ̑ϐ𓾂BɂϐɗݐςB";
#else
		if (name) return "Resist Fire";
		if (desc) return "Gives resistance to fire. This resistance can be added to which from equipment for more powerful resistance.";
#endif
    
		{
			int base = 20;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_oppose_fire(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 5:
#ifdef JP
		if (name) return "mCg";
		if (desc) return "莞ԁAq[[CɂȂB";
#else
		if (name) return "Heroism";
		if (desc) return "Removes fear, and gives bonus to hit and 10 more HP for a while.";
#endif
    
		{
			int base = 25;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_hero(creature_ptr, randint1(base) + base, FALSE);
				hp_player(creature_ptr, 10);
				set_afraid(creature_ptr, 0);
			}
		}
		break;

	case 6:
#ifdef JP
		if (name) return "ϓd";
		if (desc) return "莞ԁAdւ̑ϐ𓾂BɂϐɗݐςB";
#else
		if (name) return "Resist Lightning";
		if (desc) return "Gives resistance to electricity. This resistance can be added to which from equipment for more powerful resistance.";
#endif
    
		{
			int base = 20;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_oppose_elec(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 7:
#ifdef JP
		if (name) return "ώ_";
		if (desc) return "莞ԁA_ւ̑ϐ𓾂BɂϐɗݐςB";
#else
		if (name) return "Resist Acid";
		if (desc) return "Gives resistance to acid. This resistance can be added to which from equipment for more powerful resistance.";
#endif
    
		{
			int base = 20;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_oppose_acid(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 8:
#ifdef JP
		if (name) return "F";
		if (desc) return "莞ԁAȂ̂悤ɂȂB";
#else
		if (name) return "See Invisibility";
		if (desc) return "Gives see invisible for a while.";
#endif
    
		{
			int base = 24;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_tim_invis(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 9:
#ifdef JP
		if (name) return "";
		if (desc) return "ACeɂア􂢂B";
#else
		if (name) return "Remove Curse";
		if (desc) return "Removes normal curses from equipped items.";
#endif
    
		{
			if (cast)
			{
				if (remove_curse(creature_ptr))
				{
#ifdef JP
					msg_print("NɌĂ悤ȋCB");
#else
					msg_print("You feel as if someone is watching over you.");
#endif
				}
			}
		}
		break;

	case 10:
#ifdef JP
		if (name) return "ϓ";
		if (desc) return "莞ԁAłւ̑ϐ𓾂BɂϐɗݐςB";
#else
		if (name) return "Resist Poison";
		if (desc) return "Gives resistance to poison. This resistance can be added to which from equipment for more powerful resistance.";
#endif
    
		{
			int base = 20;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_oppose_pois(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 11:
#ifdef JP
		if (name) return "m";
		if (desc) return "mA|B";
#else
		if (name) return "Berserk";
		if (desc) return "Gives bonus to hit and HP, immunity to fear for a while. But decreases AC.";
#endif
    
		{
			int base = 25;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_shero(creature_ptr, randint1(base) + base, FALSE);
				hp_player(creature_ptr, 30);
				set_afraid(creature_ptr, 0);
			}
		}
		break;

	case 12:
#ifdef JP
		if (name) return "ȕ";
		if (desc) return "݂̎̏ԂSɒmB";
#else
		if (name) return "Self Knowledge";
		if (desc) return "Gives you useful info regarding your current resistances, the powers of your weapon and maximum limits of your stats.";
#endif
    
		{
			if (cast)
			{
				creature_knowledge(creature_ptr);
			}
		}
		break;

	case 13:
#ifdef JP
		if (name) return "Ύ׈E";
		if (desc) return "׈ȃN[`[̍UhoA𒣂B";
#else
		if (name) return "Protection from Evil";
		if (desc) return "Gives aura which protect you from evil creature's physical attack.";
#endif
    
		{
			int base = 3 * plev;
			int sides = 25;

			if (info) return info_duration(base, sides);

			if (cast)
			{
				set_protevil(creature_ptr, randint1(sides) + base, FALSE);
			}
		}
		break;

	case 14:
#ifdef JP
		if (name) return "";
		if (desc) return "ŁANOԁASAo𒼂B";
#else
		if (name) return "Cure";
		if (desc) return "Heals poison, stun, cut and hallucination completely.";
#endif
    
		{
			if (cast)
			{
				set_poisoned(creature_ptr, 0);
				set_stun(creature_ptr, 0);
				set_cut(creature_ptr, 0);
				set_image(creature_ptr, 0);
			}
		}
		break;

	case 15:
#ifdef JP
		if (name) return "@";
		if (desc) return "莞ԁAɗCAAdA_Ał̂ꂩ̑BȂƎgȂB";
#else
		if (name) return "Mana Branding";
		if (desc) return "Makes current weapon some elemental branded. You must wield weapons.";
#endif
    
		{
			int base = plev / 2;

			if (info) return info_duration(base, base);

			if (cast)
			{
				if (!choose_ele_attack(creature_ptr)) return NULL;
			}
		}
		break;

	case 16:
#ifdef JP
		if (name) return "epV[";
		if (desc) return "莞ԁAepV[\͂𓾂B";
#else
		if (name) return "Telepathy";
		if (desc) return "Gives telepathy for a while.";
#endif
    
		{
			int base = 25;
			int sides = 30;

			if (info) return info_duration(base, sides);

			if (cast)
			{
				set_tim_esp(creature_ptr, randint1(sides) + base, FALSE);
			}
		}
		break;

	case 17:
#ifdef JP
		if (name) return "Ή";
		if (desc) return "莞ԁAAC㏸B";
#else
		if (name) return "Stone Skin";
		if (desc) return "Gives bonus to AC for a while.";
#endif
    
		{
			int base = 30;
			int sides = 20;

			if (info) return info_duration(base, sides);

			if (cast)
			{
				set_shield(creature_ptr, randint1(sides) + base, FALSE);
			}
		}
		break;

	case 18:
#ifdef JP
		if (name) return "Sϐ";
		if (desc) return "莞ԁA_AdAACAłɑ΂ϐ𓾂BɂϐɗݐςB";
#else
		if (name) return "Resistance";
		if (desc) return "Gives resistance to fire, cold, electricity, acid and poison for a while. These resistances can be added to which from equipment for more powerful resistances.";
#endif
    
		{
			int base = 20;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_oppose_acid(creature_ptr, randint1(base) + base, FALSE);
				set_oppose_elec(creature_ptr, randint1(base) + base, FALSE);
				set_oppose_fire(creature_ptr, randint1(base) + base, FALSE);
				set_oppose_cold(creature_ptr, randint1(base) + base, FALSE);
				set_oppose_pois(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 19:
#ifdef JP
		if (name) return "Xs[h";
		if (desc) return "莞ԁAB";
#else
		if (name) return "Haste Self";
		if (desc) return "Hastes you for a while.";
#endif
    
		{
			int base = plev;
			int sides = 20 + plev;

			if (info) return info_duration(base, sides);

			if (cast)
			{
				set_fast(creature_ptr, randint1(sides) + base, FALSE);
			}
		}
		break;

	case 20:
#ifdef JP
		if (name) return "ǔ";
		if (desc) return "莞ԁAǂʂ蔲悤ɂȂB";
#else
		if (name) return "Walk through Wall";
		if (desc) return "Gives ability to pass walls for a while.";
#endif
    
		{
			int base = plev / 2;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_kabenuke(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 21:
#ifdef JP
		if (name) return "";
		if (desc) return "ɔ˂̑B";
#else
		if (name) return "Polish Shield";
		if (desc) return "Makes a shield a shield of reflection.";
#endif
    
		{
			if (cast)
			{
				pulish_shield(creature_ptr);
			}
		}
		break;

	case 22:
#ifdef JP
		if (name) return "S[";
		if (desc) return "1̂̃S[𐻑B";
#else
		if (name) return "Create Golem";
		if (desc) return "Creates a golem.";
#endif
    
		{
			if (cast)
			{
				if (summon_specific(NULL, creature_ptr->fy, creature_ptr->fx, plev, SUMMON_GOLEM, PM_FORCE_PET))
				{
#ifdef JP
					msg_print("S[B");
#else
					msg_print("You make a golem.");
#endif
				}
				else
				{
#ifdef JP
					msg_print("܂S[ȂB");
#else
					msg_print("No Golems arrive.");
#endif
				}
			}
		}
		break;

	case 23:
#ifdef JP
		if (name) return "@̊Z";
		if (desc) return "莞ԁA@h͂ACオAƖӖڂ̑ϐA˔\́Aგm炸AV𓾂B";
#else
		if (name) return "Magical armor";
		if (desc) return "Gives resistance to magic, bonus to AC, resistance to confusion, blindness, reflection, free action and levitation for a while.";
#endif
    
		{
			int base = 20;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_magicdef(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 24:
#ifdef JP
		if (name) return "͉";
		if (desc) return "Ehɂꂽ閂͂SɉB";
#else
		if (name) return "Remove Enchantment";
		if (desc) return "Removes all magics completely from any weapon or armor.";
#endif
    
		{
			if (cast)
			{
				if (!mundane_spell(creature_ptr, TRUE)) return NULL;
			}
		}
		break;

	case 25:
#ifdef JP
		if (name) return "􂢕";
		if (desc) return "ACeɂ͂Ȏ􂢂B";
#else
		if (name) return "Remove All Curse";
		if (desc) return "Removes normal and heavy curse from equipped items.";
#endif
    
		{
			if (cast)
			{
				if (remove_all_curse(creature_ptr))
				{
#ifdef JP
					msg_print("NɌĂ悤ȋCB");
#else
					msg_print("You feel as if someone is watching over you.");
#endif
				}
			}
		}
		break;

	case 26:
#ifdef JP
		if (name) return "SȂm";
		if (desc) return "ACe̎\͂SɒmB";
#else
		if (name) return "Knowledge True";
		if (desc) return "*Identifies* an item.";
#endif
    
		{
			if (cast)
			{
				if (!identify_fully(creature_ptr, FALSE)) return NULL;
			}
		}
		break;

	case 27:
#ifdef JP
		if (name) return "틭";
		if (desc) return "̖Cƃ_[WCB";
#else
		if (name) return "Enchant Weapon";
		if (desc) return "Attempts to increase +to-hit, +to-dam of a weapon.";
#endif
    
		{
			if (cast)
			{
				if (!enchant_spell(creature_ptr, randint0(4) + 1, randint0(4) + 1, 0)) return NULL;
			}
		}
		break;

	case 28:
#ifdef JP
		if (name) return "h";
		if (desc) return "Z̖hCB";
#else
		if (name) return "Enchant Armor";
		if (desc) return "Attempts to increase +AC of an armor.";
#endif
    
		{
			if (cast)
			{
				if (!enchant_spell(creature_ptr, 0, 0, randint0(3) + 2)) return NULL;
			}
		}
		break;

	case 29:
#ifdef JP
		if (name) return "푮t^";
		if (desc) return "Ƀ_ɑB";
#else
		if (name) return "Brand Weapon";
		if (desc) return "Makes current weapon a random ego weapon.";
#endif
    
		{
			if (cast)
			{
				brand_weapon(creature_ptr, randint0(18));
			}
		}
		break;

	case 30:
#ifdef JP
		if (name) return "lԃgv";
		if (desc) return "_Ƀe|[gˑRψقÄӎvŃe|[gˑRψقgɂB";
#else
		if (name) return "Living Trump";
		if (desc) return "Gives mutation which makes you teleport randomly or makes you able to teleport at will.";
#endif
    
		{
			if (cast)
			{
				int mutation;

				if (one_in_(7))
					/* Teleport control */
					mutation = 12;
				else
					/* Random teleportation (uncontrolled) */
					mutation = 77;

				/* Gain the mutation */
				if (gain_trait(creature_ptr, mutation, TRUE))
				{
#ifdef JP
					msg_print("Ȃ͐ĂJ[hɕςB");
#else
					msg_print("You have turned into a Living Trump.");
#endif
				}
			}
		}
		break;

	case 31:
#ifdef JP
		if (name) return "ւ̖Ɖu";
		if (desc) return "莞ԁACAAdA_̂ꂩɑ΂Ɖu𓾂B";
#else
		if (name) return "Immunity";
		if (desc) return "Gives an immunity to fire, cold, electricity or acid for a while.";
#endif
    
		{
			int base = 13;

			if (info) return info_duration(base, base);

			if (cast)
			{
				if (!choose_ele_immune(creature_ptr, base + randint1(base))) return NULL;
			}
		}
		break;
	}

	return "";
}


static cptr do_daemon_spell(creature_type *creature_ptr, int spell, int mode)
{
	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool info = (mode == SPELL_INFO) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;

#ifdef JP
	static const char s_dam[] = ":";
#else
	static const char s_dam[] = "dam ";
#endif

	int dir;
	int plev = creature_ptr->lev;

	switch (spell)
	{
	case 0:
#ifdef JP
		if (name) return "}WbNE~TC";
		if (desc) return "ア@̖B";
#else
		if (name) return "Magic Missile";
		if (desc) return "Fires a weak bolt of magic.";
#endif
    
		{
			int dice = 3 + (plev - 1) / 5;
			int sides = 4;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_bolt_or_beam(creature_ptr, beam_chance(creature_ptr) - 10, GF_MISSILE, dir, diceroll(dice, sides));
			}
		}
		break;

	case 1:
#ifdef JP
		if (name) return "m";
		if (desc) return "߂̐̂ȂN[`[mB";
#else
		if (name) return "Detect Unlife";
		if (desc) return "Detects all nonliving creatures in your vicinity.";
#endif
    
		{
			int rad = DETECT_RAD_DEFAULT;

			if (info) return info_radius(rad);

			if (cast)
			{
				detect_creatures_nonliving(creature_ptr, rad);
			}
		}
		break;

	case 2:
#ifdef JP
		if (name) return "ׂȂj";
		if (desc) return "莞ԁAACɃ{[iX𓾂B";
#else
		if (name) return "Evil Bless";
		if (desc) return "Gives bonus to hit and AC for a few turns.";
#endif
    
		{
			int base = 12;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_blessed(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 3:
#ifdef JP
		if (name) return "ωΉ";
		if (desc) return "莞ԁAւ̑ϐ𓾂BɂϐɗݐςB";
#else
		if (name) return "Resist Fire";
		if (desc) return "Gives resistance to fire, cold and electricity for a while. These resistances can be added to which from equipment for more powerful resistances.";
#endif
    
		{
			int base = 20;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_oppose_fire(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 4:
#ifdef JP
		if (name) return "Q";
		if (desc) return "N[`[1̂|ANOBRƖB";
#else
		if (name) return "Horrify";
		if (desc) return "Attempts to scare and stun a creature.";
#endif
    
		{
			int power = plev;

			if (info) return info_power(power);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fear_creature(creature_ptr, dir, power);
				stun_creature(creature_ptr, dir, power);
			}
		}
		break;

	case 5:
#ifdef JP
		if (name) return "n̖";
		if (desc) return "ñ{g̓r[B";
#else
		if (name) return "Nether Bolt";
		if (desc) return "Fires a bolt or beam of nether.";
#endif
    
		{
			int dice = 6 + (plev - 5) / 4;
			int sides = 8;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_bolt_or_beam(creature_ptr, beam_chance(creature_ptr), GF_NETHER, dir, diceroll(dice, sides));
			}
		}
		break;

	case 6:
#ifdef JP
		if (name) return "Ñ̎쏢";
		if (desc) return "Ñ̎B";
#else
		if (name) return "Summon Manes";
		if (desc) return "Summons a manes.";
#endif
    
		{
			if (cast)
			{
				if (!summon_specific(NULL, creature_ptr->fy, creature_ptr->fx, (plev * 3) / 2, SUMMON_MANES, (PM_ALLOW_GROUP | PM_FORCE_PET)))
				{
#ifdef JP
					msg_print("Ñ̎͌ȂB");
#else
					msg_print("No Manes arrive.");
#endif
				}
			}
		}
		break;

	case 7:
#ifdef JP
		if (name) return "n̉";
		if (desc) return "׈ȗ͂{[BPǂȃN[`[ɂ͑傫ȃ_[W^B";
#else
		if (name) return "Hellish Flame";
		if (desc) return "Fires a ball of evil power. Hurts good creatures greatly.";
#endif
    
		{
			int dice = 3;
			int sides = 6;
			int rad = (plev < 30) ? 2 : 3;
			int base;

			if (creature_ptr->class_idx == CLASS_MAGE ||
			    creature_ptr->class_idx == CLASS_HIGH_MAGE ||
			    creature_ptr->class_idx == CLASS_SORCERER)
				base = plev + plev / 2;
			else
				base = plev + plev / 4;


			if (info) return info_damage(dice, sides, base);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_ball(creature_ptr, GF_HELL_FIRE, dir, diceroll(dice, sides) + base, rad);
			}
		}
		break;

	case 8:
#ifdef JP
		if (name) return "f[xz";
		if (desc) return "1̂𖣗BRƖ";
#else
		if (name) return "Dominate Demon";
		if (desc) return "Attempts to charm a demon.";
#endif
    
		{
			int power = plev;

			if (info) return info_power(power);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				control_one_demon(creature_ptr, dir, power);
			}
		}
		break;

	case 9:
#ifdef JP
		if (name) return "rW";
		if (desc) return "ӂ̒n`mB";
#else
		if (name) return "Vision";
		if (desc) return "Maps nearby area.";
#endif
    
		{
			int rad = DETECT_RAD_MAP;

			if (info) return info_radius(rad);

			if (cast)
			{
				map_area(creature_ptr, rad);
			}
		}
		break;

	case 10:
#ifdef JP
		if (name) return "ϒn";
		if (desc) return "莞ԁAnւ̑ϐ𓾂B";
#else
		if (name) return "Resist Nether";
		if (desc) return "Gives resistance to nether for a while.";
#endif
    
		{
			int base = 20;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_tim_res_nether(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 11:
#ifdef JP
		if (name) return "vY}E{g";
		if (desc) return "vY}̃{g̓r[B";
#else
		if (name) return "Plasma bolt";
		if (desc) return "Fires a bolt or beam of plasma.";
#endif
    
		{
			int dice = 11 + (plev - 5) / 4;
			int sides = 8;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_bolt_or_beam(creature_ptr, beam_chance(creature_ptr), GF_PLASMA, dir, diceroll(dice, sides));
			}
		}
		break;

	case 12:
#ifdef JP
		if (name) return "t@CAE{[";
		if (desc) return "̋B";
#else
		if (name) return "Fire Ball";
		if (desc) return "Fires a ball of fire.";
#endif
    
		{
			int dam = plev + 55;
			int rad = 2;

			if (info) return info_damage(0, 0, dam);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_ball(creature_ptr, GF_FIRE, dir, dam, rad);
			}
		}
		break;

	case 13:
#ifdef JP
		if (name) return "̐n";
		if (desc) return "ɉ̑B";
#else
		if (name) return "Fire Branding";
		if (desc) return "Makes current weapon fire branded.";
#endif
    
		{
			if (cast)
			{
				brand_weapon(creature_ptr, 1);
			}
		}
		break;

	case 14:
#ifdef JP
		if (name) return "n";
		if (desc) return "傫Ȓn̋B";
#else
		if (name) return "Nether Ball";
		if (desc) return "Fires a huge ball of nether.";
#endif
    
		{
			int dam = plev * 3 / 2 + 100;
			int rad = plev / 20 + 2;

			if (info) return info_damage(0, 0, dam);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_ball(creature_ptr, GF_NETHER, dir, dam, rad);
			}
		}
		break;

	case 15:
#ifdef JP
		if (name) return "f[";
		if (desc) return "1̂B";
#else
		if (name) return "Summon Demon";
		if (desc) return "Summons a demon.";
#endif
    
		{
			if (cast)
			{
				bool pet = !one_in_(3);
				u32b mode = 0L;

				if (pet) mode |= PM_FORCE_PET;
				else mode |= PM_NO_PET;
				if (!(pet && (plev < 50))) mode |= PM_ALLOW_GROUP;

				if (summon_specific((pet ? creature_ptr : NULL), creature_ptr->fy, creature_ptr->fx, plev*2/3+randint1(plev/2), SUMMON_DEMON, mode))
				{
#ifdef JP
					msg_print("̈L[B");
#else
					msg_print("The area fills with a stench of sulphur and brimstone.");
#endif


					if (pet)
					{
#ifdef JP
						msg_print("upł܂Allv");
#else
						msg_print("'What is thy bidding... Master?'");
#endif
					}
					else
					{
#ifdef JP
						msg_print("uڂ҂Ả͓lɂ炸I O̍𒸂Iv");
#else
						msg_print("'NON SERVIAM! Wretch! I shall feast on thy mortal soul!'");
#endif
					}
				}
				else
				{
#ifdef JP
					msg_print("͌ȂB");
#else
					msg_print("No demons arrive.");
#endif
				}
				break;
			}
		}
		break;

	case 16:
#ifdef JP
		if (name) return "̖";
		if (desc) return "莞ԁAepV[\͂𓾂B";
#else
		if (name) return "Devilish Eye";
		if (desc) return "Gives telepathy for a while.";
#endif
    
		{
			int base = 30;
			int sides = 25;

			if (info) return info_duration(base, sides);

			if (cast)
			{
				set_tim_esp(creature_ptr, randint1(base) + sides, FALSE);
			}
		}
		break;

	case 17:
#ifdef JP
		if (name) return "̃N[N";
		if (desc) return "|菜A莞ԁAƗC̑ϐÃI[𓾂Bϐ͑ɂϐɗݐςB";
#else
		if (name) return "Devil Cloak";
		if (desc) return "Removes fear. Gives resistance to fire and cold, and aura of fire. These resistances can be added to which from equipment for more powerful resistances.";
#endif
    
		{
			int base = 20;

			if (info) return info_duration(base, base);

			if (cast)
			{
				int dur = randint1(base) + base;
					
				set_oppose_fire(creature_ptr, dur, FALSE);
				set_oppose_cold(creature_ptr, dur, FALSE);
				set_tim_sh_fire(creature_ptr, dur, FALSE);
				set_afraid(creature_ptr, 0);
				break;
			}
		}
		break;

	case 18:
#ifdef JP
		if (name) return "n◬";
		if (desc) return "𒆐SƂ̋oAnɕςB";
#else
		if (name) return "The Flow of Lava";
		if (desc) return "Generates a ball of fire centered on you which transforms floors to magma.";
#endif
    
		{
			int dam = (55 + plev) * 2;
			int rad = 3;

			if (info) return info_damage(0, 0, dam/2);

			if (cast)
			{
				fire_ball(creature_ptr, GF_FIRE, 0, dam, rad);
				fire_ball_hide(creature_ptr, GF_LAVA_FLOW, 0, 2 + randint1(2), rad);
			}
		}
		break;

	case 19:
#ifdef JP
		if (name) return "vY}";
		if (desc) return "vY}̋B";
#else
		if (name) return "Plasma Ball";
		if (desc) return "Fires a ball of plasma.";
#endif
    
		{
			int dam = plev * 3 / 2 + 80;
			int rad = 2 + plev / 40;

			if (info) return info_damage(0, 0, dam);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_ball(creature_ptr, GF_PLASMA, dir, dam, rad);
			}
		}
		break;

	case 20:
#ifdef JP
		if (name) return "ω";
		if (desc) return "莞ԁAɕωBωĂԂ͖{̎푰̔\͂AɈƂĂ̔\͂𓾂B";
#else
		if (name) return "Polymorph Demon";
		if (desc) return "Mimic a demon for a while. Loses abilities of original race and gets abilities as a demon.";
#endif
    
		{
			int base = 10 + plev / 2;

			if (info) return info_duration(base, base);

			if (cast)
			{
				//TODO set_mimic(creature_ptr, base + randint1(base), MIMIC_DEMON, FALSE);
			}
		}
		break;

	case 21:
#ifdef JP
		if (name) return "n̔g";
		if (desc) return "ȆSẴN[`[Ƀ_[W^BPǂȃN[`[ɓɑ傫ȃ_[W^B";
#else
		if (name) return "Nather Wave";
		if (desc) return "Damages all creatures in sight. Hurts good creatures greatly.";
#endif
    
		{
			int sides1 = plev * 2;
			int sides2 = plev * 2;

			if (info) return format("%sd%d+d%d", s_dam, sides1, sides2);

			if (cast)
			{
				dispel_creatures(creature_ptr, randint1(sides1));
				dispel_good(creature_ptr, randint1(sides2));
			}
		}
		break;

	case 22:
#ifdef JP
		if (name) return "TLoX̐ڕ";
		if (desc) return "ʍ̋B";
#else
		if (name) return "Kiss of Succubus";
		if (desc) return "Fires a ball of nexus.";
#endif
    
		{
			int dam = 100 + plev * 2;
			int rad = 4;

			if (info) return info_damage(0, 0, dam);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;
				fire_ball(creature_ptr, GF_NEXUS, dir, dam, rad);
			}
		}
		break;

	case 23:
#ifdef JP
		if (name) return "jł̎";
		if (desc) return "jł̎BHN[`[͂̂ƂHP̔Õ_[W󂯂B";
#else
		if (name) return "Doom Hand";
		if (desc) return "Attempts to make a creature's HP almost half.";
#endif
    
		{
			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;
#ifdef JP
				else msg_print("<jł̎>I");
#else
				else msg_print("You invoke the Hand of Doom!");
#endif

				fire_ball_hide(creature_ptr, GF_HAND_DOOM, dir, plev * 2, 0);
			}
		}
		break;

	case 24:
#ifdef JP
		if (name) return "mCg";
		if (desc) return "莞ԁAq[[CɂȂB";
#else
		if (name) return "Raise the Morale";
		if (desc) return "Removes fear, and gives bonus to hit and 10 more HP for a while.";
#endif
    
		{
			int base = 25;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_hero(creature_ptr, randint1(base) + base, FALSE);
				hp_player(creature_ptr, 10);
				set_afraid(creature_ptr, 0);
			}
		}
		break;

	case 25:
#ifdef JP
		if (name) return "sł̓";
		if (desc) return "莞ԁAԋt]ւ̑ϐ𓾂B";
#else
		if (name) return "Immortal Body";
		if (desc) return "Gives resistance to time for a while.";
#endif
    
		{
			int base = 20;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_tim_res_time(creature_ptr, randint1(base)+base, FALSE);
			}
		}
		break;

	case 26:
#ifdef JP
		if (name) return "C̉~";
		if (desc) return "𒆐SƂJIX̋A̋𔭐A߂̃N[`[𖣗B";
#else
		if (name) return "Insanity Circle";
		if (desc) return "Generate balls of chaos, confusion and charm centered on you.";
#endif
    
		{
			int dam = 50 + plev;
			int power = 20 + plev;
			int rad = 3 + plev / 20;

			if (info) return format("%s%d+%d", s_dam, dam/2, dam/2);

			if (cast)
			{
				fire_ball(creature_ptr, GF_CHAOS, 0, dam, rad);
				fire_ball(creature_ptr, GF_CONFUSION, 0, dam, rad);
				fire_ball(creature_ptr, GF_CHARM, 0, power, rad);
			}
		}
		break;

	case 27:
#ifdef JP
		if (name) return "ybgj";
		if (desc) return "SẴybgIɔjB";
#else
		if (name) return "Explode Pets";
		if (desc) return "Makes all pets explode.";
#endif
    
		{
			if (cast)
			{
				discharge_minion(creature_ptr);
			}
		}
		break;

	case 28:
#ifdef JP
		if (name) return "O[^[f[";
		if (desc) return "㋉f[Bɂ͐l('p','h','t'ŕ\N[`[)̎̂Ȃ΂ȂȂB";
#else
		if (name) return "Summon Greater Demon";
		if (desc) return "Summons greater demon. It need to sacrifice a corpse of human ('p','h' or 't').";
#endif
    
		{
			if (cast)
			{
				if (!cast_summon_greater_demon(creature_ptr)) return NULL;
			}
		}
		break;

	case 29:
#ifdef JP
		if (name) return "n";
		if (desc) return "Ȓn̋B";
#else
		if (name) return "Nether Storm";
		if (desc) return "Generate a huge ball of nether.";
#endif
    
		{
			int dam = plev * 15;
			int rad = plev / 5;

			if (info) return info_damage(0, 0, dam);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_ball(creature_ptr, GF_NETHER, dir, dam, rad);
			}
		}
		break;

	case 30:
#ifdef JP
		if (name) return "̎";
		if (desc) return "_[W󂯂邱ƂɂđΏۂɎ􂢂A_[W^lXȌʂNB";
#else
		if (name) return "Bloody Curse";
		if (desc) return "Puts blood curse which damages and causes various effects on a creature. You also take damage.";
#endif
    
		{
			int dam = 600;
			int rad = 0;

			if (info) return info_damage(0, 0, dam);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_ball_hide(creature_ptr, GF_BLOOD_CURSE, dir, dam, rad);
#ifdef JP
				take_hit(NULL, creature_ptr, DAMAGE_USELIFE, 20 + randint1(30), "̎", NULL, -1);
#else
				take_hit(NULL, creature_ptr, DAMAGE_USELIFE, 20 + randint1(30), "Blood curse", NULL, -1);
#endif
			}
		}
		break;

	case 31:
#ifdef JP
		if (name) return "ω";
		if (desc) return "̉ɕωBωĂԂ͖{̎푰̔\͂AɈ̉ƂĂ̔\͂𓾁Aǂj󂵂ȂB";
#else
		if (name) return "Polymorph Demonlord";
		if (desc) return "Mimic a demon lord for a while. Loses abilities of original race and gets great abilities as a demon lord. Even hard walls can't stop your walking.";
#endif
    
		{
			int base = 15;

			if (info) return info_duration(base, base);

			if (cast)
			{
				//TODO set_mimic(creature_ptr, base + randint1(base), MIMIC_DEMON_LORD, FALSE);
			}
		}
		break;
	}

	return "";
}


static cptr do_crusade_spell(creature_type *creature_ptr, int spell, int mode)
{
	floor_type *floor_ptr = get_floor_ptr(creature_ptr);

	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool info = (mode == SPELL_INFO) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;

	int dir;
	int plev = creature_ptr->lev;

	switch (spell)
	{
	case 0:
#ifdef JP
		if (name) return "";
		if (desc) return "d̃{g̓r[B";
#else
		if (name) return "Punishment";
		if (desc) return "Fires a bolt or beam of lightning.";
#endif
    
		{
			int dice = 3 + (plev - 1) / 5;
			int sides = 4;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_bolt_or_beam(creature_ptr, beam_chance(creature_ptr) - 10, GF_ELEC, dir, diceroll(dice, sides));
			}
		}
		break;

	case 1:
#ifdef JP
		if (name) return "׈݊m";
		if (desc) return "߂̎׈ȃN[`[mB";
#else
		if (name) return "Detect Evil";
		if (desc) return "Detects all evil creatures in your vicinity.";
#endif
    
		{
			int rad = DETECT_RAD_DEFAULT;

			if (info) return info_radius(rad);

			if (cast)
			{
				detect_creatures_evil(creature_ptr, rad);
			}
		}
		break;

	case 2:
#ifdef JP
		if (name) return "|";
		if (desc) return "|菜B";
#else
		if (name) return "Remove Fear";
		if (desc) return "Removes fear.";
#endif
    
		{
			if (cast)
			{
				set_afraid(creature_ptr, 0);
			}
		}
		break;

	case 3:
#ifdef JP
		if (name) return "Ј";
		if (desc) return "N[`[1̂|BRƖB";
#else
		if (name) return "Scare Creature";
		if (desc) return "Attempts to scare a creature.";
#endif
    
		{
			int power = plev;

			if (info) return info_power(power);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fear_creature(creature_ptr, dir, power);
			}
		}
		break;

	case 4:
#ifdef JP
		if (name) return "";
		if (desc) return "אڂSẴN[`[𖰂点BRƖB";
#else
		if (name) return "Sanctuary";
		if (desc) return "Attempts to sleep creatures in the adjacent squares.";
#endif
    
		{
			int power = plev;

			if (info) return info_power(power);

			if (cast)
			{
				sleep_creatures_touch(creature_ptr);
			}
		}
		break;

	case 5:
#ifdef JP
		if (name) return "";
		if (desc) return "̃e|[gB";
#else
		if (name) return "Portal";
		if (desc) return "Teleport medium distance.";
#endif
    
		{
			int range = 25 + plev / 2;

			if (info) return info_range(range);

			if (cast)
			{
				teleport_player(creature_ptr, range, 0L);
			}
		}
		break;

	case 6:
#ifdef JP
		if (name) return "X^[_Xg";
		if (desc) return "^[Qbgt߂ɑM̃{gA˂B";
#else
		if (name) return "Star Dust";
		if (desc) return "Fires many bolts of light near the target.";
#endif
    
		{
			int dice = 3 + (plev - 1) / 9;
			int sides = 2;

			if (info) return info_multi_damage_dice(dice, sides);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;
				fire_blast(creature_ptr, GF_LITE, dir, dice, sides, 10, 3);
			}
		}
		break;

	case 7:
#ifdef JP
		if (name) return "g̏";
		if (desc) return "AŁANOSB";
#else
		if (name) return "Purify";
		if (desc) return "Heals all cut, stun and poison status.";
#endif
    
		{
			if (cast)
			{
				set_cut(creature_ptr, 0);
				set_poisoned(creature_ptr, 0);
				set_stun(creature_ptr, 0);
			}
		}
		break;

	case 8:
#ifdef JP
		if (name) return "׈΂";
		if (desc) return "׈ȃN[`[1̂e|[gBRƖB";
#else
		if (name) return "Scatter Evil";
		if (desc) return "Attempts to teleport an evil creature away.";
#endif
    
		{
			int power = MAX_SIGHT * 5;

			if (info) return info_power(power);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;
				fire_ball(creature_ptr, GF_AWAY_EVIL, dir, power, 0);
			}
		}
		break;

	case 9:
#ifdef JP
		if (name) return "Ȃ";
		if (desc) return "Ȃ͂B׈ȃN[`[ɑ΂đ傫ȃ_[W^邪APǂȃN[`[ɂ͌ʂȂB";
#else
		if (name) return "Holy Orb";
		if (desc) return "Fires a ball with holy power. Hurts evil creatures greatly, but don't effect good creatures.";
#endif
    
		{
			int dice = 3;
			int sides = 6;
			int rad = (plev < 30) ? 2 : 3;
			int base;

			if (creature_ptr->class_idx == CLASS_PRIEST ||
			    creature_ptr->class_idx == CLASS_HIGH_MAGE ||
			    creature_ptr->class_idx == CLASS_SORCERER)
				base = plev + plev / 2;
			else
				base = plev + plev / 4;


			if (info) return info_damage(dice, sides, base);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_ball(creature_ptr, GF_HOLY_FIRE, dir, diceroll(dice, sides) + base, rad);
			}
		}
		break;

	case 10:
#ifdef JP
		if (name) return "";
		if (desc) return "ȆSẴAfbhyшɃ_[W^A׈ȃN[`[|B";
#else
		if (name) return "Exorcism";
		if (desc) return "Damages all undead and demons in sight, and scares all evil creatures in sight.";
#endif
    
		{
			int sides = plev;
			int power = plev;

			if (info) return info_damage(1, sides, 0);

			if (cast)
			{
				dispel_undead(creature_ptr, randint1(sides));
				dispel_demons(creature_ptr, randint1(sides));
				turn_evil(creature_ptr, power);
			}
		}
		break;

	case 11:
#ifdef JP
		if (name) return "";
		if (desc) return "ACeɂア􂢂B";
#else
		if (name) return "Remove Curse";
		if (desc) return "Removes normal curses from equipped items.";
#endif
    
		{
			if (cast)
			{
				if (remove_curse(creature_ptr))
				{
#ifdef JP
					msg_print("NɌĂ悤ȋCB");
#else
					msg_print("You feel as if someone is watching over you.");
#endif
				}
			}
		}
		break;

	case 12:
#ifdef JP
		if (name) return "F";
		if (desc) return "莞ԁAȂ̂悤ɂȂB";
#else
		if (name) return "Sense Unseen";
		if (desc) return "Gives see invisible for a while.";
#endif
    
		{
			int base = 24;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_tim_invis(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 13:
#ifdef JP
		if (name) return "Ύ׈E";
		if (desc) return "׈ȃN[`[̍UhoA𒣂B";
#else
		if (name) return "Protection from Evil";
		if (desc) return "Gives aura which protect you from evil creature's physical attack.";
#endif
    
		{
			int base = 25;
			int sides = 3 * plev;

			if (info) return info_duration(base, sides);

			if (cast)
			{
				set_protevil(creature_ptr, randint1(sides) + sides, FALSE);
			}
		}
		break;

	case 14:
#ifdef JP
		if (name) return "ق̗";
		if (desc) return "͂ȓd̃{gB";
#else
		if (name) return "Judgment Thunder";
		if (desc) return "Fires a powerful bolt of lightning.";
#endif
    
		{
			int dam = plev * 5;

			if (info) return info_damage(0, 0, dam);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;
				fire_bolt(creature_ptr, GF_ELEC, dir, dam);
			}
		}
		break;

	case 15:
#ifdef JP
		if (name) return "Ȃ䌾t";
		if (desc) return "E̎׈ȑ݂ɑ傫ȃ_[W^A̗͂񕜂AŁA|ANOԁASB";
#else
		if (name) return "Holy Word";
		if (desc) return "Damages all evil creatures in sight, heals HP somewhat, and completely heals poison, fear, stun and cut status.";
#endif
    
		{
			int dam_sides = plev * 6;
			int heal = 100;

#ifdef JP
			if (info) return format(":1d%d/%d", dam_sides, heal);
#else
			if (info) return format("dam:d%d/h%d", dam_sides, heal);
#endif

			if (cast)
			{
				dispel_evil(creature_ptr, randint1(dam_sides));
				hp_player(creature_ptr, heal);
				set_afraid(creature_ptr, 0);
				set_poisoned(creature_ptr, 0);
				set_stun(creature_ptr, 0);
				set_cut(creature_ptr, 0);
			}
		}
		break;

	case 16:
#ifdef JP
		if (name) return "Jꂽ";
		if (desc) return "꒼̑SĂ㩂Ɣj󂷂B";
#else
		if (name) return "Unbarring Ways";
		if (desc) return "Fires a beam which destroy traps and doors.";
#endif
    
		{
			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				destroy_door(creature_ptr, dir);
			}
		}
		break;

	case 17:
#ifdef JP
		if (name) return "";
		if (desc) return "׈ȃN[`[̓~߂B";
#else
		if (name) return "Arrest";
		if (desc) return "Attempts to paralyze an evil creature.";
#endif
    
		{
			int power = plev * 2;

			if (info) return info_power(power);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;
				stasis_evil(creature_ptr, dir);
			}
		}
		break;

	case 18:
#ifdef JP
		if (name) return "ȂI[";
		if (desc) return "莞ԁA׈ȃN[`[鐹ȂI[𓾂B";
#else
		if (name) return "Holy Aura";
		if (desc) return "Gives aura of holy power which injures evil creatures which attacked you for a while.";
#endif
    
		{
			int base = 20;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_tim_sh_holy(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 19:
#ifdef JP
		if (name) return "Afbh&ގU";
		if (desc) return "ȆSẴAfbhyшɃ_[W^B";
#else
		if (name) return "Dispel Undead & Demons";
		if (desc) return "Damages all undead and demons in sight.";
#endif
    
		{
			int sides = plev * 4;

			if (info) return info_damage(1, sides, 0);

			if (cast)
			{
				dispel_undead(creature_ptr, randint1(sides));
				dispel_demons(creature_ptr, randint1(sides));
			}
		}
		break;

	case 20:
#ifdef JP
		if (name) return "׈ގU";
		if (desc) return "ȆSĂ̎׈ȃN[`[Ƀ_[W^B";
#else
		if (name) return "Dispel Evil";
		if (desc) return "Damages all evil creatures in sight.";
#endif
    
		{
			int sides = plev * 4;

			if (info) return info_damage(1, sides, 0);

			if (cast)
			{
				dispel_evil(creature_ptr, randint1(sides));
			}
		}
		break;

	case 21:
#ifdef JP
		if (name) return "Ȃn";
		if (desc) return "ʏ̕ɖŎׂ̑B";
#else
		if (name) return "Holy Blade";
		if (desc) return "Makes current weapon especially deadly against evil creatures.";
#endif
    
		{
			if (cast)
			{
				brand_weapon(creature_ptr, 13);
			}
		}
		break;

	case 22:
#ifdef JP
		if (name) return "X^[o[Xg";
		if (desc) return "ȑM̋B";
#else
		if (name) return "Star Burst";
		if (desc) return "Fires a huge ball of powerful light.";
#endif
    
		{
			int dam = 100 + plev * 2;
			int rad = 4;

			if (info) return info_damage(0, 0, dam);

			if (cast)
			{
				if (!get_aim_dir(creature_ptr, &dir)) return NULL;

				fire_ball(creature_ptr, GF_LITE, dir, dam, rad);
			}
		}
		break;

	case 23:
#ifdef JP
		if (name) return "Vg";
		if (desc) return "Vg1̏B";
#else
		if (name) return "Summon Angel";
		if (desc) return "Summons an angel.";
#endif
    
		{
			if (cast)
			{
				bool pet = !one_in_(3);
				u32b mode = 0L;

				if (pet) mode |= PM_FORCE_PET;
				else mode |= PM_NO_PET;
				if (!(pet && (plev < 50))) mode |= PM_ALLOW_GROUP;

				if (summon_specific((pet ? creature_ptr : NULL), creature_ptr->fy, creature_ptr->fx, (plev * 3) / 2, SUMMON_ANGEL, mode))
				{
					if (pet)
					{
#ifdef JP
						msg_print("upł܂Allv");
#else
						msg_print("'What is thy bidding... Master?'");
#endif
					}
					else
					{
#ifdef JP
						msg_print("ủ͓lɂ炸I s҂A߂Iv");
#else
						msg_print("Mortal! Repent of thy impiousness.");
#endif
					}
				}
			}
		}
		break;

	case 24:
#ifdef JP
		if (name) return "mCg";
		if (desc) return "莞ԁAq[[CɂȂB";
#else
		if (name) return "Heroism";
		if (desc) return "Removes fear, and gives bonus to hit and 10 more HP for a while.";
#endif
    
		{
			int base = 25;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_hero(creature_ptr, randint1(base) + base, FALSE);
				hp_player(creature_ptr, 10);
				set_afraid(creature_ptr, 0);
			}
		}
		break;

	case 25:
#ifdef JP
		if (name) return "􂢑ގU";
		if (desc) return "ACeɂ͂Ȏ􂢂B";
#else
		if (name) return "Dispel Curse";
		if (desc) return "Removes normal and heavy curse from equipped items.";
#endif
    
		{
			if (cast)
			{
				if (remove_all_curse(creature_ptr))
				{
#ifdef JP
					msg_print("NɌĂ悤ȋCB");
#else
					msg_print("You feel as if someone is watching over you.");
#endif
				}
			}
		}
		break;

	case 26:
#ifdef JP
		if (name) return "׈Ǖ";
		if (desc) return "ȆSĂ̎׈ȃN[`[e|[gBRƖB";
#else
		if (name) return "Banish Evil";
		if (desc) return "Teleports all evil creatures in sight away unless resisted.";
#endif
    
		{
			int power = 100;

			if (info) return info_power(power);

			if (cast)
			{
				if (banish_evil(creature_ptr, power))
				{
#ifdef JP
					msg_print("_ȗ͂׈łI");
#else
					msg_print("The holy power banishes evil!");
#endif

				}
			}
		}
		break;

	case 27:
#ifdef JP
		if (name) return "n}Qh";
		if (desc) return "ӂ̃ACeAN[`[An`j󂷂B";
#else
		if (name) return "Armageddon";
		if (desc) return "Destroy everything in nearby area.";
#endif
    
		{
			int base = 12;
			int sides = 4;

			if (cast)
			{
				destroy_area(creature_ptr, creature_ptr->fy, creature_ptr->fx, base + randint1(sides), FALSE);
			}
		}
		break;

	case 28:
#ifdef JP
		if (name) return "ڂɂ͖ڂ";
		if (desc) return "莞ԁA_[W󂯂ƂɍUsN[`[ɑ΂ē̃_[W^B";
#else
		if (name) return "An Eye for an Eye";
		if (desc) return "Gives special aura for a while. When you are attacked by a creature, the creature are injured with same amount of damage as you take.";
#endif
    
		{
			int base = 10;

			if (info) return info_duration(base, base);

			if (cast)
			{
				set_tim_eyeeye(creature_ptr, randint1(base) + base, FALSE);
			}
		}
		break;

	case 29:
#ifdef JP
		if (name) return "_̓{";
		if (desc) return "^[Qbg̎͂ɕ̋𑽐ƂB";
#else
		if (name) return "Wrath of the God";
		if (desc) return "Drops many balls of disintegration near the target.";
#endif
    
		{
			int dam = plev * 3 + 25;
			int rad = 2;

			if (info) return info_multi_damage(dam);

			if (cast)
			{
				if (!cast_wrath_of_the_god(creature_ptr, dam, rad)) return NULL;
			}
		}
		break;

	case 30:
#ifdef JP
		if (name) return "_";
		if (desc) return "אڂN[`[ɐȂ_[W^AẼN[`[Ƀ_[WAANOAA|A^Bɑ̗͂񕜂B";
#else
		if (name) return "Divine Intervention";
		if (desc) return "Damages all adjacent creatures with holy power. Damages and attempt to slow, stun, confuse, scare and freeze all creatures in sight. And heals HP.";
#endif
    
		{
			int b_dam = plev * 11;
			int d_dam = plev * 4;
			int heal = 100;
			int power = plev * 4;

#ifdef JP
			if (info) return format("%d/%d+%d", heal, d_dam, b_dam/2);
#else
			if (info) return format("h%d/dm%d+%d", heal, d_dam, b_dam/2);
#endif

			if (cast)
			{
				project(creature_ptr, 1, creature_ptr->fy, creature_ptr->fx, b_dam, GF_HOLY_FIRE, PROJECT_KILL, -1);
				dispel_creatures(creature_ptr, d_dam);
				slow_creatures(creature_ptr);
				stun_creatures(creature_ptr, power);
				confuse_creatures(creature_ptr, power);
				turn_creatures(creature_ptr, power);
				stasis_creatures(creature_ptr, power);
				hp_player(creature_ptr, heal);
			}
		}
		break;

	case 31:
#ifdef JP
		if (name) return "";
		if (desc) return "ȆPǂȃN[`[ybgɂ悤ƂAȂȂꍇyёPǂłȂN[`[|Bɑ̉ꂽRmAq[[AjAAΎ׈E𓾂B";
#else
		if (name) return "Crusade";
		if (desc) return "Attempts to charm all good creatures in sight, and scare all non-charmed creatures, and summons great number of knights, and gives heroism, bless, speed and protection from evil.";
#endif
    
		{
			if (cast)
			{
				int base = 25;
				int sp_sides = 20 + plev;
				int sp_base = plev;

				int i;
				crusade(creature_ptr);
				for (i = 0; i < 12; i++)
				{
					int attempt = 10;
					int my, mx;

					while (attempt--)
					{
						scatter(floor_ptr, &my, &mx, creature_ptr->fy, creature_ptr->fx, 4, 0);
						if (cave_empty_bold2(floor_ptr, my, mx)) break; // Require empty grids
					}
					if (attempt < 0) continue;
					summon_specific(NULL, my, mx, plev, SUMMON_KNIGHTS, (PM_ALLOW_GROUP | PM_FORCE_PET | PM_HASTE));
				}
				set_hero(creature_ptr, randint1(base) + base, FALSE);
				set_blessed(creature_ptr, randint1(base) + base, FALSE);
				set_fast(creature_ptr, randint1(sp_sides) + sp_base, FALSE);
				set_protevil(creature_ptr, randint1(base) + base, FALSE);
				set_afraid(creature_ptr, 0);
			}
		}
		break;
	}

	return "";
}


static cptr do_music_spell(creature_type *caster_ptr, int spell, int mode)
{
	floor_type *floor_ptr = get_floor_ptr(caster_ptr);
	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool info = (mode == SPELL_INFO) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;
	bool fail = (mode == SPELL_FAIL) ? TRUE : FALSE;
	bool cont = (mode == SPELL_CONT) ? TRUE : FALSE;
	bool stop = (mode == SPELL_STOP) ? TRUE : FALSE;

#ifdef JP
	static const char s_dam[] = ":";
#else
	static const char s_dam[] = "dam ";
#endif

	int dir;
	int plev = caster_ptr->lev;

	switch (spell)
	{
	case 0:
#ifdef JP
		if (name) return "x݂̉";
		if (desc) return "ȆSẴN[`[BRƖB";
#else
		if (name) return "Song of Holding";
		if (desc) return "Attempts to slow all creatures in sight.";
#endif
    
		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		if (cast)
		{
#ifdef JP
			msg_print("ƂfBݎn߂DDD");
#else
			msg_print("You start humming a slow, steady melody...");
#endif
			start_singing(caster_ptr, spell, MUSIC_SLOW);
		}

		{
			int power = plev;

			if (info) return info_power(power);

			if (cont)
			{
				slow_creatures(caster_ptr);
			}
		}
		break;

	case 1:
#ifdef JP
		if (name) return "j̉";
		if (desc) return "AC̃{[iX𓾂B";
#else
		if (name) return "Song of Blessing";
		if (desc) return "Gives bonus to hit and AC for a few turns.";
#endif
    
		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		if (cast)
		{
#ifdef JP
			msg_print("ȃfBtŎn߂DDD");
#else
			msg_print("The holy power of the Music of the Ainur enters you...");
#endif
			start_singing(caster_ptr, spell, MUSIC_BLESS);
		}

		if (stop)
		{
			if (!caster_ptr->blessed)
			{
#ifdef JP
				msg_print("ȋCB");
#else
				msg_print("The prayer has expired.");
#endif
			}
		}

		break;

	case 2:
#ifdef JP
		if (name) return "̉F";
		if (desc) return "̃{gB";
#else
		if (name) return "Wrecking Note";
		if (desc) return "Fires a bolt of sound.";
#endif
    
		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		{
			int dice = 4 + (plev - 1) / 5;
			int sides = 4;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
				if (!get_aim_dir(caster_ptr, &dir)) return NULL;

				fire_bolt(caster_ptr, GF_SOUND, dir, diceroll(dice, sides));
			}
		}
		break;

	case 3:
#ifdef JP
		if (name) return "NO̐";
		if (desc) return "ȆSẴN[`[NOBRƖB";
#else
		if (name) return "Stun Pattern";
		if (desc) return "Attempts to stun all creatures in sight.";
#endif
    
		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		if (cast)
		{
#ifdef JP
			msg_print("Ῐf郁fBtŎn߂DDD");
#else
			msg_print("You weave a pattern of sounds to bewilder and daze...");
#endif
			start_singing(caster_ptr, spell, MUSIC_STUN);
		}

		{
			int dice = plev / 10;
			int sides = 2;

			if (info) return info_power_dice(dice, sides);

			if (cont)
			{
				stun_creatures(caster_ptr, diceroll(dice, sides));
			}
		}

		break;

	case 4:
#ifdef JP
		if (name) return "̗";
		if (desc) return "̗͂񕜂B";
#else
		if (name) return "Flow of Life";
		if (desc) return "Heals HP a little.";
#endif
    
		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		if (cast)
		{
#ifdef JP
			msg_print("̂ʂđ̂ɊC߂ĂDDD");
#else
			msg_print("Life flows through you as you sing a song of healing...");
#endif
			start_singing(caster_ptr, spell, MUSIC_L_LIFE);
		}

		{
			int dice = 2;
			int sides = 6;

			if (info) return info_heal(dice, sides, 0);

			if (cont)
			{
				hp_player(caster_ptr, diceroll(dice, sides));
			}
		}

		break;

	case 5:
#ifdef JP
		if (name) return "z̉";
		if (desc) return "Ƃ炵Ă͈͂Ŝivɖ邭B";
#else
		if (name) return "Song of the Sun";
		if (desc) return "Lights up nearby area and the inside of a room permanently.";
#endif
    
		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		{
			int dice = 2;
			int sides = plev / 2;
			int rad = plev / 10 + 1;

			if (info) return info_damage(dice, sides, 0);

			if (cast)
			{
#ifdef JP
				msg_print("P̂ӂƂ炵B");
#else
				msg_print("Your uplifting song brings brightness to dark places...");
#endif

				lite_area(caster_ptr, diceroll(dice, sides), rad);
			}
		}
		break;

	case 6:
#ifdef JP
		if (name) return "|̉";
		if (desc) return "ȆSẴN[`[|BRƖB";
#else
		if (name) return "Song of Fear";
		if (desc) return "Attempts to scare all creatures in sight.";
#endif
    
		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		if (cast)
		{
#ifdef JP
			msg_print("ǂ남ǂ낵fBtŎn߂DDD");
#else
			msg_print("You start weaving a fearful pattern...");
#endif
			start_singing(caster_ptr, spell, MUSIC_FEAR);			
		}

		{
			int power = plev;

			if (info) return info_power(power);

			if (cont)
			{
				project_hack(caster_ptr, GF_TURN_ALL, power);
			}
		}

		break;

	case 7:
#ifdef JP
		if (name) return "킢̉";
		if (desc) return "q[[CɂȂB";
#else
		if (name) return "Heroic Ballad";
		if (desc) return "Removes fear, and gives bonus to hit and 10 more HP for a while.";
#endif

		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		if (cast)
		{
#ifdef JP
			msg_print("킢̉̂̂DDD");
#else
			msg_print("You start singing a song of intense fighting...");
#endif

			(void)hp_player(caster_ptr, 10);
			(void)set_afraid(caster_ptr, 0);

			/* Recalculate hitpoints */
			caster_ptr->creature_update |= (CRU_HP);

			start_singing(caster_ptr, spell, MUSIC_HERO);
		}

		if (stop)
		{
			if (!caster_ptr->hero)
			{
#ifdef JP
				msg_print("q[[̋CB");
#else
				msg_print("The heroism wears off.");
#endif
				/* Recalculate hitpoints */
				caster_ptr->creature_update |= (CRU_HP);
			}
		}

		break;

	case 8:
#ifdef JP
		if (name) return "Imo";
		if (desc) return "߂//KimBx15őSẴN[`[A20ōƃACemł悤ɂȂBx25Ŏӂ̒n`mA40ł̊KŜivɏƂ炵A_ŴׂẴACemB̌ʂ͉̂邱ƂŏɋNB";
#else
		if (name) return "Clairaudience";
		if (desc) return "Detects traps, doors and stairs in your vicinity. And detects all creatures at level 15, treasures and items at level 20. Maps nearby area at level 25. Lights and know the whole level at level 40. These effects occurs by turns while this song continues.";
#endif
    
		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		if (cast)
		{
#ifdef JP
			msg_print("Âȉyo܂DDD");
#else
			msg_print("Your quiet music sharpens your sense of hearing...");
#endif

			/* Hack -- Initialize the turn count */
			caster_ptr->class_skills.old_skills.magic_num1[2] = 0;

			start_singing(caster_ptr, spell, MUSIC_DETECT);
		}

		{
			int rad = DETECT_RAD_DEFAULT;

			if (info) return info_radius(rad);

			if (cont)
			{
				int count = caster_ptr->class_skills.old_skills.magic_num1[2];

				if (count >= 19) wiz_lite(floor_ptr, caster_ptr, FALSE);
				if (count >= 11)
				{
					map_area(caster_ptr, rad);
					if (plev > 39 && count < 19)
						caster_ptr->class_skills.old_skills.magic_num1[2] = count + 1;
				}
				if (count >= 6)
				{
					/* There are too many hidden treasure.  So... */
					/* detect_treasure(rad); */
					detect_objects_gold(caster_ptr, rad);
					detect_objects_normal(caster_ptr, rad);

					if (plev > 24 && count < 11)
						caster_ptr->class_skills.old_skills.magic_num1[2] = count + 1;
				}
				if (count >= 3)
				{
					detect_creatures_invis(caster_ptr, rad);
					detect_creatures_normal(caster_ptr, rad);

					if (plev > 19 && count < 6)
						caster_ptr->class_skills.old_skills.magic_num1[2] = count + 1;
				}
				detect_traps(caster_ptr, rad, TRUE);
				detect_doors(caster_ptr, rad);
				detect_stairs(caster_ptr, rad);

				if (plev > 14 && count < 3)
					caster_ptr->class_skills.old_skills.magic_num1[2] = count + 1;
			}
		}

		break;

	case 9:
#ifdef JP
		if (name) return "̉";
		if (desc) return "ȆSẴN[`[ɑ΂Đ_UsB";
#else
		if (name) return "Soul Shriek";
		if (desc) return "Damages all creatures in sight with PSI damages.";
#endif

		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		if (cast)
		{
#ifdef JP
			msg_print("_PȂ̂̂DDD");
#else
			msg_print("You start singing a song of soul in pain...");
#endif
			start_singing(caster_ptr, spell, MUSIC_PSI);
		}

		{
			int dice = 1;
			int sides = plev * 3 / 2;

			if (info) return info_damage(dice, sides, 0);

			if (cont)
			{
				project_hack(caster_ptr, GF_PSI, diceroll(dice, sides));
			}
		}

		break;

	case 10:
#ifdef JP
		if (name) return "m̉";
		if (desc) return "̂}XƗׂ̃}XɗĂACeӒ肷B";
#else
		if (name) return "Song of Lore";
		if (desc) return "Identifies all items which are in the adjacent squares.";
#endif
    
		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		if (cast)
		{
#ifdef JP
			msg_print("̐E̒mꍞłDDD");
#else
			msg_print("You recall the rich lore of the world...");
#endif
			start_singing(caster_ptr, spell, MUSIC_ID);
		}

		{
			int rad = 1;

			if (info) return info_radius(rad);

			/*
			 * ̂̊JnɂʔF
			 * MPsŊӒ肪Oɉ̂fĂ܂̂h~B
			 */
			if (cont || cast)
			{
				project(caster_ptr, rad, caster_ptr->fy, caster_ptr->fx, 0, GF_IDENTIFY, PROJECT_ITEM, -1);
			}
		}

		break;

	case 11:
#ifdef JP
		if (name) return "Bق̉";
		if (desc) return "Bs\͂㏸B";
#else
		if (name) return "Hiding Tune";
		if (desc) return "Gives improved stealth.";
#endif

		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		if (cast)
		{
#ifdef JP
			msg_print("Ȃ̎piFɂƂłDDD");
#else
			msg_print("Your song carries you beyond the sight of mortal eyes...");
#endif
			start_singing(caster_ptr, spell, MUSIC_STEALTH);
		}

		if (stop)
		{
			if (!caster_ptr->tim_stealth)
			{
#ifdef JP
				msg_print("p͂ƌ悤ɂȂB");
#else
				msg_print("You are no longer hided.");
#endif
			}
		}

		break;

	case 12:
#ifdef JP
		if (name) return "e̐";
		if (desc) return "ȆSẴN[`[BRƖB";
#else
		if (name) return "Illusion Pattern";
		if (desc) return "Attempts to confuse all creatures in sight.";
#endif
    
		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		if (cast)
		{
#ifdef JP
			msg_print("ӂʂɌeꂽDDD");
#else
			msg_print("You weave a pattern of sounds to beguile and confuse...");
#endif
			start_singing(caster_ptr, spell, MUSIC_CONF);
		}

		{
			int power = plev * 2;

			if (info) return info_power(power);

			if (cont)
			{
				confuse_creatures(caster_ptr, power);
			}
		}

		break;

	case 13:
#ifdef JP
		if (name) return "jł̋";
		if (desc) return "ȆSẴN[`[ɑ΂čUsB";
#else
		if (name) return "Doomcall";
		if (desc) return "Damages all creatures in sight with booming sound.";
#endif
    
		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		if (cast)
		{
#ifdef JP
			msg_print("DDD");
#else
			msg_print("The fury of the Downfall of Numenor lashes out...");
#endif
			start_singing(caster_ptr, spell, MUSIC_SOUND);
		}

		{
			int dice = 10 + plev / 5;
			int sides = 7;

			if (info) return info_damage(dice, sides, 0);

			if (cont)
			{
				project_hack(caster_ptr, GF_SOUND, diceroll(dice, sides));
			}
		}

		break;

	case 14:
#ifdef JP
		if (name) return "tBG̉";
		if (desc) return "͂̎̂⍜𐶂ԂB";
#else
		if (name) return "Firiel's Song";
		if (desc) return "Resurrects nearby corpse and skeletons. And makes these your pets.";
#endif
    
		{
			/* Stop singing before start another */
			if (cast || fail) stop_singing(caster_ptr);

			if (cast)
			{
#ifdef JP
				msg_print("ƕ̃e[}tŎn߂DDD");
#else
				msg_print("The themes of life and revival are woven into your song...");
#endif

				animate_dead(NULL, caster_ptr->fy, caster_ptr->fx);
			}
		}
		break;

	case 15:
#ifdef JP
		if (name) return "̒";
		if (desc) return "ȆSẴN[`[𖣗BRƖB";
#else
		if (name) return "Fellowship Chant";
		if (desc) return "Attempts to charm all creatures in sight.";
#endif

		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		if (cast)
		{
#ifdef JP
			msg_print("炩ȃfBtŎn߂DDD");
#else
			msg_print("You weave a slow, soothing melody of imploration...");
#endif
			start_singing(caster_ptr, spell, MUSIC_CHARM);
		}

		{
			int dice = 10 + plev / 15;
			int sides = 6;

			if (info) return info_power_dice(dice, sides);

			if (cont)
			{
				charm_creatures(caster_ptr, diceroll(dice, sides));
			}
		}

		break;

	case 16:
#ifdef JP
		if (name) return "g";
		if (desc) return "ǂ@iށB̑̃ACe͏B";
#else
		if (name) return "Sound of disintegration";
		if (desc) return "Makes you be able to burrow into walls. Objects under your feet evaporate.";
#endif

		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		if (cast)
		{
#ifdef JP
			msg_print("ӂ郁fBtŎn߂DDD");
#else
			msg_print("You weave a violent pattern of sounds to break wall.");
#endif
			start_singing(caster_ptr, spell, MUSIC_WALL);
		}

		{
			/*
			 * ̂̊JnɂʔF
			 * MPsŌʂOɉ̂fĂ܂̂h~B
			 */
			if (cont || cast)
			{
				project(caster_ptr, 0, caster_ptr->fy, caster_ptr->fx,
					0, GF_DISINTEGRATE, PROJECT_KILL | PROJECT_ITEM | PROJECT_HIDE, -1);
			}
		}
		break;

	case 17:
#ifdef JP
		if (name) return "fϐ";
		if (desc) return "_AdAACAłɑ΂ϐ𓾂BɂϐɗݐςB";
#else
		if (name) return "Finrod's Resistance";
		if (desc) return "Gives resistance to fire, cold, electricity, acid and poison. These resistances can be added to which from equipment for more powerful resistances.";
#endif
    
		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		if (cast)
		{
#ifdef JP
			msg_print("f̗͂ɑ΂Eς̉̂̂B");
#else
			msg_print("You sing a song of perseverance against powers...");
#endif
			start_singing(caster_ptr, spell, MUSIC_RESIST);
		}

		if (stop)
		{
			if (!caster_ptr->oppose_acid)
			{
#ifdef JP
				msg_print("_ւ̑ϐꂽCB");
#else
				msg_print("You feel less resistant to acid.");
#endif
			}

			if (!caster_ptr->oppose_elec)
			{
#ifdef JP
				msg_print("dւ̑ϐꂽCB");
#else
				msg_print("You feel less resistant to elec.");
#endif
			}

			if (!caster_ptr->oppose_fire)
			{
#ifdef JP
				msg_print("΂ւ̑ϐꂽCB");
#else
				msg_print("You feel less resistant to fire.");
#endif
			}

			if (!caster_ptr->oppose_cold)
			{
#ifdef JP
				msg_print("Cւ̑ϐꂽCB");
#else
				msg_print("You feel less resistant to cold.");
#endif
			}

			if (!caster_ptr->oppose_pois)
			{
#ifdef JP
				msg_print("łւ̑ϐꂽCB");
#else
				msg_print("You feel less resistant to pois.");
#endif
			}
		}

		break;

	case 18:
#ifdef JP
		if (name) return "zrbg̃fB";
		if (desc) return "B";
#else
		if (name) return "Hobbit Melodies";
		if (desc) return "Hastes you.";
#endif

		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		if (cast)
		{
#ifdef JP
			msg_print("yȉ̂ݎn߂DDD");
#else
			msg_print("You start singing joyful pop song...");
#endif
			start_singing(caster_ptr, spell, MUSIC_SPEED);
		}

		if (stop)
		{
			if (!caster_ptr->fast)
			{
#ifdef JP
				msg_print("̑fȂȂ悤B");
#else
				msg_print("You feel yourself slow down.");
#endif
			}
		}

		break;

	case 19:
#ifdef JP
		if (name) return "c񂾐E";
		if (desc) return "߂̃N[`[e|[gBRƖB";
#else
		if (name) return "World Contortion";
		if (desc) return "Teleports all nearby creatures away unless resisted.";
#endif
    
		{
			int rad = plev / 15 + 1;
			int power = plev * 3 + 1;

			if (info) return info_radius(rad);

			/* Stop singing before start another */
			if (cast || fail) stop_singing(caster_ptr);

			if (cast)
			{
#ifdef JP
				msg_print("̂Ԃc߂DDD");
#else
				msg_print("Reality whirls wildly as you sing a dizzying melody...");
#endif

				project(caster_ptr, rad, caster_ptr->fy, caster_ptr->fx, power, GF_AWAY_ALL, PROJECT_KILL, -1);
			}
		}
		break;

	case 20:
#ifdef JP
		if (name) return "ގỦ";
		if (desc) return "ȆSẴN[`[Ƀ_[W^B׈ȃN[`[ɓɑ傫ȃ_[W^B";
#else
		if (name) return "Dispelling chant";
		if (desc) return "Damages all creatures in sight. Hurts evil creatures greatly.";
#endif
    
		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		if (cast)
		{
#ifdef JP
			msg_print("ςȂsaGӂߗĂDDD");
#else
			msg_print("You cry out in an ear-wracking voice...");
#endif
			start_singing(caster_ptr, spell, MUSIC_DISPEL);
		}

		{
			int m_sides = plev * 3;
			int e_sides = plev * 3;

			if (info) return format("%s1d%d+1d%d", s_dam, m_sides, e_sides);

			if (cont)
			{
				dispel_creatures(caster_ptr, randint1(m_sides));
				dispel_evil(caster_ptr, randint1(e_sides));
			}
		}
		break;

	case 21:
#ifdef JP
		if (name) return "T}̊Ì";
		if (desc) return "ȆSẴN[`[A点悤ƂBRƖB";
#else
		if (name) return "The Voice of Saruman";
		if (desc) return "Attempts to slow and sleep all creatures in sight.";
#endif
    
		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		if (cast)
		{
#ifdef JP
			msg_print("DA͓Iȉ̂ݎn߂DDD");
#else
			msg_print("You start humming a gentle and attractive song...");
#endif
			start_singing(caster_ptr, spell, MUSIC_SARUMAN);
		}

		{
			int power = plev;

			if (info) return info_power(power);

			if (cont)
			{
				slow_creatures(caster_ptr);
				sleep_creatures(caster_ptr);
			}
		}

		break;

	case 22:
#ifdef JP
		if (name) return "̉F";
		if (desc) return "̃r[B";
#else
		if (name) return "Song of the Tempest";
		if (desc) return "Fires a beam of sound.";
#endif
    
		{
			int dice = 15 + (plev - 1) / 2;
			int sides = 10;

			if (info) return info_damage(dice, sides, 0);

			/* Stop singing before start another */
			if (cast || fail) stop_singing(caster_ptr);

			if (cast)
			{
				if (!get_aim_dir(caster_ptr, &dir)) return NULL;

				fire_beam(caster_ptr, GF_SOUND, dir, diceroll(dice, sides));
			}
		}
		break;

	case 23:
#ifdef JP
		if (name) return "̐E";
		if (desc) return "݂̊Kč\B";
#else
		if (name) return "Ambarkanta";
		if (desc) return "Recreates current dungeon level.";
#endif
    
		{
			int base = 15;
			int sides = 20;

			if (info) return info_delay(base, sides);

			/* Stop singing before start another */
			if (cast || fail) stop_singing(caster_ptr);

			if (cast)
			{
#ifdef JP
				msg_print("͂ωn߂DDD");
#else
				msg_print("You sing of the primeval shaping of Middle-earth...");
#endif

				alter_reality(caster_ptr);
			}
		}
		break;

	case 24:
#ifdef JP
		if (name) return "j̐";
		if (desc) return "͂̃_Wh炵AǂƏ_ɓςB";
#else
		if (name) return "Wrecking Pattern";
		if (desc) return "Shakes dungeon structure, and results in random swapping of floors and walls.";
#endif

		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		if (cast)
		{
#ifdef JP
			msg_print("jIȉ̂킽DDD");
#else
			msg_print("You weave a pattern of sounds to contort and shatter...");
#endif
			start_singing(caster_ptr, spell, MUSIC_QUAKE);
		}

		{
			int rad = 10;

			if (info) return info_radius(rad);

			if (cont)
			{
				earthquake(caster_ptr, caster_ptr->fy, caster_ptr->fx, 10);
			}
		}

		break;


	case 25:
#ifdef JP
		if (name) return "؂̉";
		if (desc) return "ȆSẴN[`[Ⴢ悤ƂBRƖB";
#else
		if (name) return "Stationary Shriek";
		if (desc) return "Attempts to freeze all creatures in sight.";
#endif
    
		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		if (cast)
		{
#ifdef JP
			msg_print("ƂfBtŎn߂DDD");
#else
			msg_print("You weave a very slow pattern which is almost likely to stop...");
#endif
			start_singing(caster_ptr, spell, MUSIC_STASIS);
		}

		{
			int power = plev * 4;

			if (info) return info_power(power);

			if (cont)
			{
				stasis_creatures(caster_ptr, power);
			}
		}

		break;

	case 26:
#ifdef JP
		if (name) return "̉";
		if (desc) return "̂鏰̏ɁAN[`[ʂ蔲菢ꂽ肷邱ƂłȂȂ郋[`B";
#else
		if (name) return "Endurance";
		if (desc) return "Sets a glyph on the floor beneath you. Creatures cannot attack you if you are on a glyph, but can try to break glyph.";
#endif
    
		{
			/* Stop singing before start another */
			if (cast || fail) stop_singing(caster_ptr);

			if (cast)
			{
#ifdef JP
				msg_print("̂_ȏoDDD");
#else
				msg_print("The holy power of the Music is creating sacred field...");
#endif

				warding_glyph(caster_ptr);
			}
		}
		break;

	case 27:
#ifdef JP
		if (name) return "pY̎";
		if (desc) return "Aq[[CɂȂAȆSẴN[`[Ƀ_[W^B";
#else
		if (name) return "The Hero's Poem";
		if (desc) return "Hastes you. Gives heroism. Damages all creatures in sight.";
#endif
    
		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		if (cast)
		{
#ifdef JP
			msg_print("pỶ̂񂾁DDD");
#else
			msg_print("You chant a powerful, heroic call to arms...");
#endif
			(void)hp_player(caster_ptr, 10);
			(void)set_afraid(caster_ptr, 0);

			/* Recalculate hitpoints */
			caster_ptr->creature_update |= (CRU_HP);

			start_singing(caster_ptr, spell, MUSIC_SHERO);
		}

		if (stop)
		{
			if (!caster_ptr->hero)
			{
#ifdef JP
				msg_print("q[[̋CB");
#else
				msg_print("The heroism wears off.");
#endif
				/* Recalculate hitpoints */
				caster_ptr->creature_update |= (CRU_HP);
			}

			if (!caster_ptr->fast)
			{
#ifdef JP
				msg_print("̑fȂȂ悤B");
#else
				msg_print("You feel yourself slow down.");
#endif
			}
		}

		{
			int dice = 1;
			int sides = plev * 3;

			if (info) return info_damage(dice, sides, 0);

			if (cont)
			{
				dispel_creatures(caster_ptr, diceroll(dice, sides));
			}
		}
		break;

	case 28:
#ifdef JP
		if (name) return "@ȉ";
		if (desc) return "͂ȉ񕜂̉̂ŁAƞNOԂSB";
#else
		if (name) return "Relief of Yavanna";
		if (desc) return "Powerful healing song. Also heals cut and stun completely.";
#endif
    
		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		if (cast)
		{
#ifdef JP
			msg_print("̂ʂđ̂ɊC߂ĂDDD");
#else
			msg_print("Life flows through you as you sing the song...");
#endif
			start_singing(caster_ptr, spell, MUSIC_H_LIFE);
		}

		{
			int dice = 15;
			int sides = 10;

			if (info) return info_heal(dice, sides, 0);

			if (cont)
			{
				hp_player(caster_ptr, diceroll(dice, sides));
				set_stun(caster_ptr, 0);
				set_cut(caster_ptr, 0);
			}
		}

		break;

	case 29:
#ifdef JP
		if (name) return "Đ̉";
		if (desc) return "ׂẴXe[^Xƌol񕜂B";
#else
		if (name) return "Goddess' rebirth";
		if (desc) return "Restores all stats and experience.";
#endif
    
		{
			/* Stop singing before start another */
			if (cast || fail) stop_singing(caster_ptr);

			if (cast)
			{
#ifdef JP
				msg_print("Í̒ɌƔӂ܂B̂̊͂߂B");
#else
				msg_print("You strewed light and beauty in the dark as you sing. You feel refreshed.");
#endif
				(void)do_res_stat(caster_ptr, STAT_STR);
				(void)do_res_stat(caster_ptr, STAT_INT);
				(void)do_res_stat(caster_ptr, STAT_WIS);
				(void)do_res_stat(caster_ptr, STAT_DEX);
				(void)do_res_stat(caster_ptr, STAT_CON);
				(void)do_res_stat(caster_ptr, STAT_CHA);
				(void)restore_level(caster_ptr);
			}
		}
		break;

	case 30:
#ifdef JP
		if (name) return "TE̖p";
		if (desc) return "ɋ͂ł̋B";
#else
		if (name) return "Wizardry of Sauron";
		if (desc) return "Fires an extremely powerful tiny ball of sound.";
#endif
    
		{
			int dice = 50 + plev;
			int sides = 10;
			int rad = 0;

			if (info) return info_damage(dice, sides, 0);

			/* Stop singing before start another */
			if (cast || fail) stop_singing(caster_ptr);

			if (cast)
			{
				if (!get_aim_dir(caster_ptr, &dir)) return NULL;

				fire_ball(caster_ptr, GF_SOUND, dir, diceroll(dice, sides), rad);
			}
		}
		break;

	case 31:
#ifdef JP
		if (name) return "tBStB̒";
		if (desc) return "_[W󂯂ȂȂoA𒣂B";
#else
		if (name) return "Fingolfin's Challenge";
		if (desc) return "Generates barrier which completely protect you from almost all damages. Takes a few your turns when the barrier breaks.";
#endif
    
		/* Stop singing before start another */
		if (cast || fail) stop_singing(caster_ptr);

		if (cast)
		{
#ifdef JP
				msg_print("tBStB̖ւ̒̂DDD");
#else
				msg_print("You recall the valor of Fingolfin's challenge to the Dark Lord...");
#endif

				/* Redraw map */
				play_redraw |= (PR_MAP);
		
				// Update creatures
				caster_ptr->creature_update |= (PU_MONSTERS);
		
				/* Window stuff */
				play_window |= (PW_OVERHEAD | PW_DUNGEON);

				start_singing(caster_ptr, spell, MUSIC_INVULN);
		}

		if (stop)
		{
			if (!caster_ptr->invuln)
			{
#ifdef JP
				msg_print("Gł͂ȂȂB");
#else
				msg_print("The invulnerability wears off.");
#endif
				/* Redraw map */
				play_redraw |= (PR_MAP);

				// Update creatures
				caster_ptr->creature_update |= (PU_MONSTERS);

				/* Window stuff */
				play_window |= (PW_OVERHEAD | PW_DUNGEON);
			}
		}

		break;
	}

	return "";
}


static cptr do_hissatsu_spell(creature_type *caster_ptr, int spell, int mode)
{
	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;

	int dir;
	int plev = caster_ptr->lev;
	floor_type *floor_ptr = get_floor_ptr(caster_ptr);

	switch (spell)
	{
	case 0:
#ifdef JP
		if (name) return "эj";
		if (desc) return "2}XꂽƂɂN[`[UB";
#else
		if (name) return "Tobi-Izuna";
		if (desc) return "Attacks a two squares distant creature.";
#endif
    
		if (cast)
		{
			project_length = 2;
			if (!get_aim_dir(caster_ptr, &dir)) return NULL;

			project_hook(caster_ptr, GF_ATTACK, dir, HISSATSU_2, PROJECT_STOP | PROJECT_KILL);
		}
		break;

	case 1:
#ifdef JP
		if (name) return "܌Ja";
		if (desc) return "3ɑ΂čUB";
#else
		if (name) return "3-Way Attack";
		if (desc) return "Attacks in 3 directions in one time.";
#endif
    
		if (cast)
		{
			int cdir;
			int y, x;

			if (!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if (dir == 5) return NULL;

			for (cdir = 0;cdir < 8; cdir++)
			{
				if (cdd[cdir] == dir) break;
			}

			if (cdir == 8) return NULL;

			y = caster_ptr->fy + ddy_cdd[cdir];
			x = caster_ptr->fx + ddx_cdd[cdir];
			if (floor_ptr->cave[y][x].creature_idx)
				melee_attack(caster_ptr, y, x, 0);
			else
#ifdef JP
				msg_print("U͋؂B");
#else
				msg_print("You attack the empty air.");
#endif
			y = caster_ptr->fy + ddy_cdd[(cdir + 7) % 8];
			x = caster_ptr->fx + ddx_cdd[(cdir + 7) % 8];
			if (floor_ptr->cave[y][x].creature_idx)
				melee_attack(caster_ptr, y, x, 0);
			else
#ifdef JP
				msg_print("U͋؂B");
#else
				msg_print("You attack the empty air.");
#endif
			y = caster_ptr->fy + ddy_cdd[(cdir + 1) % 8];
			x = caster_ptr->fx + ddx_cdd[(cdir + 1) % 8];
			if (floor_ptr->cave[y][x].creature_idx)
				melee_attack(caster_ptr, y, x, 0);
			else
#ifdef JP
				msg_print("U͋؂B");
#else
				msg_print("You attack the empty air.");
#endif
		}
		break;

	case 2:
#ifdef JP
		if (name) return "u[";
		if (desc) return "茳ɖ߂Ă悤ɓB߂ĂȂƂB";
#else
		if (name) return "Boomerang";
		if (desc) return "Throws current weapon. And it'll return to your hand unless failed.";
#endif
    
		if (cast)
		{
			if (!do_cmd_throw_aux(caster_ptr, 1, TRUE, 0)) return NULL;
		}
		break;

	case 3:
#ifdef JP
		if (name) return "";
		if (desc) return "Ήϐ̂ȂN[`[ɑ_[W^B";
#else
		if (name) return "Burning Strike";
		if (desc) return "Attacks a creature with more damage unless it has resistance to fire.";
#endif
    
		if (cast)
		{
			int y, x;

			if (!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if (dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if (floor_ptr->cave[y][x].creature_idx)
				melee_attack(caster_ptr, y, x, HISSATSU_FIRE);
			else
			{
#ifdef JP
				msg_print("̕ɂ̓N[`[͂܂B");
#else
				msg_print("There is no creature.");
#endif
				return NULL;
			}
		}
		break;

	case 4:
#ifdef JP
		if (name) return "ECm";
		if (desc) return "߂̎vl邱ƂłN[`[mB";
#else
		if (name) return "Detect Ferocity";
		if (desc) return "Detects all creatures except mindless in your vicinity.";
#endif
    
		if (cast)
		{
			detect_creatures_mind(caster_ptr, DETECT_RAD_DEFAULT);
		}
		break;

	case 5:
#ifdef JP
		if (name) return "݂ˑł";
		if (desc) return "Ƀ_[W^ȂANOƂB";
#else
		if (name) return "Strike to Stun";
		if (desc) return "Attempts to stun a creature in the adjacent.";
#endif
    
		if (cast)
		{
			int y, x;

			if (!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if (dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if (floor_ptr->cave[y][x].creature_idx)
				melee_attack(caster_ptr, y, x, HISSATSU_MINEUCHI);
			else
			{
#ifdef JP
				msg_print("̕ɂ̓N[`[͂܂B");
#else
				msg_print("There is no creature.");
#endif
				return NULL;
			}
		}
		break;

	case 6:
#ifdef JP
		if (name) return "JE^[";
		if (desc) return "ɍUꂽƂɔB邽тMPB";
#else
		if (name) return "Counter";
		if (desc) return "Prepares to counterattack. When attack by a creature, strikes back using SP each time.";
#endif
    
		if (cast)
		{
			if (caster_ptr->riding)
			{
#ifdef JP
				msg_print("nɂ͖B");
#else
				msg_print("You cannot do it when riding.");
#endif
				return NULL;
			}
#ifdef JP
			msg_print("̍Uɑ΂Đg\B");
#else
			msg_print("You prepare to counter blow.");
#endif
			caster_ptr->counter = TRUE;
		}
		break;

	case 7:
#ifdef JP
		if (name) return "";
		if (desc) return "UAΑɔB";
#else
		if (name) return "Harainuke";
		if (desc) return "Attacks creature with your weapons normally, then move through counter side of the creature.";
#endif
    
		if (cast)
		{
			int y, x;

			if (caster_ptr->riding)
			{
#ifdef JP
				msg_print("nɂ͖B");
#else
				msg_print("You cannot do it when riding.");
#endif
				return NULL;
			}
	
			if (!get_rep_dir2(caster_ptr, &dir)) return NULL;
	
			if (dir == 5) return NULL;
			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];
	
			if (!floor_ptr->cave[y][x].creature_idx)
			{
#ifdef JP
				msg_print("̕ɂ̓N[`[͂܂B");
#else
				msg_print("There is no creature.");
#endif
				return NULL;
			}
	
			melee_attack(caster_ptr, y, x, 0);
	
			if (!creature_can_cross_terrain(caster_ptr, floor_ptr->cave[y][x].feat, 0) || is_trap(floor_ptr->cave[y][x].feat))
				break;
	
			y += ddy[dir];
			x += ddx[dir];
	
			if (creature_can_cross_terrain(caster_ptr, floor_ptr->cave[y][x].feat, 0) && !is_trap(floor_ptr->cave[y][x].feat) && !floor_ptr->cave[y][x].creature_idx)
			{
				msg_print(NULL);
	
				/* Move the player */
				(void)move_creature_effect(caster_ptr, NULL, y, x, MPE_FORGET_FLOW | MPE_HANDLE_STUFF | MPE_DONT_PICKUP);
			}
		}
		break;

	case 8:
#ifdef JP
		if (name) return "T[yc^";
		if (desc) return "őϐ̂ȂN[`[ɑ_[W^B";
#else
		if (name) return "Serpent's Tongue";
		if (desc) return "Attacks a creature with more damage unless it has resistance to poison.";
#endif
    
		if (cast)
		{
			int y, x;

			if (!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if (dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if (floor_ptr->cave[y][x].creature_idx)
				melee_attack(caster_ptr, y, x, HISSATSU_POISON);
			else
			{
#ifdef JP
				msg_print("̕ɂ̓N[`[͂܂B");
#else
				msg_print("There is no creature.");
#endif
				return NULL;
			}
		}
		break;

	case 9:
#ifdef JP
		if (name) return "ȃ";
		if (desc) return "̂Ȃ׈ȃN[`[ɑ_[W^邪ÃN[`[ɂ͑SʂȂB";
#else
		if (name) return "Zammaken";
		if (desc) return "Attacks an evil unliving creature with great damage. No effect to other  creatures.";
#endif
    
		if (cast)
		{
			int y, x;

			if (!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if (dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if (floor_ptr->cave[y][x].creature_idx)
				melee_attack(caster_ptr, y, x, HISSATSU_ZANMA);
			else
			{
#ifdef JP
				msg_print("̕ɂ̓N[`[͂܂B");
#else
				msg_print("There is no creature.");
#endif
				return NULL;
			}
		}
		break;

	case 10:
#ifdef JP
		if (name) return "";
		if (desc) return "U֐΂B";
#else
		if (name) return "Wind Blast";
		if (desc) return "Attacks an adjacent creature, and blow it away.";
#endif
    
		if (cast)
		{
			int y, x;

			if (!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if (dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if (floor_ptr->cave[y][x].creature_idx)
				melee_attack(caster_ptr, y, x, 0);
			else
			{
#ifdef JP
				msg_print("̕ɂ̓N[`[͂܂B");
#else
				msg_print("There is no creature.");
#endif
				return NULL;
			}
			if (dungeon_info[floor_ptr->dun_type].flags1 & DF1_NO_MELEE)
			{
				return "";
			}
			if (floor_ptr->cave[y][x].creature_idx)
			{
				int i;
				int ty = y, tx = x;
				int oy = y, ox = x;
				int m_idx = floor_ptr->cave[y][x].creature_idx;
				creature_type *m_ptr = &creature_list[m_idx];
				char m_name[80];
	
				creature_desc(m_name, m_ptr, 0);
	
				for (i = 0; i < 5; i++)
				{
					y += ddy[dir];
					x += ddx[dir];
					if (cave_empty_bold(floor_ptr, y, x))
					{
						ty = y;
						tx = x;
					}
					else break;
				}
				if ((ty != oy) || (tx != ox))
				{
#ifdef JP
					msg_format("%s𐁂΂I", m_name);
#else
					msg_format("You blow %s away!", m_name);
#endif
					floor_ptr->cave[oy][ox].creature_idx = 0;
					floor_ptr->cave[ty][tx].creature_idx = m_idx;
					m_ptr->fy = ty;
					m_ptr->fx = tx;
	
					update_creature_view(player_ptr, m_idx, TRUE);
					lite_spot(floor_ptr, oy, ox);
					lite_spot(floor_ptr, ty, tx);
	
					if (is_lighting_creature(m_ptr) || is_darken_creature(m_ptr))
						caster_ptr->creature_update |= (PU_SPECIES_LITE);
				}
			}
		}
		break;

	case 11:
#ifdef JP
		if (name) return "̖ڗ";
		if (desc) return "Eh1ʂBx45ȏŕEh̔\͂Sɒm邱ƂłB";
#else
		if (name) return "Judge";
		if (desc) return "Identifies a weapon or armor. Or *identifies* these at level 45.";
#endif
    
		if (cast)
		{
			if (plev > 44)
			{
				if (!identify_fully(caster_ptr, TRUE)) return NULL;
			}
			else
			{
				if (!ident_spell(caster_ptr, TRUE)) return NULL;
			}
		}
		break;

	case 12:
#ifdef JP
		if (name) return "ja";
		if (desc) return "󂵁AΌñN[`[ɑ_[W^B";
#else
		if (name) return "Rock Smash";
		if (desc) return "Breaks rock. Or greatly damage a creature made by rocks.";
#endif
    
		if (cast)
		{
			int y, x;

			if (!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if (dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if (floor_ptr->cave[y][x].creature_idx)
				melee_attack(caster_ptr, y, x, HISSATSU_HAGAN);
	
			if (!cave_have_flag_bold(floor_ptr, y, x, FF_HURT_ROCK)) break;
	
			/* Destroy the feature */
			cave_alter_feat(floor_ptr, y, x, FF_HURT_ROCK);
	
			/* Update some things */
			update |= (PU_FLOW);
		}
		break;

	case 13:
#ifdef JP
		if (name) return "ጎ";
		if (desc) return "U񐔂ACϐ̂ȂN[`[ɑ_[W^B";
#else
		if (name) return "Midare-Setsugekka";
		if (desc) return "Attacks a creature with increased number of attacks and more damage unless it has resistance to cold.";
#endif
    
		if (cast)
		{
			int y, x;

			if (!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if (dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if (floor_ptr->cave[y][x].creature_idx)
				melee_attack(caster_ptr, y, x, HISSATSU_COLD);
			else
			{
#ifdef JP
				msg_print("̕ɂ̓N[`[͂܂B");
#else
				msg_print("There is no creature.");
#endif
				return NULL;
			}
		}
		break;

	case 14:
#ifdef JP
		if (name) return "}˂";
		if (desc) return "N[`[ꌂœ|UJoBs1__[W^ȂB";
#else
		if (name) return "Spot Aiming";
		if (desc) return "Attempts to kill a creature instantly. If failed cause only 1HP of damage.";
#endif
    
		if (cast)
		{
			int y, x;

			if (!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if (dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if (floor_ptr->cave[y][x].creature_idx)
				melee_attack(caster_ptr, y, x, HISSATSU_KYUSHO);
			else
			{
#ifdef JP
				msg_print("̕ɂ̓N[`[͂܂B");
#else
				msg_print("There is no creature.");
#endif
				return NULL;
			}
		}
		break;

	case 15:
#ifdef JP
		if (name) return "_a";
		if (desc) return "S̈ꌂōUBU킳₷B";
#else
		if (name) return "Majingiri";
		if (desc) return "Attempts to attack with critical hit. But this attack is easy to evade for a creature.";
#endif
    
		if (cast)
		{
			int y, x;

			if (!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if (dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if (floor_ptr->cave[y][x].creature_idx)
				melee_attack(caster_ptr, y, x, HISSATSU_MAJIN);
			else
			{
#ifdef JP
				msg_print("̕ɂ̓N[`[͂܂B");
#else
				msg_print("There is no creature.");
#endif
				return NULL;
			}
		}
		break;

	case 16:
#ifdef JP
		if (name) return "̂Đg";
		if (desc) return "͂ȍUJoB̃^[܂ł̊ԁAH炤_[WB";
#else
		if (name) return "Desperate Attack";
		if (desc) return "Attacks with all of your power. But all damages you take will be doubled for one turn.";
#endif
    
		if (cast)
		{
			int y, x;

			if (!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if (dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if (floor_ptr->cave[y][x].creature_idx)
				melee_attack(caster_ptr, y, x, HISSATSU_SUTEMI);
			else
			{
#ifdef JP
				msg_print("̕ɂ̓N[`[͂܂B");
#else
				msg_print("There is no creature.");
#endif
				return NULL;
			}
			caster_ptr->sutemi = TRUE;
		}
		break;

	case 17:
#ifdef JP
		if (name) return "h܎a";
		if (desc) return "dϐ̂ȂN[`[ɔɑ傫_[W^B";
#else
		if (name) return "Lightning Eagle";
		if (desc) return "Attacks a creature with more damage unless it has resistance to electricity.";
#endif
    
		if (cast)
		{
			int y, x;

			if (!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if (dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if (floor_ptr->cave[y][x].creature_idx)
				melee_attack(caster_ptr, y, x, HISSATSU_ELEC);
			else
			{
#ifdef JP
				msg_print("̕ɂ̓N[`[͂܂B");
#else
				msg_print("There is no creature.");
#endif
				return NULL;
			}
		}
		break;

	case 18:
#ifdef JP
		if (name) return "g";
		if (desc) return "fɋߊUB";
#else
		if (name) return "Rush Attack";
		if (desc) return "Steps close to a creature and attacks at a time.";
#endif
    
		if (cast)
		{
			if (!rush_attack(caster_ptr, NULL)) return NULL;
		}
		break;

	case 19:
#ifdef JP
		if (name) return "ԗQ";
		if (desc) return "gȀ[قǑ傫З͂őS̓GUłBĂȂN[`[ɂ͌ʂȂB";
#else
		if (name) return "Bloody Maelstrom";
		if (desc) return "Attacks all adjacent creatures with power corresponding to your cut status. Then increases your cut status. No effect to unliving creatures.";
#endif
    
		if (cast)
		{
			int y = 0, x = 0;

			cave_type       *c_ptr;
			creature_type    *m_ptr;
	
			if (caster_ptr->cut < 300)
				set_cut(caster_ptr, caster_ptr->cut + 300);
			else
				set_cut(caster_ptr, caster_ptr->cut * 2);
	
			for (dir = 0; dir < 8; dir++)
			{
				y = caster_ptr->fy + ddy_ddd[dir];
				x = caster_ptr->fx + ddx_ddd[dir];
				c_ptr = &floor_ptr->cave[y][x];
	
				/* Get the creature */
				m_ptr = &creature_list[c_ptr->creature_idx];
	
				/* Hack -- attack creatures */
				if (c_ptr->creature_idx && (m_ptr->ml || cave_have_flag_bold(floor_ptr, y, x, FF_PROJECT)))
				{
					if (!creature_living(m_ptr))
					{
						char m_name[80];
	
						creature_desc(m_name, m_ptr, 0);
#ifdef JP
						msg_format("%sɂ͌ʂȂI", m_name);
#else
						msg_format("%s is unharmed!", m_name);
#endif
					}
					else melee_attack(caster_ptr, y, x, HISSATSU_SEKIRYUKA);
				}
			}
		}
		break;

	case 20:
#ifdef JP
		if (name) return "k";
		if (desc) return "nkNB";
#else
		if (name) return "Earthquake Blow";
		if (desc) return "Shakes dungeon structure, and results in random swapping of floors and walls.";
#endif
    
		if (cast)
		{
			int y,x;

			if (!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if (dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if (floor_ptr->cave[y][x].creature_idx)
				melee_attack(caster_ptr, y, x, HISSATSU_QUAKE);
			else
				earthquake(caster_ptr, caster_ptr->fy, caster_ptr->fx, 10);
		}
		break;

	case 21:
#ifdef JP
		if (name) return "n";
		if (desc) return "Ռg̃r[B";
#else
		if (name) return "Crack";
		if (desc) return "Fires a beam of shock wave.";
#endif
    
		if (cast)
		{
			int total_damage = 0, basedam, i;
			u32b flgs[TR_FLAG_SIZE];
			object_type *object_ptr;
			if (!get_aim_dir(caster_ptr, &dir)) return NULL;
#ifdef JP
			msg_print("傫U艺낵B");
#else
			msg_print("You swing your weapon downward.");
#endif
			for (i = 0; i < 2; i++)
			{
				int damage;
	
				if (!get_equipped_slot_ptr(caster_ptr, INVEN_SLOT_HAND, i+1)) break;
				object_ptr = get_equipped_slot_ptr(caster_ptr, INVEN_SLOT_HAND, i+1);
				basedam = (object_ptr->dd * (object_ptr->ds + 1)) * 50;
				damage = object_ptr->to_damage * 100;
				object_flags(object_ptr, flgs);
				if ((object_ptr->name1 == ART_VORPAL_BLADE) || (object_ptr->name1 == ART_CHAINSWORD))
				{
					/* vorpal blade */
					basedam *= 5;
					basedam /= 3;
				}
				else if (have_flag(flgs, TRAIT_VORPAL))
				{
					/* vorpal flag only */
					basedam *= 11;
					basedam /= 9;
				}
				damage += basedam;
				total_damage += damage / 200;
				if (i) total_damage = total_damage*7/10;
			}
			fire_beam(caster_ptr, GF_FORCE, dir, total_damage);
		}
		break;

	case 22:
#ifdef JP
		if (name) return "C̗Y";
		if (desc) return "ȆSN[`[ɑ΂č̍UsBɁA߂ɂN[`[{点B";
#else
		if (name) return "War Cry";
		if (desc) return "Damages all creatures in sight with sound. Aggravate nearby creatures.";
#endif
    
		if (cast)
		{
#ifdef JP
			msg_print("YтI");
#else
			msg_print("You roar out!");
#endif
			project_hack(caster_ptr, GF_SOUND, randint1(plev * 3));
			aggravate_creatures(caster_ptr);
		}
		break;

	case 23:
#ifdef JP
		if (name) return "oOi";
		if (desc) return "͂3iUJoB";
#else
		if (name) return "Musou-Sandan";
		if (desc) return "Attacks with powerful 3 strikes.";
#endif
    
		if (cast)
		{
			int i;

			if (!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if (dir == 5) return NULL;

			for (i = 0; i < 3; i++)
			{
				int y, x;
				int ny, nx;
				int m_idx;
				cave_type *c_ptr;
				creature_type *m_ptr;
	
				y = caster_ptr->fy + ddy[dir];
				x = caster_ptr->fx + ddx[dir];
				c_ptr = &floor_ptr->cave[y][x];
	
				if (c_ptr->creature_idx)
					melee_attack(caster_ptr, y, x, HISSATSU_3DAN);
				else
				{
#ifdef JP
					msg_print("̕ɂ̓N[`[͂܂B");
#else
					msg_print("There is no creature.");
#endif
					return NULL;
				}
	
				if (dungeon_info[floor_ptr->dun_type].flags1 & DF1_NO_MELEE)
				{
					return "";
				}
	
				/* Creature is dead? */
				if (!c_ptr->creature_idx) break;
	
				ny = y + ddy[dir];
				nx = x + ddx[dir];
				m_idx = c_ptr->creature_idx;
				m_ptr = &creature_list[m_idx];
	
				/* Creature cannot move back? */
				if (!creature_can_enter(ny, nx, m_ptr, 0))
				{
					/* -more- */
					if (i < 2) msg_print(NULL);
					continue;
				}
	
				c_ptr->creature_idx = 0;
				floor_ptr->cave[ny][nx].creature_idx = m_idx;
				m_ptr->fy = ny;
				m_ptr->fx = nx;
	
				update_creature_view(player_ptr, m_idx, TRUE);
	
				/* Redraw the old spot */
				lite_spot(floor_ptr, y, x);
	
				/* Redraw the new spot */
				lite_spot(floor_ptr, ny, nx);
	
				/* Player can move forward? */
				if (creature_can_cross_terrain(caster_ptr, c_ptr->feat, 0))
				{
					/* Move the player */
					if (!move_creature_effect(caster_ptr, NULL, y, x, MPE_FORGET_FLOW | MPE_HANDLE_STUFF | MPE_DONT_PICKUP)) break;
				}
				else
				{
					break;
				}

				/* -more- */
				if (i < 2) msg_print(NULL);
			}
		}
		break;

	case 24:
#ifdef JP
		if (name) return "zS̉";
		if (desc) return "Ȗ̗͂zƂȂ̗͂񕜂BȂN[`[ɂ͒ʂȂB";
#else
		if (name) return "Vampire's Fang";
		if (desc) return "Attacks with vampiric strikes which absorbs HP from a creature and gives them to you. No effect to unliving creatures.";
#endif
    
		if (cast)
		{
			int y, x;

			if (!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if (dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if (floor_ptr->cave[y][x].creature_idx)
				melee_attack(caster_ptr, y, x, HISSATSU_DRAIN);
			else
			{
#ifdef JP
					msg_print("̕ɂ̓N[`[͂܂B");
#else
					msg_print("There is no creature.");
#endif
				return NULL;
			}
		}
		break;

	case 25:
#ifdef JP
		if (name) return "f";
		if (desc) return "E̋NĂSN[`[ɞNOAA^悤ƂB";
#else
		if (name) return "Moon Dazzling";
		if (desc) return "Attempts to stun, confuse and sleep all waking creatures.";
#endif
    
		if (cast)
		{
#ifdef JP
			msg_print("sKɗh炵DDD");
#else
			msg_print("You irregularly wave your weapon...");
#endif
			project_hack(caster_ptr, GF_ENGETSU, plev * 4);
			project_hack(caster_ptr, GF_ENGETSU, plev * 4);
			project_hack(caster_ptr, GF_ENGETSU, plev * 4);
		}
		break;

	case 26:
#ifdef JP
		if (name) return "Sla";
		if (desc) return "AēgŃN[`[UBU邽тMPBMPȂȂ邩AN[`[|ȂSla͏IB";
#else
		if (name) return "Hundred Slaughter";
		if (desc) return "Performs a series of rush attacks. The series continues while killing each creature in a time and SP remains.";
#endif
    
		if (cast)
		{
			const int mana_cost_per_creature = 8;
			bool new = TRUE;
			bool mdeath;

			do
			{
				if (!rush_attack(caster_ptr, &mdeath)) break;
				if (new)
				{
					/* Reserve needed mana point */
					caster_ptr->csp -= technic_info[REALM_HISSATSU - MIN_TECHNIC][26].smana;
					new = FALSE;
				}
				else
					caster_ptr->csp -= mana_cost_per_creature;

				if (!mdeath) break;
				command_dir = 0;

				play_redraw |= PR_MANA;
				handle_stuff();
			}
			while (caster_ptr->csp > mana_cost_per_creature);

			if (new) return NULL;
	
			/* Restore reserved mana */
			caster_ptr->csp += technic_info[REALM_HISSATSU - MIN_TECHNIC][26].smana;
		}
		break;

	case 27:
#ifdef JP
		if (name) return "VėM";
		if (desc) return "Ȅꏊw肵āȀꏊƎ̊ԂɂSN[`[UȀꏊɈړB";
#else
		if (name) return "Dragonic Flash";
		if (desc) return "Runs toward given location while attacking all creatures on the path.";
#endif
    
		if (cast)
		{
			int y, x;

			if (!tgt_pt(caster_ptr, &x, &y)) return NULL;

			if (!cave_player_teleportable_bold(caster_ptr, y, x, 0L) ||
			    (distance(y, x, caster_ptr->fy, caster_ptr->fx) > MAX_SIGHT / 2) ||
			    !projectable(floor_ptr, caster_ptr->fy, caster_ptr->fx, y, x))
			{
#ifdef JP
				msg_print("sI");
#else
				msg_print("You cannot move to that place!");
#endif
				break;
			}
			if (has_trait(caster_ptr, TRAIT_PREVENT_TELEPORT))
			{
#ifdef JP
				msg_print("svcȗ͂e|[ghI");
#else
				msg_print("A mysterious force prevents you from teleporting!");
#endif
	
				break;
			}
			project(caster_ptr, 0, y, x, HISSATSU_ISSEN, GF_ATTACK, PROJECT_BEAM | PROJECT_KILL, -1);
			teleport_creature_to(caster_ptr, y, x, 0L);
		}
		break;

	case 28:
#ifdef JP
		if (name) return "ď";
		if (desc) return "1^[2xUsB";
#else
		if (name) return "Twin Slash";
		if (desc) return "double attacks at a time.";
#endif
    
		if (cast)
		{
			int x, y;
	
			if (!get_rep_dir(caster_ptr, &dir, FALSE)) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if (floor_ptr->cave[y][x].creature_idx)
			{
				melee_attack(caster_ptr, y, x, 0);
				if (floor_ptr->cave[y][x].creature_idx)
				{
					handle_stuff();
					melee_attack(caster_ptr, y, x, 0);
				}
			}
			else
			{
#ifdef JP
	msg_print("̕ɂ̓N[`[͂܂B");
#else
				msg_print("You don't see any creature in this direction");
#endif
				return NULL;
			}
		}
		break;

	case 29:
#ifdef JP
		if (name) return "ՕⓁ";
		if (desc) return "͂ȍUsA߂̏ꏊɂʂyԁB";
#else
		if (name) return "Kofuku-Zettousei";
		if (desc) return "Performs a powerful attack which even effect nearby creatures.";
#endif
    
		if (cast)
		{
			int total_damage = 0, basedam, i;
			int y, x;
			u32b flgs[TR_FLAG_SIZE];
			object_type *object_ptr;
	
			if (!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if (dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if (dungeon_info[floor_ptr->dun_type].flags1 & DF1_NO_MELEE)
			{
#ifdef JP
				msg_print("ȂU邱ƂłȂB");
#else
				msg_print("Something prevent you from attacking.");
#endif
				return "";
			}
#ifdef JP
			msg_print("傫U艺낵B");
#else
			msg_print("You swing your weapon downward.");
#endif
			for (i = 0; i < 2; i++)
			{
				int damage;
				if (!get_equipped_slot_ptr(caster_ptr, INVEN_SLOT_HAND, i+1)) break;
				object_ptr = get_equipped_slot_ptr(caster_ptr, INVEN_SLOT_HAND, i+1);
				basedam = (object_ptr->dd * (object_ptr->ds + 1)) * 50;
				damage = object_ptr->to_damage * 100;
				object_flags(object_ptr, flgs);
				if ((object_ptr->name1 == ART_VORPAL_BLADE) || (object_ptr->name1 == ART_CHAINSWORD))
				{
					/* vorpal blade */
					basedam *= 5;
					basedam /= 3;
				}
				else if (have_flag(flgs, TRAIT_VORPAL))
				{
					/* vorpal flag only */
					basedam *= 11;
					basedam /= 9;
				}
				damage += basedam;
				damage += caster_ptr->to_damage[i] * 100;
				total_damage += (damage / 100);
			}
			project(caster_ptr, (cave_have_flag_bold(floor_ptr, y, x, FF_PROJECT) ? 5 : 0), y, x, total_damage * 3 / 2, GF_METEOR, PROJECT_KILL | PROJECT_JUMP | PROJECT_ITEM, -1);
		}
		break;

	case 30:
#ifdef JP
		if (name) return "c_SE";
		if (desc) return "_[W炤Aɔɑ傫ȃ_[W^BAfbhɂ͓ɌʂB";
#else
		if (name) return "Keiun-Kininken";
		if (desc) return "Attacks a creature with extremely powerful damage. But you also takes some damages. Hurts a undead creature greatly.";
#endif
    
		if (cast)
		{
			int y, x;

			if (!get_rep_dir2(caster_ptr, &dir)) return NULL;
			if (dir == 5) return NULL;

			y = caster_ptr->fy + ddy[dir];
			x = caster_ptr->fx + ddx[dir];

			if (floor_ptr->cave[y][x].creature_idx)
				melee_attack(caster_ptr, y, x, HISSATSU_UNDEAD);
			else
			{
#ifdef JP
				msg_print("̕ɂ̓N[`[͂܂B");
#else
				msg_print("There is no creature.");
#endif
				return NULL;
			}
#ifdef JP
			take_hit(NULL, caster_ptr, DAMAGE_NOESCAPE, 100 + randint1(100), "c_SEgՌ", NULL, -1);
#else
			take_hit(NULL, caster_ptr, DAMAGE_NOESCAPE, 100 + randint1(100), "exhaustion on using Keiun-Kininken", NULL, -1);
#endif
		}
		break;

	case 31:
#ifdef JP
		if (name) return "ؕ";
		if (desc) return "umƂ́AʂƂƌBv";
#else
		if (name) return "Harakiri";
		if (desc) return "'Busido is found in death'";
#endif

		if (cast)
		{
			int i;
#ifdef JP
	if (!get_check("{ɎE܂H")) return NULL;
#else
			if (!get_check("Do you really want to commit suicide? ")) return NULL;
#endif
				/* Special Verification for suicide */
#ifdef JP
	prt("mF̂ '@' ĉB", 0, 0);
#else
			prt("Please verify SUICIDE by typing the '@' sign: ", 0, 0);
#endif
	
			flush();
			i = inkey();
			prt("", 0, 0);
			if (i != '@') return NULL;
			if (caster_ptr->total_winner)
			{
				take_hit(NULL, caster_ptr, DAMAGE_FORCE, 9999, "Seppuku", NULL, -1);
				caster_ptr->total_winner = TRUE;
			}
			else
			{
#ifdef JP
				msg_print("mƂ́AʂƂƌB");
#else
				msg_print("Meaning of Bushi-do is found in the death.");
#endif
				take_hit(NULL, caster_ptr, DAMAGE_FORCE, 9999, "Seppuku", NULL, -1);
			}
		}
		break;
	}

	return "";
}


/* Hex */
static bool item_tester_hook_weapon_except_bow(creature_type *creature_ptr, object_type *object_ptr)
{
	switch (object_ptr->tval)
	{
		case TV_SWORD:
		case TV_HAFTED:
		case TV_POLEARM:
		case TV_DIGGING:
		{
			return (TRUE);
		}
	}

	return (FALSE);
}

static bool item_tester_hook_cursed(creature_type *creature_ptr, object_type *object_ptr)
{
	return (bool)(object_is_cursed(object_ptr));
}

static cptr do_hex_spell(creature_type *creature_ptr, int spell, int mode)
{
	floor_type *floor_ptr = get_floor_ptr(creature_ptr);

	bool name = (mode == SPELL_NAME) ? TRUE : FALSE;
	bool desc = (mode == SPELL_DESC) ? TRUE : FALSE;
	bool info = (mode == SPELL_INFO) ? TRUE : FALSE;
	bool cast = (mode == SPELL_CAST) ? TRUE : FALSE;
	bool fail = (mode == SPELL_FAIL) ? TRUE : FALSE;
	bool cont = (mode == SPELL_CONT) ? TRUE : FALSE;
	bool stop = (mode == SPELL_STOP) ? TRUE : FALSE;

	bool add = TRUE;

	int plev = creature_ptr->lev;
	int power;

	switch (spell)
	{
	/*** 1st book (0-7) ***/
	case 0:
#ifdef JP
		if (name) return "ׂȂj";
		if (desc) return "jɂUxƖh͂オB";
#else
		if (name) return "Evily blessing";
		if (desc) return "Attempts to increase +to_hit of a weapon and AC";
#endif
		if (cast)
		{
			if (!creature_ptr->blessed)
			{
#ifdef JP
				msg_print("ȋCɂȂI");
#else
				msg_print("You feel righteous!");
#endif
			}
		}
		if (stop)
		{
			if (!creature_ptr->blessed)
			{
#ifdef JP
				msg_print("ȋCB");
#else
				msg_print("The prayer has expired.");
#endif
			}
		}
		break;

	case 1:
#ifdef JP
		if (name) return "y̎";
		if (desc) return "HP⏝񕜂B";
#else
		if (name) return "Cure light wounds";
		if (desc) return "Heals cut and HP a little.";
#endif
		if (info) return info_heal(1, 10, 0);
		if (cast)
		{
#ifdef JP
			msg_print("CǂȂĂB");
#else
			msg_print("You feel better and better.");
#endif
		}
		if (cast || cont)
		{
			hp_player(creature_ptr, diceroll(1, 10));
			set_cut(creature_ptr, creature_ptr->cut - 10);
		}
		break;

	case 2:
#ifdef JP
		if (name) return "̃I[";
		if (desc) return "̃I[gɂ܂ƂA񕜑xȂB";
#else
		if (name) return "Demonic aura";
		if (desc) return "Gives fire aura and regeneration.";
#endif
		if (cast)
		{
#ifdef JP
			msg_print("̂̃I[ŕꂽB");
#else
			msg_print("You have enveloped by fiery aura!");
#endif
		}
		if (stop)
		{
#ifdef JP
			msg_print("̃I[B");
#else
			msg_print("Fiery aura disappeared.");
#endif
		}
		break;

	case 3:
#ifdef JP
		if (name) return "L";
		if (desc) return "ẼN[`[ɔʂ̓ł̃_[W^B";
#else
		if (name) return "Stinking mist";
		if (desc) return "Deals few damages of poison to all creatures in your sight.";
#endif
		power = plev / 2 + 5;
		if (info) return info_damage(1, power, 0);
		if (cast || cont)
		{
			project_hack(creature_ptr, GF_POIS, randint1(power));
		}
		break;

	case 4:
#ifdef JP
		if (name) return "r͋";
		if (desc) return "p҂̘r͂㏸B";
#else
		if (name) return "Extra might";
		if (desc) return "Attempts to increase your strength.";
#endif
		if (cast)
		{
#ifdef JP
			msg_print("͂NėB");
#else
			msg_print("You feel you get stronger.");
#endif
		}
		break;

	case 5:
#ifdef JP
		if (name) return "";
		if (desc) return "Ă镐􂤁B";
#else
		if (name) return "Curse weapon";
		if (desc) return "Curses your weapon.";
#endif
		if (cast)
		{
			int item;
			char *q, *s;
			char object_name[MAX_NLEN];
			object_type *object_ptr;
			u32b f[TR_FLAG_SIZE];

#ifdef JP
			q = "ǂ􂢂܂H";
			s = "𑕔ĂȂB";
#else
			q = "Which weapon do you curse?";
			s = "You wield no weapons.";
#endif

			if (!get_item(creature_ptr, &item, q, s, (USE_EQUIP), item_tester_hook_weapon_except_bow, 0)) return FALSE;

			object_ptr = &creature_ptr->inventory[item];
			object_desc(object_name, object_ptr, OD_NAME_ONLY);
			object_flags(object_ptr, f);

#ifdef JP
			if (!get_check(format("{ %s 􂢂܂H", object_name))) return FALSE;
#else
			if (!get_check(format("Do you curse %s, reallyH", object_name))) return FALSE;
#endif

			if (!one_in_(3) &&
				(object_is_artifact(object_ptr) || have_flag(f, TRAIT_BLESSED_BRAND)))
			{
#ifdef JP
				msg_format("%s ͎􂢂𒵂˕ԂB", object_name);
#else
				msg_format("%s resists the effect.", object_name);
#endif
				if (one_in_(3))
				{
					if (object_ptr->to_damage > 0)
					{
						object_ptr->to_damage -= randint1(3) % 2;
						if (object_ptr->to_damage < 0) object_ptr->to_damage = 0;
					}
					if (object_ptr->to_hit > 0)
					{
						object_ptr->to_hit -= randint1(3) % 2;
						if (object_ptr->to_hit < 0) object_ptr->to_hit = 0;
					}
					if (object_ptr->to_ac > 0)
					{
						object_ptr->to_ac -= randint1(3) % 2;
						if (object_ptr->to_ac < 0) object_ptr->to_ac = 0;
					}
#ifdef JP
					msg_format("%s ͗򉻂Ă܂B", object_name);
#else
					msg_format("Your %s was disenchanted!", object_name);
#endif
				}
			}
			else
			{
				int power = 0;
#ifdef JP
				msg_format("|̈ÍI[Ȃ%sݍ񂾁I", object_name);
#else
				msg_format("A terrible black aura blasts your %s!", object_name);
#endif
				object_ptr->curse_flags |= (TRC_CURSED);

				if (object_is_artifact(object_ptr) || object_is_ego(object_ptr))
				{

					if (one_in_(3)) object_ptr->curse_flags |= (TRC_HEAVY_CURSE);
					if (one_in_(666))
					{
						object_ptr->curse_flags |= (TRC_TY_CURSE);
						if (one_in_(666)) object_ptr->curse_flags |= (TRC_DIVINE_CURSE);

						add_flag(object_ptr->trait_flags, TRAIT_ANTIPATHY);
						add_flag(object_ptr->trait_flags, TRAIT_VORPAL);
						add_flag(object_ptr->trait_flags, TRAIT_VAMPIRIC_BRAND);
#ifdef JP
						msg_print("III");
#else
						msg_print("Blood, Blood, Blood!");
#endif
						power = 2;
					}
				}

				object_ptr->curse_flags |= get_curse(power, object_ptr);
			}

			creature_ptr->creature_update |= (CRU_BONUS);
			add = FALSE;
		}
		break;

	case 6:
#ifdef JP
		if (name) return "׈m";
		if (desc) return "͂̎׈ȃN[`[mB";
#else
		if (name) return "Evil detection";
		if (desc) return "Detects evil creatures.";
#endif
		if (info) return info_range(MAX_SIGHT);
		if (cast)
		{
#ifdef JP
			msg_print("׈Ȑ݂̑낤ƂB");
#else
			msg_print("You attend to the presence of evil creatures.");
#endif
		}
		break;

	case 7:
#ifdef JP
		if (name) return "䖝";
		if (desc) return "^[UςA󂯂_[Wn̋Ɖ΂ƂĎ͂ɕoB";
#else
		if (name) return "Patience";
		if (desc) return "Bursts hell fire strongly after patients any damage while few turns.";
#endif
		power = MIN(200, (creature_ptr->class_skills.old_skills.magic_num1[2] * 2));
		if (info) return info_damage(0, 0, power);
		if (cast)
		{
			int a = 3 - (creature_ptr->speed - 100) / 10;
			int r = 3 + randint1(3) + MAX(0, MIN(3, a));

			if (creature_ptr->class_skills.old_skills.magic_num2[2] > 0)
			{
#ifdef JP
				msg_print("łɉ䖝ĂB");
#else
				msg_print("You are already patienting.");
#endif
				return NULL;
			}

			creature_ptr->class_skills.old_skills.magic_num2[1] = 1;
			creature_ptr->class_skills.old_skills.magic_num2[2] = r;
			creature_ptr->class_skills.old_skills.magic_num1[2] = 0;
#ifdef JP
			msg_print("Ƒς邱ƂɂB");
#else
			msg_print("You decide to patient all damages.");
#endif
			add = FALSE;
		}
		if (cont)
		{
			int rad = 2 + (power / 50);

			creature_ptr->class_skills.old_skills.magic_num2[2]--;

			if ((creature_ptr->class_skills.old_skills.magic_num2[2] <= 0) || (power >= 200))
			{
#ifdef JP
				msg_print("䖝ꂽI");
#else
				msg_print("Time for end of patioence!");
#endif
				if (power)
				{
					project(creature_ptr, rad, creature_ptr->fy, creature_ptr->fx, power, GF_HELL_FIRE,
						(PROJECT_STOP | PROJECT_GRID | PROJECT_ITEM | PROJECT_KILL), -1);
				}
				if (wizard)
				{
#ifdef JP
					msg_format("%d_̃_[WԂB", power);
#else
					msg_format("You return %d damages.", power);
#endif
				}

				/* Reset */
				creature_ptr->class_skills.old_skills.magic_num2[1] = 0;
				creature_ptr->class_skills.old_skills.magic_num2[2] = 0;
				creature_ptr->class_skills.old_skills.magic_num1[2] = 0;
			}
		}
		break;

	/*** 2nd book (8-15) ***/
	case 8:
#ifdef JP
		if (name) return "X̊Z";
		if (desc) return "X̃I[gɂ܂ƂAh͂㏸B";
#else
		if (name) return "Ice armor";
		if (desc) return "Gives fire aura and bonus to AC.";
#endif
		if (cast)
		{
#ifdef JP
			msg_print("̂X̊ZŕꂽB");
#else
			msg_print("You have enveloped by ice armor!");
#endif
		}
		if (stop)
		{
#ifdef JP
			msg_print("X̊ZB");
#else
			msg_print("Ice armor disappeared.");
#endif
		}
		break;

	case 9:
#ifdef JP
		if (name) return "d̎";
		if (desc) return "̗͂⏝𑽏񕜂B";
#else
		if (name) return "Cure serious wounds";
		if (desc) return "Heals cut and HP more.";
#endif
		if (info) return info_heal(2, 10, 0);
		if (cast)
		{
#ifdef JP
			msg_print("CǂȂĂB");
#else
			msg_print("You feel better and better.");
#endif
		}
		if (cast || cont)
		{
			hp_player(creature_ptr, diceroll(2, 10));
			set_cut(creature_ptr, (creature_ptr->cut / 2) - 10);
		}
		break;

	case 10:
#ifdef JP
		if (name) return "iz";
		if (desc) return "r𒆎~邱ƂȂǍʂ𓾂邱ƂłB";
#else
		if (name) return "Inhail potion";
		if (desc) return "Quaffs a potion without canceling of casting a spell.";
#endif
		if (cast)
		{
			creature_ptr->class_skills.old_skills.magic_num1[0] |= (1L << HEX_INHAIL);
			do_cmd_quaff_potion(creature_ptr);
			creature_ptr->class_skills.old_skills.magic_num1[0] &= ~(1L << HEX_INHAIL);
			add = FALSE;
		}
		break;

	case 11:
#ifdef JP
		if (name) return "z";
		if (desc) return "ẼN[`[ɔʂ̐͋z̃_[W^B^_[W̕A̗͂񕜂B";
#else
		if (name) return "Vampiric mist";
		if (desc) return "Deals few dameges of drain life to all creatures in your sight.";
#endif
		power = (plev / 2) + 5;
		if (info) return info_damage(1, power, 0);
		if (cast || cont)
		{
			project_hack(creature_ptr, GF_OLD_DRAIN, randint1(power));
		}
		break;

	case 12:
#ifdef JP
		if (name) return "";
		if (desc) return "̍U͂グB؂ꖡ𓾁A􂢂ɉė^_[W㏸APǂȃN[`[ɑ΂_[W2{ɂȂB";
#else
		if (name) return "Swords to runeswords";
		if (desc) return "Gives vorpal ability to your weapon. Increases damages by your weapon acccording to curse of your weapon.";
#endif
		if (cast)
		{
#ifdef JP
			msg_print("Ȃ̕킪PB");
#else
			if (!empty_hands(creature_ptr, FALSE))
				msg_print("Your weapons glow bright black.");
			else
				msg_print("Your weapon glows bright black.");
#endif
		}
		if (stop)
		{
#ifdef JP
			msg_print("̋PB");
#else
			msg_format("Brightness of weapon%s disappeared.", (empty_hands(creature_ptr, FALSE)) ? "" : "s");
#endif
		}
		break;

	case 13:
#ifdef JP
		if (name) return "̎";
		if (desc) return "UۃN[`[B";
#else
		if (name) return "Touch of confusion";
		if (desc) return "Confuses a creature when you attack.";
#endif
		if (cast)
		{
#ifdef JP
			msg_print("Ȃ̎肪ԂPn߂B");
#else
			msg_print("Your hands glow bright red.");
#endif
		}
		if (stop)
		{
#ifdef JP
			msg_print("̋PȂȂB");
#else
			msg_print("Brightness on your hands disappeard.");
#endif
		}
		break;

	case 14:
#ifdef JP
		if (name) return "̋";
		if (desc) return "p҂̘ŕApAϋv͂㏸BU񐔂̏ 1 B";
#else
		if (name) return "Building up";
		if (desc) return "Attempts to increases your strength, dexterity and constitusion.";
#endif
		if (cast)
		{
#ifdef JP
			msg_print("ĝȂCB");
#else
			msg_print("You feel your body is developed more now.");
#endif
		}
		break;

	case 15:
#ifdef JP
		if (name) return "e|[gE";
		if (desc) return "ẼN[`[̃e|[gjQoA𒣂B";
#else
		if (name) return "Anti teleport barrier";
		if (desc) return "Obstructs all teleportations by creatures in your sight.";
#endif
		power = plev * 3 / 2;
		if (info) return info_power(power);
		if (cast)
		{
#ifdef JP
			msg_print("e|[gh􂢂B");
#else
			msg_print("You feel anyone can not teleport except you.");
#endif
		}
		break;

	/*** 3rd book (16-23) ***/
	case 16:
#ifdef JP
		if (name) return "Ռ̃N[N";
		if (desc) return "dC̃I[gɂ܂ƂAȂB";
#else
		if (name) return "Cloak of shock";
		if (desc) return "Gives lightning aura and a bonus to speed.";
#endif
		if (cast)
		{
#ifdef JP
			msg_print("̂Ȃ̃I[ŕꂽB");
#else
			msg_print("You have enveloped by electrical aura!");
#endif
		}
		if (stop)
		{
#ifdef JP
			msg_print("Ȃ̃I[B");
#else
			msg_print("Electrical aura disappeared.");
#endif
		}
		break;

	case 17:
#ifdef JP
		if (name) return "v̎";
		if (desc) return "̗͂⏝񕜂B";
#else
		if (name) return "Cure critical wounds";
		if (desc) return "Heals cut and HP greatry.";
#endif
		if (info) return info_heal(4, 10, 0);
		if (cast)
		{
#ifdef JP
			msg_print("CǂȂĂB");
#else
			msg_print("You feel better and better.");
#endif
		}
		if (cast || cont)
		{
			hp_player(creature_ptr, diceroll(4, 10));
			set_stun(creature_ptr, 0);
			set_cut(creature_ptr, 0);
			set_poisoned(creature_ptr, 0);
		}
		break;

	case 18:
#ifdef JP
		if (name) return "͕";
		if (desc) return "@̓ɖ͂ď[UB";
#else
		if (name) return "Recharging";
		if (desc) return "Recharges a magic device.";
#endif
		power = plev * 2;
		if (info) return info_power(power);
		if (cast)
		{
			if (!recharge(creature_ptr, power)) return NULL;
			add = FALSE;
		}
		break;

	case 19:
#ifdef JP
		if (name) return "ҕ";
		if (desc) return "̂h点ăybgɂB";
#else
		if (name) return "Animate Dead";
		if (desc) return "Raises corpses and skeletons from dead.";
#endif
		if (cast)
		{
#ifdef JP
			msg_print("҂ւ̌Ăтn߂B");
#else
			msg_print("You start to call deads.!");
#endif
		}
		if (cast || cont)
		{
			animate_dead(NULL, creature_ptr->fy, creature_ptr->fx);
		}
		break;

	case 20:
#ifdef JP
		if (name) return "h";
		if (desc) return "ĂhɎ􂢂B";
#else
		if (name) return "Curse armor";
		if (desc) return "Curse a piece of armour that you wielding.";
#endif
		if (cast)
		{
			int item;
			char *q, *s;
			char object_name[MAX_NLEN];
			object_type *object_ptr;
			u32b f[TR_FLAG_SIZE];

#ifdef JP
			q = "ǂ􂢂܂H";
			s = "h𑕔ĂȂB";
#else
			q = "Which piece of armour do you curse?";
			s = "You wield no piece of armours.";
#endif

			if (!get_item(creature_ptr, &item, q, s, (USE_EQUIP), object_is_armour2, 0)) return FALSE;

			object_ptr = &creature_ptr->inventory[item];
			object_desc(object_name, object_ptr, OD_NAME_ONLY);
			object_flags(object_ptr, f);

#ifdef JP
			if (!get_check(format("{ %s 􂢂܂H", object_name))) return FALSE;
#else
			if (!get_check(format("Do you curse %s, reallyH", object_name))) return FALSE;
#endif

			if (!one_in_(3) &&
				(object_is_artifact(object_ptr) || have_flag(f, TRAIT_BLESSED_BRAND)))
			{
#ifdef JP
				msg_format("%s ͎􂢂𒵂˕ԂB", object_name);
#else
				msg_format("%s resists the effect.", object_name);
#endif
				if (one_in_(3))
				{
					if (object_ptr->to_damage > 0)
					{
						object_ptr->to_damage -= randint1(3) % 2;
						if (object_ptr->to_damage < 0) object_ptr->to_damage = 0;
					}
					if (object_ptr->to_hit > 0)
					{
						object_ptr->to_hit -= randint1(3) % 2;
						if (object_ptr->to_hit < 0) object_ptr->to_hit = 0;
					}
					if (object_ptr->to_ac > 0)
					{
						object_ptr->to_ac -= randint1(3) % 2;
						if (object_ptr->to_ac < 0) object_ptr->to_ac = 0;
					}
#ifdef JP
					msg_format("%s ͗򉻂Ă܂B", object_name);
#else
					msg_format("Your %s was disenchanted!", object_name);
#endif
				}
			}
			else
			{
				int power = 0;
#ifdef JP
				msg_format("|̈ÍI[Ȃ%sݍ񂾁I", object_name);
#else
				msg_format("A terrible black aura blasts your %s!", object_name);
#endif
				object_ptr->curse_flags |= (TRC_CURSED);

				if (object_is_artifact(object_ptr) || object_is_ego(object_ptr))
				{

					if (one_in_(3)) object_ptr->curse_flags |= (TRC_HEAVY_CURSE);
					if (one_in_(666))
					{
						object_ptr->curse_flags |= (TRC_TY_CURSE);
						if (one_in_(666)) object_ptr->curse_flags |= (TRC_DIVINE_CURSE);

						add_flag(object_ptr->trait_flags, TRAIT_ANTIPATHY);
						add_flag(object_ptr->trait_flags, TRAIT_RES_POIS);
						add_flag(object_ptr->trait_flags, TRAIT_RES_DARK);
						add_flag(object_ptr->trait_flags, TRAIT_RES_NETH);
#ifdef JP
						msg_print("III");
#else
						msg_print("Blood, Blood, Blood!");
#endif
						power = 2;
					}
				}

				object_ptr->curse_flags |= get_curse(power, object_ptr);
			}

			creature_ptr->creature_update |= (CRU_BONUS);
			add = FALSE;
		}
		break;

	case 21:
#ifdef JP
		if (name) return "ẽN[N";
		if (desc) return "ẽI[gɂ܂ƂAGɉẽ_[W^B";
#else
		if (name) return "Cloak of shadow";
		if (desc) return "Gives aura of shadow.";
#endif
		if (cast)
		{
			//TODO: GET outer equipment.
			object_type *object_ptr = &creature_ptr->inventory[0];

			if (!object_ptr->k_idx)
			{
#ifdef JP
				msg_print("N[NgɂĂȂI");
#else
				msg_print("You don't ware any cloak.");
#endif
				return NULL;
			}
			else if (!object_is_cursed(object_ptr))
			{
#ifdef JP
				msg_print("N[N͎ĂȂI");
#else
				msg_print("Your cloak is not cursed.");
#endif
				return NULL;
			}
			else
			{
#ifdef JP
				msg_print("ẽI[gɂ܂ƂB");
#else
				msg_print("You have enveloped by shadow aura!");
#endif
			}
		}
		if (cont)
		{
			//TODO: GET outer Equipment
			object_type *object_ptr = &creature_ptr->inventory[0];

			if ((!object_ptr->k_idx) || (!object_is_cursed(object_ptr)))
			{
				do_spell(creature_ptr, REALM_HEX, spell, SPELL_STOP);
				creature_ptr->class_skills.old_skills.magic_num1[0] &= ~(1L << spell);
				creature_ptr->class_skills.old_skills.magic_num2[0]--;
				if (!creature_ptr->class_skills.old_skills.magic_num2[0]) set_action(creature_ptr, ACTION_NONE);
			}
		}
		if (stop)
		{
#ifdef JP
			msg_print("ẽI[B");
#else
			msg_print("Shadow aura disappeared.");
#endif
		}
		break;

	case 22:
#ifdef JP
		if (name) return "ɂ𖂗͂";
		if (desc) return "ẼN[`[ɐ__[W^A͂zB";
#else
		if (name) return "Pains to mana";
		if (desc) return "Deals psychic damages to all creatures in sight, and drains some mana.";
#endif
		power = plev * 3 / 2;
		if (info) return info_damage(1, power, 0);
		if (cast || cont)
		{
			project_hack(creature_ptr, GF_PSI_DRAIN, randint1(power));
		}
		break;

	case 23:
#ifdef JP
		if (name) return "ڂɂ͖ڂ";
		if (desc) return "Ō▂@Ŏ󂯂_[WAŨN[`[ɂ^B";
#else
		if (name) return "Eye for an eye";
		if (desc) return "Returns same damage which you got to the creature which damaged you.";
#endif
		if (cast)
		{
#ifdef JP
			msg_print("Q~]ɂꂽB");
#else
			msg_print("You wish strongly you want to revenge anything.");
#endif
		}
		break;

	/*** 4th book (24-31) ***/
	case 24:
#ifdef JP
		if (name) return "BE";
		if (desc) return "̊K̑BN[`[̑Bj~B";
#else
		if (name) return "Anti multiply barrier";
		if (desc) return "Obstructs all multiplying by creatures in entire floor.";
#endif
		if (cast)
		{
#ifdef JP
			msg_print("Bj~􂢂B");
#else
			msg_print("You feel anyone can not already multiply.");
#endif
		}
		break;

	case 25:
#ifdef JP
		if (name) return "͕";
		if (desc) return "olXɕA\͒l񕜂B";
#else
		if (name) return "Restore life";
		if (desc) return "Restores life energy and status.";
#endif
		if (cast)
		{
#ifdef JP
			msg_print("͂߂n߂B");
#else
			msg_print("You feel your life energy starting to return.");
#endif
		}
		if (cast || cont)
		{
			bool flag = FALSE;
			int d = (creature_ptr->max_exp - creature_ptr->exp);
			int r = (creature_ptr->exp / 20);
			int i;

			if (d > 0)
			{
				if (d < r)
					creature_ptr->exp = creature_ptr->max_exp;
				else
					creature_ptr->exp += r;

				/* Check the experience */
				check_experience(creature_ptr);

				flag = TRUE;
			}
			for (i = STAT_STR; i < 6; i ++)
			{
				if (creature_ptr->stat_cur[i] < creature_ptr->stat_max[i])
				{
					creature_ptr->stat_cur[i] += 10;
					if (creature_ptr->stat_cur[i] > creature_ptr->stat_max[i]) creature_ptr->stat_cur[i] = creature_ptr->stat_max[i];
					creature_ptr->creature_update |= (CRU_BONUS); // Recalculate bonuses

					flag = TRUE;
				}
			}

			if (!flag)
			{
#ifdef JP
				msg_format("%s̎̉r߂B", do_spell(creature_ptr, REALM_HEX, HEX_RESTORE, SPELL_NAME));
#else
				msg_format("Finish casting '%^s'.", do_spell(creature_ptr, REALM_HEX, HEX_RESTORE, SPELL_NAME));
#endif
				creature_ptr->class_skills.old_skills.magic_num1[0] &= ~(1L << HEX_RESTORE);
				if (cont) creature_ptr->class_skills.old_skills.magic_num2[0]--;
				if (creature_ptr->class_skills.old_skills.magic_num2) creature_ptr->action = ACTION_NONE;

				/* Redraw status */
				creature_ptr->creature_update |= (CRU_BONUS | CRU_HP | CRU_MANA | CRU_SPELLS);
				play_redraw |= (PR_EXTRA);

				return "";
			}
		}
		break;

	case 26:
#ifdef JP
		if (name) return "͋z";
		if (desc) return "ꂽ̎􂢂zĖ͂񕜂B";
#else
		if (name) return "Drain curse power";
		if (desc) return "Drains curse on your weapon and heals SP a little.";
#endif
		if (cast)
		{
			int item;
			char *s, *q;
			u32b f[TR_FLAG_SIZE];
			object_type *object_ptr;
#ifdef JP
			q = "ǂ̑iz܂H";
			s = "ꂽACe𑕔ĂȂB";
#else
			q = "Which cursed equipment do you drain mana from?";
			s = "You have no cursed equipment.";
#endif

			if (!get_item(creature_ptr, &item, q, s, (USE_EQUIP), item_tester_hook_cursed, 0)) return FALSE;

			object_ptr = &creature_ptr->inventory[item];
			object_flags(object_ptr, f);

			creature_ptr->csp += (creature_ptr->lev / 5) + randint1(creature_ptr->lev / 5);
			if (have_flag(f, TRAIT_TY_CURSE) || (object_ptr->curse_flags & TRC_TY_CURSE)) creature_ptr->csp += randint1(5);
			if (creature_ptr->csp > creature_ptr->msp) creature_ptr->csp = creature_ptr->msp;

			if (object_ptr->curse_flags & TRC_DIVINE_CURSE)
			{
				/* Nothing */
			}
			else if (object_ptr->curse_flags & TRC_HEAVY_CURSE)
			{
				if (one_in_(7))
				{
#ifdef JP
					msg_print("􂢂SċzB");
#else
					msg_print("Heavy curse vanished away.");
#endif
					object_ptr->curse_flags = 0L;
				}
			}
			else if ((object_ptr->curse_flags & (TRC_CURSED)) && one_in_(3))
			{
#ifdef JP
				msg_print("􂢂SċzB");
#else
				msg_print("Curse vanished away.");
#endif
				object_ptr->curse_flags = 0L;
			}

			add = FALSE;
		}
		break;

	case 27:
#ifdef JP
		if (name) return "z̐n";
		if (desc) return "zōUB";
#else
		if (name) return "Swords to vampires";
		if (desc) return "Gives vampiric ability to your weapon.";
#endif
		if (cast)
		{
#ifdef JP
			msg_print("Ȃ̕킪~ĂB");
#else
			if (!empty_hands(creature_ptr, FALSE))
				msg_print("Your weapons want more blood now.");
			else
				msg_print("Your weapon wants more blood now.");
#endif
		}
		if (stop)
		{
#ifdef JP
			msg_print("̊]B");
#else
			msg_format("Thirsty of weapon%s disappeared.", (empty_hands(creature_ptr, FALSE)) ? "" : "s");
#endif
		}
		break;

	case 28:
#ifdef JP
		if (name) return "NǑt";
		if (desc) return "ẼN[`[NOƂB";
#else
		if (name) return "Word of stun";
		if (desc) return "Stuns all creatures in your sight.";
#endif
		power = plev * 4;
		if (info) return info_power(power);
		if (cast || cont)
		{
			stun_creatures(creature_ptr, power);
		}
		break;

	case 29:
#ifdef JP
		if (name) return "eړ";
		if (desc) return "N[`[ׂ̗̃}XɏuԈړB";
#else
		if (name) return "Moving into shadow";
		if (desc) return "Teleports you close to a creature.";
#endif
		if (cast)
		{
			int i, y, x, dir;
			bool flag;

			for (i = 0; i < 3; i++)
			{
				if (!tgt_pt(creature_ptr, &x, &y)) return FALSE;

				flag = FALSE;

				for (dir = 0; dir < 8; dir++)
				{
					int dy = y + ddy_ddd[dir];
					int dx = x + ddx_ddd[dir];
					if (dir == 5) continue;
					if(floor_ptr->cave[dy][dx].creature_idx) flag = TRUE;
				}

				if (!cave_empty_bold(floor_ptr, y, x) || (floor_ptr->cave[y][x].info & CAVE_ICKY) ||
					(distance(y, x, creature_ptr->fy, creature_ptr->fx) > plev + 2))
				{
#ifdef JP
					msg_print("ɂ͈ړłȂB");
#else
					msg_print("Can not teleport to there.");
#endif
					continue;
				}
				break;
			}

			if (flag && randint0(plev * plev / 2))
			{
				teleport_creature_to(creature_ptr, y, x, 0L);
			}
			else
			{
#ifdef JP
				msg_print("ƁI");
#else
				msg_print("Oops!");
#endif
				teleport_player(creature_ptr, 30, 0L);
			}

			add = FALSE;
		}
		break;

	case 30:
#ifdef JP
		if (name) return "@E";
		if (desc) return "ẼN[`[̖@jQoA𒣂B";
#else
		if (name) return "Anti magic barrier";
		if (desc) return "Obstructs all magic spell of creatures in your sight.";
#endif
		power = plev * 3 / 2;
		if (info) return info_power(power);
		if (cast)
		{
#ifdef JP
			msg_print("@h􂢂B");
#else
			msg_print("You feel anyone can not cast spells except you.");
#endif
		}
		break;

	case 31:
#ifdef JP
		if (name) return "Q̐鍐";
		if (desc) return "^[ɂ܂Ŏ󂯂_[WɉЗ͂̒n̍΂̒eB";
#else
		if (name) return "Revenge sentence";
		if (desc) return "Fires  a ball of hell fire to try revenging after few turns.";
#endif
		power = creature_ptr->class_skills.old_skills.magic_num1[2];
		if (info) return info_damage(0, 0, power);
		if (cast)
		{
			int r;
			int a = 3 - (creature_ptr->speed - 100) / 10;
			r = 1 + randint1(2) + MAX(0, MIN(3, a));

			if (creature_ptr->class_skills.old_skills.magic_num2[2] > 0)
			{
#ifdef JP
				msg_print("łɕQ͐鍐ς݂B");
#else
				msg_print("You already pronounced your revenge.");
#endif
				return NULL;
			}

			creature_ptr->class_skills.old_skills.magic_num2[1] = 2;
			creature_ptr->class_skills.old_skills.magic_num2[2] = r;
#ifdef JP
			msg_format("Ȃ͕Q鍐B %d ^[B", r);
#else
			msg_format("You pronounce your revenge. %d turns left.", r);
#endif
			add = FALSE;
		}
		if (cont)
		{
			creature_ptr->class_skills.old_skills.magic_num2[2]--;

			if (creature_ptr->class_skills.old_skills.magic_num2[2] <= 0)
			{
				int dir;

				if (power)
				{
					command_dir = 0;

					do
					{
#ifdef JP
						msg_print("Q̎I");
#else
						msg_print("Time to revenge!");
#endif
					}
					while (!get_aim_dir(creature_ptr, &dir));

					fire_ball(creature_ptr, GF_HELL_FIRE, dir, power, 1);

					if (wizard)
					{
#ifdef JP
						msg_format("%d_̃_[WԂB", power);
#else
						msg_format("You return %d damages.", power);
#endif
					}
				}
				else
				{
#ifdef JP
					msg_print("QCB");
#else
					msg_print("You are not a mood to revenge.");
#endif
				}
				creature_ptr->class_skills.old_skills.magic_num1[2] = 0;
			}
		}
		break;
	}

	/* start casting */
	if ((cast) && (add))
	{
		/* add spell */
		creature_ptr->class_skills.old_skills.magic_num1[0] |= 1L << (spell);
		creature_ptr->class_skills.old_skills.magic_num2[0]++;

		if (creature_ptr->action != ACTION_SPELL) set_action(creature_ptr, ACTION_SPELL);
	}

	/* Redraw status */
	if (!info)
	{
		creature_ptr->creature_update |= (CRU_BONUS | CRU_HP | CRU_MANA | CRU_SPELLS);
		play_redraw |= (PR_EXTRA | PR_HP | PR_MANA);
	}

	return "";
}


/*
 * Do everything for each spell
 */
cptr do_spell(creature_type *creature_ptr, int realm, int spell, int mode)
{
	switch (realm)
	{
	case REALM_LIFE:     return do_life_spell(creature_ptr, spell, mode);
	case REALM_SORCERY:  return do_sorcery_spell(creature_ptr, spell, mode);
	case REALM_NATURE:   return do_nature_spell(creature_ptr, spell, mode);
	case REALM_CHAOS:    return do_chaos_spell(creature_ptr, spell, mode);
	case REALM_DEATH:    return do_death_spell(creature_ptr, spell, mode);
	case REALM_TRUMP:    return do_trump_spell(creature_ptr, spell, mode);
	case REALM_ARCANE:   return do_arcane_spell(creature_ptr, spell, mode);
	case REALM_CRAFT:    return do_craft_spell(creature_ptr, spell, mode);
	case REALM_DAEMON:   return do_daemon_spell(creature_ptr, spell, mode);
	case REALM_CRUSADE:  return do_crusade_spell(creature_ptr, spell, mode);
	case REALM_MUSIC:    return do_music_spell(creature_ptr, spell, mode);
	case REALM_HISSATSU: return do_hissatsu_spell(creature_ptr, spell, mode);
	case REALM_HEX:      return do_hex_spell(creature_ptr, spell, mode);
	}

	return NULL;
}
