/* File: mspells1.c */

/*
 * Copyright (c) 1997 Ben Harrison, James E. Wilson, Robert A. Koeneke
 *
 * This software may be copied and distributed for educational, research,
 * and not for profit purposes provided that this copyright and statement
 * are included in all such copies.  Other copyrights may also apply.
 */

/* Purpose: Creature spells (attack player) */

#include "angband.h"


/*
 * And now for Intelligent creature attacks (including spells).
 *
 * Original idea and code by "DRS" (David Reeves Sward).
 * Major modifications by "BEN" (Ben Harrison).
 *
 * Give creatures more intelligent attack/spell selection based on
 * observations of previous attacks on the player, and/or by allowing
 * the creature to "cheat" and know the player status.
 *
 * Maintain an idea of the player status, and use that information
 * to occasionally eliminate "ineffective" spell attacks.  We could
 * also eliminate ineffective normal attacks, but there is no reason
 * for the creature to do this, since he gains no benefit.
 * Note that MINDLESS creatures are not allowed to use this code.
 * And non-INTELLIGENT creatures only use it partially effectively.
 *
 * Actually learn what the player resists, and use that information
 * to remove attacks or spells before using them.  This will require
 * much less space, if I am not mistaken.  Thus, each creature gets a
 * set of 32 bit flags, "smart", build from the various "SM_*" flags.
 *
 * This has the added advantage that attacks and spells are related.
 * The "smart_learn" option means that the creature "learns" the flags
 * that should be set, and "smart_cheat" means that he "knows" them.
 * So "smart_cheat" means that the "smart" field is always up to date,
 * while "smart_learn" means that the "smart" field is slowly learned.
 * Both of them have the same effect on the "choose spell" routine.
 */



/*
 * Internal probability routine
 */
static bool int_outof(species_type *r_ptr, int prob)
{
	/* Non-Smart creatures are half as "smart" */
	if (!is_smart_species(r_ptr)) prob = prob / 2;

	/* Roll the dice */
	return (randint0(100) < prob);
}



/*
 * Remove the "bad" spells from a spell list
 */
static void remove_bad_spells(creature_type *caster_ptr, u32b *f4p, u32b *f5p, u32b *f6p)
{
	species_type *r_ptr = &species_info[caster_ptr->species_idx];

	u32b f4 = (*f4p);
	u32b f5 = (*f5p);
	u32b f6 = (*f6p);

	u32b smart = 0L;


	/* Too stupid to know anything */
	if (has_trait(caster_ptr, TRAIT_STUPID)) return;


	/* Must be cheating or learning */
	if (!smart_cheat && !smart_learn) return;


	/* Update acquired knowledge */
	if (smart_learn)
	{
		/* Hack -- Occasionally forget player status */
		/* Only save SM_FRIENDLY, SM_PET or SM_CLONED */
		//TODO if (caster_ptr->smart && (randint0(100) < 1)) caster_ptr->smart &= (SM_FRIENDLY | SM_PET | SM_CLONED);

		/* Use the memorized flags */
		smart = caster_ptr->smart;
	}


	/* Cheat if requested */
	if (smart_cheat)
	{
		/* Know basic info */
		if (caster_ptr->resist_acid) smart |= (SM_RES_ACID);
		if (IS_OPPOSE_ACID(caster_ptr)) smart |= (SM_OPP_ACID);
		//if (caster_ptr->immune_acid) smart |= (SM_IMM_ACID);
		if (caster_ptr->resist_elec) smart |= (SM_RES_ELEC);
		if (IS_OPPOSE_ELEC(caster_ptr)) smart |= (SM_OPP_ELEC);
		//if (caster_ptr->immune_elec) smart |= (SM_IMM_ELEC);
		if (caster_ptr->resist_fire) smart |= (SM_RES_FIRE);
		if (IS_OPPOSE_FIRE(caster_ptr)) smart |= (SM_OPP_FIRE);
		//if (caster_ptr->immune_fire) smart |= (SM_IMM_FIRE);
		if (caster_ptr->resist_cold) smart |= (SM_RES_COLD);
		if (IS_OPPOSE_COLD(caster_ptr)) smart |= (SM_OPP_COLD);
		//if (caster_ptr->immune_cold) smart |= (SM_IMM_COLD);

		/* Know poison info */
		if (caster_ptr->resist_pois) smart |= (SM_RES_POIS);
		if (IS_OPPOSE_POIS(caster_ptr)) smart |= (SM_OPP_POIS);

		/* Know special resistances */
		if (caster_ptr->resist_neth) smart |= (SM_RES_NETH);
		if (caster_ptr->resist_lite) smart |= (SM_RES_LITE);
		if (caster_ptr->resist_dark) smart |= (SM_RES_DARK);
		if (has_trait(caster_ptr, TRAIT_FEARLESS)) smart |= (SM_RES_FEAR);
		if (has_trait(caster_ptr, TRAIT_NO_CONF)) smart |= (SM_RES_CONF);
		if (caster_ptr->resist_chaos) smart |= (SM_RES_CHAOS);
		if (caster_ptr->resist_disen) smart |= (SM_RES_DISEN);
		if (has_trait(caster_ptr, TRAIT_NO_BLIND)) smart |= (SM_RES_BLIND);
		if (caster_ptr->resist_nexus) smart |= (SM_RES_NEXUS);
		if (caster_ptr->resist_sound) smart |= (SM_RES_SOUND);
		if (caster_ptr->resist_shard) smart |= (SM_RES_SHARD);
		//if (caster_ptr->reflect) smart |= (SM_IMM_REFLECT);

		/* Know bizarre "resistances" */
		if (has_trait(caster_ptr, TRAIT_FREE_ACTION)) smart |= (SM_IMM_FREE);
		if (!caster_ptr->msp) smart |= (SM_IMM_MANA);
	}


	/* Nothing known */
	if (!smart) return;

/*
	if (smart & SM_IMM_ACID)
	{
		f4 &= ~(RF4_BR_ACID);
		f5 &= ~(RF5_BA_ACID);
		f5 &= ~(RF5_BO_ACID);
	}
	else if ((smart & (SM_OPP_ACID)) && (smart & (SM_RES_ACID)))
	{
		if (int_outof(r_ptr, 80)) f4 &= ~(RF4_BR_ACID);
		if (int_outof(r_ptr, 80)) f5 &= ~(RF5_BA_ACID);
		if (int_outof(r_ptr, 80)) f5 &= ~(RF5_BO_ACID);
	}
	else if ((smart & (SM_OPP_ACID)) || (smart & (SM_RES_ACID)))
	{
		if (int_outof(r_ptr, 30)) f4 &= ~(RF4_BR_ACID);
		if (int_outof(r_ptr, 30)) f5 &= ~(RF5_BA_ACID);
		if (int_outof(r_ptr, 30)) f5 &= ~(RF5_BO_ACID);
	}


	if (smart & (SM_IMM_ELEC))
	{
		f4 &= ~(RF4_BR_ELEC);
		f5 &= ~(RF5_BA_ELEC);
		f5 &= ~(RF5_BO_ELEC);
	}
	else if ((smart & (SM_OPP_ELEC)) && (smart & (SM_RES_ELEC)))
	{
		if (int_outof(r_ptr, 80)) f4 &= ~(RF4_BR_ELEC);
		if (int_outof(r_ptr, 80)) f5 &= ~(RF5_BA_ELEC);
		if (int_outof(r_ptr, 80)) f5 &= ~(RF5_BO_ELEC);
	}
	else if ((smart & (SM_OPP_ELEC)) || (smart & (SM_RES_ELEC)))
	{
		if (int_outof(r_ptr, 30)) f4 &= ~(RF4_BR_ELEC);
		if (int_outof(r_ptr, 30)) f5 &= ~(RF5_BA_ELEC);
		if (int_outof(r_ptr, 30)) f5 &= ~(RF5_BO_ELEC);
	}


	if (smart & (SM_IMM_FIRE))
	{
		f4 &= ~(RF4_BR_FIRE);
		f5 &= ~(RF5_BA_FIRE);
		f5 &= ~(RF5_BO_FIRE);
	}
	else if ((smart & (SM_OPP_FIRE)) && (smart & (SM_RES_FIRE)))
	{
		if (int_outof(r_ptr, 80)) f4 &= ~(RF4_BR_FIRE);
		if (int_outof(r_ptr, 80)) f5 &= ~(RF5_BA_FIRE);
		if (int_outof(r_ptr, 80)) f5 &= ~(RF5_BO_FIRE);
	}
	else if ((smart & (SM_OPP_FIRE)) || (smart & (SM_RES_FIRE)))
	{
		if (int_outof(r_ptr, 30)) f4 &= ~(RF4_BR_FIRE);
		if (int_outof(r_ptr, 30)) f5 &= ~(RF5_BA_FIRE);
		if (int_outof(r_ptr, 30)) f5 &= ~(RF5_BO_FIRE);
	}


	if (smart & (SM_IMM_COLD))
	{
		f4 &= ~(RF4_BR_COLD);
		f5 &= ~(RF5_BA_COLD);
		f5 &= ~(RF5_BO_COLD);
		f5 &= ~(RF5_BO_ICEE);
	}
	else if ((smart & (SM_OPP_COLD)) && (smart & (SM_RES_COLD)))
	{
		if (int_outof(r_ptr, 80)) f4 &= ~(RF4_BR_COLD);
		if (int_outof(r_ptr, 80)) f5 &= ~(RF5_BA_COLD);
		if (int_outof(r_ptr, 80)) f5 &= ~(RF5_BO_COLD);
		if (int_outof(r_ptr, 80)) f5 &= ~(RF5_BO_ICEE);
	}
	else if ((smart & (SM_OPP_COLD)) || (smart & (SM_RES_COLD)))
	{
		if (int_outof(r_ptr, 30)) f4 &= ~(RF4_BR_COLD);
		if (int_outof(r_ptr, 30)) f5 &= ~(RF5_BA_COLD);
		if (int_outof(r_ptr, 30)) f5 &= ~(RF5_BO_COLD);
		if (int_outof(r_ptr, 20)) f5 &= ~(RF5_BO_ICEE);
	}


	if ((smart & (SM_OPP_POIS)) && (smart & (SM_RES_POIS)))
	{
		if (int_outof(r_ptr, 80)) f4 &= ~(RF4_BR_POIS);
		if (int_outof(r_ptr, 80)) f5 &= ~(RF5_BA_POIS);
		if (int_outof(r_ptr, 60)) f4 &= ~(RF4_BA_NUKE);
		if (int_outof(r_ptr, 60)) f4 &= ~(RF4_BR_NUKE);
	}
	else if ((smart & (SM_OPP_POIS)) || (smart & (SM_RES_POIS)))
	{
		if (int_outof(r_ptr, 30)) f4 &= ~(RF4_BR_POIS);
		if (int_outof(r_ptr, 30)) f5 &= ~(RF5_BA_POIS);
	}
*/

	if (smart & (SM_RES_NETH))
	{
		/*
		if (IS_RACE(caster_ptr, RACE_LICH))
		{
			f4 &= ~(RF4_BR_NETH);
			f5 &= ~(RF5_BA_NETH);
			f5 &= ~(RF5_BO_NETH);
		}
		else
		{
			if (int_outof(r_ptr, 20)) f4 &= ~(RF4_BR_NETH);
			if (int_outof(r_ptr, 50)) f5 &= ~(RF5_BA_NETH);
			if (int_outof(r_ptr, 50)) f5 &= ~(RF5_BO_NETH);
		}
		*/
	}

/*
	if (smart & (SM_RES_LITE))
	{
		if (int_outof(r_ptr, 50)) f4 &= ~(RF4_BR_LITE);
		if (int_outof(r_ptr, 50)) f5 &= ~(RF5_BA_LITE);
	}
*/

	if (smart & (SM_RES_DARK))
	{
		/*
		if (IS_RACE(caster_ptr, RACE_VAMPIRE))
		{
			f4 &= ~(RF4_BR_DARK);
			f5 &= ~(RF5_BA_DARK);
		}
		else
		{
			if (int_outof(r_ptr, 50)) f4 &= ~(RF4_BR_DARK);
			if (int_outof(r_ptr, 50)) f5 &= ~(RF5_BA_DARK);
		}
		*/
	}

	/*
	if (smart & (SM_RES_FEAR))
	{
		f5 &= ~(RF5_SCARE);
	}
	*/

/*
	if (smart & (SM_RES_CONF))
	{
		f5 &= ~(RF5_CONF);
		if (int_outof(r_ptr, 50)) f4 &= ~(RF4_BR_CONF);
	}

	if (smart & (SM_RES_CHAOS))
	{
		if (int_outof(r_ptr, 20)) f4 &= ~(RF4_BR_CHAO);
		if (int_outof(r_ptr, 50)) f4 &= ~(RF4_BA_CHAO);
	}

	if (smart & (SM_RES_DISEN))
	{
		if (int_outof(r_ptr, 40)) f4 &= ~(RF4_BR_DISE);
	}

	if (smart & (SM_RES_BLIND))
	{
		f5 &= ~(RF5_BLIND);
	}

	if (smart & (SM_RES_NEXUS))
	{
		if (int_outof(r_ptr, 50)) f4 &= ~(RF4_BR_NEXU);
		f6 &= ~(RF6_TELE_LEVEL);
	}

	if (smart & (SM_RES_SOUND))
	{
		if (int_outof(r_ptr, 50)) f4 &= ~(RF4_BR_SOUN);
	}

	if (smart & (SM_RES_SHARD))
	{
		if (int_outof(r_ptr, 40)) f4 &= ~(RF4_BR_SHAR);
	}

	if (smart & (SM_IMM_REFLECT))
	{
		if (int_outof(r_ptr, 150)) f5 &= ~(RF5_BO_COLD);
		if (int_outof(r_ptr, 150)) f5 &= ~(RF5_BO_FIRE);
		if (int_outof(r_ptr, 150)) f5 &= ~(RF5_BO_ACID);
		if (int_outof(r_ptr, 150)) f5 &= ~(RF5_BO_ELEC);
		if (int_outof(r_ptr, 150)) f5 &= ~(RF5_BO_NETH);
		if (int_outof(r_ptr, 150)) f5 &= ~(RF5_BO_WATE);
		if (int_outof(r_ptr, 150)) f5 &= ~(RF5_BO_MANA);
		if (int_outof(r_ptr, 150)) f5 &= ~(RF5_BO_PLAS);
		if (int_outof(r_ptr, 150)) f5 &= ~(RF5_BO_ICEE);
		if (int_outof(r_ptr, 150)) f5 &= ~(RF5_MISSILE);
		if (int_outof(r_ptr, 150)) f4 &= ~(RF4_SHOOT);
	}

	if (smart & (SM_IMM_FREE))
	{
		f5 &= ~(RF5_HOLD);
		f5 &= ~(RF5_SLOW);
	}

	if (smart & (SM_IMM_MANA))
	{
		f5 &= ~(RF5_DRAIN_MANA);
	}
	*/

	/* XXX XXX XXX No spells left? */
	/* if (!f4 && !f5 && !f6) ... */

	(*f4p) = f4;
	(*f5p) = f5;
	(*f6p) = f6;
}


// Determine if there is a space near the player in which
// a summoned creature can appear
bool summon_possible(creature_type *target_ptr, int y1, int x1)
{
	floor_type *floor_ptr = get_floor_ptr(target_ptr);
	int y, x;

	/* Start at the player's location, and check 2 grids in each dir */
	for (y = y1 - 2; y <= y1 + 2; y++)
	{
		for (x = x1 - 2; x <= x1 + 2; x++)
		{
			/* Ignore illegal locations */
			if (!in_bounds(floor_ptr, y, x)) continue;

			/* Only check a circular area */
			if (distance(y1, x1, y, x)>2) continue;

			/* ...nor on the Pattern */
			if (pattern_tile(floor_ptr, y, x)) continue;

			/* Require empty floor grid in line of projection */
			if (cave_empty_bold(floor_ptr, y, x) && projectable(floor_ptr, y, x, y1, x1)) return (TRUE);
		}
	}

	return FALSE;
}

// For raised dead
bool raise_possible(creature_type *caster_ptr, creature_type *target_ptr)
{
	floor_type *floor_ptr = get_floor_ptr(caster_ptr);
	int xx, yy;
	int y = target_ptr->fy;
	int x = target_ptr->fx;
	s16b this_object_idx, next_object_idx = 0;
	cave_type *c_ptr;

	for (xx = x - 5; xx <= x + 5; xx++)
	{
		for (yy = y - 5; yy <= y + 5; yy++)
		{
			if (distance(y, x, yy, xx) > 5) continue;
			if (!los(floor_ptr, y, x, yy, xx)) continue;
			if (!projectable(floor_ptr, y, x, yy, xx)) continue;

			c_ptr = &floor_ptr->cave[yy][xx];
			/* Scan the pile of objects */
			for (this_object_idx = c_ptr->object_idx; this_object_idx; this_object_idx = next_object_idx)
			{
				/* Acquire object */
				object_type *object_ptr = &object_list[this_object_idx];

				/* Acquire next object */
				next_object_idx = object_ptr->next_object_idx;

				/* Known to be worthless? */
				if (object_ptr->tval == TV_CORPSE)
				{
					if (!creature_has_hostile_align(target_ptr, caster_ptr)) return TRUE;
				}
			}
		}
	}
	return FALSE;
}


/*
 * Originally, it was possible for a friendly to shoot another friendly.
 * Change it so a "clean shot" means no equally friendly creature is
 * between the attacker and target.
 */
/*
 * Determine if a bolt spell will hit the player.
 *
 * This is exactly like "projectable", but it will
 * return FALSE if a creature is in the way.
 * no equally friendly creature is
 * between the attacker and target.
 */
bool clean_shot(creature_type *target_ptr, int y1, int x1, int y2, int x2, bool friend)
{
	floor_type *floor_ptr = get_floor_ptr(target_ptr);

	/* Must be the same as projectable(floor_ptr, ) */
	int i, y, x;

	int grid_n = 0;
	u16b grid_g[512];

	/* Check the projection path */
	grid_n = project_path(grid_g, MAX_RANGE, floor_ptr, y1, x1, y2, x2, 0);

	/* No grid is ever projectable from itself */
	if (!grid_n) return (FALSE);

	/* Final grid */
	y = GRID_Y(grid_g[grid_n-1]);
	x = GRID_X(grid_g[grid_n-1]);

	/* May not end in an unrequested grid */
	if ((y != y2) || (x != x2)) return (FALSE);

	for (i = 0; i < grid_n; i++)
	{
		y = GRID_Y(grid_g[i]);
		x = GRID_X(grid_g[i]);

		if ((floor_ptr->cave[y][x].creature_idx > 0) && !((y == y2) && (x == x2)))
		{
			creature_type *m_ptr = &creature_list[floor_ptr->cave[y][x].creature_idx];
			if (friend == is_pet(player_ptr, m_ptr))
			{
				return (FALSE);
			}
		}
		/* Pets may not shoot through the character - TNB */
		if (creature_bold(target_ptr, y, x))
		{
			if (friend) return (FALSE);
		}
	}

	return (TRUE);
}

/*
 * Cast a bolt at the player
 * Stop if we hit a creature
 * Affect creatures and the player
 */
static void bolt(creature_type *caster_ptr, creature_type *target_ptr, int typ, int dam_hp, int monspell, bool learnable)
{
	int flg = PROJECT_STOP | PROJECT_KILL | PROJECT_PLAYER | PROJECT_REFLECTABLE;

	/* Target the player with a bolt attack */
	(void)project(caster_ptr, 0, target_ptr->fy, target_ptr->fx, dam_hp, typ, flg, (learnable ? monspell : -1));
}

static void beam(creature_type *caster_ptr, creature_type *target_ptr, int typ, int dam_hp, int monspell, bool learnable)
{
	int flg = PROJECT_BEAM | PROJECT_KILL | PROJECT_THRU | PROJECT_PLAYER;

	/* Target the player with a bolt attack */
	(void)project(caster_ptr, 0, target_ptr->fy, target_ptr->fx, dam_hp, typ, flg, (learnable ? monspell : -1));
}


/*
 * Cast a breath (or ball) attack at the player
 * Pass over any creatures that may be in the way
 * Affect grids, objects, creatures, and the player
 */
static void breath(int y, int x, creature_type *caster_ptr, int typ, int dam_hp, int rad, bool breath, int monspell, bool learnable)
{
	int flg = PROJECT_GRID | PROJECT_ITEM | PROJECT_KILL | PROJECT_PLAYER;

	/* Determine the radius of the blast */
	if ((rad < 1) && breath) rad = has_trait(caster_ptr, TRAIT_POWERFUL) ? 3 : 2;

	/* Handle breath attacks */
	if (breath) rad = 0 - rad;

	switch (typ)
	{
	case GF_ROCKET:
		flg |= PROJECT_STOP;
		break;
	case GF_DRAIN_MANA:
	case GF_MIND_BLAST:
	case GF_BRAIN_SMASH:
	case GF_CAUSE_1:
	case GF_CAUSE_2:
	case GF_CAUSE_3:
	case GF_CAUSE_4:
	case GF_HAND_DOOM:
		flg |= (PROJECT_HIDE | PROJECT_AIMED);
		break;
	}

	/* Target the player with a ball attack */
	(void)project(caster_ptr, rad, y, x, dam_hp, typ, flg, (learnable ? monspell : -1));
}


u32b get_curse(int power, object_type *object_ptr)
{
	u32b new_curse;

	while(1)
	{
		new_curse = (1 << (randint0(MAX_CURSE)+4));
		if (power == 2)
		{
			if (!(new_curse & TRC_HEAVY_MASK)) continue;
		}
		else if (power == 1)
		{
			if (new_curse & TRC_SPECIAL_MASK) continue;
		}
		else if (power == 0)
		{
			if (new_curse & TRC_HEAVY_MASK) continue;
		}
		if (new_curse == TRC_LOW_MELEE && !object_is_weapon(object_ptr)) continue;
		if (new_curse == TRC_LOW_AC && !object_is_armour(object_ptr)) continue;
		break;
	}
	return new_curse;
}

void curse_equipment(creature_type *creature_ptr, int chance, int heavy_chance)
{
	bool        changed = FALSE;
	int         curse_power = 0;
	u32b        new_curse;
	u32b oflgs[TR_FLAG_SIZE];
	//TODO SELECT
	object_type *object_ptr = get_equipped_slot_ptr(creature_ptr, INVEN_SLOT_HAND, 1);
	char object_name[MAX_NLEN];

	if (randint1(100) > chance) return;

	if (!object_ptr->k_idx) return;

	object_flags(object_ptr, oflgs);

	object_desc(object_name, object_ptr, (OD_OMIT_PREFIX | OD_NAME_ONLY));

	/* Extra, biased saving throw for blessed items */
	if (have_flag(oflgs, TRAIT_BLESSED_BRAND) && (randint1(888) > chance))
	{
#ifdef JP
		msg_format("%s%s͎􂢂𒵂˕ԂI", creature_ptr->name, object_name);
#else
		msg_format("%s's %s resist%s cursing!", creature_ptr->name, object_name, ((object_ptr->number > 1) ? "" : "s"));
#endif
			
		/* Hmmm -- can we wear multiple items? If not, this is unnecessary */
		return;
	}

	if ((randint1(100) <= heavy_chance) &&
	    (object_is_artifact(object_ptr) || object_is_ego(object_ptr)))
	{
		if (!(object_ptr->curse_flags & TRC_HEAVY_CURSE))
			changed = TRUE;
		object_ptr->curse_flags |= TRC_HEAVY_CURSE;
		object_ptr->curse_flags |= TRC_CURSED;
		curse_power++;
	}
	else
	{
		if (!object_is_cursed(object_ptr))
			changed = TRUE;
		object_ptr->curse_flags |= TRC_CURSED;
	}
	if (heavy_chance >= 50) curse_power++;

	new_curse = get_curse(curse_power, object_ptr);
	if (!(object_ptr->curse_flags & new_curse))
	{
		changed = TRUE;
		object_ptr->curse_flags |= new_curse;
	}

	if (changed)
	{
#ifdef JP
msg_format("ӂɖI[%s%sƂ܂...", creature_ptr->name, object_name);
#else
		msg_format("There is a malignant black aura surrounding %s's %s...", creature_ptr->name, object_name);
#endif

		object_ptr->feeling = FEEL_NONE;
	}
	creature_ptr->creature_update |= (CRU_BONUS);
}


/*
 * Return TRUE if a spell is good for hurting the player (directly).
 */
static bool spell_attack(byte spell)
{
	/* All RF4 spells hurt (except for shriek and dispel) */
	if (spell < 128 && spell > 98) return (TRUE);

	/* Various "ball" spells */
	if (spell >= 128 && spell <= 128 + 8) return (TRUE);

	/* "Cause wounds" and "bolt" spells */
	if (spell >= 128 + 12 && spell < 128 + 27) return (TRUE);

	/* Hand of Doom */
	if (spell == 160 + 1) return (TRUE);

	/* Psycho-Spear */
	if (spell == 160 + 11) return (TRUE);

	/* Doesn't hurt */
	return (FALSE);
}


/*
 * Return TRUE if a spell is good for escaping.
 */
static bool spell_escape(byte spell)
{
	/* Blink or Teleport */
	if (spell == 160 + 4 || spell == 160 + 5) return (TRUE);

	/* Teleport the player away */
	if (spell == 160 + 9 || spell == 160 + 10) return (TRUE);

	/* Isn't good for escaping */
	return (FALSE);
}

/*
 * Return TRUE if a spell is good for annoying the player.
 */
static bool spell_annoy(byte spell)
{
	/* Shriek */
	if (spell == 96 + 0) return (TRUE);

	/* Brain smash, et al (added curses) */
	if (spell >= 128 + 9 && spell <= 128 + 14) return (TRUE);

	/* Scare, confuse, blind, slow, paralyze */
	if (spell >= 128 + 27 && spell <= 128 + 31) return (TRUE);

	/* Teleport to */
	if (spell == 160 + 8) return (TRUE);

	/* Teleport level */
	if (spell == 160 + 10) return (TRUE);

	/* Darkness, make traps, cause amnesia */
	if (spell >= 160 + 12 && spell <= 160 + 14) return (TRUE);

	/* Doesn't annoy */
	return (FALSE);
}

/*
 * Return TRUE if a spell summons help.
 */
static bool spell_summon(byte spell)
{
	/* All summon spells */
	if (spell >= 160 + 16) return (TRUE);

	/* Doesn't summon */
	return (FALSE);
}


/*
 * Return TRUE if a spell raise-dead.
 */
static bool spell_raise(byte spell)
{
	/* All raise-dead spells */
	if (spell == 160 + 15) return (TRUE);

	/* Doesn't summon */
	return (FALSE);
}


/*
 * Return TRUE if a spell is good in a tactical situation.
 */
static bool spell_tactic(byte spell)
{
	/* Blink */
	if (spell == 160 + 4) return (TRUE);

	/* Not good */
	return (FALSE);
}

/*
 * Return TRUE if a spell makes invulnerable.
 */
static bool spell_invulner(byte spell)
{
	/* Invulnerability */
	if (spell == 160 + 3) return (TRUE);

	/* No invulnerability */
	return (FALSE);
}

/*
 * Return TRUE if a spell hastes.
 */
static bool spell_haste(byte spell)
{
	/* Haste self */
	if (spell == 160 + 0) return (TRUE);

	/* Not a haste spell */
	return (FALSE);
}


/*
 * Return TRUE if a spell world.
 */
static bool spell_world(byte spell)
{
	/* world */
	if (spell == 160 + 6) return (TRUE);

	/* Not a haste spell */
	return (FALSE);
}


/*
 * Return TRUE if a spell special.
 */
static bool spell_special(byte spell)
{
	if (gamble_arena_mode) return FALSE;

	/* world */
	if (spell == 160 + 7) return (TRUE);

	/* Not a haste spell */
	return (FALSE);
}


/*
 * Return TRUE if a spell psycho-spear.
 */
static bool spell_psy_spe(byte spell)
{
	/* world */
	if (spell == 160 + 11) return (TRUE);

	/* Not a haste spell */
	return (FALSE);
}


/*
 * Return TRUE if a spell is good for healing.
 */
static bool spell_heal(byte spell)
{
	/* Heal */
	if (spell == 160 + 2) return (TRUE);

	/* No healing */
	return (FALSE);
}


/*
 * Return TRUE if a spell is good for dispel.
 */
static bool spell_dispel(byte spell)
{
	/* Dispel */
	if (spell == 96 + 2) return (TRUE);

	/* No dispel */
	return (FALSE);
}


/*
 * Check should creature cast dispel spell.
 */
bool dispel_check(creature_type *caster_ptr, creature_type *target_ptr)
{

	/* Invulnabilty (including the song) */
	if (IS_INVULN(target_ptr)) return (TRUE);

	/* Wraith form */
	if (target_ptr->wraith_form) return (TRUE);

	/* Shield */
	if (target_ptr->shield) return (TRUE);

	/* Magic defence */
	if (target_ptr->magicdef) return (TRUE);

	/* Multi Shadow */
	if (target_ptr->multishadow) return (TRUE);

	/* Robe of dust */
	if (target_ptr->dustrobe) return (TRUE);

	/* Berserk Strength */
	if (target_ptr->shero && (target_ptr->class_idx != CLASS_BERSERKER)) return (TRUE);

	/* Elemental resistances */
	if (has_trait(caster_ptr, TRAIT_BR_ACID))
	{
		if (!has_trait(target_ptr, TRAIT_IM_ACID) && (target_ptr->oppose_acid || music_singing(target_ptr, MUSIC_RESIST))) return (TRUE);
		if (target_ptr->special_defense & DEFENSE_ACID) return (TRUE);
	}

	if (has_trait(caster_ptr, TRAIT_BR_FIRE))
	{
		if (!(is_demon_creature(target_ptr) && target_ptr->lev > 44))
		{
			if (!has_trait(target_ptr, TRAIT_IM_FIRE) && (target_ptr->oppose_fire || music_singing(target_ptr, MUSIC_RESIST))) return (TRUE);
			if (target_ptr->special_defense & DEFENSE_FIRE) return (TRUE);
		}
	}

	if (has_trait(caster_ptr, TRAIT_BR_ELEC))
	{
		if (!has_trait(target_ptr, TRAIT_IM_ELEC) && (target_ptr->oppose_elec || music_singing(target_ptr, MUSIC_RESIST))) return (TRUE);
		if (target_ptr->special_defense & DEFENSE_ELEC) return (TRUE);
	}

	if (has_trait(caster_ptr, TRAIT_BR_COLD))
	{
		if (!has_trait(target_ptr, TRAIT_IM_COLD) && (target_ptr->oppose_cold || music_singing(target_ptr, MUSIC_RESIST))) return (TRUE);
		if (target_ptr->special_defense & DEFENSE_COLD) return (TRUE);
	}

	if (has_trait(caster_ptr, TRAIT_BR_POIS) || has_trait(caster_ptr, TRAIT_BR_NUKE))
	{
		if (!((target_ptr->class_idx == CLASS_NINJA) && caster_ptr->lev > 44))
		{
			if (target_ptr->oppose_pois || music_singing(caster_ptr, MUSIC_RESIST)) return (TRUE);
			if (target_ptr->special_defense & DEFENSE_POIS) return (TRUE);
		}
	}

	/* Ultimate resistance */
	if (target_ptr->ult_res) return (TRUE);

	/* Potion of Neo Tsuyosi special */
	if (target_ptr->tsuyoshi) return (TRUE);

	/* Elemental Brands */
	if ((target_ptr->special_attack & ATTACK_ACID) && !has_trait(target_ptr, TRAIT_RES_ACID)) return (TRUE);
	if ((target_ptr->special_attack & ATTACK_FIRE) && !has_trait(target_ptr, TRAIT_RES_FIRE)) return (TRUE);
	if ((target_ptr->special_attack & ATTACK_ELEC) && !has_trait(target_ptr, TRAIT_RES_ELEC)) return (TRUE);
	if ((target_ptr->special_attack & ATTACK_COLD) && !has_trait(target_ptr, TRAIT_RES_COLD)) return (TRUE);
	if ((target_ptr->special_attack & ATTACK_POIS) && !has_trait(target_ptr, TRAIT_RES_POIS)) return (TRUE);

	/* Speed */
	if (target_ptr->speed < 145)
	{
		if (IS_FAST(target_ptr)) return (TRUE);
	}

	/* Light speed */
	if (target_ptr->lightspeed && (target_ptr->speed < 136)) return (TRUE);

	if (target_ptr->riding && (creature_list[target_ptr->riding].speed < 135))
	{
		if (creature_list[target_ptr->riding].fast) return (TRUE);
	}

	/* No need to cast dispel spell */
	return (FALSE);
}


/*
 * Have a creature choose a spell from a list of "useful" spells.
 *
 * Note that this list does NOT include spells that will just hit
 * other creatures, and the list is restricted when the creature is
 * "desperate".  Should that be the job of this function instead?
 *
 * Stupid creatures will just pick a spell randomly.  Smart creatures
 * will choose more "intelligently".
 *
 * Use the helper functions above to put spells into categories.
 *
 * This function may well be an efficiency bottleneck.
 */
static int choose_attack_spell(creature_type *caster_ptr, creature_type *target_ptr, int spells[], byte num)
{
	species_type *r_ptr = &species_info[caster_ptr->species_idx];

	byte escape[96], escape_num = 0;
	byte attack[96], attack_num = 0;
	byte summon[96], summon_num = 0;
	byte tactic[96], tactic_num = 0;
	byte annoy[96], annoy_num = 0;
	byte invul[96], invul_num = 0;
	byte haste[96], haste_num = 0;
	byte world[96], world_num = 0;
	byte special[96], special_num = 0;
	byte psy_spe[96], psy_spe_num = 0;
	byte raise[96], raise_num = 0;
	byte heal[96], heal_num = 0;
	byte dispel[96], dispel_num = 0;

	int i;

	/* Stupid creatures choose randomly */
	if (has_trait(caster_ptr, TRAIT_STUPID))
	{
		/* Pick at random */
		return (spells[randint0(num)]);
	}

	/* Categorize spells */
	for (i = 0; i < num; i++)
	{
		/* Escape spell? */
		if (spell_escape(spells[i])) escape[escape_num++] = spells[i];

		/* Attack spell? */
		if (spell_attack(spells[i])) attack[attack_num++] = spells[i];

		/* Summon spell? */
		if (spell_summon(spells[i])) summon[summon_num++] = spells[i];

		/* Tactical spell? */
		if (spell_tactic(spells[i])) tactic[tactic_num++] = spells[i];

		/* Annoyance spell? */
		if (spell_annoy(spells[i])) annoy[annoy_num++] = spells[i];

		/* Invulnerability spell? */
		if (spell_invulner(spells[i])) invul[invul_num++] = spells[i];

		/* Haste spell? */
		if (spell_haste(spells[i])) haste[haste_num++] = spells[i];

		/* World spell? */
		if (spell_world(spells[i])) world[world_num++] = spells[i];

		/* Special spell? */
		if (spell_special(spells[i])) special[special_num++] = spells[i];

		/* Psycho-spear spell? */
		if (spell_psy_spe(spells[i])) psy_spe[psy_spe_num++] = spells[i];

		/* Raise-dead spell? */
		if (spell_raise(spells[i])) raise[raise_num++] = spells[i];

		/* Heal spell? */
		if (spell_heal(spells[i])) heal[heal_num++] = spells[i];

		/* Dispel spell? */
		if (spell_dispel(spells[i])) dispel[dispel_num++] = spells[i];
	}

	/*** Try to pick an appropriate spell type ***/

	/* world */
	if (world_num && (randint0(100) < 15) && !the_world)
	{
		/* Choose haste spell */
		return (world[randint0(world_num)]);
	}

	/* special */
	if (special_num)
	{
		bool success = FALSE;
		switch(caster_ptr->species_idx)
		{
			case SPECIES_BANOR:
			case SPECIES_LUPART:
				if ((caster_ptr->chp < caster_ptr->mhp / 2) && species_info[SPECIES_BANOR].max_num && species_info[SPECIES_LUPART].max_num) success = TRUE;
				break;
			default: break;
		}
		if (success) return (special[randint0(special_num)]);
	}

	/* Still hurt badly, couldn't flee, attempt to heal */
	if (caster_ptr->chp < caster_ptr->mhp / 3 && one_in_(2))
	{
		/* Choose heal spell if possible */
		if (heal_num) return (heal[randint0(heal_num)]);
	}

	/* Hurt badly or afraid, attempt to flee */
	if (((caster_ptr->chp < caster_ptr->mhp / 3) || caster_ptr->afraid) && one_in_(2))
	{
		/* Choose escape spell if possible */
		if (escape_num) return (escape[randint0(escape_num)]);
	}

	/* special */
	if (special_num)
	{
		bool success = FALSE;
		switch (caster_ptr->species_idx)
		{
			case SPECIES_OHMU:
			case SPECIES_BANOR:
			case SPECIES_LUPART:
				break;
			case SPECIES_BANORLUPART:
				if (randint0(100) < 70) success = TRUE;
				break;
			case SPECIES_ROLENTO:
				if (randint0(100) < 40) success = TRUE;
				break;
			default:
				if (randint0(100) < 50) success = TRUE;
				break;
		}
		if (success) return (special[randint0(special_num)]);
	}

	/* Player is close and we have attack spells, blink away */
	if ((distance(target_ptr->fy, target_ptr->fx, caster_ptr->fy, caster_ptr->fx) < 4) && (attack_num || has_trait_raw(&r_ptr->flags, TRAIT_TRAPS)) && (randint0(100) < 75) && !the_world)
	{
		/* Choose tactical spell */
		if (tactic_num) return (tactic[randint0(tactic_num)]);
	}

	/* Summon if possible (sometimes) */
	if (summon_num && (randint0(100) < 40))
	{
		/* Choose summon spell */
		return (summon[randint0(summon_num)]);
	}

	/* dispel */
	if (dispel_num && one_in_(2))
	{
		/* Choose dispel spell if possible */
		if (dispel_check(caster_ptr, target_ptr))
		{
			return (dispel[randint0(dispel_num)]);
		}
	}

	/* Raise-dead if possible (sometimes) */
	if (raise_num && (randint0(100) < 40))
	{
		/* Choose raise-dead spell */
		return (raise[randint0(raise_num)]);
	}

	/* Attack spell (most of the time) */
	if (IS_INVULN(target_ptr))
	{
		if (psy_spe_num && (randint0(100) < 50))
		{
			/* Choose attack spell */
			return (psy_spe[randint0(psy_spe_num)]);
		}
		else if (attack_num && (randint0(100) < 40))
		{
			/* Choose attack spell */
			return (attack[randint0(attack_num)]);
		}
	}
	else if (attack_num && (randint0(100) < 85))
	{
		/* Choose attack spell */
		return (attack[randint0(attack_num)]);
	}

	/* Try another tactical spell (sometimes) */
	if (tactic_num && (randint0(100) < 50) && !the_world)
	{
		/* Choose tactic spell */
		return (tactic[randint0(tactic_num)]);
	}

	/* Cast globe of invulnerability if not already in effect */
	if (invul_num && !caster_ptr->invuln && (randint0(100) < 50))
	{
		/* Choose Globe of Invulnerability */
		return (invul[randint0(invul_num)]);
	}

	/* We're hurt (not badly), try to heal */
	if ((caster_ptr->chp < caster_ptr->mhp * 3 / 4) && (randint0(100) < 25))
	{
		/* Choose heal spell if possible */
		if (heal_num) return (heal[randint0(heal_num)]);
	}

	/* Haste self if we aren't already somewhat hasted (rarely) */
	if (haste_num && (randint0(100) < 20) && !caster_ptr->fast)
	{
		/* Choose haste spell */
		return (haste[randint0(haste_num)]);
	}

	/* Annoy player (most of the time) */
	if (annoy_num && (randint0(100) < 80))
	{
		/* Choose annoyance spell */
		return (annoy[randint0(annoy_num)]);
	}

	/* Choose no spell */
	return (0);
}



static bool adjacent_grid_check(creature_type *base_ptr, creature_type *m_ptr, int *yp, int *xp, int f_flag, bool (*path_check)(floor_type *, int, int, int, int))
{
	int i;
	int tonari;
	static int tonari_y[4][8] = {{-1, -1, -1,  0,  0,  1,  1,  1},
			                     {-1, -1, -1,  0,  0,  1,  1,  1},
			                     { 1,  1,  1,  0,  0, -1, -1, -1},
			                     { 1,  1,  1,  0,  0, -1, -1, -1}};
	static int tonari_x[4][8] = {{-1,  0,  1, -1,  1, -1,  0,  1},
			                     { 1,  0, -1,  1, -1,  1,  0, -1},
			                     {-1,  0,  1, -1,  1, -1,  0,  1},
			                     { 1,  0, -1,  1, -1,  1,  0, -1}};
	floor_type *floor_ptr = get_floor_ptr(base_ptr);

	if (m_ptr->fy < base_ptr->fy && m_ptr->fx < base_ptr->fx) tonari = 0;
	else if (m_ptr->fy < base_ptr->fy) tonari = 1;
	else if (m_ptr->fx < base_ptr->fx) tonari = 2;
	else tonari = 3;

	for (i = 0; i < 8; i++)
	{
		int next_x = *xp + tonari_x[tonari][i];
		int next_y = *yp + tonari_y[tonari][i];
		cave_type *c_ptr;

		/* Access the next grid */
		c_ptr = &floor_ptr->cave[next_y][next_x];

		/* Skip this feature */
		if (!cave_have_flag_grid(c_ptr, f_flag)) continue;

		if (path_check(floor_ptr, m_ptr->fy, m_ptr->fx, next_y, next_x))
		{
			*yp = next_y;
			*xp = next_x;
			return TRUE;
		}
	}

	return FALSE;
}

#define DO_SPELL_NONE    0
#define DO_SPELL_BR_LITE 1
#define DO_SPELL_BR_DISI 2
#define DO_SPELL_BA_LITE 3

/*
 * Creatures can cast spells, shoot missiles, and breathe.
 *
 * Returns "TRUE" if a spell (or whatever) was (successfully) cast.
 *
 * XXX XXX XXX This function could use some work, but remember to
 * keep it as optimized as possible, while retaining generic code.
 *
 * Verify the various "blind-ness" checks in the code.
 *
 * XXX XXX XXX Note that several effects should really not be "seen"
 * if the player is blind.  See also "effects.c" for other "mistakes".
 *
 * Perhaps creatures should breathe at locations *near* the player,
 * since this would allow them to inflict "partial" damage.
 *
 * Perhaps smart creatures should decline to use "bolt" spells if
 * there is a creature in the way, unless they wish to kill it.
 *
 * Note that, to allow the use of the "track_target" option at some
 * later time, certain non-optimal things are done in the code below,
 * including explicit checks against the "direct" variable, which is
 * currently always true by the time it is checked, but which should
 * really be set according to an explicit "projectable(floor_ptr, )" test, and
 * the use of generic "x,y" locations instead of the player location,
 * with those values being initialized with the player location.
 *
 * It will not be possible to "correctly" handle the case in which a
 * creature attempts to attack a location which is thought to contain
 * the player, but which in fact is nowhere near the player, since this
 * might induce all sorts of messages about the attack itself, and about
 * the effects of the attack, which the player might or might not be in
 * a position to observe.  Thus, for simplicity, it is probably best to
 * only allow "faulty" attacks by a creature if one of the important grids
 * (probably the initial or final grid) is in fact in view of the player.
 * It may be necessary to actually prevent spell attacks except when the
 * creature actually has line of sight to the player.  Note that a creature
 * could be left in a bizarre situation after the player ducked behind a
 * pillar and then teleported away, for example.
 *
 * Note that certain spell attacks do not use the "project()" function
 * but "simulate" it via the "direct" variable, which is always at least
 * as restrictive as the "project()" function.  This is necessary to
 * prevent "blindness" attacks and such from bending around walls, etc,
 * and to allow the use of the "track_target" option in the future.
 *
 * Note that this function attempts to optimize the use of spells for the
 * cases in which the creature has no spells, or has spells but cannot use
 * them, or has spells but they will have no "useful" effect.  Note that
 * this function has been an efficiency bottleneck in the past.
 *
 * Note the special "MFLAG_NICE" flag, which prevents a creature from using
 * any spell attacks until the player has had a single chance to move.
 */
bool make_attack_spell(creature_type *caster_ptr, creature_type *target_ptr)
{
	floor_type      *floor_ptr = get_floor_ptr(caster_ptr);
	int             k, thrown_spell = 0, rlev, failrate;
	int             racial_spell[MAX_TRAITS], num = 0;
	species_type    *r_ptr = &species_info[caster_ptr->species_idx];
	char            m_name[80];
#ifndef JP
	char            m_poss[80];
#endif
	bool            no_inate = FALSE;
	bool            do_spell = DO_SPELL_NONE;
	int             dam = 0;
	u32b mode = 0L;
	int s_num_6 = 6;
	int s_num_4 = 4;

	/* Target location */
	int x = target_ptr->fx;
	int y = target_ptr->fy;

	/* Target location for lite breath */
	int x_br_lite = 0;
	int y_br_lite = 0;

	/* Summon count */
	int count = 0;

	/* Extract the blind-ness */
	bool blind = (IS_BLIND(target_ptr) ? TRUE : FALSE);

	/* Extract the "see-able-ness" */
	bool seen = (!blind && caster_ptr->ml);

	bool maneable = player_has_los_bold(caster_ptr->fy, caster_ptr->fx);
	bool learnable = (seen && maneable && !the_world);

	/* Check "projectable" */
	bool direct;

	bool in_no_magic_dungeon = (dungeon_info[floor_ptr->dun_type].flags1 & DF1_NO_MAGIC) && floor_ptr->floor_level
		&& (!inside_quest || is_fixed_quest_idx(inside_quest));

	bool can_use_lite_area = FALSE;

	bool can_remember;

	/* Cannot cast spells when confused */
	if (caster_ptr->confused)
	{
		reset_target(caster_ptr);
		return (FALSE);
	}

	/* Cannot cast spells when nice */
	if (caster_ptr->mflag & MFLAG_NICE) return (FALSE);
	if (!is_hostile(caster_ptr)) return (FALSE);


	/* Sometimes forbid inate attacks (breaths) */
	if (randint0(100) >= (r_ptr->freq_spell * 2)) no_inate = TRUE;

	/* XXX XXX XXX Handle "track_target" option (?) */


	/*** require projectable player ***/

	/* Check range */
	if ((caster_ptr->cdis > MAX_RANGE) && !caster_ptr->target_y) return (FALSE);

	/* Check path for lite breath */
	if (has_trait(target_ptr, TRAIT_BR_LITE))
	{
		y_br_lite = y;
		x_br_lite = x;

		if (los(floor_ptr, caster_ptr->fy, caster_ptr->fx, y_br_lite, x_br_lite))
		{
			feature_type *f_ptr = &feature_info[floor_ptr->cave[y_br_lite][x_br_lite].feat];

			if (!have_flag(f_ptr->flags, FF_LOS))
			{
				//TODO if (have_flag(f_ptr->flags, FF_PROJECT) && one_in_(2)) f4 &= ~(RF4_BR_LITE);
			}
		}

		/* Check path to next grid */
		//TODO else if (!adjacent_grid_check(caster_ptr, &y_br_lite, &x_br_lite, FF_LOS, los)) f4 &= ~(RF4_BR_LITE);

		/* Don't breath lite to the wall if impossible */
		if (!has_trait(target_ptr, TRAIT_BR_LITE))
		{
			y_br_lite = 0;
			x_br_lite = 0;
		}
	}

	/* Check path */
	if (projectable(floor_ptr, caster_ptr->fy, caster_ptr->fx, y, x))
	{
		feature_type *f_ptr = &feature_info[floor_ptr->cave[y][x].feat];

		if (!have_flag(f_ptr->flags, FF_PROJECT))
		{
			/* Breath disintegration to the wall if possible */
			if (has_trait(target_ptr, TRAIT_BR_DISI) && have_flag(f_ptr->flags, FF_HURT_DISI) && one_in_(2)) do_spell = DO_SPELL_BR_DISI;

			/* Breath lite to the transparent wall if possible */
			else if (has_trait(target_ptr, TRAIT_BR_LITE) && have_flag(f_ptr->flags, FF_LOS) && one_in_(2)) do_spell = DO_SPELL_BR_LITE;
		}
	}

	/* Check path to next grid */
	else
	{
		bool success = FALSE;

		if (has_trait(target_ptr, TRAIT_BR_DISI) && (caster_ptr->cdis < MAX_RANGE/2) &&
		    in_disintegration_range(floor_ptr, caster_ptr->fy, caster_ptr->fx, y, x) &&
		    (one_in_(10) || (projectable(floor_ptr, y, x, caster_ptr->fy, caster_ptr->fx) && one_in_(2))))
		{
			do_spell = DO_SPELL_BR_DISI;
			success = TRUE;
		}
		else if (has_trait(target_ptr, TRAIT_BR_LITE) && (caster_ptr->cdis < MAX_RANGE/2) &&
		    los(floor_ptr, caster_ptr->fy, caster_ptr->fx, y, x) && one_in_(5))
		{
			do_spell = DO_SPELL_BR_LITE;
			success = TRUE;
		}
		else if (has_trait(target_ptr, TRAIT_BA_LITE) && (caster_ptr->cdis <= MAX_RANGE))
		{
			int by = y, bx = x;
			get_project_point(caster_ptr, caster_ptr->fy, caster_ptr->fx, &by, &bx, 0L);
			if ((distance(by, bx, y, x) <= 3) && los(floor_ptr, by, bx, y, x) && one_in_(5))
			{
				do_spell = DO_SPELL_BA_LITE;
				success = TRUE;
			}
		}

		if (!success) success = adjacent_grid_check(caster_ptr, target_ptr, &y, &x, FF_PROJECT, projectable);

		if (!success)
		{
			if (caster_ptr->target_y && caster_ptr->target_x)
			{
				y = caster_ptr->target_y;
				x = caster_ptr->target_x;
				//TODO f4 &= (RF4_INDIRECT_MASK);
				//TODO f5 &= (RF5_INDIRECT_MASK);
				//TODO f6 &= (RF6_INDIRECT_MASK);
				success = TRUE;
			}

			if (y_br_lite && x_br_lite && (caster_ptr->cdis < MAX_RANGE/2) && one_in_(5))
			{
				if (!success)
				{
					y = y_br_lite;
					x = x_br_lite;
					do_spell = DO_SPELL_BR_LITE;
					success = TRUE;
				}
				//TODO else f4 |= (RF4_BR_LITE);
			}
		}

		/* No spells */
		if (!success) return FALSE;
	}

	reset_target(caster_ptr);

	/* Extract the creature level */
	rlev = ((r_ptr->level >= 1) ? r_ptr->level : 1);

	/* Forbid inate attacks sometimes */
	if (no_inate)
	{
		//TODO f4 &= ~(RF4_NOMAGIC_MASK);
		//TODO f5 &= ~(RF5_NOMAGIC_MASK);
		//TODO f6 &= ~(RF6_NOMAGIC_MASK);
	}

	if (has_trait(target_ptr, TRAIT_DARKNESS))
	{
		if ((target_ptr->class_idx == CLASS_NINJA) &&
		    !is_hurt_lite_creature(caster_ptr) &&
			!is_undead_creature(caster_ptr) && 
		    !is_darken_creature(caster_ptr))
			can_use_lite_area = TRUE;

		if (!has_trait(caster_ptr, TRAIT_STUPID))
		{
			//TODO if (dungeon_info[floor_ptr->dun_type].flags1 & DF1_DARKNESS) f6 &= ~(RF6_DARKNESS);
			//TODO else if ((target_ptr->class_idx == CLASS_NINJA) && !can_use_lite_area) f6 &= ~(RF6_DARKNESS);
		}
	}

	if (in_no_magic_dungeon && !has_trait(caster_ptr, TRAIT_STUPID))
	{
		//TODO f4 &= (RF4_NOMAGIC_MASK);
		//TODO f5 &= (RF5_NOMAGIC_MASK);
		//TODO f6 &= (RF6_NOMAGIC_MASK);
	}

	if (has_trait(caster_ptr, TRAIT_SMART))
	{
		/* Hack -- allow "desperate" spells */
		if ((caster_ptr->chp < caster_ptr->mhp / 10) &&
			(randint0(100) < 50))
		{
			/* Require intelligent spells */
			//TODO f4 &= (RF4_INT_MASK);
			//TODO f5 &= (RF5_INT_MASK);
			//TODO f6 &= (RF6_INT_MASK);
		}

		/* Hack -- decline "teleport level" in some case */
		if (has_trait(target_ptr, TRAIT_TELE_LEVEL) && TELE_LEVEL_IS_INEFF(floor_ptr, target_ptr, 0))
		{
			//TODO f6 &= ~(RF6_TELE_LEVEL);
		}
	}

	/* No spells left */
	//TODO if (!f4 && !f5 && !f6) return (FALSE);

	/* Remove the "ineffective" spells */
	//TODO remove_bad_spells(caster_ptr, &f4, &f5, &f6);

	if (fight_arena_mode || gamble_arena_mode)
	{
		//TODO f4 &= ~(RF4_SUMMON_MASK);
		//TODO f5 &= ~(RF5_SUMMON_MASK);
		//TODO f6 &= ~(RF6_SUMMON_MASK | RF6_TELE_LEVEL);

		//TODO if (caster_ptr->species_idx == SPECIES_ROLENTO) f6 &= ~(RF6_SPECIAL);
	}

	/* No spells left */
	//TODO if (!f4 && !f5 && !f6) return (FALSE);

	if (!has_trait(caster_ptr, TRAIT_STUPID))
	{
		//TODO if (!target_ptr->csp) f5 &= ~(RF5_DRAIN_MANA);

		/* Check for a clean bolt shot */
		//TODO if (((f4 & RF4_BO_MASK) ||
		//TODO      (f5 & RF5_BO_MASK) ||
		//TODO      (f6 & RF6_BO_MASK)) &&
		//TODO     !clean_shot(caster_ptr->fy, caster_ptr->fx, target_ptr->fy, target_ptr->fx, FALSE))
		//TODO {
			/* Remove spells that will only hurt friends */
		//TODO 	f4 &= ~(RF4_BO_MASK);
		//TODO 	f5 &= ~(RF5_BO_MASK);
		//TODO 	f6 &= ~(RF6_BO_MASK);
		//TODO }

		/* Check for a possible summon */
		//TODO if (((f4 & RF4_SUMMON_MASK) ||
		//TODO      (f5 & RF5_SUMMON_MASK) ||
		//TODO      (f6 & RF6_SUMMON_MASK)) &&
		//TODO     !(summon_possible(y, x)))
		//TODO {
			/* Remove summoning spells */
		//TODO 	f4 &= ~(RF4_SUMMON_MASK);
		//TODO 	f5 &= ~(RF5_SUMMON_MASK);
		//TODO 	f6 &= ~(RF6_SUMMON_MASK);
		//TODO }

		/* Check for a possible raise dead */
		if (has_trait(caster_ptr, TRAIT_ANIM_DEAD) && !raise_possible(caster_ptr, target_ptr))
		{
			/* Remove raise dead spell */
			//TODO f6 &= ~(RF6_RAISE_DEAD);
		}

		/* Special moves restriction */
		if (has_trait(caster_ptr, TRAIT_SPECIAL))
		{
			if ((caster_ptr->species_idx == SPECIES_ROLENTO) && !summon_possible(target_ptr, y, x))
			{
				//TODO f6 &= ~(RF6_SPECIAL);
			}
		}

		/* No spells left */
		//TODO if (!f4 && !f5 && !f6) return (FALSE);
	}

	/* Extract the "inate" spells */
	for(k = 0; k < max_trait_idx; k++)
		if (trait_info[k].is_spell && has_trait(caster_ptr, k)) racial_spell[num++] = k;

	/* No spells left */
	if (!num) return (FALSE);

	/* Stop if player is dead or gone */
	if (!playing || gameover) return (FALSE);

	/* Stop if player is leaving */
	if (subject_change_floor) return (FALSE);

	/* Get the creature name (or "it") */
	creature_desc(m_name, caster_ptr, 0x00);

#ifndef JP
	/* Get the creature possessive ("his"/"her"/"its") */
	creature_desc(m_poss, caster_ptr, MD_PRON_VISIBLE | MD_POSSESSIVE);
#endif

	switch (do_spell)
	{
	case DO_SPELL_NONE:
		{
			int attempt = 10;
			while (attempt--)
			{
				thrown_spell = choose_attack_spell(caster_ptr, target_ptr, racial_spell, num);
				if (thrown_spell) break;
			}
		}
		break;

	case DO_SPELL_BR_LITE:
		thrown_spell = TRAIT_BR_LITE;
		break;

	case DO_SPELL_BR_DISI:
		thrown_spell = TRAIT_BR_DISI;
		break;

	case DO_SPELL_BA_LITE:
		thrown_spell = TRAIT_BA_LITE;
		break;

	default:
		return FALSE; /* Paranoia */
	}

	/* Abort if no spell was chosen */
	if (!thrown_spell) return (FALSE);

	/* Calculate spell failure rate */
	failrate = 25 - (rlev + 3) / 4;

	/* Hack -- Stupid creatures will never fail (for jellies and such) */
	if (has_trait(caster_ptr, TRAIT_STUPID)) failrate = 0;

	/* Check for spell failure (inate attacks never fail) */
	// TODO Distinction of spell failure_rate
	if ((in_no_magic_dungeon || (caster_ptr->stun && one_in_(2)) || (randint0(100) < failrate)))
	{
		disturb(player_ptr, 1, 0);
		/* Message */
#ifdef JP
		msg_format("%^s͎悤ƂsB", m_name);
#else
		msg_format("%^s tries to cast a spell, but fails.", m_name);
#endif

		return (TRUE);
	}

	/* Hex: Anti Magic Barrier */
	// TODO Distinction of spell failure_rate
	if (magic_barrier(target_ptr, caster_ptr))
	{
#ifdef JP
		msg_format("@oA%^s̎B", m_name);
#else
		msg_format("Anti magic barrier cancels the spell which %^s casts.");
#endif
		return (TRUE);
	}

	/* Projectable? */
	direct = creature_bold(caster_ptr, y, x);

	can_remember = is_original_ap_and_seen(target_ptr, caster_ptr);

	/* Cast the spell. */
	switch (thrown_spell)
	{
		case TRAIT_SHRIEK:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
msg_format("%^s񍂂؂萺B", m_name);
#else
			msg_format("%^s makes a high pitched shriek.", m_name);
#endif

			aggravate_creatures(caster_ptr);
			break;
		}

		case TRAIT_DISPEL:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s͋Ԃ₢B", m_name);
			else msg_format("%^s͏̎OB", m_name);
#else
			if (blind) msg_format("%^s mumbles powerfully.", m_name);
			else msg_format("%^s invokes a dispel magic.", m_name);
#endif
			dispel_creature(target_ptr);
			if (target_ptr->riding) dispel_creature(&creature_list[target_ptr->riding]);

#ifdef JP
			if ((target_ptr->chara_idx == CHARA_COMBAT) || (get_equipped_slot_ptr(target_ptr, INVEN_SLOT_BOW, 1)->name1 == ART_CRIMSON))
				msg_print("₪ȁI");
			else if (target_ptr->chara_idx == CHARA_CHARGEMAN)
				msg_print("ア̂߂͂߂񂾁I");
#endif
			learn_trait(target_ptr, TRAIT_DISPEL);
			break;
		}

		case TRAIT_ROCKET:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s˂B", m_name);
#else
			if (blind) msg_format("%^s shoots something.", m_name);
#endif

#ifdef JP
			else msg_format("%^sPbg𔭎˂B", m_name);
#else
			else msg_format("%^s fires a rocket.", m_name);
#endif

			dam = ((caster_ptr->chp / 4) > 800 ? 800 : (caster_ptr->chp / 4));
			breath(y, x, caster_ptr, GF_ROCKET,
				dam, 2, FALSE, TRAIT_ROCKET, learnable);
			update_smart_learn(caster_ptr, DRS_SHARD);
			break;
		}

		case TRAIT_SHOOT:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sȉ𔭂B", m_name);
#else
			if (blind) msg_format("%^s makes a strange noise.", m_name);
#endif

#ifdef JP
			else msg_format("%^sB", m_name);
#else
			else msg_format("%^s fires an arrow.", m_name);
#endif

			//TODO Fix damage calc.
			dam = diceroll(caster_ptr->blow[0].d_dice, caster_ptr->blow[0].d_side);
			bolt(caster_ptr, target_ptr, GF_ARROW, dam, TRAIT_SHOOT, learnable);
			update_smart_learn(caster_ptr, DRS_REFLECT);
			break;
		}

		case TRAIT_BR_ACID:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s̃uXfB", m_name);
#else
			if (blind) msg_format("%^s breathes.", m_name);
#endif

#ifdef JP
			else msg_format("%^s_̃uXfB", m_name);
#else
			else msg_format("%^s breathes acid.", m_name);
#endif

			dam = ((caster_ptr->chp / 3) > 1600 ? 1600 : (caster_ptr->chp / 3));
			breath(y, x, caster_ptr, GF_ACID, dam, 0, TRUE, TRAIT_BR_ACID, learnable);
			update_smart_learn(caster_ptr, DRS_ACID);
			break;
		}

		case TRAIT_BR_ELEC:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s̃uXfB", m_name);
#else
			if (blind) msg_format("%^s breathes.", m_name);
#endif

#ifdef JP
			else msg_format("%^sȂ̃uXfB", m_name);
#else
			else msg_format("%^s breathes lightning.", m_name);
#endif

			dam = ((caster_ptr->chp / 3) > 1600 ? 1600 : (caster_ptr->chp / 3));
			breath(y, x, caster_ptr, GF_ELEC, dam,0, TRUE, TRAIT_BR_ELEC, learnable);
			update_smart_learn(caster_ptr, DRS_ELEC);
			break;
		}

		case TRAIT_BR_FIRE:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s̃uXfB", m_name);
#else
			if (blind) msg_format("%^s breathes.", m_name);
#endif

#ifdef JP
			else msg_format("%^sΉ̃uXfB", m_name);
#else
			else msg_format("%^s breathes fire.", m_name);
#endif

			dam = ((caster_ptr->chp / 3) > 1600 ? 1600 : (caster_ptr->chp / 3));
			breath(y, x, caster_ptr, GF_FIRE, dam,0, TRUE, TRAIT_BR_FIRE, learnable);
			update_smart_learn(caster_ptr, DRS_FIRE);
			break;
		}

		case TRAIT_BR_COLD:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s̃uXfB", m_name);
#else
			if (blind) msg_format("%^s breathes.", m_name);
#endif

#ifdef JP
			else msg_format("%^sC̃uXfB", m_name);
#else
			else msg_format("%^s breathes frost.", m_name);
#endif

			dam = ((caster_ptr->chp / 3) > 1600 ? 1600 : (caster_ptr->chp / 3));
			breath(y, x, caster_ptr, GF_COLD, dam,0, TRUE, TRAIT_BR_COLD, learnable);
			update_smart_learn(caster_ptr, DRS_COLD);
			break;
		}

		case TRAIT_BR_POIS:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s̃uXfB", m_name);
#else
			if (blind) msg_format("%^s breathes.", m_name);
#endif

#ifdef JP
			else msg_format("%^sKX̃uXfB", m_name);
#else
			else msg_format("%^s breathes gas.", m_name);
#endif

			dam = ((caster_ptr->chp / 3) > 800 ? 800 : (caster_ptr->chp / 3));
			breath(y, x, caster_ptr, GF_POIS, dam, 0, TRUE, TRAIT_BR_POIS, learnable);
			update_smart_learn(caster_ptr, DRS_POIS);
			break;
		}

		case TRAIT_BR_NETH:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s̃uXfB", m_name);
#else
			if (blind) msg_format("%^s breathes.", m_name);
#endif

#ifdef JP
			else msg_format("%^sñuXfB", m_name);
#else
			else msg_format("%^s breathes nether.", m_name);
#endif

			dam = ((caster_ptr->chp / 6) > 550 ? 550 : (caster_ptr->chp / 6));
			breath(y, x, caster_ptr, GF_NETHER, dam,0, TRUE, TRAIT_BR_NETH, learnable);
			update_smart_learn(caster_ptr, DRS_NETH);
			break;
		}

		case TRAIT_BR_LITE:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s̃uXfB", m_name);
#else
			if (blind) msg_format("%^s breathes.", m_name);
#endif

#ifdef JP
			else msg_format("%^sM̃uXfB", m_name);
#else
			else msg_format("%^s breathes light.", m_name);
#endif

			dam = ((caster_ptr->chp / 6) > 400 ? 400 : (caster_ptr->chp / 6));
			breath(y_br_lite, x_br_lite, caster_ptr, GF_LITE, dam,0, TRUE, TRAIT_BR_LITE, learnable);
			update_smart_learn(caster_ptr, DRS_LITE);
			break;
		}

		case TRAIT_BR_DARK:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s̃uXfB", m_name);
#else
			if (blind) msg_format("%^s breathes.", m_name);
#endif

#ifdef JP
			else msg_format("%^sÍ̃uXfB", m_name);
#else
			else msg_format("%^s breathes darkness.", m_name);
#endif

			dam = ((caster_ptr->chp / 6) > 400 ? 400 : (caster_ptr->chp / 6));
			breath(y, x, caster_ptr, GF_DARK, dam,0, TRUE, TRAIT_BR_DARK, learnable);
			update_smart_learn(caster_ptr, DRS_DARK);
			break;
		}

		case TRAIT_BR_CONF:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s̃uXfB", m_name);
#else
			if (blind) msg_format("%^s breathes.", m_name);
#endif

#ifdef JP
			else msg_format("%^s̃uXfB", m_name);
#else
			else msg_format("%^s breathes confusion.", m_name);
#endif

			dam = ((caster_ptr->chp / 6) > 450 ? 450 : (caster_ptr->chp / 6));
			breath(y, x, caster_ptr, GF_CONFUSION, dam,0, TRUE, TRAIT_BR_CONF, learnable);
			update_smart_learn(caster_ptr, DRS_CONF);
			break;
		}

		case TRAIT_BR_SOUN:
		{
			disturb(player_ptr, 1, 0);
			if (caster_ptr->species_idx == SPECIES_JAIAN)
#ifdef JP
				msg_format("u{HG``````v");
#else
				msg_format("'Booooeeeeee'");
#endif
#ifdef JP
				else if (blind) msg_format("%^s̃uXfB", m_name);
#else
				else if (blind) msg_format("%^s breathes.", m_name);
#endif

#ifdef JP
			else msg_format("%^s̃uXfB", m_name);
#else
			else msg_format("%^s breathes sound.", m_name);
#endif

			dam = ((caster_ptr->chp / 6) > 450 ? 450 : (caster_ptr->chp / 6));
			breath(y, x, caster_ptr, GF_SOUND, dam,0, TRUE, TRAIT_BR_SOUN, learnable);
			update_smart_learn(caster_ptr, DRS_SOUND);
			break;
		}

		case TRAIT_BR_CHAO:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s̃uXfB", m_name);
#else
			if (blind) msg_format("%^s breathes.", m_name);
#endif

#ifdef JP
			else msg_format("%^sJIX̃uXfB", m_name);
#else
			else msg_format("%^s breathes chaos.", m_name);
#endif

			dam = ((caster_ptr->chp / 6) > 600 ? 600 : (caster_ptr->chp / 6));
			breath(y, x, caster_ptr, GF_CHAOS, dam,0, TRUE, TRAIT_BR_CHAO, learnable);
			update_smart_learn(caster_ptr, DRS_CHAOS);
			break;
		}

		case TRAIT_BR_DISE:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s̃uXfB", m_name);
#else
			if (blind) msg_format("%^s breathes.", m_name);
#endif

#ifdef JP
			else msg_format("%^s򉻂̃uXfB", m_name);
#else
			else msg_format("%^s breathes disenchantment.", m_name);
#endif

			dam = ((caster_ptr->chp / 6) > 500 ? 500 : (caster_ptr->chp / 6));
			breath(y, x, caster_ptr, GF_DISENCHANT, dam,0, TRUE, TRAIT_BR_DISE, learnable);
			update_smart_learn(caster_ptr, DRS_DISEN);
			break;
		}

		case TRAIT_BR_NEXU:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s̃uXfB", m_name);
#else
			if (blind) msg_format("%^s breathes.", m_name);
#endif

#ifdef JP
			else msg_format("%^sʍ̃uXfB", m_name);
#else
			else msg_format("%^s breathes nexus.", m_name);
#endif

			dam = ((caster_ptr->chp / 3) > 250 ? 250 : (caster_ptr->chp / 3));
			breath(y, x, caster_ptr, GF_NEXUS, dam,0, TRUE, TRAIT_BR_NEXU, learnable);
			update_smart_learn(caster_ptr, DRS_NEXUS);
			break;
		}

		case TRAIT_BR_TIME:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s̃uXfB", m_name);
#else
			if (blind) msg_format("%^s breathes.", m_name);
#endif

#ifdef JP
			else msg_format("%^sԋt]̃uXfB", m_name);
#else
			else msg_format("%^s breathes time.", m_name);
#endif

			dam = ((caster_ptr->chp / 3) > 150 ? 150 : (caster_ptr->chp / 3));
			breath(y, x, caster_ptr, GF_TIME, dam,0, TRUE, TRAIT_BR_TIME, learnable);
			break;
		}

		case TRAIT_BR_INER:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s̃uXfB", m_name);
#else
			if (blind) msg_format("%^s breathes.", m_name);
#endif

#ifdef JP
			else msg_format("%^sx݂̃uXfB", m_name);
#else
			else msg_format("%^s breathes inertia.", m_name);
#endif

			dam = ((caster_ptr->chp / 6) > 200 ? 200 : (caster_ptr->chp / 6));
			breath(y, x, caster_ptr, GF_INERTIA, dam,0, TRUE, TRAIT_BR_INER, learnable);
			break;
		}

		case TRAIT_BR_GRAV:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s̃uXfB", m_name);
#else
			if (blind) msg_format("%^s breathes.", m_name);
#endif

#ifdef JP
			else msg_format("%^sd͂̃uXfB", m_name);
#else
			else msg_format("%^s breathes gravity.", m_name);
#endif

			dam = ((caster_ptr->chp / 3) > 200 ? 200 : (caster_ptr->chp / 3));
			breath(y, x, caster_ptr, GF_GRAVITY, dam,0, TRUE, TRAIT_BR_GRAV, learnable);
			break;
		}

		case TRAIT_BR_SHAR:
		{
			disturb(player_ptr, 1, 0);
			if (caster_ptr->species_idx == SPECIES_BOTEI)
#ifdef JP
				msg_format("u{rJb^[IIIv");
#else
				msg_format("'Boty-Build cutter!!!'");
#endif
#ifdef JP
			else if (blind) msg_format("%^s̃uXfB", m_name);
#else
			else if (blind) msg_format("%^s breathes.", m_name);
#endif

#ifdef JP
			else msg_format("%^sjЂ̃uXfB", m_name);
#else
			else msg_format("%^s breathes shards.", m_name);
#endif

			dam = ((caster_ptr->chp / 6) > 500 ? 500 : (caster_ptr->chp / 6));
			breath(y, x, caster_ptr, GF_SHARDS, dam,0, TRUE, TRAIT_BR_SHAR, learnable);
			update_smart_learn(caster_ptr, DRS_SHARD);
			break;
		}

		case TRAIT_BR_PLAS:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s̃uXfB", m_name);
#else
			if (blind) msg_format("%^s breathes.", m_name);
#endif

#ifdef JP
			else msg_format("%^svY}̃uXfB", m_name);
#else
			else msg_format("%^s breathes plasma.", m_name);
#endif

			dam = ((caster_ptr->chp / 6) > 150 ? 150 : (caster_ptr->chp / 6));
			breath(y, x, caster_ptr, GF_PLASMA, dam,0, TRUE, TRAIT_BR_PLAS, learnable);
			break;
		}

		case TRAIT_BR_WALL:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s̃uXfB", m_name);
#else
			if (blind) msg_format("%^s breathes.", m_name);
#endif

#ifdef JP
			else msg_format("%^stH[X̃uXfB", m_name);
#else
			else msg_format("%^s breathes force.", m_name);
#endif

			dam = ((caster_ptr->chp / 6) > 200 ? 200 : (caster_ptr->chp / 6));
			breath(y, x, caster_ptr, GF_FORCE, dam,0, TRUE, TRAIT_BR_WALL, learnable);
			break;
		}

		case TRAIT_BR_MANA:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s̃uXfB", m_name);
#else
			if (blind) msg_format("%^s breathes.", m_name);
#endif

#ifdef JP
			else msg_format("%^s͂̃uXfB", m_name);
#else
			else msg_format("%^s breathes mana.", m_name);
#endif
			dam = ((caster_ptr->chp / 3) > 250 ? 250 : (caster_ptr->chp / 3));
			breath(y, x, caster_ptr, GF_MANA, dam,0, TRUE, TRAIT_BR_MANA, learnable);
			break;
		}

		case TRAIT_BA_NUKE:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^s˔\B", m_name);
#else
			else msg_format("%^s casts a ball of radiation.", m_name);
#endif

			dam = (rlev + diceroll(10, 6)) * (has_trait(caster_ptr, TRAIT_POWERFUL) ? 2 : 1);
			breath(y, x, caster_ptr, GF_NUKE, dam, 2, FALSE, TRAIT_BA_NUKE, learnable);
			update_smart_learn(caster_ptr, DRS_POIS);
			break;
		}

		case TRAIT_BR_NUKE:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s̃uXfB", m_name);
#else
			if (blind) msg_format("%^s breathes.", m_name);
#endif

#ifdef JP
			else msg_format("%^sːp̃uXfB", m_name);
#else
			else msg_format("%^s breathes toxic waste.", m_name);
#endif

			dam = ((caster_ptr->chp / 3) > 800 ? 800 : (caster_ptr->chp / 3));
			breath(y, x, caster_ptr, GF_NUKE, dam,0, TRUE, TRAIT_BR_NUKE, learnable);
			update_smart_learn(caster_ptr, DRS_POIS);
			break;
		}

		case TRAIT_BA_CHAO:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s낵ɂԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles frighteningly.", m_name);
#endif

#ifdef JP
			else msg_format("%^sOXB", m_name);/*nuke me*/
#else
			else msg_format("%^s invokes a raw Logrus.", m_name);
#endif

			dam = (has_trait(caster_ptr, TRAIT_POWERFUL) ? (rlev * 3) : (rlev * 2))+ diceroll(10, 10);
			breath(y, x, caster_ptr, GF_CHAOS, dam, 4, FALSE, TRAIT_BA_CHAO, learnable);
			update_smart_learn(caster_ptr, DRS_CHAOS);
			break;
		}

		case TRAIT_BR_DISI:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s̃uXfB", m_name);
#else
			if (blind) msg_format("%^s breathes.", m_name);
#endif

#ifdef JP
			else msg_format("%^s̃uXfB", m_name);
#else
			else msg_format("%^s breathes disintegration.", m_name);
#endif

			dam = ((caster_ptr->chp / 6) > 150 ? 150 : (caster_ptr->chp / 6));
			breath(y, x, caster_ptr, GF_DISINTEGRATE, dam,0, TRUE, TRAIT_BR_DISI, learnable);
			break;
		}


		case TRAIT_BA_ACID:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
else msg_format("%^sAVbhE{[̎B", m_name);
#else
			else msg_format("%^s casts an acid ball.", m_name);
#endif

			dam = (randint1(rlev * 3) + 15) * (has_trait(caster_ptr, TRAIT_POWERFUL) ? 2 : 1);
			breath(y, x, caster_ptr, GF_ACID, dam, 2, FALSE, TRAIT_BA_ACID, learnable);
			update_smart_learn(caster_ptr, DRS_ACID);
			break;
		}

		case TRAIT_BA_ELEC:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
else msg_format("%^sT_[E{[̎B", m_name);
#else
			else msg_format("%^s casts a lightning ball.", m_name);
#endif

			dam = (randint1(rlev * 3 / 2) + 8) * (has_trait(caster_ptr, TRAIT_POWERFUL) ? 2 : 1);
			breath(y, x, caster_ptr, GF_ELEC, dam, 2, FALSE, TRAIT_BA_ELEC, learnable);
			update_smart_learn(caster_ptr, DRS_ELEC);
			break;
		}

		case TRAIT_BA_FIRE:
		{
			disturb(player_ptr, 1, 0);

			if (caster_ptr->species_idx == SPECIES_ROLENTO)
			{
#ifdef JP
				if (blind)
					msg_format("%s𓊂B", m_name);
				else 
					msg_format("%s͎֒e𓊂B", m_name);
#else
				if (blind)
					msg_format("%^s throws something.", m_name);
				else
					msg_format("%^s throws a hand grenade.", m_name);
#endif
			}
			else
			{
#ifdef JP
				if (blind) msg_format("%^sԂ₢B", m_name);
#else
				if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
				else msg_format("%^st@CAE{[̎B", m_name);
#else
				else msg_format("%^s casts a fire ball.", m_name);
#endif
			}

			dam = (randint1(rlev * 7 / 2) + 10) * (has_trait(caster_ptr, TRAIT_POWERFUL) ? 2 : 1);
			breath(y, x, caster_ptr, GF_FIRE, dam, 2, FALSE, TRAIT_BA_FIRE, learnable);
			update_smart_learn(caster_ptr, DRS_FIRE);
			break;
		}

		case TRAIT_BA_COLD:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^sACXE{[̎B", m_name);
#else
			else msg_format("%^s casts a frost ball.", m_name);
#endif

			dam = (randint1(rlev * 3 / 2) + 10) * (has_trait(caster_ptr, TRAIT_POWERFUL) ? 2 : 1);
			breath(y, x, caster_ptr, GF_COLD, dam, 2, FALSE, TRAIT_BA_COLD, learnable);
			update_smart_learn(caster_ptr, DRS_COLD);
			break;
		}

		case TRAIT_BA_POIS:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^sL_̎B", m_name);
#else
			else msg_format("%^s casts a stinking cloud.", m_name);
#endif

			dam = diceroll(12, 2) * (has_trait(caster_ptr, TRAIT_POWERFUL) ? 2 : 1);
			breath(y, x, caster_ptr, GF_POIS, dam, 2, FALSE, TRAIT_BA_POIS, learnable);
			update_smart_learn(caster_ptr, DRS_POIS);
			break;
		}

		case TRAIT_BA_NETH:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^sn̎B", m_name);
#else
			else msg_format("%^s casts a nether ball.", m_name);
#endif

			dam = 50 + diceroll(10, 10) + (rlev * (has_trait(caster_ptr, TRAIT_POWERFUL) ? 2 : 1));
			breath(y, x, caster_ptr, GF_NETHER, dam, 2, FALSE, TRAIT_BA_NETH, learnable);
			update_smart_learn(caster_ptr, DRS_NETH);
			break;
		}

		case TRAIT_BA_WATE:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^s悤ȐgUB", m_name);
#else
			else msg_format("%^s gestures fluidly.", m_name);
#endif

#ifdef JP
			msg_print("Ȃ͉QɈݍ܂ꂽB");
#else
			msg_print("You are engulfed in a whirlpool.");
#endif

			dam = (has_trait(caster_ptr, TRAIT_POWERFUL) ? randint1(rlev * 3) : randint1(rlev * 2)) + 50;
			breath(y, x, caster_ptr, GF_WATER, dam, 4, FALSE, TRAIT_BA_WATE, learnable);
			break;
		}

		case TRAIT_BA_MANA:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s͋Ԃ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles powerfully.", m_name);
#endif

#ifdef JP
			else msg_format("%^s̗͂̎OB", m_name);
#else
			else msg_format("%^s invokes a mana storm.", m_name);
#endif

			dam = (rlev * 4) + 50 + diceroll(10, 10);
			breath(y, x, caster_ptr, GF_MANA, dam, 4, FALSE, TRAIT_BA_MANA, learnable);
			break;
		}

		case TRAIT_BA_DARK:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s͋Ԃ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles powerfully.", m_name);
#endif

#ifdef JP
			else msg_format("%^sÍ̗̎OB", m_name);
#else
			else msg_format("%^s invokes a darkness storm.", m_name);
#endif

			dam = (rlev * 4) + 50 + diceroll(10, 10);
			breath(y, x, caster_ptr, GF_DARK, dam, 4, FALSE, TRAIT_BA_DARK, learnable);
			update_smart_learn(caster_ptr, DRS_DARK);
			break;
		}

		case TRAIT_DRAIN_MANA:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);

			dam = (randint1(rlev) / 2) + 1;
			breath(y, x, caster_ptr, GF_DRAIN_MANA, dam, 0, FALSE, TRAIT_DRAIN_MANA, learnable);
			update_smart_learn(caster_ptr, DRS_MANA);
			break;
		}

		case TRAIT_MIND_BLAST:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
			if (!seen)
			{
#ifdef JP
				msg_print("Ȃ̐_ɔOĂ悤B");
#else
				msg_print("You feel something focusing on your mind.");
#endif

			}
			else
			{
#ifdef JP
				msg_format("%^sȂ̓ƂɂłB", m_name);
#else
				msg_format("%^s gazes deep into your eyes.", m_name);
#endif

			}

			dam = diceroll(7, 7);
			breath(y, x, caster_ptr, GF_MIND_BLAST, dam, 0, FALSE, TRAIT_MIND_BLAST, learnable);
			break;
		}

		case TRAIT_BRAIN_SMASH:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
			if (!seen)
			{
#ifdef JP
				msg_print("Ȃ̐_ɔOĂ悤B");
#else
				msg_print("You feel something focusing on your mind.");
#endif

			}
			else
			{
#ifdef JP
				msg_format("%^sȂ̓ƌĂB", m_name);
#else
				msg_format("%^s looks deep into your eyes.", m_name);
#endif

			}

			dam = diceroll(12, 12);
			breath(y, x, caster_ptr, GF_BRAIN_SMASH, dam, 0, FALSE, TRAIT_BRAIN_SMASH, learnable);
			break;
		}

		case TRAIT_CAUSE_1:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^sȂwĎB", m_name);
#else
			else msg_format("%^s points at you and curses.", m_name);
#endif

			dam = diceroll(3, 8);
			breath(y, x, caster_ptr, GF_CAUSE_1, dam, 0, FALSE, TRAIT_CAUSE_1, learnable);
			break;
		}

		case TRAIT_CAUSE_2:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^sȂwċ낵ɎB", m_name);
#else
			else msg_format("%^s points at you and curses horribly.", m_name);
#endif

			dam = diceroll(8, 8);
			breath(y, x, caster_ptr, GF_CAUSE_2, dam, 0, FALSE, TRAIT_CAUSE_2, learnable);
			break;
		}

		case TRAIT_CAUSE_3:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s吺ŋ񂾁B", m_name);
#else
			if (blind) msg_format("%^s mumbles loudly.", m_name);
#endif

#ifdef JP
			else msg_format("%^sȂwċ낵ɎI", m_name);
#else
			else msg_format("%^s points at you, incanting terribly!", m_name);
#endif

			dam = diceroll(10, 15);
			breath(y, x, caster_ptr, GF_CAUSE_3, dam, 0, FALSE, TRAIT_CAUSE_3, learnable);
			break;
		}

		case TRAIT_CAUSE_4:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^suO͊ɎłvƋ񂾁B", m_name);
#else
			if (blind) msg_format("%^s screams the word 'DIE!'", m_name);
#endif

#ifdef JP
			else msg_format("%^sȂ̔E˂āuO͊ɎłvƋ񂾁B", m_name);
#else
			else msg_format("%^s points at you, screaming the word DIE!", m_name);
#endif

			dam = diceroll(15, 15);
			breath(y, x, caster_ptr, GF_CAUSE_4, dam, 0, FALSE, TRAIT_CAUSE_4, learnable);
			break;
		}

		case TRAIT_BO_ACID:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^sAVbhE{g̎B", m_name);
#else
			else msg_format("%^s casts a acid bolt.", m_name);
#endif

			dam = (diceroll(7, 8) + (rlev / 3)) * (has_trait(caster_ptr, TRAIT_POWERFUL) ? 2 : 1);
			bolt(caster_ptr, target_ptr, GF_ACID, dam, TRAIT_BO_ACID, learnable);
			update_smart_learn(caster_ptr, DRS_ACID);
			update_smart_learn(caster_ptr, DRS_REFLECT);
			break;
		}

		case TRAIT_BO_ELEC:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^sT_[E{g̎B", m_name);
#else
			else msg_format("%^s casts a lightning bolt.", m_name);
#endif

			dam = (diceroll(4, 8) + (rlev / 3)) * (has_trait(caster_ptr, TRAIT_POWERFUL) ? 2 : 1);
			bolt(caster_ptr, target_ptr, GF_ELEC, dam, TRAIT_BO_ELEC, learnable);
			update_smart_learn(caster_ptr, DRS_ELEC);
			update_smart_learn(caster_ptr, DRS_REFLECT);
			break;
		}

		case TRAIT_BO_FIRE:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^st@CAE{g̎B", m_name);
#else
			else msg_format("%^s casts a fire bolt.", m_name);
#endif

			dam = (diceroll(9, 8) + (rlev / 3)) * (has_trait(caster_ptr, TRAIT_POWERFUL) ? 2 : 1);
			bolt(caster_ptr, target_ptr, GF_FIRE, dam, TRAIT_BO_FIRE, learnable);
			update_smart_learn(caster_ptr, DRS_FIRE);
			update_smart_learn(caster_ptr, DRS_REFLECT);
			break;
		}

		case TRAIT_BO_COLD:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^sACXE{g̎B", m_name);
#else
			else msg_format("%^s casts a frost bolt.", m_name);
#endif

			dam = (diceroll(6, 8) + (rlev / 3)) * (has_trait(caster_ptr, TRAIT_POWERFUL) ? 2 : 1);
			bolt(caster_ptr, target_ptr, GF_COLD, dam, TRAIT_BO_COLD, learnable);
			update_smart_learn(caster_ptr, DRS_COLD);
			update_smart_learn(caster_ptr, DRS_REFLECT);
			break;
		}

		case TRAIT_BA_LITE:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^s͋Ԃ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles powerfully.", m_name);
#endif

#ifdef JP
			else msg_format("%^sX^[o[Xg̎OB", m_name);
#else
			else msg_format("%^s invokes a starburst.", m_name);
#endif

			dam = (rlev * 4) + 50 + diceroll(10, 10);
			breath(y, x, caster_ptr, GF_LITE, dam, 4, FALSE, TRAIT_BA_LITE, learnable);
			update_smart_learn(caster_ptr, DRS_LITE);
			break;
		}

		case TRAIT_BO_NETH:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^sn̖̎B", m_name);
#else
			else msg_format("%^s casts a nether bolt.", m_name);
#endif

			dam = 30 + diceroll(5, 5) + (rlev * 4) / (has_trait(caster_ptr, TRAIT_POWERFUL) ? 2 : 3);
			bolt(caster_ptr, target_ptr, GF_NETHER, dam, TRAIT_BO_NETH, learnable);
			update_smart_learn(caster_ptr, DRS_NETH);
			update_smart_learn(caster_ptr, DRS_REFLECT);
			break;
		}

		case TRAIT_BO_WATE:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^sEH[^[E{g̎B", m_name);
#else
			else msg_format("%^s casts a water bolt.", m_name);
#endif

			dam = diceroll(10, 10) + (rlev * 3 / (has_trait(caster_ptr, TRAIT_POWERFUL) ? 2 : 3));
			bolt(caster_ptr, target_ptr, GF_WATER, dam, TRAIT_BO_WATE, learnable);
			update_smart_learn(caster_ptr, DRS_REFLECT);
			break;
		}

		case TRAIT_BO_MANA:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^s̖͂̎B", m_name);
#else
			else msg_format("%^s casts a mana bolt.", m_name);
#endif

			dam = randint1(rlev * 7 / 2) + 50;
			bolt(caster_ptr, target_ptr, GF_MANA, dam, TRAIT_BO_MANA, learnable);
			update_smart_learn(caster_ptr, DRS_REFLECT);
			break;
		}

		case TRAIT_BO_PLAS:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^svY}E{g̎B", m_name);
#else
			else msg_format("%^s casts a plasma bolt.", m_name);
#endif

			dam = 10 + diceroll(8, 7) + (rlev * 3 / (has_trait(caster_ptr, TRAIT_POWERFUL) ? 2 : 3));
			bolt(caster_ptr, target_ptr, GF_PLASMA, dam, TRAIT_BO_PLAS, learnable);
			update_smart_learn(caster_ptr, DRS_REFLECT);
			break;
		}

		case TRAIT_BO_ICEE:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^sɊ̖̎B", m_name);
#else
			else msg_format("%^s casts an ice bolt.", m_name);
#endif

			dam = diceroll(6, 6) + (rlev * 3 / (has_trait(caster_ptr, TRAIT_POWERFUL) ? 2 : 3));
			bolt(caster_ptr, target_ptr, GF_ICE, dam, TRAIT_BO_ICEE, learnable);
			update_smart_learn(caster_ptr, DRS_COLD);
			update_smart_learn(caster_ptr, DRS_REFLECT);
			break;
		}

		case TRAIT_MISSILE:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^s}WbNE~TC̎B", m_name);
#else
			else msg_format("%^s casts a magic missile.", m_name);
#endif

			dam = diceroll(2, 6) + (rlev / 3);
			bolt(caster_ptr, target_ptr, GF_MISSILE, dam, TRAIT_MISSILE, learnable);
			update_smart_learn(caster_ptr, DRS_REFLECT);
			break;
		}

		case TRAIT_SCARE:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₭ƁA낵ȉB", m_name);
#else
			if (blind) msg_format("%^s mumbles, and you hear scary noises.", m_name);
#endif

#ifdef JP
			else msg_format("%^s낵ȌooB", m_name);
#else
			else msg_format("%^s casts a fearful illusion.", m_name);
#endif

			if (has_trait(target_ptr, TRAIT_FEARLESS))
			{
#ifdef JP
				msg_print("|ɐNȂB");
#else
				msg_print("You refuse to be frightened.");
#endif

			}
			else if (randint0(100 + rlev/2) < target_ptr->skill_rob)
			{
#ifdef JP
				msg_print("|ɐNȂB");
#else
				msg_print("You refuse to be frightened.");
#endif

			}
			else
			{
				(void)set_afraid(target_ptr, target_ptr->afraid + randint0(4) + 4);
			}
			learn_trait(target_ptr, TRAIT_SCARE);
			update_smart_learn(caster_ptr, DRS_FEAR);
			break;
		}

		case TRAIT_BLIND:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^sĂȂ̖ڂ܂I", m_name);
#else
			else msg_format("%^s casts a spell, burning your eyes!", m_name);
#endif

			if (has_trait(target_ptr, TRAIT_NO_BLIND))
			{
#ifdef JP
				msg_print("ʂȂI");
#else
				msg_print("You are unaffected!");
#endif

			}
			else if (randint0(100 + rlev/2) < target_ptr->skill_rob)
			{
#ifdef JP
				msg_print("͂𒵂˕ԂI");
#else
				msg_print("You resist the effects!");
#endif

			}
			else
			{
				(void)set_blind(target_ptr, 12 + randint0(4));
			}
			learn_trait(target_ptr, TRAIT_BLIND);
			update_smart_learn(caster_ptr, DRS_BLIND);
			break;
		}

		case TRAIT_CONF:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₭ƁAY܂B", m_name);
#else
			if (blind) msg_format("%^s mumbles, and you hear puzzling noises.", m_name);
#endif

#ifdef JP
			else msg_format("%^sUfIȌooB", m_name);
#else
			else msg_format("%^s creates a mesmerising illusion.", m_name);
#endif

			if (has_trait(target_ptr, TRAIT_NO_CONF))
			{
#ifdef JP
				msg_print("oɂ͂܂ȂB");
#else
				msg_print("You disbelieve the feeble spell.");
#endif

			}
			else if (randint0(100 + rlev/2) < target_ptr->skill_rob)
			{
#ifdef JP
				msg_print("oɂ͂܂ȂB");
#else
				msg_print("You disbelieve the feeble spell.");
#endif

			}
			else
			{
				(void)set_confused(target_ptr, target_ptr->confused + randint0(4) + 4);
			}
			learn_trait(target_ptr, TRAIT_CONF);
			update_smart_learn(caster_ptr, DRS_CONF);
			break;
		}

		case TRAIT_SLOW:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			msg_format("%^sȂ̋ؗ͂z낤ƂI", m_name);
#else
			msg_format("%^s drains power from your muscles!", m_name);
#endif

			if (has_trait(target_ptr, TRAIT_FREE_ACTION))
			{
#ifdef JP
				msg_print("ʂȂI");
#else
				msg_print("You are unaffected!");
#endif

			}
			else if (randint0(100 + rlev/2) < target_ptr->skill_rob)
			{
#ifdef JP
				msg_print("͂𒵂˕ԂI");
#else
				msg_print("You resist the effects!");
#endif

			}
			else
			{
				(void)set_slow(target_ptr, target_ptr->slow + randint0(4) + 4, FALSE);
			}
			learn_trait(target_ptr, TRAIT_SLOW);
			update_smart_learn(caster_ptr, DRS_FREE);
			break;
		}

		case TRAIT_HOLD:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^sȂ̖ڂƌ߂I", m_name);
#else
			else msg_format("%^s stares deep into your eyes!", m_name);
#endif

			if (has_trait(target_ptr, TRAIT_FREE_ACTION))
			{
#ifdef JP
				msg_print("ʂȂI");
#else
				msg_print("You are unaffected!");
#endif

			}
			else if (randint0(100 + rlev/2) < target_ptr->skill_rob)
			{
#ifdef JP
				msg_format("͂𒵂˕ԂI");
#else
				msg_format("You resist the effects!");
#endif

			}
			else
			{
				(void)set_paralyzed(target_ptr, target_ptr->paralyzed + randint0(4) + 4);
			}
			learn_trait(target_ptr, TRAIT_HOLD);
			update_smart_learn(caster_ptr, DRS_FREE);
			break;
		}

		case TRAIT_HASTE:
		{
			disturb(player_ptr, 1, 0);
			if (blind)
			{
#ifdef JP
				msg_format("%^sԂ₢B", m_name);
#else
				msg_format("%^s mumbles.", m_name);
#endif

			}
			else
			{
#ifdef JP
				msg_format("%^s̑̂ɔO𑗂B", m_name);
#else
				msg_format("%^s concentrates on %s body.", m_name, m_poss);
#endif

			}

			/* Allow quick speed increases to base+10 */
			if (set_fast(caster_ptr, caster_ptr->fast + 100, FALSE))
			{
#ifdef JP
				msg_format("%^s̓ȂB", m_name);
#else
				msg_format("%^s starts moving faster.", m_name);
#endif
			}
			break;
		}

		case TRAIT_HAND_DOOM:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			msg_format("%^s<jł̎>I", m_name);
#else
			msg_format("%^s invokes the Hand of Doom!", m_name);
#endif
			dam = (((s32b) ((40 + randint1(20)) * (target_ptr->chp))) / 100);
			breath(y, x, caster_ptr, GF_HAND_DOOM, dam, 0, FALSE, TRAIT_HAND_DOOM, learnable);
			break;
		}

		case TRAIT_HEAL:
		{
			disturb(player_ptr, 1, 0);

			/* Message */
			if (blind)
			{
#ifdef JP
				msg_format("%^sԂ₢B", m_name);
#else
				msg_format("%^s mumbles.", m_name);
#endif

			}
			else
			{
#ifdef JP
				msg_format("%^s̏ɏWB", m_name);
#else
				msg_format("%^s concentrates on %s wounds.", m_name, m_poss);
#endif

			}

			/* Heal some */
			caster_ptr->chp += (rlev * 6);

			/* Fully healed */
			if (caster_ptr->chp >= caster_ptr->mhp)
			{
				/* Fully healed */
				caster_ptr->chp = caster_ptr->mhp;

				/* Message */
				if (seen)
				{
#ifdef JP
					msg_format("%^s͊SɎI", m_name);
#else
					msg_format("%^s looks completely healed!", m_name);
#endif

				}
				else
				{
#ifdef JP
					msg_format("%^s͊SɎ悤I", m_name);
#else
					msg_format("%^s sounds completely healed!", m_name);
#endif

				}
			}

			/* Partially healed */
			else
			{
				/* Message */
				if (seen)
				{
#ifdef JP
					msg_format("%^s̗͑͂񕜂悤B", m_name);
#else
					msg_format("%^s looks healthier.", m_name);
#endif

				}
				else
				{
#ifdef JP
					msg_format("%^s̗͑͂񕜂悤B", m_name);
#else
					msg_format("%^s sounds healthier.", m_name);
#endif

				}
			}

			/* Redraw (later) if needed */
			if (&creature_list[health_who] == caster_ptr) play_redraw |= (PR_HEALTH);
			if (&creature_list[target_ptr->riding] == caster_ptr) play_redraw |= (PR_UHEALTH);

			/* Cancel fear */
			if (caster_ptr->afraid)
			{
				/* Cancel fear */
				(void)set_afraid(caster_ptr, 0);

				/* Message */
#ifdef JP
				msg_format("%^s͗EC߂B", m_name);
#else
				msg_format("%^s recovers %s courage.", m_name, m_poss);
#endif
			}
			break;
		}

		case TRAIT_INVULNER:
		{
			disturb(player_ptr, 1, 0);

			/* Message */
			if (!seen)
			{
#ifdef JP
				msg_format("%^s͋Ԃ₢B", m_name);
#else
				msg_format("%^s mumbles powerfully.", m_name);
#endif

			}
			else
			{
#ifdef JP
				msg_format("%s͖̋̎B", m_name);
#else
				msg_format("%^s casts a Globe of Invulnerability.", m_name);
#endif

			}

			if (!caster_ptr->invuln) (void)set_invuln(caster_ptr, randint1(4) + 4, FALSE);
			break;
		}

		case TRAIT_BLINK:
		{
			disturb(player_ptr, 1, 0);
			if (teleport_barrier(target_ptr, caster_ptr))
			{
#ifdef JP
				msg_format("@̃oA%^s̃e|[gזB", m_name);
#else
				msg_format("Magic barrier obstructs teleporting of %^s.", m_name);
#endif
			}
			else
			{
#ifdef JP
				msg_format("%^suɏB", m_name);
#else
				msg_format("%^s blinks away.", m_name);
#endif
				teleport_away(caster_ptr, 10, 0L);
				update |= (PU_MONSTERS);
			}
			break;
		}

		case TRAIT_TPORT:
		{
			disturb(player_ptr, 1, 0);
			if (teleport_barrier(target_ptr, caster_ptr))
			{
#ifdef JP
				msg_format("@̃oA%^s̃e|[gזB", m_name);
#else
				msg_format("Magic barrier obstructs teleporting of %^s.", m_name);
#endif
			}
			else
			{
#ifdef JP
				msg_format("%^se|[gB", m_name);
#else
				msg_format("%^s teleports away.", m_name);
#endif
				teleport_away_followable(caster_ptr);
			}
			break;
		}

		case TRAIT_WORLD:
		{
			int who = 0;
			disturb(player_ptr, 1, 0);
			if(caster_ptr->species_idx == SPECIES_DIO) who = 1;
			else if(caster_ptr->species_idx == SPECIES_WONG) who = 3;
			dam = who;
			if (!process_the_world(player_ptr, randint1(2)+2, who, TRUE)) return (FALSE);
			break;
		}

		case TRAIT_SPECIAL:
		{
			int k;

			disturb(player_ptr, 1, 0);
			switch (caster_ptr->species_idx)
			{
			case SPECIES_OHMU:
				/* Moved to process_nonplayer(), like multiplication */
				return FALSE;

			case SPECIES_BANORLUPART:
				{
					int dummy_hp = (caster_ptr->chp + 1) / 2;
					int dummy_mhp = caster_ptr->mhp/2;
					int dummy_y = caster_ptr->fy;
					int dummy_x = caster_ptr->fx;

					if (fight_arena_mode || gamble_arena_mode || !summon_possible(caster_ptr, caster_ptr->fy, caster_ptr->fx)) return FALSE;
					delete_species_idx(&creature_list[floor_ptr->cave[caster_ptr->fy][caster_ptr->fx].creature_idx]);
					summon_named_creature(0, floor_ptr, dummy_y, dummy_x, SPECIES_BANOR, mode);
					creature_list[hack_m_idx_ii].chp = dummy_hp;
					creature_list[hack_m_idx_ii].mhp = dummy_mhp;
					summon_named_creature(0, floor_ptr, dummy_y, dummy_x, SPECIES_LUPART, mode);
					creature_list[hack_m_idx_ii].chp = dummy_hp;
					creature_list[hack_m_idx_ii].mhp = dummy_mhp;

#ifdef JP
					msg_print("wo[m[Ep[gx􂵂I");
#else
					msg_print("Banor=Rupart splits in two person!");
#endif

					break;
				}

			case SPECIES_BANOR:
			case SPECIES_LUPART:
				{
					int dummy_hp = 0;
					int dummy_mhp = 0;
					int dummy_y = caster_ptr->fy;
					int dummy_x = caster_ptr->fx;

					if (!species_info[SPECIES_BANOR].cur_num || !species_info[SPECIES_LUPART].cur_num) return (FALSE);
					for (k = 1; k < creature_max; k++)
					{
						if (creature_list[k].species_idx == SPECIES_BANOR || creature_list[k].species_idx == SPECIES_LUPART)
						{
							dummy_hp += creature_list[k].chp;
							dummy_mhp += creature_list[k].mhp;
							if (creature_list[k].species_idx != caster_ptr->species_idx)
							{
								dummy_y = creature_list[k].fy;
								dummy_x = creature_list[k].fx;
							}
							delete_species_idx(&creature_list[k]);
						}
					}
					summon_named_creature(0, floor_ptr, dummy_y, dummy_x, SPECIES_BANORLUPART, mode);
					creature_list[hack_m_idx_ii].chp = dummy_hp;
					creature_list[hack_m_idx_ii].mhp = dummy_mhp;

#ifdef JP
					msg_print("wo[m[xƁwp[gx̂I");
#else
					msg_print("Banor and Rupart combine into one!");
#endif

					break;
				}

			case SPECIES_ROLENTO:
#ifdef JP
				if (blind) msg_format("%^sʂɓB", m_name);
				else msg_format("%^s͎֒e΂܂B", m_name);
#else
				if (blind) msg_format("%^s spreads something.", m_name);
				else msg_format("%^s throws some hand grenades.", m_name);
#endif

				{
					int num = 1 + randint1(3);

					for (k = 0; k < num; k++)
					{
						count += summon_named_creature(caster_ptr, floor_ptr, y, x, SPECIES_SHURYUUDAN, mode);
					}
				}
#ifdef JP
				if (blind && count) msg_print("̂̂ԋ߂ɂ΂܂鉹B");
#else
				if (blind && count) msg_print("You hear many things are scattered nearby.");
#endif
				break;

			default:
				if (r_ptr->d_char == 'B')
				{
					disturb(player_ptr, 1, 0);
					if (one_in_(3) || !direct)
					{
#ifdef JP
						msg_format("%^s͓ˑRE!", m_name);
#else
						msg_format("%^s suddenly go out of your sight!", m_name);
#endif
						teleport_away(caster_ptr, 10, TELEPORT_NONMAGICAL);
						update |= (PU_MONSTERS);
					}
					else
					{
						int get_damage = 0;

#ifdef JP
						msg_format("%^sȂ͂ŋ󒆂瓊ƂB", m_name);
#else
						msg_format("%^s holds you, and drops from the sky.", m_name);
#endif
						dam = diceroll(4, 8);
						teleport_creature_to(target_ptr, caster_ptr->fy, caster_ptr->fx, TELEPORT_NONMAGICAL | TELEPORT_PASSIVE);

						sound(SOUND_FALL);

						if (has_trait(target_ptr, TRAIT_CAN_FLY))
						{
#ifdef JP
							msg_print("Ȃ͐ÂɒnB");
#else
							msg_print("You float gently down to the ground.");
#endif
						}
						else
						{
#ifdef JP
							msg_print("Ȃ͒nʂɒ@ꂽB");
#else
							msg_print("You crashed into the ground.");
#endif
							dam += diceroll(6, 8);
						}

						/* Mega hack -- this special action deals damage to the player. Therefore the code of "eyeeye" is necessary.
						   -- henkma
						 */
						get_damage = take_hit(NULL, target_ptr, DAMAGE_NOESCAPE, dam, m_name, NULL, -1);
						if (target_ptr->tim_eyeeye && get_damage > 0 && !gameover)
						{
#ifdef JP
							msg_format("U%sgI", m_name);
#else
							char m_name_self[80];

							/* hisself */
							creature_desc(m_name_self, caster_ptr, MD_PRON_VISIBLE | MD_POSSESSIVE | MD_OBJECTIVE);

							msg_format("The attack of %s has wounded %s!", m_name, m_name_self);
#endif
							project(caster_ptr, 0, caster_ptr->fy, caster_ptr->fx, get_damage, GF_MISSILE, PROJECT_KILL, -1);
							set_tim_eyeeye(target_ptr, target_ptr->tim_eyeeye-5, TRUE);
						}

						if (target_ptr->riding) melee_attack(caster_ptr, target_ptr->fy, target_ptr->fx, 0);
					}
					break;
				}

				/* Something is wrong */
				else return FALSE;
			}
			break;
		}

		case TRAIT_TELE_TO:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			msg_format("%^sȂ߂B", m_name);
#else
			msg_format("%^s commands you to return.", m_name);
#endif

			teleport_creature_to(target_ptr, caster_ptr->fy, caster_ptr->fx, TELEPORT_PASSIVE);
			learn_trait(target_ptr, TRAIT_TELE_TO);
			break;
		}

		case TRAIT_TELE_AWAY:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			msg_format("%^sɃe|[gꂽB", m_name);

			if ((target_ptr->chara_idx == CHARA_COMBAT) || (get_equipped_slot_ptr(target_ptr, INVEN_SLOT_BOW, 1)->name1 == ART_CRIMSON))
				msg_print("`");

			if (target_ptr->chara_idx == CHARA_CHARGEMAN)
				msg_print("ȂĎ񂾁I");
#else
			msg_format("%^s teleports you away.", m_name);
#endif

			learn_trait(target_ptr, TRAIT_TELE_AWAY);
			teleport_player_away(caster_ptr, 100);
			break;
		}

		case TRAIT_TELE_LEVEL:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sȌtԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles strangely.", m_name);
#endif

#ifdef JP
			else msg_format("%^sȂ̑wB", m_name);
#else
			else msg_format("%^s gestures at your feet.", m_name);
#endif

			if (target_ptr->resist_nexus)
			{
#ifdef JP
				msg_print("ʂȂI");
#else
				msg_print("You are unaffected!");
#endif

			}
			else if (randint0(100 + rlev/2) < target_ptr->skill_rob)
			{
#ifdef JP
				msg_print("͂𒵂˕ԂI");
#else
				msg_print("You resist the effects!");
#endif

			}
			else
			{
				teleport_level(target_ptr, 0);
			}
			learn_trait(target_ptr, TRAIT_TELE_LEVEL);
			update_smart_learn(caster_ptr, DRS_NEXUS);
			break;
		}

		case TRAIT_PSY_SPEAR:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^šB", m_name);
#else
			else msg_format("%^s throw a Psycho-Spear.", m_name);
#endif

			dam = has_trait(caster_ptr, TRAIT_POWERFUL) ? (randint1(rlev * 2) + 150) : (randint1(rlev * 3 / 2) + 100);
			beam(caster_ptr, target_ptr, GF_PSY_SPEAR, dam, TRAIT_PSY_SPEAR, learnable);
			break;
		}

		case TRAIT_DARKNESS:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else if (can_use_lite_area) msg_format("%^sӂ𖾂邭Ƃ炵B", m_name);
			else msg_format("%^sÈł̒ŎUB", m_name);
#else
			else if (can_use_lite_area) msg_format("%^s cast a spell to light up.", m_name);
			else msg_format("%^s gestures in shadow.", m_name);
#endif

			if (can_use_lite_area) (void)lite_area(caster_ptr, 0, 3);
			else
			{
				learn_trait(target_ptr, TRAIT_DARKNESS);
				(void)unlite_area(caster_ptr, 0, 3);
			}
			break;
		}

		case TRAIT_TRAPS:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢Ď׈ɔ΂񂾁B", m_name);
#else
			if (blind) msg_format("%^s mumbles, and then cackles evilly.", m_name);
#endif

#ifdef JP
			else msg_format("%^sĎ׈ɔ΂񂾁B", m_name);
#else
			else msg_format("%^s casts a spell and cackles evilly.", m_name);
#endif

			learn_trait(target_ptr, TRAIT_TRAPS);
			(void)trap_creation(caster_ptr, y, x);
			break;
		}

		case TRAIT_FORGET:
		{
			if (!direct) return (FALSE);
			disturb(player_ptr, 1, 0);
#ifdef JP
			msg_format("%^sȂ̋L悤ƂĂB", m_name);
#else
			msg_format("%^s tries to blank your mind.", m_name);
#endif


			if (randint0(100 + rlev/2) < target_ptr->skill_rob)
			{
#ifdef JP
				msg_print("͂𒵂˕ԂI");
#else
				msg_print("You resist the effects!");
#endif

			}
			else if (lose_all_info(target_ptr))
			{
#ifdef JP
				msg_print("LĂ܂B");
#else
				msg_print("Your memories fade away.");
#endif

			}
			learn_trait(target_ptr, TRAIT_FORGET);
			break;
		}

	case TRAIT_ANIM_DEAD:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^sҕ̎B", m_name);
#else
			else msg_format("%^s casts a spell to revive corpses.", m_name);
#endif
			animate_dead(caster_ptr, caster_ptr->fy, caster_ptr->fx);
			break;
		}

		case TRAIT_S_KIN:
		{
			disturb(player_ptr, 1, 0);
			if (caster_ptr->species_idx == SPECIES_SERPENT || caster_ptr->species_idx == SPECIES_ZOMBI_SERPENT)
			{
#ifdef JP
				if (blind)
					msg_format("%^sԂ₢B", m_name);
				else
					msg_format("%^s_W̎B", m_name);
#else
				if (blind)
					msg_format("%^s mumbles.", m_name);
				else
					msg_format("%^s magically summons guardians of dungeons.", m_name);
#endif
			}
			else
			{
#ifdef JP
				if (blind)
					msg_format("%^sԂ₢B", m_name);
				else
					msg_format("%^s͖@%sB",
					m_name,
					(is_unique_creature(caster_ptr) ?
					"艺" : ""));
#else
				if (blind)
					msg_format("%^s mumbles.", m_name);
				else
					msg_format("%^s magically summons %s %s.",
					m_name, m_poss,
					is_unique_creature(caster_ptr) ?
					"minions" : "kin"));
#endif
			}

			switch (caster_ptr->species_idx)
			{
			case SPECIES_MENELDOR:
			case SPECIES_GWAIHIR:
			case SPECIES_THORONDOR:
				{
					int num = 4 + randint1(3);
					for (k = 0; k < num; k++)
					{
						count += summon_specific(caster_ptr, y, x, rlev, SUMMON_EAGLES, (PM_ALLOW_GROUP | PM_ALLOW_UNIQUE));
					}
				}
				break;

			case SPECIES_RICHARD_STOLENMAN:
				{
					int num = 2 + randint1(3);
					for (k = 0; k < num; k++)
					{
						count += summon_named_creature(caster_ptr, floor_ptr, y, x, SPECIES_IE, mode);
					}
				}
				break;

			case SPECIES_SERPENT:
			case SPECIES_ZOMBI_SERPENT:
				{
					int num = 2 + randint1(3);

					if (species_info[SPECIES_JORMUNGAND].cur_num < species_info[SPECIES_JORMUNGAND].max_num && one_in_(6))
					{
#ifdef JP
						msg_print("nʂ琅oI");
#else
						msg_print("Water blew off from the ground!");
#endif
						fire_ball_hide(caster_ptr, GF_WATER_FLOW, 0, 3, 8);
					}

					for (k = 0; k < num; k++)
					{
						count += summon_specific(caster_ptr, y, x, rlev, SUMMON_GUARDIANS, (PM_ALLOW_GROUP | PM_ALLOW_UNIQUE));
					}
				}
				break;

			case SPECIES_CALDARM:
				{
					int num = randint1(3);
					for (k = 0; k < num; k++)
					{
						count += summon_named_creature(caster_ptr, floor_ptr, y, x, SPECIES_LOCKE_CLONE, mode);
					}
				}
				break;

			case SPECIES_LOUSY:
				{
					int num = 2 + randint1(3);
					for (k = 0; k < num; k++)
					{
						count += summon_specific(caster_ptr, y, x, rlev, SUMMON_LOUSE, PM_ALLOW_GROUP);
					}
				}
				break;

			default:
				//summon_kin_type = r_ptr->d_char; /* Big hack */

				for (k = 0; k < 4; k++)
				{
					count += summon_specific(caster_ptr, y, x, rlev, SUMMON_KIN, PM_ALLOW_GROUP);
				}
				break;
			}
#ifdef JP
			if (blind && count) msg_print("̂̂ԋ߂ɌꂽB");
#else
			if (blind && count) msg_print("You hear many things appear nearby.");
#endif

			break;
		}

		case TRAIT_S_CYBER:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^sTCo[f[I", m_name);
#else
			else msg_format("%^s magically summons Cyberdemons!", m_name);
#endif

#ifdef JP
			if (blind && count) msg_print("dȑ߂ŕB");
#else
			if (blind && count) msg_print("You hear heavy steps nearby.");
#endif

			summon_cyber(caster_ptr, y, x);
			break;
		}

		case TRAIT_S_MONSTER:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^s@ŒԂI", m_name);
#else
			else msg_format("%^s magically summons help!", m_name);
#endif

			for (k = 0; k < 1; k++)
			{
				count += summon_specific(caster_ptr, y, x, rlev, 0, (PM_ALLOW_GROUP | PM_ALLOW_UNIQUE));
			}
#ifdef JP
			if (blind && count) msg_print("ԋ߂ɌꂽB");
#else
			if (blind && count) msg_print("You hear something appear nearby.");
#endif

			break;
		}

		case TRAIT_S_MONSTERS:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^s@ŃN[`[I", m_name);
#else
			else msg_format("%^s magically summons creatures!", m_name);
#endif

			for (k = 0; k < s_num_6; k++)
			{
				count += summon_specific(caster_ptr, y, x, rlev, 0, (PM_ALLOW_GROUP | PM_ALLOW_UNIQUE));
			}
#ifdef JP
			if (blind && count) msg_print("̂̂ԋ߂ɌꂽB");
#else
			if (blind && count) msg_print("You hear many things appear nearby.");
#endif

			break;
		}

		case TRAIT_S_ANT:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^s@ŃAB", m_name);
#else
			else msg_format("%^s magically summons ants.", m_name);
#endif

			for (k = 0; k < s_num_6; k++)
			{
				count += summon_specific(caster_ptr, y, x, rlev, SUMMON_ANT, PM_ALLOW_GROUP);
			}
#ifdef JP
			if (blind && count) msg_print("̂̂ԋ߂ɌꂽB");
#else
			if (blind && count) msg_print("You hear many things appear nearby.");
#endif

			break;
		}

		case TRAIT_S_SPIDER:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^s@ŃNB", m_name);
#else
			else msg_format("%^s magically summons spiders.", m_name);
#endif

			for (k = 0; k < s_num_6; k++)
			{
				count += summon_specific(caster_ptr, y, x, rlev, SUMMON_SPIDER, PM_ALLOW_GROUP);
			}
#ifdef JP
			if (blind && count) msg_print("̂̂ԋ߂ɌꂽB");
#else
			if (blind && count) msg_print("You hear many things appear nearby.");
#endif

			break;
		}

		case TRAIT_S_HOUND:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^s@ŃnEhB", m_name);
#else
			else msg_format("%^s magically summons hounds.", m_name);
#endif

			for (k = 0; k < s_num_4; k++)
			{
				count += summon_specific(caster_ptr, y, x, rlev, SUMMON_HOUND, PM_ALLOW_GROUP);
			}
#ifdef JP
			if (blind && count) msg_print("̂̂ԋ߂ɌꂽB");
#else
			if (blind && count) msg_print("You hear many things appear nearby.");
#endif

			break;
		}

		case TRAIT_S_HYDRA:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^s@ŃqhB", m_name);
#else
			else msg_format("%^s magically summons hydras.", m_name);
#endif

			for (k = 0; k < s_num_4; k++)
			{
				count += summon_specific(caster_ptr, y, x, rlev, SUMMON_HYDRA, PM_ALLOW_GROUP);
			}
#ifdef JP
			if (blind && count) msg_print("̂̂ԋ߂ɌꂽB");
#else
			if (blind && count) msg_print("You hear many things appear nearby.");
#endif

			break;
		}

		case TRAIT_S_ANGEL:
		{
			int num = 1;

			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^s@œVgI", m_name);
#else
			else msg_format("%^s magically summons an angel!", m_name);
#endif

			if (is_unique_species(r_ptr))
			{
				num += r_ptr->level/40;
			}

			for (k = 0; k < num; k++)
			{
				count += summon_specific(caster_ptr, y, x, rlev, SUMMON_ANGEL, PM_ALLOW_GROUP);
			}

			if (count < 2)
			{
#ifdef JP
				if (blind && count) msg_print("ԋ߂ɌꂽB");
#else
				if (blind && count) msg_print("You hear something appear nearby.");
#endif
			}
			else
			{
#ifdef JP
				if (blind) msg_print("̂̂ԋ߂ɌꂽB");
#else
				if (blind) msg_print("You hear many things appear nearby.");
#endif
			}

			break;
		}

		case TRAIT_S_DEMON:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^s͖@ōׂ̋{삩爫I", m_name);
#else
			else msg_format("%^s magically summons a demon from the Courts of Chaos!", m_name);
#endif

			for (k = 0; k < 1; k++)
			{
				count += summon_specific(caster_ptr, y, x, rlev, SUMMON_DEMON, PM_ALLOW_GROUP);
			}
#ifdef JP
			if (blind && count) msg_print("ԋ߂ɌꂽB");
#else
			if (blind && count) msg_print("You hear something appear nearby.");
#endif

			break;
		}

		case TRAIT_S_UNDEAD:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^s@ŃAfbh̋GI", m_name);
#else
			else msg_format("%^s magically summons an undead adversary!", m_name);
#endif

			for (k = 0; k < 1; k++)
			{
				count += summon_specific(caster_ptr, y, x, rlev, SUMMON_UNDEAD, PM_ALLOW_GROUP);
			}
#ifdef JP
			if (blind && count) msg_print("ԋ߂ɌꂽB");
#else
			if (blind && count) msg_print("You hear something appear nearby.");
#endif

			break;
		}

		case TRAIT_S_DRAGON:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^s@ŃhSI", m_name);
#else
			else msg_format("%^s magically summons a dragon!", m_name);
#endif

			for (k = 0; k < 1; k++)
			{
				count += summon_specific(caster_ptr, y, x, rlev, SUMMON_DRAGON, PM_ALLOW_GROUP);
			}
#ifdef JP
			if (blind && count) msg_print("ԋ߂ɌꂽB");
#else
			if (blind && count) msg_print("You hear something appear nearby.");
#endif

			break;
		}

		case TRAIT_S_HI_UNDEAD:
		{
			disturb(player_ptr, 1, 0);

			if (((caster_ptr->species_idx == SPECIES_MORGOTH) || (caster_ptr->species_idx == SPECIES_SAURON) || (caster_ptr->species_idx == SPECIES_ANGMAR)) && ((species_info[SPECIES_NAZGUL].cur_num+2) < species_info[SPECIES_NAZGUL].max_num))
			{
				int cy = y;
				int cx = x;

#ifdef JP
				if (blind) msg_format("%^sԂ₢B", m_name);
#else
				if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
				else msg_format("%^s@ŗHSI", m_name);
#else
				else msg_format("%^s magically summons rangers of Nazgul!", m_name);
#endif
				msg_print(NULL);

				for (k = 0; k < 30; k++)
				{
					if (!summon_possible(target_ptr, cy, cx) || !cave_empty_bold(floor_ptr, cy, cx))
					{
						int j;
						for (j = 100; j > 0; j--)
						{
							scatter(floor_ptr, &cy, &cx, y, x, 2, 0);
							if (cave_empty_bold(floor_ptr, cy, cx)) break;
						}
						if (!j) break;
					}
					if (!cave_empty_bold(floor_ptr, cy, cx)) continue;

					if (summon_named_creature(caster_ptr, floor_ptr, cy, cx, SPECIES_NAZGUL, mode))
					{
						y = cy;
						x = cx;
						count++;
						if (count == 1)
#ifdef JP
							msg_format("uHS%dAiYOEubNIv", count);
#else
							msg_format("A Nazgul says 'Nazgul-Rangers Number %d, Nazgul-Black!'",count);
#endif
						else
#ifdef JP
							msg_format("u%dAiYOEubNIv", count);
#else
							msg_format("Another one says 'Number %d, Nazgul-Black!'",count);
#endif
						msg_print(NULL);
					}
				}
#ifdef JP
				msg_format("u%dlāAOW[Iv", count);
#else
				msg_format("They say 'The %d meets! We are the Ring-Ranger!'.", count);
#endif
				msg_print(NULL);
			}
			else
			{
#ifdef JP
				if (blind) msg_format("%^sԂ₢B", m_name);
#else
				if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
				else msg_format("%^s@ŋ͂ȃAfbhI", m_name);
#else
				else msg_format("%^s magically summons greater undead!", m_name);
#endif

				for (k = 0; k < s_num_6; k++)
				{
					count += summon_specific(caster_ptr, y, x, rlev, SUMMON_HI_UNDEAD, (PM_ALLOW_GROUP | PM_ALLOW_UNIQUE));
				}
			}
			if (blind && count)
			{
#ifdef JP
				msg_print("ԋ߂ŉ̂̂鉹B");
#else
				msg_print("You hear many creepy things appear nearby.");
#endif

			}
			break;
		}

		case TRAIT_S_HI_DRAGON:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^s@ŌÑhSI", m_name);
#else
			else msg_format("%^s magically summons ancient dragons!", m_name);
#endif

			for (k = 0; k < s_num_4; k++)
			{
				count += summon_specific(caster_ptr, y, x, rlev, SUMMON_HI_DRAGON, (PM_ALLOW_GROUP | PM_ALLOW_UNIQUE));
			}
			if (blind && count)
			{
#ifdef JP
				msg_print("̗͋̂ԋ߂ɌꂽB");
#else
				msg_print("You hear many powerful things appear nearby.");
#endif

			}
			break;
		}

		case TRAIT_S_AMBERITES:
		{
			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^sAo[̉I", m_name);
#else
			else msg_format("%^s magically summons Lords of Amber!", m_name);
#endif



			for (k = 0; k < s_num_4; k++)
			{
				count += summon_specific(caster_ptr, y, x, rlev, SUMMON_AMBERITES, (PM_ALLOW_GROUP | PM_ALLOW_UNIQUE));
			}
			if (blind && count)
			{
#ifdef JP
				msg_print("s̎҂߂Ɍ̂B");
#else
				msg_print("You hear immortal beings appear nearby.");
#endif

			}
			break;
		}

		case TRAIT_S_UNIQUE:
		{
			bool uniques_are_summoned = FALSE;
			int non_unique_type = SUMMON_HI_UNDEAD;

			disturb(player_ptr, 1, 0);
#ifdef JP
			if (blind) msg_format("%^sԂ₢B", m_name);
#else
			if (blind) msg_format("%^s mumbles.", m_name);
#endif

#ifdef JP
			else msg_format("%^s@œʂȋGI", m_name);
#else
			else msg_format("%^s magically summons special opponents!", m_name);
#endif

			for (k = 0; k < s_num_4; k++)
			{
				count += summon_specific(caster_ptr, y, x, rlev, SUMMON_UNIQUE, (PM_ALLOW_GROUP | PM_ALLOW_UNIQUE));
			}

			if (count) uniques_are_summoned = TRUE;

			if ((caster_ptr->sub_align & (SUB_ALIGN_GOOD | SUB_ALIGN_EVIL)) == (SUB_ALIGN_GOOD | SUB_ALIGN_EVIL))
				non_unique_type = 0;
			else if (caster_ptr->sub_align & SUB_ALIGN_GOOD)
				non_unique_type = SUMMON_ANGEL;

			for (k = count; k < s_num_4; k++)
			{
				count += summon_specific(caster_ptr, y, x, rlev, non_unique_type, (PM_ALLOW_GROUP | PM_ALLOW_UNIQUE));
			}

			if (blind && count)
			{
#ifdef JP
				msg_format("%sԋ߂ɌꂽB", uniques_are_summoned ? "͋" : "");
#else
				msg_format("You hear many %s appear nearby.", uniques_are_summoned ? "powerful things" : "things");
#endif
			}
			break;
		}
	}

	if ((target_ptr->action == ACTION_LEARN) && thrown_spell > 175)
	{
		learn_trait(target_ptr, thrown_spell - 96);
	}

	if (seen && maneable && !the_world && (target_ptr->class_idx == CLASS_IMITATOR))
	{
		if (thrown_spell != TRAIT_SPECIAL)
		{
			if (target_ptr->mane_num == MAX_MANE)
			{
				int i;
				target_ptr->mane_num--;
				for (i = 0;i < target_ptr->mane_num;i++)
				{
					target_ptr->mane_spell[i] = target_ptr->mane_spell[i+1];
					target_ptr->mane_dam[i] = target_ptr->mane_dam[i+1];
				}
			}
			target_ptr->mane_spell[target_ptr->mane_num] = thrown_spell - 96;
			target_ptr->mane_dam[target_ptr->mane_num] = dam;
			target_ptr->mane_num++;
			new_mane = TRUE;

			play_redraw |= (PR_IMITATION);
		}
	}

	/* Remember what the creature did to us */
	if (can_remember)
	{
		/* Inate spell */
		if (thrown_spell < 32 * 4)
		{
			r_ptr->r_flags4 |= (1L << (thrown_spell - 32 * 3));
			if (r_ptr->r_cast_spell < MAX_UCHAR) r_ptr->r_cast_spell++;
		}

		/* Bolt or Ball */
		else if (thrown_spell < 32 * 5)
		{
			r_ptr->r_flags5 |= (1L << (thrown_spell - 32 * 4));
			if (r_ptr->r_cast_spell < MAX_UCHAR) r_ptr->r_cast_spell++;
		}

		/* Special spell */
		else if (thrown_spell < 32 * 6)
		{
			r_ptr->r_flags6 |= (1L << (thrown_spell - 32 * 5));
			if (r_ptr->r_cast_spell < MAX_UCHAR) r_ptr->r_cast_spell++;
		}
	}


	/* Always take note of creatures that kill you */
	if (gameover && (r_ptr->r_deaths < MAX_SHORT) && !fight_arena_mode)
	{
		r_ptr->r_deaths++; /* Ignore appearance difference */
	}

	/* A spell was cast */
	return (TRUE);
}
