//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on 12/11/2011.
//

#include "TKContainer.h"
#include "TKCell.h"
#include "TKCellCode.h"
#include "DNStorage.h"
#include "DNUtils.h"
#include "dnplugininfo.h"

#include <string>

const std::string TKContainer::CELLTYPE_JSBASIC      = "B";
const std::string TKContainer::CELLTYPE_OUT          = "O";
const std::string TKContainer::CELLTYPE_IN           = "I";
const std::string TKContainer::CELLTYPE_BASICSTORAGE = "BS";
const std::string TKContainer::CELLTYPE_PLUGIN_IN    = "PI";
const std::string TKContainer::CELLTYPE_PLUGIN_OUT   = "PO";
const std::string TKContainer::CELLCODENAME_EMPTY    = "_DNEMPTY";

TKContainer::TKContainer() : mStorage(NULL), mEmptyCellClass(NULL)
{
}

void TKContainer::init()
{
    mEmptyCellClass = addCellCode(CELLCODENAME_EMPTY,CELLTYPE_JSBASIC,"");
}

TKContainer::~TKContainer()
{
    doDestroy();
    releaseDataStore();
}

bool TKContainer::doInit()
{
    DNLocker locker(&mLock);

    for ( TKCellMap::iterator it = mCells.begin(); it != mCells.end(); ++it ) {
        it->second->doInit();
    }
    return true;
}

bool TKContainer::doTickInputInterfaces(float time)
{
    DNLocker locker(&mLock);

    for ( TKCellMap::iterator it = mInpInterfaceCells.begin(); it != mInpInterfaceCells.end(); ++it )
    {
        it->second->updateReceptorValue();
    }

    for ( TKCellMap::iterator it = mInpInterfaceCells.begin(); it != mInpInterfaceCells.end(); ++it )
    {
        it->second->enterDoTick(time,true);
    }
    return true;
}

bool TKContainer::doTickOutputInterfaces(float time)
{
    DNLocker locker(&mLock);

    for ( TKCellMap::iterator it = mOutInterfaceCells.begin(); it != mOutInterfaceCells.end(); ++it )
    {
        it->second->updateReceptorValue();
    }

    for ( TKCellMap::iterator it = mOutInterfaceCells.begin(); it != mOutInterfaceCells.end(); ++it )
    {
        it->second->enterDoTick(time,true);
    }
    return true;
}

bool TKContainer::doTickSignalScan(float time)
{
    DNLocker locker(&mLock);

    for ( TKCellMap::iterator it = mNonInterfaceCells.begin(); it != mNonInterfaceCells.end(); ++it )
    {
        it->second->updateReceptorValue();
    }

    for ( TKCellMap::iterator it = mNonInterfaceCells.begin(); it != mNonInterfaceCells.end(); ++it )
    {
        it->second->enterDoTick(time,false);
    }
    return true;
}

bool TKContainer::doTickFullScan(float time)
{
    DNLocker locker(&mLock);

    for ( TKCellMap::iterator it = mNonInterfaceCells.begin(); it != mNonInterfaceCells.end(); ++it )
    {
        it->second->updateReceptorValue();
    }

    for ( TKCellMap::iterator it = mNonInterfaceCells.begin(); it != mNonInterfaceCells.end(); ++it )
    {
        it->second->enterDoTick(time,true);
    }
    return true;
}

bool TKContainer::doDestroy()
{
    DNLocker locker(&mLock);

    for ( TKCellMap::iterator it = mCells.begin(); it != mCells.end(); ++it ) {
        (it->second)->doDestroy();
    }
    for ( TKCellMap::iterator it = mCells.begin(); it != mCells.end(); ++it ) {
        delete it->second;
    }
    mCells.clear();
    mInpInterfaceCells.clear();
    mOutInterfaceCells.clear();

    return true;
}

TKCell* TKContainer::addCell(std::string theLocation, std::string theName, std::string type, std::string customScript)
{
    TKCell *cell = NULL;
    bool    noScript = false;

    if (type == CELLTYPE_JSBASIC || type.length() == 0)
    {
        cell = cellFactory(theLocation, theName, type, false, false);
    }
    else if (type == CELLTYPE_IN)
    {
        cell = cellFactory(theLocation, theName, type, true, false);
        noScript = true;
    }
    else if (type == CELLTYPE_OUT)
    {
        cell = cellFactory(theLocation, theName, type, false, true);
        noScript = true;
    }
    else if (type == CELLTYPE_BASICSTORAGE)
    {
        cell = cellFactory(theLocation, theName, type, false, false);
    }
    else if (type == CELLTYPE_PLUGIN_IN)
    {
        DNPluginInfo info = DNPluginInfo::create(theName);
        if (info.isValid)
        {
            cell = pluginCellFactory(theLocation, theName, type, info.pluginName, info.pluginValueName, true, false);
        }
        else
        {
            std::string message = std::string("Failed to initialize cell '").append(theLocation).append("#").append(theName);
            message.append("'\nThis is a plugin input cell. The name of the cell should be described by following form:\n'pluginValueName'@'pluginName'");
            dnNotifyError("Initialization failed", message);
        }
        noScript = true;
    }
    else if (type == CELLTYPE_PLUGIN_OUT)
    {
        DNPluginInfo info = DNPluginInfo::create(theName);
        if (info.isValid)
        {
            cell = pluginCellFactory(theLocation, theName, type, info.pluginName, info.pluginValueName, false, true);
        }
        else
        {
            std::string message = std::string("Failed to initialize cell '").append(theLocation).append("#").append(theName);
            message.append("'\nThis is a plugin output cell. The name of the cell should be described by following form:\n'pluginValueName'@'pluginName'");
            dnNotifyError("Initialization failed", message);
        }
        noScript = true;
    }
    else
    {
        std::string message = std::string("Failed to construct cell '").append(theLocation).append("#").append(theName);
        message.append("'\nThe cellcode requires type '").append(type).append("' but it's not a supported type.");
        dnNotifyError("Initialization failed", message);
    }

    if (cell)
    {
        std::string fqnName = getFQNString(theLocation.c_str(), theName.c_str());
        mCells.insert(TKCellMap::value_type(fqnName, cell));
        bool interface = false;
        if (cell->canInterfaceIn())
        {
            mInpInterfaceCells.insert(TKCellMap::value_type(fqnName, cell));
            interface = true;
        }
        if (cell->canInterfaceOut())
        {
            mOutInterfaceCells.insert(TKCellMap::value_type(fqnName, cell));
            interface = true;
        }
        if (!interface)
        {
            mNonInterfaceCells.insert(TKCellMap::value_type(fqnName, cell));
        }
        if (!noScript)
        {
            cell->setCellCode(mEmptyCellClass,(const void*)customScript.c_str());
        }
        else
        {
            if (customScript.length()>0)
            {
                std::string message = std::string("Failed to construct cell '").append(theLocation).append("#").append(theName);
                message.append("'\nThe cellcode is type '").append(type).append("'. This type doesn't support custom script.");
                dnNotifyError("Initialization failed",message);
            }
        }
    }
    else
    {
        std::string fqnName = getFQNString(theLocation.c_str(),theName.c_str());
        std::string message = "Failed to create a cell. ";
        message += fqnName;
        message += "\n";
        message += "Out of memory?";
        dnNotifyError("Initialization failed", message);
    }

    return cell;
}

TKCell* TKContainer::addCell(std::string theLocation, std::string theName, TKCellCode *cellCode, std::string customScript)
{
    TKCell *cell = NULL;

    std::string type = cellCode->getCellAPIName();

    if (type == CELLTYPE_JSBASIC)
    {
        cell = cellFactory(theLocation, theName, type, false, false);
    }
    else if (type == CELLTYPE_BASICSTORAGE)
    {
        cell = cellFactory(theLocation, theName, type, false, false);
    }
    else if (type == CELLTYPE_OUT)
    {
        std::string message = std::string("Failed to construct cell '").append(theLocation).append("#").append(theName);
        message.append("'\nThe cellcode type '").append(type).append("' doesn't support to have a CellCode class.");
        dnNotifyError("Initialization failed", message);
    }
    else if (type == CELLTYPE_IN)
    {
        std::string message = std::string("Failed to construct cell '").append(theLocation).append("#").append(theName);
        message.append("'\nThe cellcode requires type '").append(type).append("' doesn't support to have a CellCode class.");
        dnNotifyError("Initialization failed", message);
    }
    else
    {
        std::string message = std::string("Failed to construct cell '").append(theLocation).append("#").append(theName);
        message.append("'\nThe cellcode requires type '").append(type).append("' but it's not a supported type.");
        dnNotifyError("Initialization failed",message);
    }


    if (cell)
    {
        std::string fqnName = getFQNString(theLocation.c_str(), theName.c_str());
        mCells.insert(TKCellMap::value_type(fqnName, cell));
        cell->setCellCode(cellCode,(const void*)customScript.c_str());

        bool interface = false;
        if (cell->canInterfaceIn())
        {
            mInpInterfaceCells.insert(TKCellMap::value_type(fqnName, cell));
            interface = true;
        }
        if (cell->canInterfaceOut())
        {
            mOutInterfaceCells.insert(TKCellMap::value_type(fqnName, cell));
            interface = true;
        }
        if (!interface)
        {
            mNonInterfaceCells.insert(TKCellMap::value_type(fqnName, cell));
        }
    }
    else
    {
        std::string fqnName = getFQNString(theLocation.c_str(),theName.c_str());
        std::string message = "Failed to create a cell. ";
        message += fqnName;
        message += "\n";
        message += "Out of memory?";
        dnNotifyError("Initialization failed", message);
    }

    return cell;
}

TKCellCode* TKContainer::addCellCode(std::string theName, std::string theAPIType, std::string code)
{
    TKCellCode *cellCode = cellCodeFactory(theName, theAPIType, code);

    if (cellCode)
    {
        mCellCodes.insert( std::map<std::string, TKCellCode*>::value_type(theName, cellCode));
    }
    return cellCode;
}

bool TKContainer::setDataStore(std::string storagePath)
{
    mStorage = new DNStorage(storagePath.c_str());
    return mStorage->isValid();
}

bool TKContainer::releaseDataStore()
{
    mLock.lock();
    if (mStorage)
    {
        delete mStorage;
        mStorage = NULL;
    }
    mLock.unlock();
    return true;
}

TKCell* TKContainer::getCell(std::string theFQNName)
{
    TKCellMap::iterator it = mCells.find(theFQNName);
    if (it == mCells.end())
    {
        return NULL;
    }
    else
    {
        return it->second;
    }
}

TKCell* TKContainer::getInputInterfaceCell(std::string theFQNName)
{
    TKCellMap::iterator iti = mInpInterfaceCells.find(theFQNName);
    if (iti == mInpInterfaceCells.end())
    {
        return NULL;
    }
    else
    {
        return iti->second;
    }
}

TKCell* TKContainer::getOutputInterfaceCell(std::string theFQNName)
{
    TKCellMap::iterator ito = mOutInterfaceCells.find(theFQNName);
    if (ito == mOutInterfaceCells.end())
    {
        return NULL;
    }
    else
    {
        return ito->second;
    }
}

TKCellCode* TKContainer::getCellCode(std::string theCellCodeName)
{
    TKCellCodeMap::iterator it = mCellCodes.find(theCellCodeName);
    if (it == mCellCodes.end())
    {
        return NULL;
    }
    else
    {
        return it->second;
    }
}
