//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on 1/28/2012.
//


#include "TKLock.h"
#include "TKContainer.h"
#include "DNCellInterfaceable.h"
#include "DNTimeKeeper.h"
#include "DNServerHTTP.h"
#include "DNServerSerialPort.h"
#include "TKContainer.h"
#include "DNContainerBuilder.h"
#include "DNEngine.h"
#include "DNAlert.h"
#include "DNXML.h"
#include "DNXMLElement.h"
#include "DNGlobal.h"
#include "DNUtils.h"
#include "TKLog.h"
#include "DNSettings.h"
#include "dnpluginmanager.h"

#include <stdlib.h>
#include <sstream>
#include <iostream>

DNEngine::DNEngine(const char *contentPath) :
    mContainer(NULL),mPortNumber(9080),mHTTPServer(NULL),mSerialServer(NULL),mSerialServerEnabled(false),mValid(false), mDoTickThread(NULL), mNumUpdateScan(5)
{
    mTimeKeeper = new DNTimeKeeper();
    mContainer = TKContainer::createContainer();
    DNPluginManager::construct();
    mUIPath = "/ui/index.html";

    dnGlobal()->updateRunningStatus(DNGlobal::STOPPED);
    dnGlobal()->resetErrorStatus();

    std::string basePath(contentPath);
    std::string containerRoot = basePath;
    containerRoot.append("/Container");
    
    bool succeeded = false;
    succeeded = parseSettingFile(contentPath);
    if (!succeeded || !dnGlobal()->isErrorStatusNormal())
    {
        dnNotifyError("property.xml parse error", "failed to parse setting file /property.xml");
        return;
    }

    std::string dataStorePath = containerRoot;
    dataStorePath.append("/data.db");

    succeeded = mContainer->setDataStore(dataStorePath.c_str());
    if (!succeeded)
    {
        dnNotifyError("Initialization failed","Failed to the setup data store");
        return;
    }

    succeeded = parseContainerFile(containerRoot.c_str());
    if (!succeeded)
    {
        dnNotifyError("Initialization failed", "Failed to parse container file");
        return;
    }

    if (mSerialServerEnabled)
    {
        mSerialServerEnabled = startSerialServer();
    }
    else
    {
        stopSerialServer();
    }

    mValid = true;
}

DNEngine::~DNEngine()
{
    mValid = false;
    
    if (mHTTPServer)
    {
        stopHTTPServer();
    }

    if (mSerialServer)
    {
        stopSerialServer();
    }

    if (mContainer)
    {
        delete mContainer;
        mContainer = NULL;
    }
            
    if (mTimeKeeper)
    {
        delete mTimeKeeper;
        mTimeKeeper = NULL;
    }
    DNPluginManager::destroy();
}


bool DNEngine::parseSettingFile(const char *contentRoot)
{
    bool valid = false;

    DNXML *xml = DNXML::createXMLFromFile(contentRoot, "property.xml");
    if (xml)
    {
        float tickIntervalFull = 1.0;
        float tickIntervalSignal = 0;
        float tickIntervalInput = 0;
        float tickIntervalOutput = 0;

        valid = true;
        DNXMLElement *element = xml->getRoot();
        if (!element)
        {
            valid = false;
            std::string message = "Failed to load property.xml file";
            dnNotifyError("Initialization failed",message);
        }

        if (valid && element->name != "dennco")
        {
            valid = false;
            std::string message = "First element of property.xml should be <dennco>";
            dnNotifyError("Initialization failed",message);
        }

        if (valid)
        {
            DNXMLElement *e = element->inner;
            while(e)
            {
                std::string pname = upperString(e->name);

                if (pname == "TICKINTERVALSEC")
                {
                    std::istringstream is(e->text);
                    float t = 0.0;
                    is >> t;
                    if (t>0)
                    {
                        tickIntervalFull = t;
                    }
                    else
                    {
                        valid = false;
                        std::string message = "Error in property.xml. TickIntervalSec is not configured properly.";
                        dnNotifyError("Initialization failed",message);
                    }
                }
                else if (pname == "TICKINTERVALSEC_FULLSCAN")
                {
                    std::istringstream is(e->text);
                    float t = 0.0;
                    is >> t;
                    if (t>0)
                    {
                    }
                    else
                    {
                        valid = false;
                        std::string message = "Error in property.xml. TickIntervalSec_FullScan is not configured properly.";
                        dnNotifyError("Initialization failed",message);
                    }

                }
                else if (pname == "TICKINTERVALSEC_SIGNALSCAN")
                {
                    std::istringstream is(e->text);
                    float t = 0.0;
                    is >> t;
                    if (t>0)
                    {
                        tickIntervalSignal = t;
                    }
                    else
                    {
                        valid = false;
                        std::string message = "Error in property.xml. TickIntervalSec_SignalScan is not configured properly.";
                        dnNotifyError("Initialization failed",message);
                    }

                }
                else if (pname == "TICKINTERVALSEC_INPUTSCAN")
                {
                    std::istringstream is(e->text);
                    float t = 0.0;
                    is >> t;
                    if (t>0)
                    {
                        tickIntervalInput = t;
                    }
                    else
                    {
                        valid = false;
                        std::string message = "Error in property.xml. TickIntervalSec_InputScan is not configured properly.";
                        dnNotifyError("Initialization failed",message);
                    }

                }
                else if (pname == "TICKINTERVALSEC_OUTPUTSCAN")
                {
                    std::istringstream is(e->text);
                    float t = 0.0;
                    is >> t;
                    if (t>0)
                    {
                        tickIntervalOutput = t;
                    }
                    else
                    {
                        valid = false;
                        std::string message = "Error in property.xml. TickIntervalSec_OutputScan is not configured properly.";
                        dnNotifyError("Initialization failed",message);
                    }
                }
                else if (pname == "UIPATH")
                {
                    mUIPath = e->text;
                }
                else if (pname == "ENABLEHTTPSERVER")
                {

                }
                else if (pname == "ENABLESERIALSERVER")
                {
                    if ( upperString(e->text) == "YES")
                    {
                        mSerialServerEnabled = true;
                    }
                    else
                    {
                        mSerialServerEnabled = false;
                    }
                }
                else if (pname == "SERIALSPEED")
                {
                    DNSettings::setValue(DNSettings::SERIAL_RATE, e->text);
                }
                e = e->next;
            }
        }

        delete xml;

        if (tickIntervalSignal <= 0)
        {
            tickIntervalSignal = tickIntervalFull / 5.0f;
        }
        if (tickIntervalInput <= 0)
        {
            tickIntervalInput = tickIntervalFull;
        }
        if (tickIntervalOutput <= 0)
        {
            tickIntervalOutput = tickIntervalFull;
        }

        mTimeKeeper->setIntevalSec(tickIntervalFull, tickIntervalSignal, tickIntervalInput, tickIntervalOutput);
    }
    return valid;
}

bool DNEngine::parseContainerFile(const char *containerRoot)
{
    DNContainerBuilder builder(mContainer);
    return builder.buildContainerFromXHTML(containerRoot);
}


bool DNEngine::startHTTPServer(int portNumber)
{
    if (mHTTPServer && mHTTPServer->isRunning())
    {
        mHTTPServer->stop();
        delete mHTTPServer;
        mHTTPServer = NULL;
    }
    mHTTPServer = new DNServerHTTP(this);
    if (mHTTPServer)
    {
        mHTTPServer->setPortNumber(portNumber);
        mHTTPServer->start();
    }
    return true;
}

void DNEngine::stopHTTPServer()
{
    if (mHTTPServer)
    {
        mHTTPServer->stop();
        delete mHTTPServer;
        mHTTPServer = NULL;
    }    
}

bool DNEngine::startSerialServer()
{
    stopSerialServer();
    mSerialServer = new DNServerSerialPort(this);
    if (mSerialServer)
    {
        if (mSerialServer->setup())
        {
            mSerialServer->start();
            return true;
        }
        else
        {
            return false;
        }
    }
    return false;
}

void DNEngine::stopSerialServer()
{
    if (mSerialServer)
    {
        mSerialServer->stop();
        delete mSerialServer;
        mSerialServer = NULL;
    }
}

void DNEngine::setTickIntervalSec(float full, float signal, float input, float output)
{
    if (mTimeKeeper)
        mTimeKeeper->setIntevalSec(full, signal, input, output);
}

bool DNEngine::startEngine()
{
    mContainer->doInit();

    if (!mDoTickThread)
        mDoTickThread = DNThread::createThread(DNEngine::doTickThread, this);

    return mDoTickThread->start();
}

bool DNEngine::stopEngine()
{
    bool r = false;
    dnGlobal()->updateRunningStatus(DNGlobal::STOPPING);
    if (mDoTickThread)
    {
        r = mDoTickThread->waitForExit(5000);
        delete mDoTickThread;
        mDoTickThread = NULL;
    }
    if (mContainer)
    {
        mContainer->doDestroy();
        mContainer->releaseDataStore();
    }

    if (mSerialServer)
        mSerialServer->stop();

    return r;
}

void DNEngine::doTickThread(void *self)
{
    if (!dnGlobal()->updateRunningStatus(DNGlobal::RUNNIING))
    {
        return;
    }

    DNEngine *engine = (DNEngine*)self;
    
    while(dnGlobal()->getRunningStatus() == DNGlobal::RUNNIING && engine->isValid())
    {
        int state =  engine->mTimeKeeper->sleepUntilNextInterval();

        if (state & DNTimeKeeper::STATE_IN_INPUT_SCAN_TIME)
        {
            engine->mContainer->doTickInputInterfaces(engine->mTimeKeeper->getTickTime());
        }
        if (state & DNTimeKeeper::STATE_IN_FULL_SCAN_TIME)
        {
            engine->mContainer->doTickFullScan(engine->mTimeKeeper->getTickTime());
        }
        else if (state & DNTimeKeeper::STATE_IN_SIGNAL_SCAN_TIME)
        {
            //scan non-interface cells which has updated receptor value
            engine->mContainer->doTickSignalScan(engine->mTimeKeeper->getTickTime());
        }
        if (state & DNTimeKeeper::STATE_IN_OUTPUT_SCAN_TIME)
        {
            //scan output cells
            engine->mContainer->doTickOutputInterfaces(engine->mTimeKeeper->getTickTime());
        }
    }

    dnGlobal()->updateRunningStatus(DNGlobal::STOPPED);
}

float DNEngine::doClientGetRequest(const char* path)
{
    if (dnGlobal()->getRunningStatus() != DNGlobal::RUNNIING)
        return 0;

    std::string fqn = getFQNString("/",path);
    DNCellInterfaceable *cell = dynamic_cast<DNCellInterfaceable*>(mContainer->getOutputInterfaceCell(fqn));
    float result = 0;
    if (cell)
    {
        result = cell->getValue();
    }
    else
    {
        std::string message = "The requested path ";
        message += path;
        message += " doesn't exist.";
        dnNotifyWarning("Failed to process the client request. (getValue)", message);
    }
    return result;
}

bool DNEngine::doClientSetRequest(const char* path, const char* value)
{
    return doClientSetRequest(path, (float)atof(value));
}

bool DNEngine::doClientSetRequest(const char* path, float value)
{
    if (dnGlobal()->getRunningStatus() != DNGlobal::RUNNIING)
        return false;

    std::string fqn = getFQNString("/",path);
    DNCellInterfaceable *cell = dynamic_cast<DNCellInterfaceable*>(mContainer->getInputInterfaceCell(fqn));
    bool result = false;
    if (cell)
    {
        cell->setValue(value);
        result = true;
    }
    else
    {
        std::string message = "The requested path ";
        message += path;
        message += " doesn't exist.";
        dnNotifyWarning("Failed to process the client request. (setValue)", message);
    }
    return result;
}

std::string DNEngine::getContainerRootPath()
{
    if (mContainer)
        return mContainer->getContainerRootPath();
    else
        return "";
}

std::string DNEngine::getUIPath()
{
    return mUIPath;
}

//static
void DNEngine::callbackSerialPortClosed(DNEngine *engine)
{
    (void)engine;

    //TODO
}

