//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on 12/15/2011.
//

#include "TKJSContainer.h"
#include "TKCell.h"
#include "TKJSCellCode.h"
#include "TKJSBasicCell.h"
#include "TKUICell.h"
#include "TKLog.h"
#include "DNUtils.h"
#include "versioninfo.h"

#include <string>

const std::string TKJSContainer::CELLTYPE_JSBASIC = "TKJSBasicCell";
const std::string TKJSContainer::CELLTYPE_UIBASE = "TKUIBaseCell";

static JSValueRef jsGlobalPrint(
								JSContextRef        ctx,
								JSObjectRef         jobj,
								JSObjectRef         jobjThis,
								size_t              argLen,
								const JSObjectRef   args[],
								JSValueRef*         jobjExp);

static JSValueRef jsGlobalFQN(
								JSContextRef        ctx,
								JSObjectRef         jobj,
								JSObjectRef         jobjThis,
								size_t              argLen,
								const JSObjectRef   args[],
								JSValueRef*         jobjExp);

//static
TKContainer* TKContainer::createContainer()
{
    return new TKJSContainer;
}

TKJSContainer::TKJSContainer()
{
    mContext = JSGlobalContextCreate(NULL);
    mGlobalObject = JSContextGetGlobalObject(mContext);

    JSStringRef jstrPrint = JSStringCreateWithUTF8CString("print");
    JSObjectRef jfuncPrint = JSObjectMakeFunctionWithCallback(mContext, jstrPrint, (JSObjectCallAsFunctionCallback)jsGlobalPrint);
    JSObjectSetProperty(mContext, mGlobalObject, jstrPrint, jfuncPrint, kJSPropertyAttributeNone, NULL);
    JSStringRelease(jstrPrint);

    JSStringRef jstrFQN = JSStringCreateWithUTF8CString("FQN");
    JSObjectRef jfuncFQN = JSObjectMakeFunctionWithCallback(mContext, jstrFQN, (JSObjectCallAsFunctionCallback)jsGlobalFQN);
    JSObjectSetProperty(mContext, mGlobalObject, jstrFQN, jfuncFQN, kJSPropertyAttributeNone, NULL);
    JSStringRelease(jstrFQN);

    JSStringRef jstrDGlobal = JSStringCreateWithUTF8CString("global");
    mJGlobalObject = JSObjectMake(mContext, NULL, NULL);
    JSObjectSetProperty(mContext, mGlobalObject, jstrDGlobal, mJGlobalObject, kJSPropertyAttributeNone, NULL);
    JSStringRelease(jstrDGlobal);

    JSStringRef jstrContainerVersion = JSStringCreateWithUTF8CString("container_version");
    JSStringRef jstrContainerVersionValue = JSStringCreateWithUTF8CString(CONTAINER_VERSION);
    JSValueRef  jContainerVersionValue = JSValueMakeString(mContext, jstrContainerVersionValue);
    JSObjectSetProperty(mContext, mJGlobalObject, jstrContainerVersion, jContainerVersionValue, kJSPropertyAttributeNone, NULL);
    JSStringRelease(jstrContainerVersion);
    JSStringRelease(jstrContainerVersionValue);

}

TKJSContainer::~TKJSContainer()
{
    JSGlobalContextRelease(mContext);
    mContext = 0;
}

TKCell* TKJSContainer::createCell(std::string theLocation, std::string theName, TKCellCode *cellCode, std::string startupScript)
{
	TKCell *cell = NULL;
	
    std::string type = cellCode->getCellAPIName();
    
	if (type == CELLTYPE_JSBASIC)
	{
		TKJSBasicCell *basicCell = new TKJSBasicCell(this, theLocation, theName);
		if (basicCell)
		{
            //do some init here if needed.
            
			cell = basicCell;
		}
	}
	else if (type == CELLTYPE_UIBASE)
    {
        cell = new TKUICell(this, theLocation, theName);
    }
    else
	{
		TKLog::printf("ERROR in TKJSContainer::createCell. Type name:%s is not correct.", type.c_str());
	}

	
	if (cell)
	{
        std::string fqnName = getFQNString(theLocation.c_str(), theName.c_str());
		mCells.insert(TKCellMap::value_type(fqnName, cell));
        cell->setCellCode(cellCode,(const void*)startupScript.c_str());
        
        if (cell->isInterface())
        {
            mInterfaceCells.insert(TKCellMap::value_type(fqnName, cell));
        }
	}
    else
    {
        std::string fqnName = getFQNString(theLocation.c_str(),theName.c_str());
        TKLog::printf("Failed to create a cell. %s", fqnName.c_str());
    }
    
	return cell;

}

TKCellCode* TKJSContainer::createCellCode(std::string theName, std::string theAPIType, std::string code)
{
    TKJSCellCode *cellCode = new TKJSCellCode(theName, theAPIType, this, code);
    
    if (cellCode)
    {
        mCellCodes.insert( std::map<std::string, TKCellCode*>::value_type(theName, cellCode));
    }
    return cellCode;
}


void TKJSContainer::setValue(std::string key, float value)
{    
    JSStringRef jstr = JSStringCreateWithUTF8CString(key.c_str());
    JSValueRef jvalue = JSValueMakeNumber(mContext, value);
    JSObjectSetProperty(mContext, mJGlobalObject, jstr, jvalue, kJSPropertyAttributeNone, NULL);
    JSStringRelease(jstr);
}

float TKJSContainer::getValue(std::string key)
{
    JSStringRef jstr = JSStringCreateWithUTF8CString(key.c_str());
    float result = 0.0;
    if (JSObjectHasProperty(mContext, mJGlobalObject, jstr))
    {
        JSValueRef jvalue = JSObjectGetProperty(mContext, mJGlobalObject, jstr, NULL);
        JSStringRelease(jstr);
        result = JSValueToNumber(mContext, jvalue,NULL);
    }
    return result;
}

bool TKJSContainer::doTick(float time)
{
	for ( TKCellMap::iterator it = mCells.begin(); it != mCells.end(); ++it ) {
		it->second->doTick(time);
	}
	
	return true;
}

JSValueRef jsGlobalPrint(
								JSContextRef        ctx,
								JSObjectRef         jobj,
								JSObjectRef         jobjThis,
								size_t              argLen,
								const JSObjectRef   args[],
								JSValueRef*         jobjExp) {
	
    if (argLen) {
        JSStringRef     jstrArg = JSValueToStringCopy(ctx, args[0], jobjExp);
        size_t          len     = JSStringGetMaximumUTF8CStringSize(jstrArg);
        char*           szArg   =  (char*)malloc(len);
        JSStringGetUTF8CString(jstrArg, szArg, len);
        TKLog::printf("%s", szArg);
        JSStringRelease(jstrArg);
        free(szArg);
    }
	
    return JSValueMakeUndefined(ctx);
}

JSValueRef jsGlobalFQN(
								JSContextRef        ctx,
								JSObjectRef         jobj,
								JSObjectRef         jobjThis,
								size_t              argLen,
								const JSObjectRef   args[],
								JSValueRef*         jobjExp) {
	
    JSValueRef jsResult = NULL;;
    
    if (argLen >= 2) {
        JSStringRef     jstrLocation    = JSValueToStringCopy(ctx, args[0], jobjExp);
        JSStringRef     jstrName        = JSValueToStringCopy(ctx, args[1], jobjExp);
        
        size_t          locationLen = JSStringGetMaximumUTF8CStringSize(jstrLocation);
        size_t          nameLen     = JSStringGetMaximumUTF8CStringSize(jstrName);
        char*           cLocation   =  (char*)malloc(locationLen);
        char*           cName       =  (char*)malloc(nameLen);
        JSStringGetUTF8CString(jstrLocation, cLocation, locationLen);
        JSStringGetUTF8CString(jstrName, cName, nameLen);
        std::string fqnString = getFQNString(cLocation, cName);
        
        jsResult = JSValueMakeString(ctx,JSStringCreateWithUTF8CString(fqnString.c_str()));
        
        free(cLocation);
        free(cName);
    }
    else
    {
        jsResult = JSValueMakeUndefined(ctx);        
    }
	
    return jsResult;
}


