//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on 3/28/2012.
//
#include "dnqsbasicstoragecell.h"

#include "TKContainer.h"
#include "dnqscontainer.h"
#include "dnqsbasiccell.h"
#include "TKLog.h"
#include "TKDebug.h"
#include "DNUtils.h"
#include "DNStorage.h"

QScriptString DNQSBasicStorageCell::mLengthQSString;
QScriptString DNQSBasicStorageCell::mXQSString;
QScriptString DNQSBasicStorageCell::mYQSString;
QScriptString DNQSBasicStorageCell::mZQSString;
QScriptString DNQSBasicStorageCell::mVQSString;
QScriptString DNQSBasicStorageCell::mPushQSString;

DNQSBasicStorageCell::DNQSBasicStorageCell(DNQSContainer *container, std::string location, std::string name, bool canInterfaceIn, bool canInterfaceOut) :
    DNQSBasicCell(container, location, name, canInterfaceIn, canInterfaceOut)
{

    mQSStorage = mEngine->newObject();
    mPath = getFQNString(mLocation.c_str(), mName.c_str());
    mQSStorage.setProperty("setValue",mEngine->newFunction(scriptFunction_storage_setValue, 2));
    mQSStorage.setProperty("getValue",mEngine->newFunction(scriptFunction_storage_getValue, 1));
    mQSStorage.setProperty("setXYZVArray",mEngine->newFunction(scriptFunction_storage_setXYZVArray, 2));
    mQSStorage.setProperty("getXYZVArray",mEngine->newFunction(scriptFunction_storage_getXYZVArray, 1));
    mQSStorage.setData(mEngine->newQObject(this));
    mQSAPIInstance.setProperty("storage", mQSStorage,QScriptValue::ReadOnly|QScriptValue::Undeletable);

    if (!mLengthQSString.isValid())
    {
        mLengthQSString = mEngine->toStringHandle("length");
        mVQSString = mEngine->toStringHandle("v");
        mXQSString = mEngine->toStringHandle("x");
        mYQSString = mEngine->toStringHandle("y");
        mZQSString = mEngine->toStringHandle("z");
        mPushQSString = mEngine->toStringHandle("push");
    }
}

DNQSBasicStorageCell::~DNQSBasicStorageCell()
{
}

int DNQSBasicStorageCell::getCount_storage(const char* key)
{
    DNStorage *storage = mContainer->getDataStore();
    if (storage)
        return storage->getCount(mPath.c_str(),key);
    else
        return 0;
}

float DNQSBasicStorageCell::getValue_storage(const char *key)
{
    DNStorage *storage = mContainer->getDataStore();
    if (storage)
        return storage->getValue(mPath.c_str(), key);
    else
        return 0.0;
}

bool DNQSBasicStorageCell::setValue_storage(const char *key, float value)
{
    DNStorage *storage = mContainer->getDataStore();
    if (storage)
        return storage->setValue(mPath.c_str(), key, value);
    else
        return false;
}

int DNQSBasicStorageCell::getXYZVArrayCount_storage(const char* key)
{
    DNStorage *storage = mContainer->getDataStore();
    if (storage)
    {
        return storage->countXYZVData(mPath.c_str(), key);
    }
    else
        return 0;

}

DNStorageXYZVRecords* DNQSBasicStorageCell::getXYZVArray_storage(const char *key)
{
    DNStorage *storage = mContainer->getDataStore();
    if (storage)
        return storage->queryXYZVData(mPath.c_str(), key);
    else
        return NULL;

}

bool DNQSBasicStorageCell::setXYZVArray_storage(const char *key, QScriptValue array)
{
    if (!array.isArray())
        return false;

    DNStorage *storage = mContainer->getDataStore();
    if (storage == NULL)
        return false;

    int length = array.property(mLengthQSString).toInt32();
    const char *path = mPath.c_str();

    storage->startTransaction();

    storage->deleteXYZVData(path,key);

    bool r = true;

    for (int i = 0; i < length; i++)
    {
        QScriptValue item = array.property(i);
        float x = 0;
        float y = 0;
        float z = 0;
        float v = 0;

        QScriptValue sx =  item.property(mXQSString);
        if (sx.isNumber())
        {
            x = sx.toNumber();
        }
        QScriptValue sy =  item.property(mYQSString);
        if (sy.isNumber())
        {
            y = sy.toNumber();
        }
        QScriptValue sz =  item.property(mZQSString);
        if (sz.isNumber())
        {
            z = sz.toNumber();
        }
        QScriptValue sv =  item.property(mVQSString);
        if (sv.isNumber())
        {
            v = sv.toNumber();
        }
        r = storage->insertXYZVData(path,key,i,x,y,z,v);
        if (!r) break;
    }

    if (r)
        storage->commitTransaction();
    else
        storage->rollbackTransaction();

    return r;
}

//static
QScriptValue DNQSBasicStorageCell::scriptFunction_storage_setValue(QScriptContext *context, QScriptEngine *engine)
{
    QScriptValue data = context->thisObject().data();
    TKASSERT(data.isQObject());
    DNQSBasicStorageCell *cell = (DNQSBasicStorageCell*) data.toQObject();
    TKASSERT(cell);
    if (cell && context->argumentCount() >= 2)
    {
        cell->setValue_storage(context->argument(0).toString().toUtf8().constData(),context->argument(1).toNumber());
    }
    return QScriptValue(engine, true);
}

//static
QScriptValue DNQSBasicStorageCell::scriptFunction_storage_getValue(QScriptContext *context, QScriptEngine *engine)
{
    QScriptValue data = context->thisObject().data();
    TKASSERT(data.isQObject());
    DNQSBasicStorageCell *cell = (DNQSBasicStorageCell*) data.toQObject();
    TKASSERT(cell);
    if (cell && context->argumentCount() >= 1)
    {
        QByteArray qkey = context->argument(0).toString().toUtf8();
        const char *key = qkey.constData();
        if (cell->getCount_storage(key) == 0)
        {
            return engine->undefinedValue();
        }
        else
        {
            return QScriptValue(cell->getValue_storage(key));
        }
    }
    return engine->undefinedValue();
}

//static
QScriptValue DNQSBasicStorageCell::scriptFunction_storage_setXYZVArray(QScriptContext *context, QScriptEngine *engine)
{
    QScriptValue data = context->thisObject().data();
    TKASSERT(data.isQObject());
    DNQSBasicStorageCell *cell = (DNQSBasicStorageCell*) data.toQObject();
    TKASSERT(cell);
    bool r = false;
    if (cell && context->argumentCount() >= 2)
    {
        QByteArray qkey = context->argument(0).toString().toUtf8();
        const char *key = qkey.constData();
        QScriptValue array = context->argument(1);
        r = cell->setXYZVArray_storage(key, array);
    }
    return QScriptValue(engine, r);
}

//static
QScriptValue DNQSBasicStorageCell::scriptFunction_storage_getXYZVArray(QScriptContext *context, QScriptEngine *engine)
{
    QScriptValue thisObj = context->thisObject();
    QScriptValue data = thisObj.data();
    TKASSERT(data.isQObject());
    DNQSBasicStorageCell *cell = (DNQSBasicStorageCell*) data.toQObject();
    TKASSERT(cell);
    if (cell && context->argumentCount() >= 1)
    {
        QByteArray qkey = context->argument(0).toString().toUtf8();
        const char *key = qkey.constData();

        if (cell->getXYZVArrayCount_storage(key) == 0)
        {
            return engine->undefinedValue();
        }
        else
        {
            DNStorageXYZVRecords *records = cell->getXYZVArray_storage(key);
            if (records)
            {
                QScriptValue result = engine->newArray(0);
                QScriptValue pushScriptFunction = result.property(mPushQSString);
                for (int i = 0; i < records->length; i++)
                {
                    QScriptValue item = engine->newObject();
                    item.setProperty(mXQSString, QScriptValue(records->data[i].x));
                    item.setProperty(mYQSString, QScriptValue(records->data[i].y));
                    item.setProperty(mZQSString, QScriptValue(records->data[i].z));
                    item.setProperty(mVQSString, QScriptValue(records->data[i].v));
                    QScriptValueList args;
                    args << item;
                    pushScriptFunction.call(result, args);
#ifdef DEBUG
                    if (engine->hasUncaughtException())
                    {
                        QScriptValue error = engine->uncaughtException();
                        QString errorString = error.toString();
                        TKLog::debugPrintf("Script error at scriptFunction_storage_getXYZVArray.  Error Message: %s", errorString.toLocal8Bit().data());
                        engine->clearExceptions();
                    }
#endif //DEBUG
                }
                delete records;
                return result;
            }
        }
    }
    return engine->undefinedValue();
}

