//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on 2/25/2012.
//
#include "qtdntimekeeperimpl.h"

#include <QThread>
#include <QDateTime>

#include "DNTimeKeeper.h"

class SleeperThread : public QThread
{
public:
    static void msleep(unsigned long msecs)
    {
        QThread::msleep(msecs);
    }
};

DNTimeKeeperImpl *DNTimeKeeperImpl::create()
{
    return new QtDNTimeKeeperImpl;
}

QtDNTimeKeeperImpl::QtDNTimeKeeperImpl() :
    mFullScanIntervalTime(1000),mSignalScanIntervalTime(1000), mInputScanIntervalTime(1000), mOutputScanIntervalTime(1000),
    mNextFullScanTime(0), mNextSignalScanTime(0), mNextInputScanTime(0), mNextOutputScanTime(0),
    mStartTime(0),mNextTime(0)
{
}

QtDNTimeKeeperImpl::~QtDNTimeKeeperImpl()
{
}

bool  QtDNTimeKeeperImpl::setIntevalSec(float full, float signal, float input, float output)
{
    mFullScanIntervalTime = full > 0 ? full * 1000 : 1;
    mSignalScanIntervalTime = signal > 0 ? signal * 1000 : 1;
    mInputScanIntervalTime = input > 0 ? input * 1000 : 1;
    mOutputScanIntervalTime = output > 0 ? output * 1000 : 1;

    start();

    return true;
}

int  QtDNTimeKeeperImpl::sleepUntilNextInterval()
{
    qint64 currentTime = getTickTimeMS();
    qint64 sleepTime = mNextTime - currentTime;
    if (sleepTime < 1) sleepTime = 1;
    SleeperThread::msleep(sleepTime);

    mNextTime = currentTime + 1000;
    int state = 0;

    if (mNextFullScanTime <= currentTime)
    {
        state |= DNTimeKeeper::STATE_IN_FULL_SCAN_TIME;
        mNextFullScanTime = currentTime + mFullScanIntervalTime - (currentTime % mFullScanIntervalTime);
    }
    if (mNextFullScanTime < mNextTime)
        mNextTime = mNextFullScanTime;

    if (mNextSignalScanTime <= currentTime)
    {
        state |= DNTimeKeeper::STATE_IN_SIGNAL_SCAN_TIME;
        mNextSignalScanTime = currentTime + mSignalScanIntervalTime - (currentTime % mSignalScanIntervalTime);

    }
    if (mNextSignalScanTime < mNextTime)
        mNextTime = mNextSignalScanTime;

    if (mNextInputScanTime <= currentTime)
    {
        state |= DNTimeKeeper::STATE_IN_INPUT_SCAN_TIME;
        mNextInputScanTime = currentTime + mInputScanIntervalTime - (currentTime % mInputScanIntervalTime);
    }
    if (mNextInputScanTime < mNextTime)
        mNextTime = mNextInputScanTime;

    if (mNextOutputScanTime <= currentTime)
    {
        state |= DNTimeKeeper::STATE_IN_OUTPUT_SCAN_TIME;
        mNextOutputScanTime = currentTime + mOutputScanIntervalTime - (currentTime % mOutputScanIntervalTime);
    }
    if (mNextOutputScanTime < mNextTime)
        mNextTime = mNextOutputScanTime;

    return state;
}

float QtDNTimeKeeperImpl::getTickTime()
{
    return getTickTimeMS() / 1000.0;
}

qint64 QtDNTimeKeeperImpl::getTickTimeMS() const
{
    return QDateTime::currentMSecsSinceEpoch() - mStartTime;
}

bool QtDNTimeKeeperImpl::start()
{
    mStartTime = QDateTime::currentMSecsSinceEpoch();

    mNextFullScanTime = 0;
    mNextSignalScanTime = mSignalScanIntervalTime;
    mNextInputScanTime = 0;
    mNextOutputScanTime = mOutputScanIntervalTime;
    mNextTime = 0;

    return true;
}

bool QtDNTimeKeeperImpl::stop()
{
    mStartTime = 0;
    return true;
}
