//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Nov-3, 2012.
//
#include "dccellscriptseditortabwidget.h"

#include <QSplitter>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QMessageBox>
#include <QMutex>
#include <QMutexLocker>

#include "dccodeeditor.h"
#include "dccell.h"
#include "dccellscriptseditorpagewidget.h"
#include "dccellscriptseditorwindow.h"

DCCellScriptsEditorTabWidget::DCCellScriptsEditorTabWidget(DCCellScriptsEditorWindow *parent) :
    QTabWidget(parent), d_parent(parent)
{
    connect(this, SIGNAL(currentChanged(int)), this, SLOT(slotCurrentIndexChanged(int)));
    connect(this, SIGNAL(tabCloseRequested(int)), this, SLOT(slotTabCloseRequested(int)));
}


DCCellScriptsEditorTabWidget::~DCCellScriptsEditorTabWidget()
{

}

int DCCellScriptsEditorTabWidget::getIndexOf(const DCCell *cell) const
{
    int index = -1;
    for (int i = 0; i < count(); i++)
    {
        QMap<QWidget*, DCCell*>::const_iterator it = d_targets.find(widget(i));
        if (it != d_targets.end())
        {
            if (it.value() == cell)
            {
                index = i;
                break;
            }
        }
    }
    return index;
}

bool DCCellScriptsEditorTabWidget::getIsModified(const DCCell *cell) const
{
    int index = getIndexOf(cell);
    if (index == -1)
        return false;

    return getIsModified(index);
}

bool DCCellScriptsEditorTabWidget::getIsModified(int index) const
{
    DCCellScriptsEditorPageWidget *pageWidget = dynamic_cast<DCCellScriptsEditorPageWidget*>(widget(index));
    return pageWidget->getIsModified();
}

DCCell* DCCellScriptsEditorTabWidget::getCurrentEditCell() const
{
    if (currentIndex() == -1)
        return NULL;

    QMap<QWidget*, DCCell*>::const_iterator it = d_targets.find(currentWidget());
    if (it != d_targets.end())
    {
        return it.value();
    }
    return NULL;
}

DCCell* DCCellScriptsEditorTabWidget::getCellForIndex(int i) const
{
    if (count() < i)
        return NULL;

    QMap<QWidget*, DCCell*>::const_iterator it = d_targets.find(widget(i));
    if (it != d_targets.end())
    {
        return it.value();
    }
    return NULL;
}

void DCCellScriptsEditorTabWidget::tabRemoved(int index)
{
    (void)index;

    QMap<QWidget*, DCCell*>  removedTargets = d_targets;

    for (int i = 0; i < count(); i++)
    {
        removedTargets.take(widget(i));
    }

    QMap<QWidget*, DCCell*>::iterator it = removedTargets.begin();
    while (it != removedTargets.end())
    {
        emit editTabRemoved(it.value());
        d_targets.remove(it.key());
        it.key()->deleteLater();
        ++it;
    }
}

void DCCellScriptsEditorTabWidget::slotCurrentIndexChanged(int index)
{
    DCCellScriptsEditorPageWidget *page = dynamic_cast<DCCellScriptsEditorPageWidget*>(widget(index));
    if (!page)
        return;

    if (page->getIsCustomScriptModifiedByExternalEditor())
    {
        slotTabPageCustomScriptModifiedByExternal(page);
    }

    if (page->getIsCellCodeModifiedByExternalEditor())
    {
        slotTabPageCellCodeModifiedByExternal(page);
    }

    QMap<QWidget*, DCCell*>::iterator it = d_targets.find(page);
    if (it != d_targets.end())
    {
        emit activeEditTabChanged(it.value());
    }
}

void DCCellScriptsEditorTabWidget::slotCellDestroyed(DCCell *cell)
{
    removeTabForCell(cell);
}

void DCCellScriptsEditorTabWidget::slotTabCloseRequested(int index)
{
    QMap<QWidget*, DCCell*>::iterator it = d_targets.find(widget(index));
    if (it != d_targets.end())
    {
        emit tabCloseRequested(it.value());
    }
}

void DCCellScriptsEditorTabWidget::slotTabPageContentModifiedStatusChanged(QWidget *tabPage, bool modified)
{
    int index = indexOf(tabPage);
    if (index == -1)
        return;

    QMap<QWidget*, DCCell*>::iterator it = d_targets.find(tabPage);
    if (it != d_targets.end())
    {
        setTabText(index, getTabTitle(it.value(), tabPage));
        emit editTabContentModifiedStatusChanged(it.value(), modified);
    }
}

void DCCellScriptsEditorTabWidget::slotTabPageCustomScriptModifiedByExternal(QWidget *tabPage)
{
    static QList<QWidget*> pendingPages;
    static QMutex mutex;

    if (currentWidget() == tabPage)
    {
        DCCellScriptsEditorPageWidget *pageWidget = dynamic_cast<DCCellScriptsEditorPageWidget*>(currentWidget());
        if (pageWidget->getIsCustomScriptExternalMode())
        {
            pageWidget->reloadCustomScript();
        }
        else
        {
            {
                QMutexLocker locker(&mutex);
                if (pendingPages.contains(tabPage))
                {
                    return;
                }
                pendingPages.append(tabPage);
            }

            QMessageBox::StandardButton ret;
            ret = QMessageBox::warning(this, tr("Cell scripts editor"),
                tr("The custom script has been modified by other editor.\n"
                "Do you want to reload ?"),
                QMessageBox::Yes | QMessageBox::No);
            if (ret == QMessageBox::Yes)
            {
                pageWidget->reloadCustomScript();
            }
            {
                QMutexLocker locker(&mutex);
                pendingPages.removeOne(tabPage);
            }
        }
    }
}

void DCCellScriptsEditorTabWidget::slotTabPageCellCodeModifiedByExternal(QWidget *tabPage)
{
    static QList<QWidget*> pendingPages;
    static QMutex mutex;

    if (currentWidget() == tabPage)
    {
        DCCellScriptsEditorPageWidget *pageWidget = dynamic_cast<DCCellScriptsEditorPageWidget*>(currentWidget());
        if (pageWidget->getIsCellCodeScriptExternalMode())
        {
            pageWidget->reloadCellCodeScript();
        }
        else
        {
            {
                QMutexLocker locker(&mutex);
                if (pendingPages.contains(tabPage))
                {
                    return;
                }
                pendingPages.append(tabPage);
            }

            QMessageBox::StandardButton ret;
            ret = QMessageBox::warning(this, tr("Cell scripts editor"),
                tr("The cell code script has been modified by other editor.\n"
                "Do you want to reload ?"),
                QMessageBox::Yes | QMessageBox::No);
            if (ret == QMessageBox::Yes)
            {
                pageWidget->reloadCellCodeScript();
            }
            {
                QMutexLocker locker(&mutex);
                pendingPages.removeOne(tabPage);
            }
        }
    }
}

void DCCellScriptsEditorTabWidget::slotTabPageContentSaved(QWidget *tabPage)
{
    int index = indexOf(tabPage);
    if (index == -1)
        return;

    QMap<QWidget*, DCCell*>::iterator it = d_targets.find(tabPage);
    if (it != d_targets.end())
    {
        QString tabTitle = getTabTitle(it.value(), tabPage);
        setTabText(index, tabTitle);

        emit editTabContentModifiedStatusChanged(it.value(), false);
    }
}

void DCCellScriptsEditorTabWidget::setCurrentTabReadOnly(bool readOnly)
{
    if (currentIndex() == -1)
        return;

    DCCellScriptsEditorPageWidget *pageWidget = dynamic_cast<DCCellScriptsEditorPageWidget*>(currentWidget());
    if (pageWidget)
    {
        return pageWidget->setReadOnly(readOnly);
    }
}

QString DCCellScriptsEditorTabWidget::getTabTitle(DCCell *cell, QWidget *tabPageWidget)
{
    QString tabTitle = QString::fromStdString(cell->getName());
    DCCellScriptsEditorPageWidget *w = dynamic_cast<DCCellScriptsEditorPageWidget*>(tabPageWidget);
    if (w && w->getIsModified())
        tabTitle.append(" * ");

    return tabTitle;
}

void DCCellScriptsEditorTabWidget::openTab(DCCell *target, bool focusInCellCodeScriptFirst)
{
    DCCellScriptsEditorPageWidget *targetWidget = NULL;
    int index = -1;
    for (int i = 0; i < count(); i++)
    {
        QMap<QWidget*, DCCell*>::iterator it = d_targets.find(widget(i));
        if (it != d_targets.end())
        {
            if (it.value() == target)
            {
                index = i;
                targetWidget = dynamic_cast<DCCellScriptsEditorPageWidget*>(it.key());
                break;
            }
        }
    }

    if (index >= 0)
    {
        setCurrentIndex(index);
    }
    else
    {
        targetWidget = new DCCellScriptsEditorPageWidget(target, this);
        d_targets.insert(targetWidget, target);

        setCurrentIndex(addTab(targetWidget, getTabTitle(target, targetWidget)));
        connect(targetWidget, SIGNAL(editingCellDestroyed(DCCell*)), this, SLOT(slotCellDestroyed(DCCell*)));
        connect(targetWidget, SIGNAL(cellScriptsModifiedStatusChanged(QWidget*, bool)), this, SLOT(slotTabPageContentModifiedStatusChanged(QWidget*, bool)));
        connect(targetWidget, SIGNAL(cellScriptsSaved(QWidget*)), this, SLOT(slotTabPageContentSaved(QWidget*)));
        connect(targetWidget, SIGNAL(customScriptModifiedByExternalEditor(QWidget*, qint64)), this, SLOT(slotTabPageCustomScriptModifiedByExternal(QWidget*)));
        connect(targetWidget, SIGNAL(cellCodeScriptModifiedByExternalEditor(QWidget*, qint64)), this, SLOT(slotTabPageCellCodeModifiedByExternal(QWidget*)));
    }

    if (targetWidget)
    {
        if (focusInCellCodeScriptFirst)
            targetWidget->focusCellCodeScript();
        else
            targetWidget->focusCustomScript();
    }
}

 QSet<DCVCPage*> DCCellScriptsEditorTabWidget::saveActiveTabContentToFile(bool saveModifiedOnly)
{
     QSet<DCVCPage*> empty;

    if (currentIndex() == -1)
        return empty;

    DCCellScriptsEditorPageWidget *pageWidget = dynamic_cast<DCCellScriptsEditorPageWidget*>(currentWidget());
    if (pageWidget)
    {
        return pageWidget->saveScriptsToFile(saveModifiedOnly);
    }
    return empty;
}

QSet<DCVCPage*> DCCellScriptsEditorTabWidget::saveContentToFile(int tabIndex, bool saveModifiedOnly)
{
    QSet<DCVCPage*> empty;

    if (count() < tabIndex)
        return empty;

    DCCellScriptsEditorPageWidget *pageWidget = dynamic_cast<DCCellScriptsEditorPageWidget*>(widget(tabIndex));
    if (pageWidget)
    {
        return pageWidget->saveScriptsToFile(saveModifiedOnly);
    }
    return empty;
}

bool DCCellScriptsEditorTabWidget::saveCustomScriptOnlyToFile(int tabIndex, bool saveModifiedOnly)
{
    if (count() < tabIndex)
        return false;

    DCCellScriptsEditorPageWidget *pageWidget = dynamic_cast<DCCellScriptsEditorPageWidget*>(widget(tabIndex));
    if (pageWidget)
    {
        return pageWidget->saveCustomScriptOnlyToFile(saveModifiedOnly);
    }
    return false;

}

bool DCCellScriptsEditorTabWidget::saveCellCodeScriptOnlyToFile(int tabIndex, bool saveModifiedOnly)
{
    if (count() < tabIndex)
        return false;

    DCCellScriptsEditorPageWidget *pageWidget = dynamic_cast<DCCellScriptsEditorPageWidget*>(widget(tabIndex));
    if (pageWidget)
    {
        return pageWidget->saveCellCodeScriptOnlyToFile(saveModifiedOnly);
    }
    return false;

}

void DCCellScriptsEditorTabWidget::removeTabForCell(DCCell *cell)
{

    int index = getIndexOf(cell);

    if (index >= 0)
    {
        removeTab(index);
    }
}

void DCCellScriptsEditorTabWidget::removeAllTab()
{
    while(count() > 0)
        removeTab(0);
}
