//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#include "TKLog.h"
#include "DNGlobal.h"
#include "dcconsole.h"

#include "dccreator.h"

#include "dccontent.h"
#include "dccontainer.h"
#include "dcscene.h"

#include "dccell.h"
#include "dcaxon.h"
#include "dcaxonterminal.h"
#include "dcreceptor.h"

#include "command/dccommandevent.h"
#include "command/dccommand.h"

#include "utils/dcdialogutil.h"
#include "utils/dccommandutil.h"
#include "dialog/dcinputreceptornamedialog.h"
#include "codeeditor/dccellscriptseditorwindow.h"

#include <QUndoCommand>
#include <map>

DCCreator::DCCreator(QMainWindow *mainwindow)
    :  d_mainWindow(mainwindow), d_vcontent(NULL), d_scene(NULL),
      d_persMode(DC_PERSMODE_PAGEEDIT), d_contentRootPath("")
{
    d_undoStack = new QUndoStack(this);

    d_console = new DCConsole;
    TKLog::setDestination(d_console);
}

DCCreator::~DCCreator()
{
    emit sceneChanged(this, NULL);
    d_scene = NULL;

    if (d_vcontent)
    {
        delete d_vcontent;
    }
    if (d_console)
    {
        delete d_console;
    }
}

bool DCCreator::event(QEvent *event)
{
    if (!d_scene)
        return false;

    if(event->type() == DCCommandEvent::EVENT_TYPEID)
    {
        DCCommandEvent *commandEvent = dynamic_cast<DCCommandEvent*>(event);
        if (commandEvent)
        {
            QUndoCommand *command = commandEvent->getCommand();
            if (command)
            {
                d_undoStack->push(command);
                emit commandExecuted(command);
            }
            return true;
        }
    }
    else if (event->type() == DCUndoEvent::EVENT_TYPEID)
    {
        d_undoStack->undo();
        emit commandExecuted(NULL);
        return true;
    }
    return false;
}

void DCCreator::resetVisualizer()
{
    emit sceneChanged(this,0);
}

void DCCreator::initMode()
{
    d_persMode = DC_PERSMODE_NAVIGATION;
    d_scene->initMode(DCScene::DCV_PERSMODE_NAVIGATION, DCScene::DCV_EDITMODE_LAYOUT);
}

bool DCCreator::loadContent(const QString &contentRoot)
{
    d_console->clearLog();

    d_contentRootPath = contentRoot;
    if (d_vcontent)
    {
        if (d_scene)
            d_scene->disconnect(this);
        resetVisualizer();
        delete d_vcontent;
    }
    d_vcontent = new DCContent(this, contentRoot.toStdString());

    if (d_vcontent->isValid())
    {
        d_scene = d_vcontent->getContainer()->getScene();

        d_scene->loadSceneAll();

        //setup signal - slot connection for scene
        connect(d_scene, SIGNAL(selectedCellObjectChanged(const void*)), this, SLOT(slotSceneSelectedCellObjectChanged(const void*)));
        connect(d_scene, SIGNAL(selectedPageChanged(const void*)), this, SLOT(slotSceneSelectedPageChanged(const void*)));
        connect(d_scene, SIGNAL(viewAngleChanged(const void*)), this, SLOT(slotSceneViewAngleChanged(const void*)));
        connect(d_scene, SIGNAL(viewCenterChanged(const void*)), this, SLOT(slotSceneViewCenterChanged(const void*)));
        connect(d_scene, SIGNAL(viewScaleChanged(const void*)), this, SLOT(slotSceneViewScaleChanged(const void*)));
        connect(d_scene, SIGNAL(viewSettingChanged(const void*)), this, SLOT(slotSceneViewSettingChanged(const void*)));
        connect(d_scene, SIGNAL(viewEditModeChanged(const void*)), this, SLOT(slotSceneViewEditModeChanged(const void*)));

        emit contentRootPathChanged(this, d_contentRootPath);
        emit sceneChanged(this, d_scene);
        initMode();

        getMainWindow()->statusBar()->showMessage(tr("Content loaded"), 2000);

        return true;
    }
    else
    {
        d_scene = NULL;
        emit contentRootPathChanged(this, d_contentRootPath);
        emit sceneChanged(this, NULL);

        if (!dnGlobal()->isErrorStatusNormal())
        {
            QMessageBox msgBox(QMessageBox::Warning, QString::fromStdString(dnGlobal()->getMessage1()), QString::fromStdString(dnGlobal()->getMessage2()));
            msgBox.setInformativeText(d_console->getLog(10));
            msgBox.exec();
        }
        return false;
    }
}

bool DCCreator::savePage(DCVCPage *page, bool showResultInMessageBox)
{
    bool r = false;
    if (d_scene && d_vcontent)
    {
        DCCellScriptsEditorWindow::saveScriptsBelongTo(page, true);
        r = d_vcontent->saveForPage(d_vcontent->getContentRootPath(), page);
    }
    if (showResultInMessageBox)
    {
        QMessageBox msgBox;
        if (r)
        {
            msgBox.setText(tr("Page is saved successfully"));
        }
        else
        {
            msgBox.setText(tr("Error!! Failed to save the page file"));
        }
        msgBox.exec();
    }
    return r;
}

bool DCCreator::savePage(const QSet<DCVCPage*> &pages, bool showResultInMessageBox)
{
    bool r = true;
    QSet<DCVCPage*>::const_iterator i = pages.constBegin();
     while (i != pages.constEnd())
     {
         if (!savePage(*i, showResultInMessageBox))
         {
             r = false;
        }
         ++i;
     }
     return r;
}

bool DCCreator::saveAll(bool showResultInMessageBox)
{
    bool r = false;

    if (DCCellScriptsEditorWindow::getIsVisible())
    {
        DCCellScriptsEditorWindow::saveScriptsAll(true);
    }

    if (d_scene && d_vcontent)
    {
        r = d_vcontent->saveAll(d_vcontent->getContentRootPath());
    }
    if (showResultInMessageBox)
    {
        QMessageBox msgBox;
        if (r)
        {
            msgBox.setText(tr("Content is saved successfully"));
        }
        else
        {
            msgBox.setText(tr("Error!! Failed to save content files"));
        }
        msgBox.exec();
    }
    getMainWindow()->statusBar()->showMessage(tr("Content saved"), 2000);

    return r;
}


void DCCreator::selectPage(const void *requester, DCVCPage *page, bool multipleSelection)
{
    if (d_scene)
    {
        d_scene->selectPage(requester, page, multipleSelection);
    }
}

void DCCreator::unselectPage(const void *requester, DCVCPage *page)
{
    if (d_scene)
    {
        d_scene->unselectPage(requester, page);
    }
}

bool DCCreator::selectPage(const void *requester, const QString &locationPath, bool multipleSelection)
{
    bool r = false;
    if (d_scene)
    {
        r = d_scene->selectPage(requester, locationPath, multipleSelection);
    }
    return r;
}

bool DCCreator::unselectPage(const void *requester, const QString &locationPath)
{
    bool r = false;
    if (d_scene)
    {
        r = d_scene->unselectPage(requester, locationPath);
    }
    return r;
}


void DCCreator::unselectPageAll(const void *requester)
{
    if (d_scene)
    {
        d_scene->unselectPageAll(requester);
    }
}

void DCCreator::selectCellObject(const void *requester, DCVComponent *object, bool multipleSelection)
{
    if (d_scene)
        d_scene->selectCellObject(requester, object, multipleSelection);
}

void DCCreator::unselectCellObject(const void *requester, DCVComponent *object)
{
    if (d_scene)
        d_scene->unselectCellObject(requester, object);
}

void DCCreator::unselectCellObjectAll(const void *requester)
{
    if (d_scene)
        d_scene->unselectCellObjectAll(requester);
}

void DCCreator::changeSceneScale(const void *requester, float scale)
{
    if (d_scene)
    {
        if (scale <= 10) scale =10;
        if (scale >= 1000) scale = 1000;
        d_scene->setScale(requester, scale);
    }
}

void DCCreator::rotateScene(const void *requester, float xangle_delta, float yangle_delta)
{
    if (d_scene)
    {
        float xangle = d_scene->getXAngle() + xangle_delta;
        if (xangle < 0) xangle = 0;
        if (xangle > 90) xangle = 90;

        float yangle = d_scene->getYAngle() + yangle_delta;
        while (yangle < 0) yangle += 360;
        while (yangle > 360) yangle -= 360;

        d_scene->setAngle(requester, xangle,yangle);
    }
}

void DCCreator::translateBrowsModeScene(const void *requester, float dx, float dy)
{
    if (d_scene)
    {
        float scale = d_scene->getScale();
        float x = d_scene->getCenterX() - dx/scale*2;
        float y = d_scene->getCenterBrowsModeY() + dy/scale*2;
        d_scene->setBrowsModeCenter(requester, x,y);
    }
}

void DCCreator::translateEditModeScene(const void *requester, float dx, float dy)
{
    if (d_scene)
    {
        float scale = d_scene->getScale();
        float x = d_scene->getCenterX() - dx/scale*2;
        float y = d_scene->getCenterEditModeY() - dy/scale*2;
        d_scene->setPageModeCenter(requester, x,y);
    }
}

void DCCreator::resetSceneTranslation()
{

}

bool DCCreator::changePersMode(const void *requester, DCPersMode mode)
{
    bool r = false;
    if (d_scene)
    {
        d_persMode = mode;
        switch(mode)
        {
        case DC_PERSMODE_NAVIGATION:
            r = d_scene->changePersMode(requester, DCScene::DCV_PERSMODE_NAVIGATION);
            break;
        case DC_PERSMODE_PAGEEDIT:
            r = d_scene->changePersMode(requester, DCScene::DCV_PERSMODE_PAGEEDIT);
            break;
        }
        d_scene->updateVisiblity();
    }
    return r;
}

void DCCreator::doUndo(const void *requester)
{
    (void)requester;
    //TODO
}

void DCCreator::doUndoImmidiate()
{
    d_undoStack->undo();
}

void DCCreator::doCommandStartAddAxonTerminalFromAxon(const void *requester, DCCell *axonCell)
{
    DCCommandUtil::postStartAddAxonTerminalCommandFromAxon(requester, this, axonCell);
}

void DCCreator::doCommandStartAddAxonTerminalFromReceptor(const void *requester, DCCell *receptorCell)
{
    DCInputReceptorNameDialog dialog(receptorCell);
    dialog.exec();
    if (dialog.getIsOk() && dialog.getInputText().length() > 0)
    {
        DCCommandUtil::postStartAddAxonTerminalCommandFromReceptor(requester, this, receptorCell, dialog.getInputText());
    }
}

void DCCreator::doCommandCommitAddAxonTerminal(const void *requester, DCAxon *axon, DCCell *receptorCell)
{
    DCInputReceptorNameDialog dialog(receptorCell);
    dialog.exec();
    if (dialog.getIsOk() && dialog.getInputText().length() > 0)
    {
        DCCommandUtil::postCommitAddAxonTerminalCommand(requester, this, axon, receptorCell, dialog.getInputText());
    }
    else
    {
        doCommandCancelAddAxonTerminal(requester);
    }
}

void DCCreator::doCommandCommitAddAxonTerminal(const void *requester, DCCell *axonCell, DCReceptor *receptor)
{
    DCCommandUtil::postCommitAddAxonTerminalCommand(requester, this, axonCell, receptor);
}

void DCCreator::doCommandCancelAddAxonTerminal(const void *requester)
{
//TODO:
    // may need to consider the case multiple commands are executed after
    // startAddAxon command is executed.
    DCCommandUtil::postUndoRequestCommand(requester, this);
}

void DCCreator::doCommandRemoveAxonTerminal(const void *requester, DCCell *axonCell, DCAxonTerminal *axonTerminal)
{
    DCReceptor *receptor = axonTerminal->getTarget();
    if (!receptor)
        return;

    DCCell *targetCell = receptor->getOwnerCell();
    if (!targetCell)
        return;

    QMessageBox msgBox;
    msgBox.setText(tr("Delete axon terminal"));
    QString msg;
    msg.append("Cell ");
    msg.append(QString::fromStdString(axonCell->getName()));
    msg.append("\nDo you want to delete the connection to ");
    msg.append(QString::fromStdString(targetCell->getLocation()));
    msg.append("#");
    msg.append(QString::fromStdString(targetCell->getName()));
    msg.append("(");
    msg.append(QString::fromStdString(targetCell->getReceptorName(receptor)));
    msg.append(")?");
    msgBox.setInformativeText(msg);
    msgBox.setStandardButtons(QMessageBox::Cancel | QMessageBox::Yes);
    msgBox.setDefaultButton(QMessageBox::Cancel);
    int ret = msgBox.exec();

    if (ret == QMessageBox::Yes)
    {
        DCCommandUtil::postRemoveAxonTerminalCommand(requester, this, axonCell, axonTerminal);
    }
}

void DCCreator::doCommandRemoveAxonTerminal(const void *requester, DCCell *receptorCell, const QString& receptorName)
{

    QMessageBox msgBox;
    msgBox.setText(tr("Delete receptor"));
    QString msg;
    msg.append("Cell ");
    msg.append(QString::fromStdString(receptorCell->getName()));
    msg.append("\nDo you want to delete receptor ");
    msg.append(receptorName);
    msg.append("?\n");
    msgBox.setInformativeText(msg);
    msgBox.setStandardButtons(QMessageBox::Cancel | QMessageBox::Yes);
    msgBox.setDefaultButton(QMessageBox::Cancel);
    int ret = msgBox.exec();

    if (ret == QMessageBox::Yes)
    {
        DCCommandUtil::postRemoveAxonTerminalCommand(requester, this, receptorCell, receptorName);
    }
}

void DCCreator::doCommandAssignCellCodeClassToCell(const void *requester, DCCell *cell, DCCellCode *cellCode)
{
    DCCommandUtil::postAssignCellCodeClassToCellCommand(requester, this, cell, cellCode);
}

void DCCreator::doCommandUnassignCellCodeClassFromCell(const void *requester, DCCell *cell)
{
    DCCommandUtil::postUnassignCellCodeClassFromCellCommand(requester, this, cell);
}

void DCCreator::doCommandAddCellCodeClass(const void *requester, DCContainer *container, const QString& name, const QString& type)
{
    DCCommandUtil::postAddCellCodeClassCommand(requester, this, container, name, type);
}

void DCCreator::doCommandChangeCellCodeClassType(const void *requester, DCCellCode* cellCode, const QString &newType)
{
    DCCommandUtil::postChangeCellCodeClassTypeCommand(requester, this, cellCode, newType);
}

void DCCreator::doCommandAddCell(const void *requester, DCContainer *container, const QString &containerBasedPath, const QString &name, const QString &type, float pageX, float pageY)
{
    DCCommandUtil::postAddCellCommand(requester, this, container, containerBasedPath, name, type, pageX, pageY);
}

void DCCreator::doCommandChangeCellType(const void *requester, DCCell *cell, const QString &newType)
{
    DCCommandUtil::postChangeCellTypeCommand(requester, this, cell, newType);
}

void DCCreator::doCommandRenameCell(const void *requester, DCCell *cell, const QString &newContainerBasedPath, const QString &newName)
{
    DCCommandUtil::postRenameCellCommand(requester, this, cell, newContainerBasedPath, newName);
}

void DCCreator::doCommandRemoveCell(const void *requester, DCContainer *container, DCCell *cell)
{
    DCCommandUtil::postRemoveCellCommand(requester, this, container, cell);
}

void DCCreator::doCommandRemoveCellCode(const void *requester, DCContainer *container, DCCellCode *cellCode)
{
    DCCommandUtil::postRemoveCellCodeCommand(requester, this, container, cellCode);
}

void DCCreator::doCommandAddPage(const void *requester, const QString &containerBasedPath)
{
    DCCommandUtil::postAddPageCommand(requester, this, containerBasedPath);
}

void DCCreator::doCommandMovePage(const void *requester, const QString &oldContainerBasedPath, const QString &newContainerBasedPath)
{
    DCCommandUtil::postMovePageCommand(requester, this, oldContainerBasedPath, newContainerBasedPath);
}

void DCCreator::doCommandRemovePage(const void *requester, DCVCPage *page)
{
    DCCommandUtil::postRemovePageCommand(requester, this, page);
}

void DCCreator::doCommandAddDirectory(const void *requester, const QString &sysFilePath)
{
    DCCommandUtil::postAddDirectoryCommand(requester, this, sysFilePath);
}

void DCCreator::doCommandRenameDirectory(const void *requester, const QString &oldSysFilePath, const QString &newSysFilePath)
{
    DCCommandUtil::postRenameDirectoryCommand(requester, this, oldSysFilePath, newSysFilePath);
}

void DCCreator::doCommandRemoveDirectory(const void *requester, const QString &sysFilePath)
{
    DCCommandUtil::postRemoveDirectoryCommand(requester, this, sysFilePath);
}

bool DCCreator::doCommandRenameReceptorName(const void *requester, DCCell *cell, const QString &oldName, const QString &newName, bool doImmediate)
{
    if (doImmediate)
    {
        DCCommand *command = DCCommandUtil::createRenameReceptorNameCommand(requester, this, cell, oldName, newName);
        d_undoStack->push(command);
        return command->getCommandResult() == DCCommand::DCCOMMAND_SUCCEEDED;
    }
    else
    {
        DCCommandUtil::postRenameReceptorNameCommand(requester, this, cell, oldName, newName);
        return true;
    }
}

DCContainer* DCCreator::getCurrentContainer() const
{
    if (d_vcontent)
        return d_vcontent->getContainer();
    else
        return NULL;
}

void DCCreator::slotSceneSelectedPageChanged(const void *requester)
{
    emit sceneSelectedPageChanged(requester, d_scene);
}

void DCCreator::slotSceneSelectedCellObjectChanged(const void *requester)
{
    emit sceneSelectedCellChanged(requester, d_scene);
}

void DCCreator::slotSceneViewScaleChanged(const void *requester)
{
    emit sceneViewScaleChanged(requester, d_scene);
}

void DCCreator::slotSceneViewAngleChanged(const void *requester)
{
    emit sceneViewAngleChanged(requester, d_scene);
}

void DCCreator::slotSceneViewCenterChanged(const void *requester)
{
    emit sceneViewCenterChanged(requester, d_scene);
}

void DCCreator::slotSceneViewSettingChanged(const void *requester)
{
    emit sceneViewSettingChanged(requester, d_scene);
}

void DCCreator::slotSceneViewEditModeChanged(const void *requester)
{
    emit sceneViewEditModeChanged(requester, d_scene);
}

