//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on 12/11/2011.
//

#include "TKCell.h"
#include "TKReceptor.h"
#include "TKContainer.h"
#include "TKCellCode.h"
#include "TKAxon.h"
#include "TKCellCodeInstance.h"
#include "TKLog.h"
#include <string>

TKCell::TKCell(TKContainer *container, std::string location, std::string name, bool canIntarfaceIn, bool canInterfaceOut)
: mName(name), mLocation(location), mContainer(container), mAxon(NULL), mCellCodeInstance(NULL),
  mCanInterfaceIn(canIntarfaceIn), mCanInterfaceOut(canInterfaceOut), mReceptorValueUpdated(false)
{
}

TKCell::~TKCell()
{
    if (mCellCodeInstance)
    {
        delete mCellCodeInstance;
        mCellCodeInstance = NULL;
    }

    for ( TKReceptorMap::iterator it = mReceptors.begin(); it != mReceptors.end(); ++it ) {
        delete it->second;
    }
    mReceptors.clear();

    if (mAxon)
    {
        delete mAxon;
        mAxon = NULL;
    }
}

void TKCell::init()
{
    if (!mAxon)
        mAxon = mContainer->axonFactory(this);
}

float TKCell::getAxonValue() const
{
    if (!mAxon)
        return 0.0;
    return mAxon->getValue();
}

void  TKCell::setAxonValue(float value)
{
    if (!mAxon)
        return;
    mAxon->setValue(value);
}

bool TKCell::setCellCode(TKCellCode *code, const void *data)
{
    if (mCellCodeInstance)
        delete mCellCodeInstance;

    mCellCodeInstance = code->createCellCodeInstance(this, data);
    return mCellCodeInstance != NULL;
}

bool TKCell::connectTo(std::string connectionName, TKCell *targetCell)
{
    TKReceptor *receptor = targetCell->createReceptor(connectionName);
    if (receptor)
    {
        TKAxonTerminal *terminal = mAxon->addTerminal();
        receptor->setTarget(terminal);

        return true;
    }
    else
    {
        return false;
    }
}

TKReceptor* TKCell::createReceptor(std::string name)
{
    TKReceptor *receptor;
    TKReceptorMap::iterator it = mReceptors.find(name);
    if (it != mReceptors.end())
    {
        receptor = NULL;
    }
    else
    {
        receptor = mContainer->receptorFactory(this);
        if (receptor)
        {
            mReceptors.insert( TKReceptorMap::value_type( name, receptor ) );
        }
    }

    return receptor;
}

bool TKCell::enterDoTick(float time, bool forceUpdate)
{
    if (!forceUpdate && !mReceptorValueUpdated)
        return false;

    bool r = doTick(time);
    mReceptorValueUpdated = false;

    return r;
}

void TKCell::updateReceptorValue()
{
    for ( TKReceptorMap::iterator it = mReceptors.begin(); it != mReceptors.end(); ++it )
    {
        if (it->second->update())
        {
            mReceptorValueUpdated = true;
        }
    }
}
