//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on 2/25/2012.
//
#include "qtdnxmlimpl.h"

#include "DNXMLElement.h"
#include "TKLog.h"
#include "DNGlobal.h"
#include "DNUtils.h"

DNXMLImpl *DNXMLImpl::createXMLFromFileImpl(const char *docRoot, const char *docPath)
{
    QString qDocRoot = QString::fromLocal8Bit(docRoot);
    QString qDocPath = QString::fromLocal8Bit(docPath);

    if (qDocRoot.lastIndexOf("/") != qDocRoot.length()-1 && qDocPath.at(0) != QChar('/'))
    {
        qDocRoot.append("/");
    }
    QString filePath = qDocRoot;
    filePath.append(qDocPath);

    return new QtDNXMLImpl(filePath);
}

QtDNXMLImpl::QtDNXMLImpl(QString filePath) : mHandler(NULL), mValid(false), mFilePath(filePath)
{
    QFile file(filePath);
    if (file.open(QFile::ReadOnly | QFile::Text)) {
        QXmlInputSource xmlInputSource(&file);
        mHandler = new QtXMLHandler(filePath);
        QXmlSimpleReader reader;
        reader.setContentHandler(mHandler);
        reader.setErrorHandler(mHandler);

        mValid = reader.parse(xmlInputSource);
    }

    if (!mValid)
    {
        std::string message = "failed to load file:";
        message.append(filePath.toStdString());
        dnNotifyError("Error at loading XML file", message);
    }
}

QtDNXMLImpl::~QtDNXMLImpl()
{
    if (mHandler)
        delete mHandler;
}

DNXMLElement* QtDNXMLImpl::getRoot()
{
    if (mHandler && mValid)
    {
        return mHandler->getRoot();
    }
    else
    {
        return NULL;
    }
}

QtXMLHandler::QtXMLHandler(QString filePath) : mRootElement(NULL), mCurrentElement(NULL), mFilePath(filePath)
{
}

QtXMLHandler::~QtXMLHandler()
{
    if (mRootElement)
    {
        delete mRootElement;
    }
}

//QXML handler implementation
bool QtXMLHandler::startDocument()
{
    return true;
}

bool QtXMLHandler::endDocument()
{
    return true;
}

bool QtXMLHandler::startElement(const QString & namespaceURI, const QString & localName, const QString & qName, const QXmlAttributes & atts )
{
    (void)namespaceURI; (void)localName;

    if (mRootElement  == NULL)
    {
        mRootElement = new DNXMLElement(qName.toStdString());
        mCurrentElement = mRootElement;
    }
    else
    {
        if (!mCurrentElement->inner)
        {
            DNXMLElement *outer = mCurrentElement;
            mCurrentElement->inner = new DNXMLElement(qName.toStdString());
            mCurrentElement = mCurrentElement->inner;
            mCurrentElement->outer = outer;
            mCurrentElement->depth = outer->depth + 1;
        }
        else
        {
            DNXMLElement *prev = mCurrentElement->inner;
            while(prev->next) prev = prev->next;
            prev->next = new DNXMLElement(qName.toStdString());
            mCurrentElement = prev->next;
            mCurrentElement->outer = prev->outer;
            mCurrentElement->depth = prev->depth;
        }
    }

    if (mCurrentElement)
    {
        for(int i = 0; i < atts.count(); i++)
        {
            QString qname = atts.qName(i);
            QString value = atts.value(i);
            mCurrentElement->setAttribute(qname.toStdString(), value.toStdString());
        }

    }
    return true;
}

bool QtXMLHandler::endElement (const QString & namespaceURI, const QString & localName, const QString & qName )
{
    (void)namespaceURI; (void)localName; (void)qName;

    mCurrentElement = mCurrentElement->outer;
    return true;
}

bool QtXMLHandler::characters (const QString & ch )
{
    mCurrentElement->text.append(ch.toStdString());
    return true;
}

bool QtXMLHandler::error (const QXmlParseException & exception )
{
    std::string message = exception.message().toStdString();
    TKLog::printf(TKLog::WARNING, "Error while reading XML file!! \n%s", message.c_str());
    return true;
}

bool QtXMLHandler::fatalError(const QXmlParseException & exception )
{
    QString message = QString().sprintf("ERROR while reading XML file!! file.%s (line:%d) \nReason:%s",
                                      mFilePath.toLocal8Bit().data(), exception.lineNumber(), exception.message().toLocal8Bit().data());
    dnNotifyError("Error at loading XML file", message.toStdString());
    return true;
}


