//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#include "TKLog.h"

#include "mainwindow.h"
#include "ui_mainwindow.h"

#include "dccreator.h"
#include "dccontent.h"
#include "dcglvisualizerwidget.h"
#include "dctreeviewwidget.h"
#include "dccellcodescripttreeviewwidget.h"
#include "TKConsole.h"
#include "utils/dcresources.h"
#include "utils/dcskeltoncreatorutil.h"
#include "dialog/dcinputnewcontentdirdialog.h"
#include "dialog/dcmanagecellcodedialog.h"
#include "dialog/dccodeeditorexternaleditorsettingdialog.h"
#include "dccellscriptseditorwindow.h"

#include <QCloseEvent>
#include <QFileDialog>
#include <QMessageBox>
#include <QSettings>
#include <QLocalSocket>

#if defined(Q_WS_WIN)
    const QString DENNCO_ENGINE = "QtDennco.exe";
#elif defined(Q_WS_MAC)
    const QString DENNCO_ENGINE = "QtDennco.app";
#elif defined(Q_OS_UNIX)
    const QString DENNCO_ENGINE = "./QtDennco";
#endif

class SleeperThread : public QThread
{
public:
    static void msleep(unsigned long msecs)
    {
        QThread::msleep(msecs);
    }
};

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);

    DCResources::initResources();

    createActions();
    createMenus();
    createToolBars();
    createStatusBar();

    readSettings();

    TKLog::setDestination(new TKConsole);

    d_creator = new DCCreator(this);
    d_visualizerWidget = new DCGLVisualizerWidget(d_creator, this) ;
    setCentralWidget(d_visualizerWidget);

    d_treeViewWidget = new DCTreeViewWidget(this, d_creator);
    ui->treeViewDock->layout()->addWidget(d_treeViewWidget);

    d_cellCodeScriptTreeViewWidget = new DCCellCodeScriptTreeViewWidget(this, d_creator);
    ui->cellCodeScriptTreeViewDock->layout()->addWidget(d_cellCodeScriptTreeViewWidget);

    DCCellScriptsEditorWindow::construct(d_creator);

    setCurrentContent("");
    setUnifiedTitleAndToolBarOnMac(true);

    tabifyDockWidget(ui->dock2,ui->dock1);
}

MainWindow::~MainWindow()
{
    delete ui;
    if (d_creator)
        delete d_creator;
}


void MainWindow::closeEvent(QCloseEvent *event)
{
    if (maybeSave()) {
        writeSettings();
        event->accept();
    } else {
        event->ignore();
    }
}

void MainWindow::newFile()
{
    QString bdir = QDir::home().absolutePath();
    if (d_contentOpenHistory.length()>0)
    {
        QDir d(d_contentOpenHistory.at(0));
        d.cdUp();
        bdir = d.absolutePath();
    }
    DCInputNewContentDirDialog dialog(bdir);
    dialog.exec();

    QString dirName = dialog.getResult();
    if (dirName.length() > 0 && DCSkeltonCreatorUtil::createNewContent(dirName))
    {
        loadContent(dirName);
    }
}

void MainWindow::open()
{
    if (maybeSave())
    {
        QString dir = QDir::home().absolutePath();
        if (d_contentOpenHistory.length()>0)
        {
            QDir d(d_contentOpenHistory.at(0));
            d.cdUp();
            dir = d.absolutePath();
        }
        QString dirName = QFileDialog::getExistingDirectory(this, tr("Open a content"),dir);
        if (!dirName.isEmpty())
            loadContent(dirName);
    }
}

bool MainWindow::openRecent(int idx)
{
    if (maybeSave())
    {
        if (d_contentOpenHistory.length() > idx)
        {
            loadContent(QString(d_contentOpenHistory.at(idx)));
            return true;
        }
    }
    return false;
}

void MainWindow::openRecent1()
{
    openRecent(0);
}

void MainWindow::openRecent2()
{
    openRecent(1);
}

void MainWindow::openRecent3()
{
    openRecent(2);
}

void MainWindow::openRecent4()
{
    openRecent(3);
}

void MainWindow::openRecent5()
{
    openRecent(4);
}

bool MainWindow::save(bool showMessage)
{
    if (!d_creator)
        return false;

    return d_creator->saveAll(showMessage);
}

bool MainWindow::save()
{
    return save(true);
}

bool MainWindow::saveAs()
{
    //TODO
    return false;
}

void MainWindow::doExternalEditorSetting()
{
    DCCodeEditorExternalEditorSettingDialog dialog(this);
    dialog.exec();
}

void MainWindow::about()
{
   QMessageBox::about(this, tr("About Application"),
            tr("TODO: exampalin@about <b>dennco creator</b>  "
               "TODO: exampalin@about <b>dennco creator</b> "
               "TODO: exampalin@about <b>dennco creator</b>"));
}

void MainWindow::documentWasModified()
//! [15] //! [16]
{
//    setWindowModified(textEdit->document()->isModified());
}
//! [16]

void MainWindow::createActions()
{
    newAct = new QAction(QIcon(":/images/new.png"), tr("&New"), this);
    newAct->setShortcuts(QKeySequence::New);
    newAct->setStatusTip(tr("Create a new file"));
    connect(newAct, SIGNAL(triggered()), this, SLOT(newFile()));

    openAct = new QAction(QIcon(":/images/open.png"), tr("&Open..."), this);
    openAct->setShortcuts(QKeySequence::Open);
    openAct->setStatusTip(tr("Open an existing file"));
    connect(openAct, SIGNAL(triggered()), this, SLOT(open()));

    QAction *openRecentAct = new QAction(this);
    connect(openRecentAct, SIGNAL(triggered()), this, SLOT(openRecent1()));
    openRecentActs.append(openRecentAct);

    openRecentAct = new QAction(this);
    connect(openRecentAct, SIGNAL(triggered()), this, SLOT(openRecent2()));
    openRecentActs.append(openRecentAct);

    openRecentAct = new QAction(this);
    connect(openRecentAct, SIGNAL(triggered()), this, SLOT(openRecent3()));
    openRecentActs.append(openRecentAct);

    openRecentAct = new QAction(this);
    connect(openRecentAct, SIGNAL(triggered()), this, SLOT(openRecent4()));
    openRecentActs.append(openRecentAct);

    openRecentAct = new QAction(this);
    connect(openRecentAct, SIGNAL(triggered()), this, SLOT(openRecent5()));
    openRecentActs.append(openRecentAct);

    saveAct = new QAction(QIcon(":/images/save.png"), tr("&Save"), this);
    saveAct->setShortcuts(QKeySequence::Save);
    saveAct->setStatusTip(tr("Save the document to disk"));
    connect(saveAct, SIGNAL(triggered()), this, SLOT(save()));

    saveAsAct = new QAction(tr("Save &As..."), this);
    saveAsAct->setShortcuts(QKeySequence::SaveAs);
    saveAsAct->setStatusTip(tr("Save the document under a new name"));
    connect(saveAsAct, SIGNAL(triggered()), this, SLOT(saveAs()));

    exitAct = new QAction(tr("E&xit"), this);
    exitAct->setShortcuts(QKeySequence::Quit);
    exitAct->setStatusTip(tr("Exit the application"));
    connect(exitAct, SIGNAL(triggered()), this, SLOT(close()));

    manageCellCodeAct = new QAction(tr("Manage Cell Code..."), this);
    connect(manageCellCodeAct, SIGNAL(triggered()), this, SLOT(manageCellCode()));

    cutAct = new QAction(QIcon(":/images/cut.png"), tr("Cu&t"), this);
    cutAct->setShortcuts(QKeySequence::Cut);
    cutAct->setStatusTip(tr("Cut the current selection's contents to the "
                            "clipboard"));
//    connect(cutAct, SIGNAL(triggered()), textEdit, SLOT(cut()));

    copyAct = new QAction(QIcon(":/images/copy.png"), tr("&Copy"), this);
    copyAct->setShortcuts(QKeySequence::Copy);
    copyAct->setStatusTip(tr("Copy the current selection's contents to the "
                             "clipboard"));
//    connect(copyAct, SIGNAL(triggered()), textEdit, SLOT(copy()));

    pasteAct = new QAction(QIcon(":/images/paste.png"), tr("&Paste"), this);
    pasteAct->setShortcuts(QKeySequence::Paste);
    pasteAct->setStatusTip(tr("Paste the clipboard's contents into the current "
                              "selection"));

    //    connect(pasteAct, SIGNAL(triggered()), textEdit, SLOT(paste()));

    playAct = new QAction(QIcon(":/images/playbutton.png"), tr("Play"), this);
    playAct->setStatusTip(tr("Play the editing content"));

    connect(playAct, SIGNAL(triggered()), this, SLOT(playContent()));

    externalEditorSettingAct = new QAction(tr("&External editor..."), this);
    externalEditorSettingAct->setStatusTip(tr("Setup properties for external code editor."));
    connect(externalEditorSettingAct, SIGNAL(triggered()), this, SLOT(doExternalEditorSetting()));

    aboutAct = new QAction(tr("&About"), this);
    aboutAct->setStatusTip(tr("Show the application's About box"));
    connect(aboutAct, SIGNAL(triggered()), this, SLOT(about()));

    cutAct->setEnabled(false);
    copyAct->setEnabled(false);
    playAct->setEnabled(false);
//    connect(textEdit, SIGNAL(copyAvailable(bool)),
//            cutAct, SLOT(setEnabled(bool)));
//    connect(textEdit, SIGNAL(copyAvailable(bool)),
//            copyAct, SLOT(setEnabled(bool)));
}

void MainWindow::createMenus()
{
    fileMenu = menuBar()->addMenu(tr("&File"));
    fileMenu->addAction(newAct);
    fileMenu->addAction(openAct);
    openRecentMenu = fileMenu->addMenu(tr("Opened recently"));
    fileMenu->addAction(saveAct);
    fileMenu->addAction(saveAsAct);
    fileMenu->addSeparator();
    fileMenu->addAction(exitAct);

    editMenu = menuBar()->addMenu(tr("&Edit"));
    editMenu->addAction(manageCellCodeAct);
    editMenu->addSeparator();
    editMenu->addAction(cutAct);
    editMenu->addAction(copyAct);
    editMenu->addAction(pasteAct);

    menuBar()->addSeparator();

    editMenu = menuBar()->addMenu(tr("&Play"));
    editMenu->addAction(playAct);

    menuBar()->addSeparator();

    settingMenu = menuBar()->addMenu(tr("&Setting"));
    settingMenu->addAction(externalEditorSettingAct);
    helpMenu = menuBar()->addMenu(tr("&Help"));
    helpMenu->addAction(aboutAct);
}

void MainWindow::createOpenRecentMenu()
{
    openRecentMenu->clear();
    for (int i = 0; i < d_contentOpenHistory.length() && i < openRecentActs.length(); i++)
    {
        openRecentActs.at(i)->setText(d_contentOpenHistory.at(i));
        openRecentMenu->addAction(openRecentActs.at(i));
    }
}

void MainWindow::createToolBars()
{
    fileToolBar = addToolBar(tr("File"));
    fileToolBar->addAction(newAct);
    fileToolBar->addAction(openAct);
    fileToolBar->addAction(saveAct);

    editToolBar = addToolBar(tr("Edit"));
    editToolBar->addAction(cutAct);
    editToolBar->addAction(copyAct);
    editToolBar->addAction(pasteAct);

    playToolBar = addToolBar(tr("Play"));
    playToolBar->addAction(playAct);
}

void MainWindow::createStatusBar()
//! [32] //! [33]
{
    statusBar()->showMessage(tr("Ready"));
}
//! [33]

void MainWindow::readSettings()
{
    QSettings settings("dennco project", "dennco creator");
    QPoint pos = settings.value("pos", QPoint(200, 200)).toPoint();
    QSize size = settings.value("size", QSize(400, 400)).toSize();
    d_contentOpenHistory = settings.value("contents", QStringList()).toStringList();

    resize(size);
    move(pos);
    createOpenRecentMenu();
}

void MainWindow::writeSettings()
{
    QSettings settings("dennco project", "dennco creator");
    settings.setValue("pos", pos());
    settings.setValue("size", size());
    settings.setValue("contents", d_contentOpenHistory);
}

//static
QByteArray MainWindow::readSettingsForCellScriptsEditorGeometory()
{
    QSettings settings("dennco project", "dennco creator");
    return settings.value("segeometory").toByteArray();
}

//static
void MainWindow::writeSettingsForCellScriptsEditorGeometory(const QByteArray &value)
{
    QSettings settings("dennco project", "dennco creator");
    settings.setValue("segeometory", value);
}

//static
QString MainWindow::readSettingsForExternalScriptEditorPath()
{
    QSettings settings("dennco project", "dennco creator");
    return settings.value("external editor path").toString();
}

//static
QString MainWindow::readSettingsForExternalScriptEditorParameters()
{
    QSettings settings("dennco project", "dennco creator");
    return settings.value("external editor param").toString();
}

//static
void MainWindow::writeSettingsForExternalScriptEditorPath(const QString &arg)
{
    QSettings settings("dennco project", "dennco creator");
    settings.setValue("external editor path", arg);
}

//static
void MainWindow::writeSettingsForExternalScriptEditorParameters(const QString &arg)
{
    QSettings settings("dennco project", "dennco creator");
    settings.setValue("external editor param", arg);
}

bool MainWindow::maybeSave()
{
    if (d_creator->getCurrentContent())
    {
        if (!DCCellScriptsEditorWindow::closeEditor())
            return false;

        if (d_creator->getCurrentContent()->getIsModified())
        {
            QMessageBox::StandardButton ret;
            ret = QMessageBox::warning(this, tr("Application"),
                tr("The content has been modified.\n"
                "Do you want to save your changes?"),
                QMessageBox::Save | QMessageBox::Discard | QMessageBox::Cancel);
            if (ret == QMessageBox::Save)
                return save(false);
            else if (ret == QMessageBox::Cancel)
                return false;
        }
    }
    return true;
}

void MainWindow::loadContent(const QString &contentDirectory)
{
    if (!d_creator)
        return;

    if (d_creator->loadContent(contentDirectory))
    {
        playAct->setEnabled(true);
    }
    else
    {
        playAct->setEnabled(false);
    }

    QString absContentDirectory = QDir().absoluteFilePath(contentDirectory);
    int idx = d_contentOpenHistory.indexOf(absContentDirectory);
    if (idx != -1)
    {
        d_contentOpenHistory.removeAt(idx);
    }
    d_contentOpenHistory.prepend(absContentDirectory);
    while (d_contentOpenHistory.length()>5)
    {
        d_contentOpenHistory.removeLast();
    }
    setCurrentContent(contentDirectory);
    createOpenRecentMenu();

    QDir dir(contentDirectory);
    QString title = dir.dirName();
    title.append(" - dennco creator");
    setWindowTitle(title);
}

void MainWindow::setCurrentContent(const QString &contentDirectory)
{
    curContent = contentDirectory;
    setWindowModified(false);

    QString shownName = curContent;
    if (curContent.isEmpty())
        shownName = "untitled.txt";
    setWindowFilePath(shownName);
}

QString MainWindow::strippedName(const QString &fullFileName)
{
    return QFileInfo(fullFileName).fileName();
}

void MainWindow::playContent()
{
    if (!d_creator || !d_creator->getCurrentContent())
        return;

    maybeSave();

    if (d_player.state() != QProcess::NotRunning)
    {
        QLocalSocket *socket = new QLocalSocket(this);
        connect(socket, SIGNAL(disconnected()), socket,SLOT(deleteLater()));

        for (int retry = 0; retry < 20; retry++)
        {
            socket->connectToServer(d_IPCServerName);
            if (socket->waitForConnected())
            {
                break;
            }
            SleeperThread::msleep(500);
        }
        QString requestData = "close,";
        socket->write(requestData.toLocal8Bit());
        socket->flush();
        socket->waitForBytesWritten();
        socket->close();

        if (!d_player.waitForFinished())
            d_player.kill();
    }

    d_IPCServerName = "denncoCreator_";
    QTextStream s(&d_IPCServerName);
    s.setIntegerBase(16);
    s << qrand();

    if (d_player.state() == QProcess::NotRunning)
    {
        QStringList args;
#ifdef Q_WS_MAC
        args << QCoreApplication::applicationDirPath() + "/../../../" + DENNCO_ENGINE;
        args << "--args";
#endif
        args << "-creatorControlled";
        args << d_IPCServerName;

#ifdef Q_WS_MAC
        d_player.start("open" , args);
#else
        d_player.start(DENNCO_ENGINE, args);
#endif
        d_player.waitForStarted();
    }

    if (d_player.state() == QProcess::Running)
    {
        QLocalSocket *socket = new QLocalSocket(this);
        connect(socket, SIGNAL(disconnected()), socket,SLOT(deleteLater()));
        for (int retry = 0; retry < 20; retry++)
        {
            socket->connectToServer(d_IPCServerName);
            if (socket->waitForConnected())
            {
                break;
            }
            SleeperThread::msleep(500);
        }
        QString requestData = "load,";
        requestData.append(d_creator->getCurrentContent()->getContentRootPath());
        socket->write(requestData.toLocal8Bit());
        socket->flush();
        socket->waitForBytesWritten();
        socket->close();
    }
}

void MainWindow::manageCellCode()
{
    if (!d_creator)
        return;

    DCManageCellCodeDialog dialog(d_creator, NULL, this);
    dialog.exec();
}


//static
bool MainWindow::openExternalEditorFor(const QString &path)
{
    static QList<QProcess*> s_processes;
    static QMutex s_mutex;

    QMutexLocker locker(&s_mutex);

    QString editorPath = readSettingsForExternalScriptEditorPath();
    if (editorPath.isEmpty() || editorPath.length() == 0)
    {
        DCCodeEditorExternalEditorSettingDialog dialog;
        dialog.exec();
        editorPath = readSettingsForExternalScriptEditorPath();
    }
    if (editorPath.isEmpty() || editorPath.length() == 0)
        return false;

    QString parameterOrg = readSettingsForExternalScriptEditorParameters();
    QString parameter = "";
    int i = 0;
    int j = parameterOrg.indexOf(QRegExp("%F[ \t\"']|%F$"),i);
    while (j >= 0)
    {
        parameter += parameterOrg.mid(i, j-i);
        parameter += path;
        i = j + 2;
        if (i > parameterOrg.length())
            break;
        j = parameterOrg.indexOf(QRegExp("%F[ \t\"']|%F$"),i);
    }
    if (i < parameterOrg.length())
        parameter += parameterOrg.mid(i);

    QList<QProcess*>::iterator it = s_processes.begin();
    while (it != s_processes.end())
    {
        if ((*it)->state() != QProcess::Running)
        {
            delete (*it);
            it = s_processes.erase(it);
        }
        else
            ++it;
    }
    QProcess *newProcess = new QProcess;
    newProcess->start("\"" + editorPath + "\" " + parameter);

    s_processes.append(newProcess);

    return true;
}
