//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Oct-26, 2012.
//
#include "dcpropertyeditor.h"

#include <QGridLayout>
#include <QLabel>
#include <QMessageBox>
#include <QCloseEvent>
#include <QTextStream>
#include <QDomDocument>
#include <QApplication>
#include <QFile>

static DCPropertyEditor *s_instance = NULL;

DCPropertyEditor::DCPropertyEditor(QWidget *parent) :
    QDialog(parent), d_isEditing(false), d_isInInitialization(false),
    d_originalTickInterval(0), d_originalHTTPServer(false), d_originalSerialServer(false)
{
    QVBoxLayout *layout_root = new QVBoxLayout();

    QGridLayout *glayout = new QGridLayout;
    glayout->addWidget(new QLabel("TickIntervalSec (sec)"),0,0);
    glayout->addWidget(new QLabel("UIPath"),1,0);
    glayout->addWidget(new QLabel("EnableHTTPServer"),2,0);
    glayout->addWidget(new QLabel("EnableSerialServer"), 3,0);
    glayout->addWidget(new QLabel("SerialSpeed"), 4, 0);

    d_spinBoxTickInterval = new QDoubleSpinBox(this);
    d_spinBoxTickInterval->setDecimals(3);
    d_textFieldUIPath = new QLineEdit(this);
    d_checkBoxHttpServer = new QCheckBox(this);
    d_checkBoxSerialServer = new QCheckBox(this);
    d_comboBoxSerialSpeed = new QComboBox(this);

    d_comboBoxSerialSpeed->addItem("1200");
    d_comboBoxSerialSpeed->addItem("2400");
    d_comboBoxSerialSpeed->addItem("4800");
    d_comboBoxSerialSpeed->addItem("19200");
    d_comboBoxSerialSpeed->addItem("38400");
    d_comboBoxSerialSpeed->addItem("75600");
    d_comboBoxSerialSpeed->addItem("115200");

    glayout->addWidget(d_spinBoxTickInterval, 0, 1);
    glayout->addWidget(d_textFieldUIPath, 1, 1);
    glayout->addWidget(d_checkBoxHttpServer, 2, 1);
    glayout->addWidget(d_checkBoxSerialServer, 3, 1);
    glayout->addWidget(d_comboBoxSerialSpeed, 4, 1);

    layout_root->addLayout(glayout);

    d_saveButton = new QPushButton(tr("save"));
    d_closeButton = new QPushButton(tr("close"));

    QHBoxLayout *hlayout = new QHBoxLayout;
    hlayout->addStretch();
    hlayout->addWidget(d_saveButton);
    hlayout->addWidget(d_closeButton);

    layout_root->addLayout(hlayout);

    setLayout(layout_root);

    connect(d_spinBoxTickInterval, SIGNAL(valueChanged(double)), this, SLOT(propertyChanged()));
    connect(d_textFieldUIPath, SIGNAL(textChanged(QString)), this, SLOT(propertyChanged()));
    connect(d_checkBoxHttpServer, SIGNAL(clicked()), this, SLOT(propertyChanged()));
    connect(d_checkBoxSerialServer, SIGNAL(clicked()), this, SLOT(propertyChanged()));
    connect(d_comboBoxSerialSpeed, SIGNAL(currentIndexChanged(int)), this, SLOT(propertyChanged()));

    connect(d_saveButton, SIGNAL(clicked()), this, SLOT(saveChange()));
    connect(d_closeButton, SIGNAL(clicked()), this, SLOT(close()));
}

DCPropertyEditor::~DCPropertyEditor()
{

}

bool DCPropertyEditor::maybeSave()
{
    if (d_isEditing)
    {
        if (getIsModified())
        {
            int button = QMessageBox::question(this, tr("File modified"), tr("File modified. Do you want to save the change?"), QMessageBox::Save | QMessageBox::No | QMessageBox::Cancel, QMessageBox::Save );
            if (button == QMessageBox::Save)
            {
                saveChange();
                return true;
            }
            else if (button == QMessageBox::No)
            {
                return true;
            }
            else if (button == QMessageBox::Cancel)
            {
                return false;
            }
        }
    }
    return true;
}

void DCPropertyEditor::updateOriginalData()
{
    d_originalTickInterval = d_spinBoxTickInterval->value();
    d_originalUIPath = d_textFieldUIPath->text();
    d_originalHTTPServer = d_checkBoxHttpServer->isChecked();
    d_originalSerialServer = d_checkBoxSerialServer->isChecked();
    d_originalSerialSpeed = d_comboBoxSerialSpeed->currentText();
}

DCPropertyEditor* DCPropertyEditor::getEditor()
{
    if (!s_instance)
    {
        s_instance = new DCPropertyEditor;
    }

    return s_instance;
}


void DCPropertyEditor::destroyEditor()
{
    if (s_instance)
    {
        delete s_instance;
        s_instance = NULL;
    }
}

void DCPropertyEditor::closeEvent(QCloseEvent *event)
{
    if (maybeSave())
    {
        event->accept();
    }
    else
    {
        event->ignore();
    }
}

bool DCPropertyEditor::startEditing(const QString &sysFilePath)
{
    d_isInInitialization = true;

    setWindowTitle(tr("property editor"));

    d_editingFile = sysFilePath;

    QFile file(d_editingFile);
    if (file.exists())
    {
        QDomDocument doc;
        doc.setContent(&file);

        QDomNodeList tickIntervalSec = doc.elementsByTagName("TickIntervalSec");
        if (tickIntervalSec.count() > 0)
        {
            d_spinBoxTickInterval->setValue(tickIntervalSec.at(0).toElement().text().toDouble());
        }
        else
        {
            d_spinBoxTickInterval->setValue(1.0);
        }

        QDomNodeList uiPath = doc.elementsByTagName("UIPath");
        if (uiPath.count() > 0)
        {
            d_textFieldUIPath->setText(uiPath.at(0).toElement().text());
        }
        else
        {
            d_textFieldUIPath->setText("/ui/index.html");
        }

        QDomNodeList enableHTTP = doc.elementsByTagName("EnableHTTPServer");
        if (enableHTTP.count() > 0)
        {
            d_checkBoxHttpServer->setChecked(enableHTTP.at(0).toElement().text().toUpper() == "YES");
        }
        else
        {
            d_checkBoxHttpServer->setChecked(false);
        }

        QDomNodeList enableSerial = doc.elementsByTagName("EnableSerialServer");
        if (enableSerial.count() > 0)
        {
            d_checkBoxSerialServer->setChecked(enableSerial.at(0).toElement().text().toUpper() == "YES");
        }
        else
        {
            d_checkBoxSerialServer->setChecked(false);
        }

        QDomNodeList serialSpeed = doc.elementsByTagName("SerialSpeed");
        if (serialSpeed.count() > 0)
        {
            int index = d_comboBoxSerialSpeed->findText(serialSpeed.at(0).toElement().text());
            if (index>=0)
            {
                d_comboBoxSerialSpeed->setCurrentIndex(index);
            }
        }
        d_comboBoxSerialSpeed->setEnabled(d_checkBoxSerialServer->isChecked());
        d_saveButton->setEnabled(false);

        updateOriginalData();
        show();
        activateWindow();
        d_isEditing = true;
    }
    else
    {
        d_isEditing = false;
    }

    d_isInInitialization = false;

    return d_isEditing;
}

bool DCPropertyEditor::saveChange()
{
    QDomDocument doc;
    QDomElement rootElement = doc.createElement("dennco");

    QDomElement tickIntervalSecElement = doc.createElement("TickIntervalSec");
    tickIntervalSecElement.appendChild(doc.createTextNode(d_spinBoxTickInterval->cleanText()));
    rootElement.appendChild(tickIntervalSecElement);

    QDomElement uiPathElement = doc.createElement("UIPath");
    uiPathElement.appendChild(doc.createTextNode(d_textFieldUIPath->text()));
    rootElement.appendChild(uiPathElement);

    QDomElement enableHTTPElement = doc.createElement("EnableHTTPServer");
    enableHTTPElement.appendChild(doc.createTextNode(d_checkBoxHttpServer->isChecked() ? "yes" : "no"));
    rootElement.appendChild(enableHTTPElement);

    QDomElement enableSerialElement = doc.createElement("EnableSerialServer");
    bool enableSerial = d_checkBoxSerialServer->isChecked();
    if (enableSerial)
    {
        enableSerialElement.appendChild(doc.createTextNode("yes"));
        QDomElement serialSpeedElement = doc.createElement("SerialSpeed");
        serialSpeedElement.appendChild(doc.createTextNode(d_comboBoxSerialSpeed->currentText()));
        rootElement.appendChild(enableSerialElement);
        rootElement.appendChild(serialSpeedElement);
    }
    else
    {
        enableSerialElement.appendChild(doc.createTextNode("no"));
        rootElement.appendChild(enableSerialElement);
    }

    doc.appendChild(rootElement);

    bool saved = false;

    QFile file(d_editingFile);
    if (file.open(QFile::WriteOnly | QFile::Truncate))
    {
        QTextStream out(&file);
        doc.save(out, 4);
        saved = true;
        updateOriginalData();
    }
    file.close();

    propertyChanged();

    return saved;
}

bool DCPropertyEditor::getIsModified() const
{
    if (d_isInInitialization)
        return false;

    bool modified = false;

    if (d_originalTickInterval != d_spinBoxTickInterval->value())
        modified = true;

    if (!modified && d_originalUIPath != d_textFieldUIPath->text())
        modified = true;

    if (!modified && d_originalHTTPServer != d_checkBoxHttpServer->isChecked())
        modified = true;

    if (!modified && d_originalSerialServer != d_checkBoxSerialServer->isChecked())
        modified = true;

    if (!modified && d_originalSerialSpeed != d_comboBoxSerialSpeed->currentText())
        modified = true;

    return modified;
}

void DCPropertyEditor::propertyChanged()
{
    if (d_isInInitialization)
        return;

    d_comboBoxSerialSpeed->setEnabled(d_checkBoxSerialServer->isChecked());

    if (getIsModified())
    {
        setWindowTitle(tr("property editor *"));
        d_saveButton->setEnabled(true);
    }
    else
    {
        setWindowTitle(tr("property editor"));
        d_saveButton->setEnabled(false);
    }
}
