//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Dec-24/2012.
//
#include "dccellcodescripttreeviewwidget.h"

#include "dccreator.h"
#include "dccontainer.h"
#include "dccellcode.h"
#include "mainwindow.h"
#include "dialog/dcaddcellcodeclassdialog.h"
#include "utils/dcqtitemmodel.h"

const static QString S_ScriptTreeStyle =
        "QTreeView::branch:!has-children:adjoins-item {"
        "border-image: url(:/docIcon.png) 0;"
        "}";

class ScriptTreeModelItem : public DCQtItemModelItem
{
    QVector<QVariant> mItemUserData;
public:
    ScriptTreeModelItem(const QVector<QVariant> &data, DCQtItemModelItem *parent)
    : DCQtItemModelItem(data, parent), mItemUserData(data)
    {
    }

    ScriptTreeModelItem(const QStringList &headers, DCQtItemModelItem *parent = 0)
        : DCQtItemModelItem(headers, parent)
    {
        QVector<QVariant> rootData;
        foreach (QString header, headers)
        {
            rootData << header;
        }
        mItemUserData = rootData;
    }

    virtual ~ScriptTreeModelItem()
    {
    }


    QVariant userData(int column) const
    {
        return mItemUserData.value(column);
    }

    virtual bool insertChildren(int position, int count, int columns)
    {
        if (position < 0 || position > mChildItems.size())
            return false;

        for (int row = 0; row < count; ++row) {
            QVector<QVariant> data(columns);
            DCQtItemModelItem *item = new ScriptTreeModelItem(data, this);
            mChildItems.insert(position, item);
        }

        return true;
    }

    virtual bool insertColumns(int position, int columns)
    {
        if (!DCQtItemModelItem::insertColumns(position, columns))
            return false;

        for (int column = 0; column < columns; ++column)
            mItemUserData.insert(position, QVariant());

        return true;
    }

    virtual bool removeColumns(int position, int columns)
    {
        if (!DCQtItemModelItem::removeColumns(position, columns))
            return false;

        for (int column = 0; column < columns; ++column)
            mItemUserData.remove(position);

        return true;
    }

    bool setUserData(int column, const QVariant &value)
    {
        if (column < 0 || column >= mItemUserData.size())
            return false;

        mItemUserData[column] = value;
        return true;
    }
};

class ScriptTreeModel : public DCQtItemModel
{
public:
    ScriptTreeModel(const QStringList &headers, QObject *parent = 0)
        : DCQtItemModel(headers, new ScriptTreeModelItem(headers), parent)
    {}

    virtual ~ScriptTreeModel()
    {
    }


    virtual QVariant data(const QModelIndex &index, int role) const
    {
        if (!index.isValid())
            return QVariant();

        if (role == Qt::UserRole)
        {
            ScriptTreeModelItem *item = dynamic_cast<ScriptTreeModelItem*>(getItem(index));
            if (item)
                return item->userData(index.column());
        }
        else
        {
            return DCQtItemModel::data(index,role);
        }

        return QVariant();
    }


    virtual bool setData(const QModelIndex &index, const QVariant &value, int role)
    {
        bool result = false;
        if (role == Qt::UserRole)
        {
            ScriptTreeModelItem *item = dynamic_cast<ScriptTreeModelItem*>(getItem(index));
            if (item)
                result = item->setUserData(index.column(), value);
        }
        else
        {
            return DCQtItemModel::setData(index,value,role);
        }

        if (result)
            emit dataChanged(index, index);

        return result;
    }

};


DCCellCodeScriptTreeViewWidget::DCCellCodeScriptTreeViewWidget(QWidget *parent, DCCreator *creator) :
    QTreeView(parent), d_creator(creator)
{
    QStringList headers;
    headers << "Name";
    d_model = new ScriptTreeModel(headers);

    setIndentation(15);
    setModel(d_model);
    setStyleSheet(S_ScriptTreeStyle);

    connect(d_creator, SIGNAL(contentRootPathChanged(const void*, QString)), this, SLOT(contentRootPathChanged(const void*, QString)));
    connect(d_creator, SIGNAL(destroyed(QObject*)), this, SLOT(creatorDestroyed()));
    connect(d_creator, SIGNAL(commandExecuted(const QUndoCommand*)), this, SLOT(commandExecuted(const QUndoCommand*)));

    setContextMenuPolicy(Qt::CustomContextMenu);
    connect(this, SIGNAL(customContextMenuRequested(const QPoint&)),this, SLOT(doContextMenu(const QPoint&)));
}

DCCellCodeScriptTreeViewWidget::~DCCellCodeScriptTreeViewWidget()
{
    if (d_model)
        d_model->deleteLater();
}


void DCCellCodeScriptTreeViewWidget::rebuildModel()
{
    if (!d_model)
        return;

    if (!d_creator || !d_creator->getCurrentContainer())
        return;

    d_model->removeAllItems();

    QVector<QString> pathNames;
    DCContainer *container = d_creator->getCurrentContainer();
    const TKCellCodeMap* cellCodeScripts = container->getCellCodes();
    TKCellCodeMap::const_iterator it = cellCodeScripts->begin();
    while(it != cellCodeScripts->end())
    {
        pathNames.append(QString::fromStdString(it->first));
        it++;
    }
    qSort(pathNames);

    QVector<QString>::const_iterator sit = pathNames.begin();
    while(sit != pathNames.end())
    {
        QStringList nodes = (*sit).split("/");

        QModelIndex index;
        if (nodes.length() < 1)
        {
            ++sit;
            continue;
        }

        for (int i = 1; i < nodes.length() -1; i++)
        {
            QString n = nodes.at(i);
            index = childNode(n, index);
        }
        QString ln = nodes.at(nodes.length()-1);
        int p = ln.lastIndexOf("#");
        if (p <= 0)
        {
            //invalid
            ++sit;
            continue;
        }
        QString fName = ln.left(p);
        QString cName = ln.mid(p+1);
        index = childNode(fName, index);
        index = childNode(cName, index);
        d_model->setData(index, QVariant(*sit), Qt::UserRole);
        ++sit;
    }
    expandAll();
}

void DCCellCodeScriptTreeViewWidget::selectPath(const QString path)
{
    QStringList nodes = path.split("/");

    QModelIndex index;
    if (path.length() < 1)
    {
        return;
    }

    for (int i = 1; i < nodes.length() -1; i++)
    {
        QString n = nodes.at(i);
        index = childNode(n, index, false);
        if (!index.isValid())
            break;
    }

    QString ln = nodes.at(nodes.length()-1);
    int p = ln.lastIndexOf("#");
    if (p>0)
    {
        QString fName = ln.left(p);
        QString cName = ln.mid(p+1);

        index = childNode(fName, index, false);
        if (index.isValid())
        {
            index = childNode(cName, index, false);
            selectionModel()->select(index, QItemSelectionModel::Select);
        }
    }
}

QModelIndex DCCellCodeScriptTreeViewWidget::childNode(const QString& nodeName, const QModelIndex& currentIndex, bool createWhenNotExist)
{
    QModelIndex childIndex;

    for (int j = 0; j < d_model->rowCount(currentIndex); j++)
    {
        if (d_model->data(d_model->index(j,0,currentIndex), Qt::EditRole).toString() == nodeName)
        {
            childIndex = d_model->index(j,0,currentIndex);
            break;
        }
    }
    if (!childIndex.isValid() && createWhenNotExist)
    {
        int position =  d_model->rowCount(currentIndex);;
        d_model->insertRows(position,1,currentIndex);
        childIndex = d_model->index(position,0,currentIndex);
        d_model->setData(childIndex,QVariant(nodeName));
    }
    return childIndex;
}

void DCCellCodeScriptTreeViewWidget::contentRootPathChanged(const void *requester, QString rootPath)
{
    (void)requester; (void)rootPath;

    rebuildModel();
}

void DCCellCodeScriptTreeViewWidget::creatorDestroyed()
{
    d_creator = NULL;
}

void DCCellCodeScriptTreeViewWidget::commandExecuted(const QUndoCommand *)
{
    rebuildModel();
}

void DCCellCodeScriptTreeViewWidget::doContextMenu(const QPoint& pos)
{
    if (!d_creator || !d_creator->getCurrentContainer())
        return;

    QPoint globalPos = mapToGlobal(pos);

    QMenu menu;

    QAction *addCellCodeScriptAction = NULL;
    QAction *removeCellCodeScriptAction = NULL;

    QString path = "";
    QModelIndex index;
    QModelIndexList indexes = selectedIndexes();
    addCellCodeScriptAction = menu.addAction(tr("Add cell code script..."));
    if (indexes.length() > 0)
    {
        index = indexes.at(0);
        QString pathName = d_model->data(index, Qt::UserRole).toString();
        if (!pathName.isEmpty() && pathName.length() > 0)
        {
            removeCellCodeScriptAction  = menu.addAction(tr("Remove..."));
            path = pathName;
        }
        else
        {
            QModelIndex topIndex;
            while (index != topIndex)
            {
                path.prepend(d_model->data(index, Qt::EditRole).toString());
                index = index.parent();
                path.prepend("/");

            }
        }
    }

    QAction* selectedItem = menu.exec(globalPos);
    if (selectedItem == addCellCodeScriptAction)
    {
        int s = path.lastIndexOf("#");
        if (s>0)
        {
            path = path.left(s);
        }
        DCAddCellCodeClassDialog dialog(d_creator, path);
        dialog.exec();
        QString addedPathName = dialog.getAddedCellCodeClassName();
        if (!addedPathName.isEmpty() && addedPathName.length()>0)
        {
            selectPath(addedPathName);
        }
    }
    else if (selectedItem == removeCellCodeScriptAction)
    {
        qDebug() << "Remove cell code script!!";
        qDebug() << path;
        DCContainer *container = d_creator->getCurrentContainer();
        if (container)
        {
            DCCellCode *cellCode = dynamic_cast<DCCellCode*>(container->getCellCode(path.toStdString()));
            if (cellCode)
            {
                QString msg = "Deleting Cell code script:\n" + path + "\nDo you want to continue?";
                QMessageBox msgBox(QMessageBox::Warning, tr("Delete cell code script"), msg);
                msgBox.setStandardButtons(QMessageBox::Cancel | QMessageBox::Yes);
                int ret = msgBox.exec();
                if (ret == QMessageBox::Yes)
                {
                    d_creator->doCommandRemoveCellCode(this, container, cellCode);
                }
            }
        }
    }
}

void DCCellCodeScriptTreeViewWidget::mouseDoubleClickEvent(QMouseEvent *event)
{
    (void) event;

    if (!d_creator || !d_creator->getCurrentContainer())
        return;

    QModelIndex index = currentIndex();
    if (index.isValid())
    {
        QString pathName = d_model->data(index, Qt::UserRole).toString();

        if (!pathName.isEmpty() && pathName.length()>0)
        {
            DCContainer *container = d_creator->getCurrentContainer();
            if (container)
            {
                DCCellCode *cellCode = dynamic_cast<DCCellCode*>(container->getCellCode(pathName.toStdString()));
                if (cellCode)
                {
                    MainWindow::openExternalEditorFor(cellCode->getWorkFilePathForCellCodeScript());
                }
            }
        }
    }
}
