//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#include "dccontent.h"

#include "dccontainer.h"
#include "DNContainerBuilder.h"
#include "DNGlobal.h"
#include "DNXML.h"
#include "DNXMLElement.h"
#include "DNUtils.h"
#include "dccontainersaver.h"
#include "uieditor/dcuieditor.h"
#include "propertyeditor/dcpropertyeditor.h"

#include <stdlib.h>
#include <sstream>
#include <iostream>

DCContent::DCContent(DCCreator *creator, std::string contentPath) :
    d_creator(creator), d_valid(false), d_container(0)
{
    d_contentRootPath = QString::fromStdString(contentPath);
    d_container = (DCContainer*) TKContainer::createContainer();
    d_container->setContent(this);

    dnGlobal()->updateRunningStatus(DNGlobal::STOPPED);
    dnGlobal()->resetErrorStatus();

    std::string containerRoot = contentPath;
    containerRoot.append("/Container");

    bool succeeded = false;
    succeeded = parseSettingFile(contentPath.c_str());
    if (!succeeded || !dnGlobal()->isErrorStatusNormal())
    {
        dnNotifyError("property.xml parse error", "failed to parse setting file /property.xml");
        return;
    }

    std::string dataStorePath = containerRoot;
    dataStorePath.append("/data.db");

    succeeded = d_container->setDataStore(dataStorePath.c_str());
    if (!succeeded)
    {
        dnNotifyError("Initialization failed","Failed to the setup data store");
        return;
    }

    succeeded = parseContainerFile(containerRoot.c_str());
    if (!succeeded)
    {
        dnNotifyError("Initialization failed", "Failed to parse container file");
        return;
    }

    d_valid = true;
}

DCContent::~DCContent()
{
    d_valid = false;
    if (d_container)
    {
        delete d_container;
        d_container = 0;
    }
}

bool DCContent::getIsModified() const
{
    if (d_valid && d_container && d_container->getIsModified())
    {
        return true;
    }

    if (DCPropertyEditor::getEditor()->isVisible())
        DCPropertyEditor::getEditor()->close();

    if (DCUIEditor::getEditor()->isVisible())
        DCUIEditor::getEditor()->close();

    return false;
}

bool DCContent::parseSettingFile(const char *contentRoot)
{
    bool valid = false;

    DNXML *xml = DNXML::createXMLFromFile(contentRoot, "property.xml");
    if (xml)
    {
        valid = true;
        DNXMLElement *element = xml->getRoot();
        if (!element)
        {
            valid = false;
            std::string message = "Failed to load property.xml file";
            dnNotifyError("Initialization failed",message);
        }

        if (valid && element->name != "dennco")
        {
            valid = false;
            std::string message = "First element of property.xml should be <dennco>";
            dnNotifyError("Initialization failed",message);
        }

        if (valid)
        {
            DNXMLElement *e = element->inner;
            while(e)
            {
                std::string pname = upperString(e->name);

                if (pname == "TICKINTERVALSEC")
                {
                    std::istringstream is(e->text);
                    float t = 0.0;
                    is >> t;
                    if (t>0)
                    {
                        //TODO
                    }
                    else
                    {
                        valid = false;
                        std::string message = "Error in property.xml. TickIntervalSec is not configured properly.";
                        dnNotifyError("Initialization failed",message);
                    }
                }
                else if (pname == "UIPATH")
                {
                    // TODO
                }
                else if (pname == "ENABLEHTTPSERVER")
                {

                }
                else if (pname == "ENABLESERIALSERVER")
                {
                    if ( upperString(e->text) == "YES")
                    {
                        //TODO
                    }
                    else
                    {
                        //TODO
                    }
                }
                else if (pname == "SERIALSPEED")
                {
                    //TODO
                }
                e = e->next;
            }
        }

        delete xml;
    }
    return valid;
}

bool DCContent::parseContainerFile(const char *containerRoot)
{
    DNContainerBuilder builder(d_container);
    d_container->beganBuildContainer();
    bool r = builder.buildContainerFromXHTML(containerRoot);
    d_container->endedBuildContainer();

    return r;
}

bool DCContent::saveAll(const QString& contentRoot)
{
    DCContainerSaver containerSaver(d_container);

    QString containerRoot = contentRoot + "/Container";
    return containerSaver.saveAll(containerRoot);
}

bool DCContent::saveForPage(const QString &contentRoot, DCVCPage *page)
{
    DCContainerSaver containerSaver(d_container);

    QString containerRoot = contentRoot + "/Container";
    return containerSaver.saveForPage(containerRoot, page);
}
