//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Nov-4, 2012.
//
#include "dcmanagecellcodedialog.h"

#include "dccontainer.h"
#include "dccreator.h"
#include "dcscene.h"
#include "dccellcode.h"
#include "dcvcpage.h"
#include "command/dccommand.h"
#include "utils/dcqtitemmodel.h"
#include "utils/dcutil.h"
#include "dcsinglecolumntableview.h"
#include "dialog/dcaddcellcodeclassdialog.h"

#include <QLineEdit>
#include <QTableView>
#include <QHeaderView>

DCManageCellCodeDialog::DCManageCellCodeDialog(DCCreator *creator, DCCellCode* cellCode, QWidget *parent) :
    QDialog(parent), d_creator(creator), d_cellCode(cellCode), d_reloading(false)
{
    setWindowTitle(tr("Manage Cell Code"));
    d_container = d_creator->getCurrentContainer();

    d_textField = new QLineEdit;
    d_table = new DCSingleColumnTableView();
    QStringList headers;
    headers << "Cell code";
    d_tableModel = new DCQtItemModel(headers);
    d_tableModel->setReadOnly(0,true);
    d_table->setModel(d_tableModel);

    d_comboBox = new QComboBox;
    d_comboBox->addItems(QStringList(d_container->getAvailableScriptableCellTypes()));

    setLayout(new QVBoxLayout);
    d_table->horizontalHeader()->setResizeMode(QHeaderView::Stretch);

    d_addButton = new QPushButton(tr("Add"));
    d_closeButton = new QPushButton(tr("Close"));
    d_saveButton = new QPushButton(tr("Save"));
    QHBoxLayout *buttonLayout = new QHBoxLayout;
    buttonLayout->addWidget(d_addButton);
    buttonLayout->addItem(new QSpacerItem(10,10, QSizePolicy::Expanding));
    buttonLayout->addWidget(d_saveButton);
    buttonLayout->addWidget(d_closeButton);
    d_saveButton->setEnabled(false);
    d_closeButton->setDefault(true);

    QGridLayout *glayout = new QGridLayout;
    glayout->addWidget(new QLabel(tr("name")),0,0);
    glayout->addWidget(d_textField,0,1);
    glayout->addWidget(new QLabel(tr("type")),1,0);
    glayout->addWidget(d_comboBox);
    ((QVBoxLayout*)layout())->addLayout(glayout);
    layout()->addWidget(d_table);
    d_statusText = new QLabel;
    d_message = new QLabel;
    d_message->setStyleSheet("color : red;");
    layout()->addWidget(d_statusText);
    layout()->addWidget(d_message);
    ((QVBoxLayout*)layout())->addLayout(buttonLayout);

    reloadViewData();

    connect(d_textField, SIGNAL(textChanged(QString)), this, SLOT(textInputChanged(QString)));
    connect(d_comboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(comboBoxSelectionChanged()));
    connect(d_addButton, SIGNAL(clicked()), this, SLOT(addButtonClicked()));
    connect(d_saveButton, SIGNAL(clicked()), this, SLOT(saveButtonClicked()));
    connect(d_closeButton, SIGNAL(clicked()), this, SLOT(close()));
    connect(d_table,SIGNAL(selectionChangedSignal(QItemSelection,QItemSelection)), this, SLOT(listSelectionChanged(QItemSelection,QItemSelection)));
    connect(d_creator, SIGNAL(commandExecuted(const QUndoCommand*)), this, SLOT(commandExecuted(const QUndoCommand*)));
}

DCManageCellCodeDialog::~DCManageCellCodeDialog()
{
    if (d_tableModel)
        d_tableModel->deleteLater();
}

void DCManageCellCodeDialog::closeEvent(QCloseEvent *event)
{
    if (maybeSave())
    {
        event->accept();
    }
    else
    {
        event->ignore();
    }
}

bool DCManageCellCodeDialog::maybeSave()
{
    if (!d_cellCode)
        return true;

    bool typeChanged = QString::fromStdString(d_cellCode->getCellAPIName()) != d_comboBox->currentText().trimmed();
    bool nameChanged = DCUtil::getNameFromFQNPath(QString::fromStdString(d_cellCode->getFQNName())) != d_textField->text().trimmed();

    bool r = true;

    if (typeChanged || nameChanged)
    {
        QMessageBox::StandardButton ret;
        ret = QMessageBox::warning(this, tr("Cell code manager"),
            tr("The cell code has been modified.\n"
            "Do you want to save your changes?"),
            QMessageBox::Save | QMessageBox::Discard | QMessageBox::Cancel);
        if (ret == QMessageBox::Save)
        {
            if (typeChanged)
            {
                d_creator->doCommandChangeCellCodeClassType(this, d_cellCode, d_comboBox->currentText().trimmed());
            }
            if (nameChanged)
            {
                //TODO
                QMessageBox::warning(this, "Not implemented", "Sorry! Renaming cell code name is not implemented now..\nIt will be supported in the future..");
                d_textField->setText(DCUtil::getNameFromFQNPath(QString::fromStdString(d_cellCode->getFQNName())));
            }
        }
        else if (ret == QMessageBox::Cancel)
        {
            r = false;
        }
    }
    return r;
}

bool DCManageCellCodeDialog::updateWindowStatus()
{
    d_message->setText("");
    d_statusText->setText("");

    if (!d_cellCode)
        return true;

    QString name = DCUtil::getNameFromFQNPath(QString::fromStdString(d_cellCode->getFQNName()));
    bool modified = name != d_textField->text().trimmed();
    if (!modified)
        modified = QString::fromStdString(d_cellCode->getCellAPIName()) != d_comboBox->currentText().trimmed();

    QString windowTitle = name;
    if (modified)
    {
        windowTitle.append("*");
        d_saveButton->setEnabled(true);
        d_saveButton->setDefault(true);
    }
    else
    {
        d_saveButton->setEnabled(false);
        d_closeButton->setEnabled(true);
    }
    windowTitle.append(" - Manage Cell Code");

    setWindowTitle(windowTitle);

    return modified;
}

void DCManageCellCodeDialog::textInputChanged(const QString &text)
{
    (void)text;

    updateWindowStatus();
}

void DCManageCellCodeDialog::comboBoxSelectionChanged()
{
    updateWindowStatus();
}

void DCManageCellCodeDialog::saveButtonClicked()
{
    bool typeChanged = QString::fromStdString(d_cellCode->getCellAPIName()) != d_comboBox->currentText().trimmed();
    bool nameChanged = DCUtil::getNameFromFQNPath(QString::fromStdString(d_cellCode->getFQNName())) != d_textField->text().trimmed();

    if (typeChanged)
    {
        d_creator->doCommandChangeCellCodeClassType(this, d_cellCode, d_comboBox->currentText().trimmed());
    }
    if (nameChanged)
    {
        //TODO
        QMessageBox::warning(this, "Not implemented", "Sorry! Renaming cell code name is not implemented now..\nIt will be supported in the future..");
        d_textField->setText(DCUtil::getNameFromFQNPath(QString::fromStdString(d_cellCode->getFQNName())));
    }
}

void DCManageCellCodeDialog::addButtonClicked()
{
    DCAddCellCodeClassDialog dialog(d_creator, "", this);
    dialog.exec();
}

void DCManageCellCodeDialog::deleteButtonClicked()
{
    //TODO
    QMessageBox::warning(this, "Not implemented", "Sorry! Delete cell code is not implemented now..\nIt will be supported in the future..");
}


void DCManageCellCodeDialog::listSelectionChanged(const QItemSelection &selected, const QItemSelection &deselected)
{
    if (d_reloading)
        return;

    DCCellCode *newCellCode = NULL;
    if (selected.count()>0)
    {
        QString selectedPath = d_tableModel->data(selected.at(0).indexes().at(0), Qt::DisplayRole).toString();
        newCellCode = dynamic_cast<DCCellCode*>(d_container->getCellCode(selectedPath.toStdString()));
    }

    if (newCellCode != d_cellCode)
    {
        if (maybeSave())
        {
            d_cellCode = newCellCode;
            reloadViewData();
            updateWindowStatus();
        }
        else
        {
            QItemSelectionModel s(d_tableModel);
            s.select(deselected,QItemSelectionModel::Select);
            d_table->setSelectionModel(&s);
        }
    }

    updateWindowStatus();
}

void DCManageCellCodeDialog::commandExecuted(const QUndoCommand *executedCommand)
{
    (void)executedCommand;

    reloadViewData();
}

void DCManageCellCodeDialog::reloadViewData()
{
    d_reloading = true;

    const TKCellCodeMap* cellcodes = d_container->getCellCodes();
    TKCellCode *emtpyCellCodeClass = d_container->getEmptyCellCodeClass();
    int selection = -1;
    int row = 0;
    d_tableModel->removeAllItems();
    for ( TKCellCodeMap::const_iterator it = cellcodes->begin(); it != cellcodes->end(); ++it ) {
        if (it->second != emtpyCellCodeClass)
        {
            if (it->second == d_cellCode)
            {
                selection = row;
            }
            d_tableModel->insertString(QString::fromStdString(it->first));
            row++;
        }
    }
    d_table->setModel(d_tableModel);
    if (selection >= 0)
    {
        d_table->selectRow(selection);
    }

    if (d_cellCode)
    {
        d_textField->setText(DCUtil::getNameFromFQNPath(QString::fromStdString(d_cellCode->getFQNName())));

        QString type = QString::fromStdString(d_cellCode->getCellAPIName());
        for (int i = 0; i < d_comboBox->count(); i++)
        {
            if (d_comboBox->itemText(i) == type)
            {
                d_comboBox->setCurrentIndex(i);
                break;
            }
        }
    }

    updateWindowStatus();
    d_reloading = false;
}
