//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on 1/7/2012.
//

#include "DNUtils.h"
#include "TKLog.h"
#include "TKDebug.h"
#include "DNGlobal.h"

#ifndef _MSC_VER
    #include <string.h>
    #include <stdlib.h>
#endif

#include <algorithm>

void trimString(std::string& str)
{
    std::string::size_type pos1 = str.find_first_not_of(' ');
    std::string::size_type pos2 = str.find_last_not_of(' ');
    std::string::size_type pos1_2 = str.find_first_not_of('\t');
    std::string::size_type pos2_2 = str.find_last_not_of('\t');
    
    pos1 = pos1 < pos1_2 ? pos1_2 : pos1;
    pos2 = pos2 < pos2_2 ? pos2_2 : pos2;
    
    str = str.substr(pos1 == std::string::npos ? 0 : pos1, 
                     pos2 == std::string::npos ? str.length() - 1 : pos2 - pos1 + 1);
}

std::string upperString( const std::string &src )
{
    std::string data = src;
    std::transform(data.begin(), data.end(), data.begin(), ::toupper);

    return data;
}

std::string parseCellCodeForScriptEngine(std::string jname, std::string cellCode)
{
#ifdef DEBUG
    DEBUG_TRACE("TKJSCellCode::TKJSCellCode() parse for %s\n=== Original code =====\n%s\n", jname.c_str(), cellCode.c_str());
#endif

    const char *c = cellCode.c_str();

    int  inComment1 = 0;   //  //
    int  inComment2 = 0;   //  /* ... */
    int  inComment2_end = 0;
    int  inFunction = 0;
    int  inVar = 0;
    int  inQuote = 0;
    int  inDoubleQuote = 0;
    int  nest = 0;
    const char *sp = 0;
    int  l = 0;
    int  lp = 0;
    std::string funcs = "";
    std::string vars = "";
    std::string fff = "";

    const char* str_comment1 = "//";
    const char* str_comment2 = "/*";
    const char* str_comment2_end = "*/";
    const char* str_function = "function";
    const char* str_var = "var";

    while (*c)
    {
        if (*c == '\r' || *c == '\n')
        {
            lp = 0;
        }
        else if (*c != ' ' && *c != '\t')
        {
            lp++;
        }

        if (inComment1 == -1)
        {
            if (*c == '\r' || *c == '\n' )
                inComment1 = 0;
            c++;
            continue;
        }
        else if (inComment2 == -1)
        {
            if (*c == str_comment2_end[inComment2_end])
            {
                inComment2_end ++;
                if (str_comment2_end[inComment2_end] == 0)
                {
                    inComment2 = 0;
                    inComment2_end = 0;
                }
            }
            else
            {
                inComment2_end = 0;
            }
            c++;
            continue;
        }

        if (inQuote == -1)
        {
            if (*c == '\\')
            {
                c++;
                if (*c != 0)c++;
                continue;
            }
            if (*c == '\'')
            {
                inQuote = 0;
            }
            c++;
            continue;
        }
        else if (inDoubleQuote == -1)
        {
            if (*c == '\\')
            {
                c++;
                if (*c != 0)c++;
                continue;
            }
            if (*c == '\"')
            {
                inDoubleQuote = 0;
            }
            c++;
            continue;
        }

        if (*c == '\'')
        {
            inQuote = -1;
            c++;
            continue;
        }

        if (*c == '\"')
        {
            inDoubleQuote = -1;
            c++;
            continue;
        }

        //IN COMMENT?
        if (*c == str_comment1[inComment1])
        {
            inComment1++;
            if (str_comment1[inComment1] == 0)
            {
                inComment1 = -1;
                c++;
                continue;
            }
        }
        else
        {
            inComment1 = 0;
        }

        if (*c == str_comment2[inComment2])
        {
            inComment2++;
            if (str_comment2[inComment2] == 0)
            {
                inComment2 = -1;
                c++;
                continue;
            }
        }
        else
        {
            inComment2 = 0;
        }

        if (!inComment1 && !inComment2 && !inQuote && !inDoubleQuote && *c == '{')
        {
            nest++;
            if (inFunction == -1 && nest == 1)
            {
                fff = "";
                fff.append(sp,c-sp+1);
                unsigned long fp = fff.find( "(", 0 );
                if( fp != std::string::npos )
                {
                    fff = fff.substr(0, fp);
                    trimString(fff);
                    sp += fp;
                }
                else
                {
                    inFunction  = 0;
                }
            }
        }
        else if (!inComment1 && !inComment2 && !inQuote && !inDoubleQuote && *c == '}')
        {
            nest--;
            if (nest == 0)
            {
                if (inFunction == -1)
                {
                    //FUNCTION DEF
                    funcs.append(jname);
                    funcs.append(".prototype.");
                    funcs.append(fff);
                    funcs.append(" = function");
                    funcs.append(sp ,c-sp+1);
                    funcs.append("\r\n");
                    inFunction = 0;
                }
            }
            c++;
            continue;
        }

        if (*c == ';')
        {
            if (inVar == -1)
            {
                //VAR DEF
                std::string t = "";
                t.append(sp,c-sp+1);
                trimString(t);
                vars.append("this.");
                vars.append(t);
                vars.append("\r\n");
                inVar = 0;
            }
            c++;
            continue;
        }

        if (inFunction == -1)
        {
            if (l == 0) {
                if (*c != ' ' && *c != '\t' && *c != '\r' && *c != '\n')
                {
                    inFunction =0;
                    c++;
                    l = 0;
                    continue;
                }
            }
            c++;
            l++;
            continue;
        }

        if (inVar == -1)
        {
            if (l == 0) {
                if (*c != ' ' && *c != '\t' && *c != '\r' && *c != '\n' && *c != '=')
                {
                    inVar =0;
                    c++;
                    l = 0;
                    continue;
                }
            }
            c++;
            l++;
            continue;
        }


        if (nest == 0 && *c == str_function[inFunction] && (inFunction+1) == lp)
        {
            inFunction++;
            if (str_function[inFunction] == 0)
            {
                inFunction = -1;
                sp = c+1;
                l = 0;
            }
            c++;
            continue;
        }

        if (nest == 0 && *c == str_var[inVar] && (inVar+1) == lp)
        {
            inVar++;
            if (str_var[inVar] == 0)
            {
                inVar = -1;
                sp = c+1;
                l = 0;
            }
            c++;
            continue;
        }

        c++;

    }

    std::string stmt = "";
    stmt.append("function ");
    stmt.append(jname);
    stmt.append("(ownerCell)\n{\nthis.cell = ownerCell;\n");
    stmt.append(vars);
    stmt.append("\n}\n");
    stmt.append(funcs);
#ifdef DEBUG
    DEBUG_TRACE("\n\n==== Translated JS statement:=====\n%s\n=====\n", stmt.c_str());
#endif
    return stmt;
}

std::string getFQNString(const char *location, const char *name)
{
    std::string path;
    std::string node;
    int l = (int)strlen(location) - 1;
    bool f = false;
    while (l >= 0)
    {
        if (location[l] == '/')
        {
            f = true;
            break;
        }
        l--;
    }
    if( f )
    {
        if (l == 0)
        {
            path = "/";
            node = location;
        }
        else
        {
            node = location;
            path = node.substr(0,l);
            node = node.substr(l);
        }
    }
    else
    {
        path = "";
        node = location;
    }
    
    const char *c = name;
    int i = 0;
    const char *p = c;
    while(*c)
    {
        if (*c == '/')
        {
            if (i >= 2 && *(c - 2) == '.' && *(c - 1) == '.')
            {
                unsigned long l = path.find_last_of("/");
                if( l != std::string::npos )
                {
                    if (l == 0)
                    {
                        path = "/";
                    }
                    else if (l > 0)
                    {
                        path = path.substr(0, l);
                    }
                }
                else
                {
                    path = "";
                }
                p = c;
            }
            else if (i >= 1 && *(c - 1) == '.')
            {
                p = c;
            }
            else if (i >= 1 && *(c - 1) == '/')
            {
                p = c; 
            }
            else if (i == 0)
            {
                if (path.length() > 0 && path.at(0) == '/')
                {
                    path = "/";
                }
                else
                {
                    path = "";
                }
            }
            else
            {
                size_t pl = path.length();
                if (pl == 0)
                {
                    if (*p == '/')
                    {
                        p++;
                    }                    
                }
                else if (pl == 1)
                {
                    char p1 = path.at(0);
                    if (p1 == '/' || p1 == '.')
                    {
                        if (*p == '/')
                        {
                            p++;
                        }
                    }
                }
                else
                {
                    if (*p != '/')
                    {
                        path.append("/");
                    }
                }
                std::string s(p, c-p); 
                path.append(s);
                p = c;
            }
        }
        c++;
        i++;
        if (i > 1024) break;
    }
    
    if (*p == '/')
    {
        node = p;
    }
    else if(*p == '#')
    {
        node.append(p);
    }
    else
    {
        const char *p2 = p;
        bool hasAnchor = false;
        while(*p2)
        {
            if (*p2 == '#')
            {
                hasAnchor = true;
                break;
            }
            p2++;
        }
        if (hasAnchor)
        {
            node = "/";
            node.append(p);        
        }
        else
        {
            node.append("#");
            node.append(p);
        }
    }
    
    int pl = (int)path.length();
    std::string fqn;
    if (pl == 0)
    {
        if (node.at(0) == '/' && node.length() > 1)
        {
            fqn = node.substr(1);
        }
        else
        {
            fqn = node;
        }
        
    }
    else if (pl == 1)
    {
        if (path.at(0) == '/' && node.length() > 0 && node.at(0) == '/')
        {
            fqn = node;
        }
        else
        {
            fqn = path.append(node);
        }
    }
    else
    {
        fqn = path.append(node);
    }
    
    return fqn;
}

std::string getJSEscapeString(const char *cstring)
{
    const char *c = cstring; 
    std::string result = "";
    while(*c)
    {
        if (*c == '/')
        {
            result.append("_s");
        }
        else if (*c == '_')
        {
            result.append("__");
        }
        else if (*c == '.')
        {
            result.append("_d");
        }
        else if (*c == '#')
        {
            result.append("_S");
        }
        else
        {
            result.append(1,*c);
        }
        
        c++;
    }

    return result;
}

static TKLock NotifyLock;

void dnNotifyError(std::string title, std::string message)
{
    NotifyLock.lock();

    std::string::size_type pos = 0;
    while(pos = message.find("\n", pos), pos != std::string::npos) {
        message.replace(pos, 1, "\n           ");
        pos += 1;
    }

    std::string outMessage = "[ERROR!!!]:";
    outMessage.append(title).append("\n           ");
    outMessage.append(message);

    TKLog::printf(TKLog::ERROR, "%s", outMessage.c_str());
    if (dnGlobal()->updateErrorStatus(DNGlobal::ERROR))
    {
        dnGlobal()->setMessage1(title);
        dnGlobal()->setMessage2(message);
    }
    else
    {
        dnGlobal()->setMessage1(title);
    }
    NotifyLock.unlock();
}

void dnNotifyWarning(std::string title, std::string message)
{
    NotifyLock.lock();

    std::string::size_type pos = 0;
    while(pos = message.find("\n", pos), pos != std::string::npos) {
        message.replace(pos, 1, "\n           ");
        pos += 1;
    }

    std::string outMessage = "[WARNING] :";
    outMessage.append(title).append("\n           ");
    outMessage.append(message);

    TKLog::printf(TKLog::WARNING, "%s", outMessage.c_str());
    if (dnGlobal()->updateErrorStatus(DNGlobal::WARNING))
    {
        dnGlobal()->setMessage1(title);
        dnGlobal()->setMessage2(message);
    }
    else
    {
        dnGlobal()->setMessage1(title);
    }
    NotifyLock.unlock();
}
