//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on 3/24/2012.
//
#include "qtdnstorageimpl.h"

#include "DNStorage.h"
#include "DNStorageImpl.h"
#include "TKLog.h"
#include "DNUtils.h"

#include <QtSql>
#include <DNGlobal.h>

//static
DNStorageImpl* DNStorageImpl::create(const char *storagePath)
{
    return new QtDNStorageImpl(storagePath);
}

QtDNStorageImpl::~QtDNStorageImpl()
{
    if (mQueries)
    {
        delete mQueries;
        mQueries = NULL;
    }

    if (mDatabase)
    {
        mDatabase->database.rollback();
        QString connectionName = mDatabase->database.connectionName();
        if (mDatabase->database.isValid() && mDatabase->database.isOpen())
        {
            mDatabase->database.close();
        }

        delete mDatabase;
        QSqlDatabase::removeDatabase(connectionName);
        mDatabase = NULL;
    }
}

QtDNStorageImpl::QtDNStorageImpl(const char *storagePath) : mDatabase(NULL),mQueries(NULL)
{
    mStoragePath = QString::fromLocal8Bit(storagePath);
    mIsValid = initDB();
}

bool QtDNStorageImpl::setValue(const char *path, const char *key, float value)
{
    if (!mIsValid)
        return false;

    QString qPath(path);
    QString qKey(key);
    mQueries->propertiesCountQuery.addBindValue(qPath);
    mQueries->propertiesCountQuery.addBindValue(qKey);
    if (!mQueries->propertiesCountQuery.exec())
    {
        std::string message = "SQL error while processing cell ";
        message.append(path);
        message.append(":");
        message.append(key);
        message.append("\n");
        message.append(mQueries->propertiesCountQuery.lastError().text().toStdString());
        dnNotifyWarning("SQL ERROR", message);
        return false;
    }
    mQueries->propertiesCountQuery.next();
    int cnt = mQueries->propertiesCountQuery.value(0).toInt();
    if (cnt == 0)
    {
        mQueries->propertiesInsertQuery.addBindValue(qPath);
        mQueries->propertiesInsertQuery.addBindValue(qKey);
        mQueries->propertiesInsertQuery.addBindValue(value);
        if (!mQueries->propertiesInsertQuery.exec())
        {
            std::string message = "SQL error while processing cell ";
            message.append(path);
            message.append(":");
            message.append(key);
            message.append("\n");
            message.append(mQueries->propertiesCountQuery.lastError().text().toStdString());
            dnNotifyWarning("SQL ERROR", message);
            return false;
        }
    }
    else
    {
        mQueries->propertiesUpdateQuery.addBindValue(value);
        mQueries->propertiesUpdateQuery.addBindValue(qPath);
        mQueries->propertiesUpdateQuery.addBindValue(qKey);
        if (!mQueries->propertiesUpdateQuery.exec())
        {
            std::string message = "SQL error while processing cell ";
            message.append(path);
            message.append(":");
            message.append(key);
            message.append("\n");
            message.append(mQueries->propertiesCountQuery.lastError().text().toStdString());
            dnNotifyWarning("SQL ERROR", message);
            return false;
        }
    }

    return true;
}

float QtDNStorageImpl::getValue(const char *path, const char *key)
{
    if (mQueries == NULL || !mIsValid)
        return 0;

    QString qPath(path);
    QString qKey(key);
    mQueries->propertiesGetQuery.addBindValue(qPath);
    mQueries->propertiesGetQuery.addBindValue(qKey);
    if (!mQueries->propertiesGetQuery.exec())
    {
        std::string message = "SQL error while processing cell ";
        message.append(path);
        message.append(":");
        message.append(key);
        message.append("\n");
        message.append(mQueries->propertiesCountQuery.lastError().text().toStdString());
        dnNotifyWarning("SQL ERROR", message);
        return false;
    }
    mQueries->propertiesGetQuery.next();
    return mQueries->propertiesGetQuery.value(0).toFloat();
}

int QtDNStorageImpl::getCount(const char *path, const char *key)
{
    if (!mIsValid)
        return 0;

    QString qPath(path);
    QString qKey(key);
    mQueries->propertiesCountQuery.addBindValue(qPath);
    mQueries->propertiesCountQuery.addBindValue(qKey);
    if (!mQueries->propertiesCountQuery.exec())
    {
        std::string message = "SQL error while processing cell ";
        message.append(path);
        message.append(":");
        message.append(key);
        message.append("\n");
        message.append(mQueries->propertiesCountQuery.lastError().text().toStdString());
        dnNotifyWarning("SQL ERROR", message);
        return 0;
    }

    mQueries->propertiesCountQuery.next();
    return mQueries->propertiesCountQuery.value(0).toInt();
}


int QtDNStorageImpl::countXYZVData(const char *path, const char *key)
{
    if (mQueries == NULL || !mIsValid)
        return 0;
    QString qPath(path);
    QString qKey(key);
    mQueries->xyzvDataCountQuery.addBindValue(qPath);
    mQueries->xyzvDataCountQuery.addBindValue(qKey);
    if (!mQueries->xyzvDataCountQuery.exec())
    {
        std::string message = "SQL error while processing cell ";
        message.append(path);
        message.append(":");
        message.append(key);
        message.append("\n");
        message.append(mQueries->xyzvDataCountQuery.lastError().text().toStdString());
        dnNotifyWarning("SQL ERROR", message);
        return 0;
    }

    mQueries->xyzvDataCountQuery.next();
    return mQueries->xyzvDataCountQuery.value(0).toInt();
}

bool QtDNStorageImpl::insertXYZVData(const char *path, const char *key, int index, float x, float y, float z, float v)
{
    if (mQueries == NULL || !mIsValid)
        return 0;

    QString qPath(path);
    QString qKey(key);
    mQueries->xyzvDataInsetQuery.addBindValue(qPath);
    mQueries->xyzvDataInsetQuery.addBindValue(qKey);
    mQueries->xyzvDataInsetQuery.addBindValue(index);

    mQueries->xyzvDataInsetQuery.addBindValue(x);
    mQueries->xyzvDataInsetQuery.addBindValue(y);
    mQueries->xyzvDataInsetQuery.addBindValue(z);
    mQueries->xyzvDataInsetQuery.addBindValue(v);
    if (!mQueries->xyzvDataInsetQuery.exec())
    {
        std::string message = "SQL error while processing cell ";
        message.append(path);
        message.append(":");
        message.append(key);
        message.append("\n");
        message.append(mQueries->xyzvDataInsetQuery.lastError().text().toStdString());
        dnNotifyWarning("SQL ERROR", message);
        return false;
    }

    return true;
}

DNStorageXYZVRecords* QtDNStorageImpl::queryXYZVData(const char *path, const char *key)
{
    if (mQueries == NULL || !mIsValid)
        return 0;

    QString qPath(path);
    QString qKey(key);
    mQueries->xyzvDataCountQuery.addBindValue(qPath);
    mQueries->xyzvDataCountQuery.addBindValue(qKey);
    if (!mQueries->xyzvDataCountQuery.exec())
    {
        std::string message = "SQL error while processing cell ";
        message.append(path);
        message.append(":");
        message.append(key);
        message.append("\n");
        message.append(mQueries->xyzvDataCountQuery.lastError().text().toStdString());
        dnNotifyWarning("SQL ERROR", message);
        return 0;
    }
    mQueries->xyzvDataCountQuery.next();
    int length = mQueries->xyzvDataCountQuery.value(0).toInt();
    DNStorageXYZVRecords *records = new DNStorageXYZVRecords(length);
    mQueries->xyzvDataGetQuery.addBindValue(qPath);
    mQueries->xyzvDataGetQuery.addBindValue(qKey);
    if (!mQueries->xyzvDataGetQuery.exec())
    {
        std::string message = "SQL error while processing cell ";
        message.append(path);
        message.append(":");
        message.append(key);
        message.append("\n");
        message.append(mQueries->xyzvDataGetQuery.lastError().text().toStdString());
        dnNotifyWarning("SQL ERROR", message);
        if (records)
            delete records;
        return 0;
    }

    int i = 0;
    while(mQueries->xyzvDataGetQuery.next() && i < length)
    {
        records->data[i].index = mQueries->xyzvDataGetQuery.value(0).toInt();
        records->data[i].x = mQueries->xyzvDataGetQuery.value(1).toFloat();
        records->data[i].y = mQueries->xyzvDataGetQuery.value(2).toFloat();
        records->data[i].z = mQueries->xyzvDataGetQuery.value(3).toFloat();
        records->data[i].v = mQueries->xyzvDataGetQuery.value(4).toFloat();
        i++;
    }
    records->length = i;

    return records;
}

bool QtDNStorageImpl::deleteXYZVData(const char *path, const char *key)
{
    if (mQueries == NULL || !mIsValid)
        return false;

    QString qPath(path);
    QString qKey(key);
    mQueries->xyzvDataRemoveQuery.addBindValue(qPath);
    mQueries->xyzvDataRemoveQuery.addBindValue(qKey);
    if (!mQueries->xyzvDataRemoveQuery.exec())
    {
        std::string message = "SQL error while processing cell ";
        message.append(path);
        message.append(":");
        message.append(key);
        message.append("\n");
        message.append(mQueries->xyzvDataRemoveQuery.lastError().text().toStdString());
        dnNotifyWarning("SQL ERROR", message);
        return false;
    }
    return true;
}


bool QtDNStorageImpl::flush()
{
    if (mDatabase == NULL || !mIsValid)
        return false;

    return mDatabase->database.commit();
}

bool QtDNStorageImpl::startTransaction()
{
    if (mDatabase == NULL || !mIsValid)
        return false;

    return mDatabase->database.transaction();
}

bool QtDNStorageImpl::commitTransaction()
{
    if (mDatabase == NULL || !mIsValid)
        return false;

    return mDatabase->database.commit();
}

bool QtDNStorageImpl::rollbackTransaction()
{
    if (mDatabase == NULL || !mIsValid)
        return false;

    return mDatabase->database.rollback();
}

bool QtDNStorageImpl::initDB()
{
    mDatabase = new QtDNDatabase;

    mDatabase->database = QSqlDatabase::addDatabase("QSQLITE");
    mDatabase->database.setDatabaseName(mStoragePath);

    if (!mDatabase->database.open())
    {
        std::string message = "Failied to open storage file ";
        message.append(mStoragePath.toStdString());
        message.append("\n");
        message.append(mDatabase->database.lastError().text().toStdString());
        dnNotifyError("Persistent storage initialization error.", message);
        return false;
    }

    QStringList tables = mDatabase->database.tables();
    if (!tables.contains("properties"))
    {
        QSqlQuery q;

        if (!q.exec("CREATE TABLE properties(cell TEXT, prokey TEXT, value REAL, PRIMARY KEY (cell, prokey))"))
        {
            std::string message = "Failied to create data table (properties) in ";
            message.append(mStoragePath.toStdString());
            message.append("\n");
            message.append(mDatabase->database.lastError().text().toStdString());
            dnNotifyError("Persistent storage initialization error.", message);
            return false;
        }
    }

    if (!tables.contains("xyzvdata"))
    {
        QSqlQuery q;

        if (!q.exec("CREATE TABLE xyzvdata(cell TEXT, dskey TEXT, dsindex INTEGER, x REAL, y REAL, z REAL, v REAL, PRIMARY KEY (cell, dskey, dsindex))"))
        {
            std::string message = "Failied to create data table (dataset) in ";
            message.append(mStoragePath.toStdString());
            message.append("\n");
            message.append(mDatabase->database.lastError().text().toStdString());
            dnNotifyError("Persistent storage initialization error.", message);
            return false;
        }
    }

    mQueries = new QtDNDBQueries;

    mQueries->propertiesCountQuery.prepare("SELECT count(*) FROM properties WHERE cell=? and prokey=?");
    mQueries->propertiesInsertQuery.prepare("INSERT INTO properties VALUES(?,?,?)");
    mQueries->propertiesUpdateQuery.prepare("UPDATE properties SET value=? WHERE cell=? and prokey=?");
    mQueries->propertiesGetQuery.prepare("SELECT value FROM properties WHERE cell=? and prokey=?");

    mQueries->xyzvDataInsetQuery.prepare("INSERT INTO xyzvdata VALUES (?,?,?, ?,?,?,?)");
    mQueries->xyzvDataCountQuery.prepare("SELECT count(*) FROM xyzvdata WHERE cell=? and dskey=?");
    mQueries->xyzvDataGetQuery.prepare("SELECT dsindex, x, y, z, v FROM xyzvdata WHERE cell=? and dskey=?");
    mQueries->xyzvDataRemoveQuery.prepare("DELETE FROM xyzvdata WHERE cell=? and dskey=?");

    if (mDatabase->database.lastError().type() != QSqlError::NoError)
    {
        std::string message = "Failied to prepare database ";
        message.append(mStoragePath.toStdString());
        message.append("\n");
        message.append(mDatabase->database.lastError().text().toStdString());
        dnNotifyError("Persistent storage initialization error.", message);
        return false;
    }

    return true;
}
