//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#include "dcqtitemmodel.h"

#include <QtGui>

static const QSize s_iconSize(16,16);

DCQtItemModelItem::DCQtItemModelItem(const QVector<QVariant> &data, DCQtItemModelItem *parent)
    : mItemData(data), mParentItem(parent)
{
}

DCQtItemModelItem::DCQtItemModelItem(const QStringList &headers, DCQtItemModelItem *parent)
    :mParentItem(parent)
{
    QVector<QVariant> rootData;
    foreach (QString header, headers)
    {
        rootData << header;
    }
    mItemData = rootData;
}


DCQtItemModelItem::~DCQtItemModelItem()
{
    qDeleteAll(mChildItems);
}

DCQtItemModelItem *DCQtItemModelItem::child(int number)
{
    return mChildItems.value(number);
}

int DCQtItemModelItem::childCount() const
{
    return mChildItems.count();
}

int DCQtItemModelItem::childNumber() const
{
    if (mParentItem)
        return mParentItem->mChildItems.indexOf(const_cast<DCQtItemModelItem*>(this));

    return 0;
}

int DCQtItemModelItem::columnCount() const
{
    return mItemData.count();
}

QVariant DCQtItemModelItem::data(int column) const
{
    return mItemData.value(column);
}

bool DCQtItemModelItem::insertChildren(int position, int count, int columns)
{
    if (position < 0 || position > mChildItems.size())
        return false;

    for (int row = 0; row < count; ++row) {
        QVector<QVariant> data(columns);
        DCQtItemModelItem *item = new DCQtItemModelItem(data, this);
        mChildItems.insert(position, item);
    }

    return true;
}

bool DCQtItemModelItem::insertColumns(int position, int columns)
{
    if (position < 0 || position > mItemData.size())
        return false;

    for (int column = 0; column < columns; ++column)
        mItemData.insert(position, QVariant());

    foreach (DCQtItemModelItem *child, mChildItems)
        child->insertColumns(position, columns);

    return true;
}

DCQtItemModelItem *DCQtItemModelItem::parent()
{
    return mParentItem;
}


bool DCQtItemModelItem::removeChildren(int position, int count)
{
    if (position < 0 || position + count > mChildItems.size())
        return false;

    for (int row = 0; row < count; ++row)
        delete mChildItems.takeAt(position);

    return true;
}


bool DCQtItemModelItem::removeColumns(int position, int columns)
{
    if (position < 0 || position + columns > mItemData.size())
        return false;

    for (int column = 0; column < columns; ++column)
        mItemData.remove(position);

    foreach (DCQtItemModelItem *child, mChildItems)
        child->removeColumns(position, columns);

    return true;
}

bool DCQtItemModelItem::setData(int column, const QVariant &value)
{
    if (column < 0 || column >= mItemData.size())
        return false;

    mItemData[column] = value;
    return true;
}



DCQtItemModel::DCQtItemModel(const QStringList &headers, DCQtItemModelItem *rootItem, QObject *parent)
    : QAbstractItemModel(parent)
{
    QVector<QVariant> rootData;
    foreach (QString header, headers)
    {
        rootData << header;
        mReadOnlyAttrib.push_back(true);
    }

    if (rootItem)
        mRootItem = rootItem;
    else
        mRootItem = new DCQtItemModelItem(rootData);
}

DCQtItemModel::~DCQtItemModel()
{
    delete mRootItem;
}

int DCQtItemModel::columnCount(const QModelIndex & /* parent */) const
{
    return mRootItem->columnCount();
}

QVariant DCQtItemModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid())
        return QVariant();

    DCQtItemModelItem *item = getItem(index);

    if (role == Qt::DecorationRole)
    {
        if (item->data(index.column()).type() == QVariant::Pixmap)
            return item->data(index.column());
    }
    else if (role == Qt::SizeHintRole)
    {
        if (item->data(index.column()).type() == QVariant::Pixmap)
        {
//            return s_iconSize;
        }
    }
    else if (role == Qt::DisplayRole || role == Qt::EditRole)
    {
        if (item->data(index.column()).type() == QVariant::Pixmap)
        {
            return " ";
        }
        else
        {
            return item->data(index.column());
        }
    }

    return QVariant();
}

Qt::ItemFlags DCQtItemModel::flags(const QModelIndex &index) const
{
    if (!index.isValid())
        return 0;

    Qt::ItemFlags flags = Qt::NoItemFlags;

    if (index.column() < mReadOnlyAttrib.size())
    {
        flags |= Qt::ItemIsEnabled | Qt::ItemIsSelectable;

        if (mReadOnlyAttrib[index.column()] == false)
        {
            flags |= Qt::ItemIsEditable;
        }
    }

    return flags;
}

DCQtItemModelItem *DCQtItemModel::getItem(const QModelIndex &index) const
{
    if (index.isValid()) {
        DCQtItemModelItem *item = static_cast<DCQtItemModelItem*>(index.internalPointer());
        if (item) return item;
    }
    return mRootItem;
}

QVariant DCQtItemModel::headerData(int section, Qt::Orientation orientation,
                               int role) const
{
    if (orientation == Qt::Horizontal && role == Qt::DisplayRole)
        return mRootItem->data(section);

    return QVariant();
}

QModelIndex DCQtItemModel::index(int row, int column, const QModelIndex &parent) const
{
    if (parent.isValid() && parent.column() != 0)
        return QModelIndex();

    DCQtItemModelItem *parentItem = getItem(parent);

    DCQtItemModelItem *childItem = parentItem->child(row);
    if (childItem)
        return createIndex(row, column, childItem);
    else
        return QModelIndex();
}

bool DCQtItemModel::insertColumns(int position, int columns, const QModelIndex &parent)
{
    bool success;

    beginInsertColumns(parent, position, position + columns - 1);
    success = mRootItem->insertColumns(position, columns);
    endInsertColumns();

    return success;
}

bool DCQtItemModel::insertRows(int position, int rows, const QModelIndex &parent)
{
    DCQtItemModelItem *parentItem = getItem(parent);
    bool success;

    beginInsertRows(parent, position, position + rows - 1);
    success = parentItem->insertChildren(position, rows, mRootItem->columnCount());
    endInsertRows();

    return success;
}

QModelIndex DCQtItemModel::parent(const QModelIndex &index) const
{
    if (!index.isValid())
        return QModelIndex();

    DCQtItemModelItem *childItem = getItem(index);
    DCQtItemModelItem *parentItem = childItem->parent();

    if (parentItem == mRootItem)
        return QModelIndex();

    return createIndex(parentItem ? parentItem->childNumber() : 0, 0, parentItem);
}

bool DCQtItemModel::removeColumns(int position, int columns, const QModelIndex &parent)
{
    bool success;

    beginRemoveColumns(parent, position, position + columns - 1);
    success = mRootItem->removeColumns(position, columns);
    endRemoveColumns();

    if (mRootItem->columnCount() == 0)
        removeRows(0, rowCount());

    return success;
}

bool DCQtItemModel::removeRows(int position, int rows, const QModelIndex &parent)
{
    if (rows == 0)
        return true;

    DCQtItemModelItem *parentItem = getItem(parent);
    bool success = true;

    beginRemoveRows(parent, position, position + rows - 1);
    success = parentItem->removeChildren(position, rows);
    endRemoveRows();

    return success;
}

bool DCQtItemModel::removeAllItems()
{
    return removeRows(0, rowCount());
}

int DCQtItemModel::rowCount(const QModelIndex &parent) const
{
    DCQtItemModelItem *parentItem = getItem(parent);

    return parentItem->childCount();
}

bool DCQtItemModel::setData(const QModelIndex &index, const QVariant &value,
                        int role)
{
    if (role != Qt::EditRole)
        return false;

    DCQtItemModelItem *item = getItem(index);
    bool result = item->setData(index.column(), value);

    if (result)
        emit dataChanged(index, index);

    return result;
}

bool DCQtItemModel::setHeaderData(int section, Qt::Orientation orientation,
                              const QVariant &value, int role)
{
    if (role != Qt::EditRole || orientation != Qt::Horizontal)
        return false;

    bool result = mRootItem->setData(section, value);

    if (result)
        emit headerDataChanged(orientation, section, section);

    return result;
}

bool DCQtItemModel::insertStringList(const QStringList &stringList, const QModelIndex &parent)
{
    DCQtItemModelItem *parentItem = getItem(parent);
    int position = parentItem->childCount();

    insertRows(position,1,parent);
    for (int i = 0; i < stringList.length(); i++)
    {
        setData(index(position,i,parent),QVariant(stringList.at(i)));
    }
    return true;
}

bool DCQtItemModel::insertString(const QString &string, const QModelIndex &parent)
{
    DCQtItemModelItem *parentItem = getItem(parent);
    int position = parentItem->childCount();
    insertRows(position,1,parent);
    setData(index(position,0,parent),QVariant(string));
    return true;
}


void DCQtItemModel::setReadOnly(int column, bool isReadOnly)
{
    if (column < mReadOnlyAttrib.size())
    {
        mReadOnlyAttrib[column] = isReadOnly;
    }
}
