//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#include "dcvcaxon.h"

#include "dcaxon.h"
#include "dcaxonterminal.h"
#include "dccell.h"
#include "dcvpagecomponent.h"
#include "dccuberenderer.h"

#include <math.h>

static const double PI = 3.14159265358979323846264338327950288419717;
static DCCubeRenderer *s_shapeTerminalSocket = NULL;

DCVCAxon::DCVCAxon(DCAxon *owner) : d_owner(owner), d_shouldUpdateShape(true)
{
    // default size
    owner->setViewLength(owner->getOwnerCell()->getViewSize()/2 + 0.1);
}

DCVCAxon::~DCVCAxon()
{
}

void DCVCAxon::prepareChildrenForDraw(bool isAnimationInterval)
{
    int len =  d_owner->getNumberOfTerminals();
    float matrix[16];
    glGetFloatv(GL_MODELVIEW_MATRIX, matrix);
     for (int i = 0; i < len; ++i)
     {
         d_owner->getTerminalAt(i)->setViewMatrixForAxonPoint(matrix);
         d_owner->getTerminalAt(i)->getVComponent()->prepareForDraw(isAnimationInterval);
     }
     if (d_owner->hasEditingTerminal())
     {
         d_owner->setEditingCursorViewMatrix(matrix);
     }
}

void DCVCAxon::drawChildren(bool isAnimationInterval)
{
    int len =  d_owner->getNumberOfTerminals();
    for (int i = 0; i < len; ++i)
    {
        d_owner->getTerminalAt(i)->getVComponent()->draw(isAnimationInterval);
    }
    if (d_owner->hasEditingTerminal())
    {
        d_owner->getEditingTerminal()->draw(isAnimationInterval);
    }
}

void DCVCAxon::drawChildrenForSelection(QList<DCVComponent *> *itemList)
{
    int len =  d_owner->getNumberOfTerminals();
    for (int i = 0; i < len; ++i)
    {
        d_owner->getTerminalAt(i)->getVComponent()->drawForSelection(itemList);
    }
    if (d_owner->hasEditingTerminal())
    {
        d_owner->getEditingTerminal()->drawForSelection(itemList);
    }
}

void DCVCAxon::translate()
{
    float rad = d_owner->getViewAngle() / 180 * PI;
    float length = d_owner->getViewLength();
    glTranslatef(cos(rad)* length, -0.245f, sin(rad)* length);
}

void DCVCAxon::setSelected(bool selected, bool updateChildren)
{
    DCVComponent::setSelected(selected, updateChildren);

    DCCell *cell = getOwnerCell();
    if (cell)
        cell->getVComponent()->setSelected(selected, false);

    if (updateChildren)
    {
        int len =  d_owner->getNumberOfTerminals();
        for (int i = 0; i < len; ++i)
        {
            d_owner->getTerminalAt(i)->getVComponent()->setSelected(selected, true);
        }
    }
}

void DCVCAxon::setVisible(DCVVisibility visibleSelf, DCVVisibility visibleChildren)
{
    DCVComponent::setVisible(visibleSelf, visibleChildren);
    d_owner->setTerminalVisible(visibleChildren != DCV_VISIBLE_NONE);
}

DCVCPage* DCVCAxon::getPageBelonging() const
{
    return getOwnerCell()->getPageBelonging();
}

void DCVCAxon::renderOwnShape(bool isAnimationInterval, bool renderAsWireframe)
{
    (void)isAnimationInterval;

    float rad = d_owner->getViewAngle() / 180 * PI;
    float length = d_owner->getViewLength();
    if (renderAsWireframe)
        glColor4f(1.0, 1.0, 1.0, 0.5);
    else
        glColor4f(1.0, 1.0, 1.0, 1.0);

    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glEnable( GL_LINE_SMOOTH );
    glHint( GL_LINE_SMOOTH_HINT, GL_NICEST );

    glBegin(GL_LINES);
    glVertex3f(0, 0, 0);
    glVertex3f(-cos(rad)* length, 0, -sin(rad)* length);
    glEnd();

    glTranslatef(0,0.05f,0);

    if (renderAsWireframe)
    {
        //render bounding box
        float hwidth = 0.05f;
        float hheight = 0.05f;
        glLineWidth(1);
        glBegin(GL_LINE_LOOP);
        glVertex3f(hwidth, hheight, hwidth);
        glVertex3f(-hwidth, hheight, hwidth);
        glVertex3f(-hwidth, hheight, -hwidth);
        glVertex3f(hwidth, hheight, -hwidth);
        glEnd();
        glBegin(GL_LINE_LOOP);
        glVertex3f(hwidth, -hheight, hwidth);
        glVertex3f(-hwidth, -hheight, hwidth);
        glVertex3f(-hwidth, -hheight, -hwidth);
        glVertex3f(hwidth, -hheight, -hwidth);
        glEnd();
        glBegin(GL_LINES);
        glVertex3f(hwidth, -hheight, hwidth);
        glVertex3f(hwidth, hheight, hwidth);
        glVertex3f(-hwidth, -hheight, hwidth);
        glVertex3f(-hwidth, hheight, hwidth);
        glVertex3f(-hwidth, -hheight, -hwidth);
        glVertex3f(-hwidth, hheight, -hwidth);
        glVertex3f(hwidth, -hheight, -hwidth);
        glVertex3f(hwidth, hheight, -hwidth);
        glEnd();
    }
    else
    {
        if (s_shapeTerminalSocket == NULL)
        {
            s_shapeTerminalSocket = new DCCubeRenderer(0.1f, 0.1f, 0,0, true);
            s_shapeTerminalSocket->setFaceColor(0,0,0);
        }
        s_shapeTerminalSocket->draw();
    }
    glDisable(GL_BLEND);
}

DCCell* DCVCAxon::getOwnerCell() const
{
    return d_owner->getOwnerCell();
}

bool DCVCAxon::startDrag(float x, float y, float z, bool isResizingDrag)
{
    (void)x; (void)y; (void)z; (void)isResizingDrag;

    return true;
}

bool DCVCAxon::dragging(float x, float y, float z, bool isResizingDrag)
{
    (void)x; (void)y; (void)z; (void)isResizingDrag;

    if (isResizingDrag)
        return false;

    DCCell *cell = getOwnerCell();
    float cellx = cell->getViewPageX();
    float celly = cell->getViewPageY();

    float dx = x - cellx;
    float dy = z - celly;

    float len, minX, minY, minLenP, lenP;
    cell->getViewHCrossPoint(dx,dy,&minX, &minY);
    minLenP = minX * minX + minY * minY;
    lenP = dx * dx + dy * dy;
    len = minLenP < lenP ? sqrt(lenP) : sqrt(minLenP);
    float angle = atan2(dy,dx) / PI * 180;

    d_owner->setViewLength(len);
    d_owner->setViewAngle(angle);

    return true;
}

bool DCVCAxon::endDrag(float x, float y, float z, bool isResizingDrag)
{
    (void)x; (void)y; (void)z; (void)isResizingDrag;

    return true;

}

void DCVCAxon::updateShape()
{
    d_shouldUpdateShape = true;
}

void DCVCAxon::saveAttributesToXML(QDomDocument *document, QDomElement* element) const
{
    (void)document; (void)element;

    element->setAttribute("length", d_owner->getViewLength());
    element->setAttribute("angle", d_owner->getViewAngle());
}

void DCVCAxon::loadAttributesFromXML(QDomElement element)
{
    if (element.hasAttribute("length"))
        d_owner->setViewLength(element.attribute("length","0.6").toFloat());

    if (element.hasAttribute("angle"))
        d_owner->setViewAngle(element.attribute("angle", "0").toFloat());
}
