//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#ifndef DCCREATOR_H
#define DCCREATOR_H

class DCContent;
class DCVCPage;
class DCCell;
class DCCellCode;
class DCAxon;
class DCReceptor;
class DCAxonTerminal;
class DCScene;
class DCVComponent;
class DCContainer;
class DCCommand;
class DCConsole;

#include <QtGui>

class DCCreator : public QObject
{
    Q_OBJECT

public:
    enum DCPersMode { DC_PERSMODE_NAVIGATION, DC_PERSMODE_PAGEEDIT };

private:
    QMainWindow         *d_mainWindow;
    DCContent           *d_vcontent;
    DCScene             *d_scene;
    DCPersMode          d_persMode;
    QString             d_contentRootPath;

    QUndoStack          *d_undoStack;
    DCConsole           *d_console;

public:
    DCCreator(QMainWindow *mainwindow);
    virtual ~DCCreator();

    DCContent*      getCurrentContent() const { return d_vcontent; }
    DCContainer*    getCurrentContainer() const;
    QMainWindow*    getMainWindow() const { return d_mainWindow; }
    DCScene*        getScene() const { return d_scene; }
    DCPersMode      getPersMode() const { return d_persMode; }

    virtual bool event(QEvent *event);

    void resetVisualizer();
    void initMode();

    bool loadContent(const QString &contentRoot);
    bool savePage(DCVCPage *page, bool showResultInMessageBox = false);
    bool savePage(const QSet<DCVCPage*>& pages, bool showResultInMessageBox = false);
    bool saveAll(bool showResultInMessageBox = false);

    void selectPage(const void *requester, DCVCPage *page, bool multipleSelection);
    void unselectPage(const void *requester, DCVCPage *page);
    bool selectPage(const void *requester, const QString &locationPath, bool multipleSelection);
    bool unselectPage(const void *requester, const QString &locationPath);
    void unselectPageAll(const void *requester);

    void selectCellObject(const void *requester, DCVComponent *object, bool multipleSelection = false);
    void unselectCellObject(const void *requester, DCVComponent *object);
    void unselectCellObjectAll(const void *requester);

    void changeSceneScale(const void *requester, float scale);
    void rotateScene(const void *requester, float xangle_delta, float yangle_delta);
    void translateBrowsModeScene(const void *requester, float dx, float dy);
    void translateEditModeScene(const void *requester, float dx, float dy);
    void resetSceneTranslation();

    bool changePersMode(const void *requester, DCPersMode mode);


    void doUndo(const void *requester);
    void doUndoImmidiate();

    //Editing actions
    void doCommandStartAddAxonTerminalFromAxon(const void *requester, DCCell *axnoCell);
    void doCommandStartAddAxonTerminalFromReceptor(const void *requester, DCCell *receptorCell);
    void doCommandCommitAddAxonTerminal(const void *requester, DCAxon *axon, DCCell *receptorCell);
    void doCommandCommitAddAxonTerminal(const void *requester, DCCell *axonCell, DCReceptor *receptor);
    void doCommandCancelAddAxonTerminal(const void *requester);
    void doCommandRemoveAxonTerminal(const void *requester, DCCell *axonCell, DCAxonTerminal *axonTerminal);
    void doCommandRemoveAxonTerminal(const void *requester, DCCell *receptorCell, const QString& receptorName);
    void doCommandAssignCellCodeClassToCell(const void *requester, DCCell *cell, DCCellCode *cellCode);
    void doCommandUnassignCellCodeClassFromCell(const void *requester, DCCell *cell);

    void doCommandAddCellCodeClass(const void *requester, DCContainer *container, const QString& name, const QString& type);
    void doCommandChangeCellCodeClassType(const void *requester, DCCellCode *cellCode, const QString& newType);
    void doCommandAddCell(const void *requester, DCContainer *container, const QString& containerBasedPath, const QString& name, const QString& type, float pageX, float pageY);
    void doCommandRenameCell(const void *requester, DCCell *cell, const QString& newContainerBasedPath, const QString& newName);
    void doCommandChangeCellType(const void *requester, DCCell *cell, const QString& newType);
    void doCommandRemoveCell(const void *requester, DCContainer *container, DCCell* cell);
    void doCommandRemoveCellCode(const void *requester, DCContainer *container, DCCellCode* cellCode);
    void doCommandAddPage(const void *requester, const QString& containerBasedPath);
    void doCommandMovePage(const void *requester, const QString &oldContainerBasedPath, const QString &newContainerBasedPath);
    void doCommandRemovePage(const void *requester, DCVCPage *page);

    void doCommandAddDirectory(const void *requester, const QString& sysFilePath);
    void doCommandRenameDirectory(const void *requester, const QString& oldSysFilePath, const QString& newSysFilePath);
    void doCommandRemoveDirectory(const void *requester, const QString& sysFilePath);

    bool doCommandRenameReceptorName(const void *requester, DCCell *cell, const QString& oldName, const QString& newName, bool doImmediate = false);

private slots:
    void slotSceneSelectedPageChanged(const void *requester);
    void slotSceneSelectedCellObjectChanged(const void *requester);
    void slotSceneViewScaleChanged(const void *requester);
    void slotSceneViewAngleChanged(const void *requester);
    void slotSceneViewCenterChanged(const void *requester);
    void slotSceneViewSettingChanged(const void *requester);
    void slotSceneViewEditModeChanged(const void *requester);

signals:
    void contentRootPathChanged(const void *requester, QString rootPath);
    void sceneChanged(const void *requester, DCScene *scene);
    void sceneSelectedPageChanged(const void *requester, const DCScene *scene);
    void sceneSelectedCellChanged(const void *requester, const DCScene *scene);
    void sceneViewScaleChanged(const void *requester, const DCScene *scene);
    void sceneViewAngleChanged(const void *requester, const DCScene *scene);
    void sceneViewCenterChanged(const void *requester, const DCScene *scene);
    void sceneViewTranslationChanged(const void *requester, const DCScene *scene);
    void sceneViewSettingChanged(const void *requester, const DCScene *scene);
    void sceneViewEditModeChanged(const void *requester, const DCScene *scene);

    void commandExecuted(const QUndoCommand *executedCommand);
};

#endif // DCCREATOR_H
