﻿using NUnit.Framework;
using Sasa.QualityTools.DrFx.Core.Tests.Unit.TestingUtilities;
using Sasa.QualityTools.DrFx.Core.Translation;
using System;
using System.Collections.Generic;
using System.IO;
using System.Xml;

namespace Sasa.QualityTools.DrFx.Core.Tests.Unit.Translation
{
    [TestFixture]
    public class TranslatorTest
    {
        /// <summary>
        /// 単一メッセージの翻訳が期待通り動作することをテストします。
        /// </summary>
        /// <param name="issue">翻訳対象の違反メッセージ。</param>
        /// <returns>翻訳結果。</returns>
        [TestCase("Microsoft.XXX", "CA0000", "Original 'XXX' Original",
            Result = "Translated 'XXX' Translated",
            Description = "翻訳ルールに一致するメッセージが翻訳されることをテストします。")]
        [TestCase("Microsoft.XX", "CA0000", "Original 'XXX' Original",
            Result = "Original 'XXX' Original",
            Description = "カテゴリ－が一致しない場合は翻訳されないことをテストします。")]
        [TestCase("Microsoft.XXX", "CA0001", "Original 'XXX' Original",
            Result = "Original 'XXX' Original",
            Description = "チェック ID が一致しない場合は翻訳されないことをテストします。")]
        [TestCase("Microsoft.XXX", "CA0000", "org 'XXX' org",
            Result = "org 'XXX' org",
            Description = "メッセージパターンが一致しない場合は翻訳されないことをテストします。")]
        public string TestTranslateSingleMessage(string category, string checkId, string issue)
        {
            string ruleContents =
                @"<?xml version=""1.0"" encoding=""utf-8""?>" +
                @"<Rules>" +
                @"  <Rule CheckId=""CA0000"" Category=""Microsoft.XXX"">" +
                @"    <Name>TranslatedName</Name>" +
                @"    <Resolution>" +
                @"      <Content>Translated {0} Translated</Content>" +
                @"      <Pattern>^Original (.+?) Original$</Pattern>" +
                @"    </Resolution>" +
                @"  </Rule>" +
                @"</Rules>";

            Translator translator = new Translator();
            Stream stream = StreamGenerator.GenerateWith(ruleContents);
            translator.AddRules(DrFxRuleLoader.Load(stream));

            return translator.TranslateIssue(category, checkId, issue);
        }

        /// <summary>
        /// FxCop レポートがルールに従って翻訳されることをテストします。
        /// </summary>
        [Test]
        [Description(" FxCop レポートがルールに従って翻訳されることをテストします。")]
        public void TestTranslateReport()
        {
            string reportContents =
                @"<?xml version=""1.0"" encoding=""utf-8""?>" +
                @"<FxCopReport Version=""1.36"">" +
                @"  <Message CheckId=""CA0000"" Category=""Microsoft.XXX"" TypeName=""Original"">" +
                @"    <Issue>Original '111' Original</Issue>" +
                @"  </Message>" +
                @"  <Rules>" +
                @"    <Rule CheckId=""CA0000"" Category=""Microsoft.XXX"">" +
                @"      <Name>OriginalName</Name>" +
                @"    </Rule>" +
                @"  </Rules>" +
                @"</FxCopReport>";
            string ruleContents =
                @"<?xml version=""1.0"" encoding=""utf-8""?>" +
                @"<Rules>" +
                @"  <Rule CheckId=""CA0000"" Category=""Microsoft.XXX"">" +
                @"    <Name>TranslatedName</Name>" +
                @"    <Resolution>" +
                @"      <Content>Translated {0} Translated</Content>" +
                @"      <Pattern>^Original (.+?) Original$</Pattern>" +
                @"    </Resolution>" +
                @"  </Rule>" +
                @"</Rules>";
            string expected =
                @"<?xml version=""1.0"" encoding=""utf-8""?>" + "\r\n" +
                @"<FxCopReport Version=""1.36"">" + "\r\n" +
                @"  <Message CheckId=""CA0000"" Category=""Microsoft.XXX"" TypeName=""Original"">" + "\r\n" +
                @"    <Issue>Translated '111' Translated</Issue>" + "\r\n" +
                @"  </Message>" + "\r\n" +
                @"  <Rules>" + "\r\n" +
                @"    <Rule CheckId=""CA0000"" Category=""Microsoft.XXX"">" + "\r\n" +
                @"      <Name>TranslatedName</Name>" + "\r\n" +
                @"    </Rule>" + "\r\n" +
                @"  </Rules>" + "\r\n" +
                @"</FxCopReport>";


            Translator translator = new Translator();
            Stream stream = StreamGenerator.GenerateWith(ruleContents);
            translator.AddRules(DrFxRuleLoader.Load(stream));

            Assert.AreEqual(expected, translator.Translate(reportContents));
        }

        [Test]
        public void TestConvertXmlDocumentToString()
        {
            const string content = @"<?xml version=""1.0"" encoding=""utf-8""?><Engines><Google/><Bing/><Yahoo/></Engines>";
            const string expected =
                @"<?xml version=""1.0"" encoding=""utf-8""?>" + "\r\n" +
                @"<Engines>" + "\r\n" +
                @"  <Google />" + "\r\n" +
                @"  <Bing />" + "\r\n" +
                @"  <Yahoo />" + "\r\n" +
                @"</Engines>";

            XmlDocument doc = new XmlDocument();
            doc.LoadXml(content);
            string actual = Translator.ConvertXmlDocumentToString(doc);

            Assert.AreEqual(expected, actual);
        }
    }
}
