﻿/*
 * DrFx - FxCop Report Translator and Visualizer.
 * Copyright (C) 2010 Sasa Yuan
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */



using System;

namespace Sasa.QualityTools.DrFx.Console
{
    /// <summary>
    /// オプション定義を表します。
    /// </summary>
    internal class Option
    {
        /// <summary>
        /// オプションのシンボル。
        /// </summary>
        internal char Symbol { get; private set; }

        /// <summary>
        /// オプションが引数を持つかどうか。
        /// </summary>
        internal bool HasArgument { get; private set; }



        /// <summary>
        /// 引数で指定された値を使用して新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="symbol">オプションのシンボル。</param>
        /// <param name="hasArgument">オプションが引数を持つかどうか。</param>
        internal Option(char symbol, bool hasArgument)
        {
            Symbol = symbol;
            HasArgument = hasArgument;
        }



        /// <summary>
        /// 指定された <paramref name="option"/> と等値であるかどうかを比較します。
        /// </summary>
        /// <param name="option">オプション定義を表す値。</param>
        /// <returns>シンボルと引数の有無を表す値が完全に一致する場合は true。そうでない場合は false。</returns>
        public bool Equals(Option option)
        {
            return (Object.Equals(option, null)) ? false : Symbol == option.Symbol && HasArgument == option.HasArgument;
        }

        /// <summary>
        /// 指定されたオブジェクトと等値であるかどうかを比較します。
        /// </summary>
        /// <param name="obj">任意のオブジェクト。</param>
        /// <returns><paramref name="obj"/> が <see cref="Sasa.QualityTools.DrFx.Console.Option"/> 型であり、
        /// シンボルと引数の有無を表す値が完全に一致する場合は true。そうでない場合は false。</returns>
        public override bool Equals(object obj)
        {
            Option option = obj as Option;
            return Equals(option);
        }

        /// <summary>
        /// このインスタンスのハッシュコードを取得します。
        /// </summary>
        /// <returns>このインスタンスのハッシュを表す値。</returns>
        public override int GetHashCode()
        {
            return Symbol.GetHashCode() ^ HasArgument.GetHashCode();
        }

        /// <summary>
        /// <paramref name="option1"/> と <paramref name="option2"/> が等値であるかどうかを比較します。
        /// </summary>
        /// <param name="option1">比較対象のインスタンス。</param>
        /// <param name="option2">比較対象のインスタンス。</param>
        /// <returns>指定された 2 つのインスタンスが等値の場合は true。そうでない場合は false。</returns>
        public static bool operator ==(Option option1, Option option2)
        {
            if (Object.Equals(option1, null) && Object.Equals(option2, null))
            {
                return true;
            }
            else
            {
                return (Object.Equals(option1, null)) ? option2.Equals(option1) : option1.Equals(option2);
            }
        }

        /// <summary>
        /// <paramref name="option1"/> と <paramref name="option2"/> が等値でないかどうかを比較します。
        /// </summary>
        /// <param name="option1">比較対象のインスタンス。</param>
        /// <param name="option2">比較対象のインスタンス。</param>
        /// <returns>指定された 2 つのインスタンスが異なる値の場合は true。そうでない場合は false。</returns>
        public static bool operator !=(Option option1, Option option2)
        {
            return !(option1 == option2);
        }
    }
}
