/*
* CVS identifier:
*
* $Id: BitstreamReaderAgent.java,v 1.27 2002/07/25 14:59:32 grosbois Exp $
*
* Class:                   BitstreamReaderAgent
*
* Description:             The generic interface for bit stream
*                          transport agents.
*
*
*
* COPYRIGHT:
* 
* This software module was originally developed by Raphal Grosbois and
* Diego Santa Cruz (Swiss Federal Institute of Technology-EPFL); Joel
* Askelf (Ericsson Radio Systems AB); and Bertrand Berthelot, David
* Bouchard, Flix Henry, Gerard Mozelle and Patrice Onno (Canon Research
* Centre France S.A) in the course of development of the JPEG2000
* standard as specified by ISO/IEC 15444 (JPEG 2000 Standard). This
* software module is an implementation of a part of the JPEG 2000
* Standard. Swiss Federal Institute of Technology-EPFL, Ericsson Radio
* Systems AB and Canon Research Centre France S.A (collectively JJ2000
* Partners) agree not to assert against ISO/IEC and users of the JPEG
* 2000 Standard (Users) any of their rights under the copyright, not
* including other intellectual property rights, for this software module
* with respect to the usage by ISO/IEC and Users of this software module
* or modifications thereof for use in hardware or software products
* claiming conformance to the JPEG 2000 Standard. Those intending to use
* this software module in hardware or software products are advised that
* their use may infringe existing patents. The original developers of
* this software module, JJ2000 Partners and ISO/IEC assume no liability
* for use of this software module or modifications thereof. No license
* or right to this software module is granted for non JPEG 2000 Standard
* conforming products. JJ2000 Partners have full right to use this
* software module for his/her own purpose, assign or donate this
* software module to any third party and to inhibit third parties from
* using this software module for non JPEG 2000 Standard conforming
* products. This copyright notice must be included in all copies or
* derivative works of this software module.
* 
* Copyright (c) 1999/2000 JJ2000 Partners.
* */
using System;
using CSJ2K.j2k.quantization.dequantizer;
using CSJ2K.j2k.wavelet.synthesis;
using CSJ2K.j2k.entropy.decoder;
using CSJ2K.j2k.codestream;
using CSJ2K.j2k.wavelet;
using CSJ2K.j2k.decoder;
using CSJ2K.j2k.image;
using CSJ2K.j2k.util;
using CSJ2K.j2k.io;
using CSJ2K.j2k;
namespace CSJ2K.j2k.codestream.reader
{
	
	/// <summary> This is the generic interface for bit stream reader agents. A bit stream
	/// reader agent is an entity that allows reading from a bit stream and
	/// requesting compressed code-blocks. It can be a simple file reader, or a
	/// network connection, or anything else.
	/// 
	/// <p>The bit stream reader agent allows to make request for compressed block
	/// data in any order. The amount of data returned would normally depend on the
	/// data available at the time of the request, be it from a file or from a
	/// network connection.</p>
	/// 
	/// <p>The bit stream reader agent has the notion of a current tile, and
	/// coordinates are relative to the current tile, where applicable.</p>
	/// 
	/// <p>Resolution level 0 is the lowest resolution level, i.e. the LL subband
	/// alone.</p>
	/// 
	/// </summary>
	public abstract class BitstreamReaderAgent : CodedCBlkDataSrcDec
	{
		/// <summary> Returns the horizontal code-block partition origin. Allowable values
		/// are 0 and 1, nothing else.
		/// 
		/// </summary>
		virtual public int CbULX
		{
			get
			{
				return hd.CbULX;
			}
			
		}
		/// <summary> Returns the vertical code-block partition origin. Allowable values are
		/// 0 and 1, nothing else.
		/// 
		/// </summary>
		virtual public int CbULY
		{
			get
			{
				return hd.CbULY;
			}
			
		}
		/// <summary> Returns the number of components in the image.
		/// 
		/// </summary>
		/// <returns> The number of components in the image.
		/// 
		/// </returns>
		virtual public int NumComps
		{
			get
			{
				return nc;
			}
			
		}
		/// <summary> Returns the index of the current tile, relative to a standard scan-line
		/// order.
		/// 
		/// </summary>
		/// <returns> The current tile's index (starts at 0).
		/// 
		/// </returns>
		virtual public int TileIdx
		{
			get
			{
				return ctY * ntX + ctX;
			}
			
		}
		/// <summary> Returns the parameters that are used in this class and implementing
		/// classes. It returns a 2D String array. Each of the 1D arrays is for a
		/// different option, and they have 3 elements. The first element is the
		/// option name, the second one is the synopsis and the third one is a long
		/// description of what the parameter is. The synopsis or description may
		/// be 'null', in which case it is assumed that there is no synopsis or
		/// description of the option, respectively. Null may be returned if no
		/// options are supported.
		/// 
		/// </summary>
		/// <returns> the options name, their synopsis and their explanation, or null
		/// if no options are supported.
		/// 
		/// </returns>
		public static System.String[][] ParameterInfo
		{
			get
			{
				return pinfo;
			}
			
		}
		/// <summary> Returns the image resolution level to reconstruct from the
		/// codestream. This value cannot be computed before every main and tile
		/// headers are read.
		/// 
		/// </summary>
		/// <returns> The image  resolution level
		/// 
		/// </returns>
		virtual public int ImgRes
		{
			get
			{
				return targetRes;
			}
			
		}
		/// <summary> Return the target decoding rate in bits per pixel.
		/// 
		/// </summary>
		/// <returns> Target decoding rate in bpp.
		/// 
		/// </returns>
		virtual public float TargetRate
		{
			get
			{
				return trate;
			}
			
		}
		/// <summary> Return the actual decoding rate in bits per pixel.
		/// 
		/// </summary>
		/// <returns> Actual decoding rate in bpp.
		/// 
		/// </returns>
		virtual public float ActualRate
		{
			get
			{
				arate = anbytes * 8f / hd.MaxCompImgWidth / hd.MaxCompImgHeight;
				return arate;
			}
			
		}
		/// <summary> Return the target number of read bytes.
		/// 
		/// </summary>
		/// <returns> Target decoding rate in bytes.
		/// 
		/// </returns>
		virtual public int TargetNbytes
		{
			get
			{
				return tnbytes;
			}
			
		}
		/// <summary> Return the actual number of read bytes.
		/// 
		/// </summary>
		/// <returns> Actual decoding rate in bytes.
		/// 
		/// </returns>
		virtual public int ActualNbytes
		{
			get
			{
				return anbytes;
			}
			
		}
		/// <summary>Returns the horizontal offset of tile partition </summary>
		virtual public int TilePartULX
		{
			get
			{
				return hd.getTilingOrigin(null).x;
			}
			
		}
		/// <summary>Returns the vertical offset of tile partition </summary>
		virtual public int TilePartULY
		{
			get
			{
				return hd.getTilingOrigin(null).y;
			}
			
		}
		/// <summary>Returns the nominal tile width </summary>
		virtual public int NomTileWidth
		{
			get
			{
				return hd.NomTileWidth;
			}
			
		}
		/// <summary>Returns the nominal tile height </summary>
		virtual public int NomTileHeight
		{
			get
			{
				return hd.NomTileHeight;
			}
			
		}
		
		/// <summary>The decoder specifications </summary>
		protected internal DecoderSpecs decSpec;
		
		/// <summary> Whether or not the components in the current tile uses a derived
		/// quantization step size (only relevant in non reversible quantization
		/// mode). This field is actualized by the setTile method in
		/// FileBitstreamReaderAgent.
		/// 
		/// </summary>
		/// <seealso cref="FileBitstreamReaderAgent.initSubbandsFields">
		/// 
		/// </seealso>
		protected internal bool[] derived = null;
		
		/// <summary> Number of guard bits off all component in the current tile. This field
		/// is actualized by the setTile method in FileBitstreamReaderAgent.
		/// 
		/// </summary>
		/// <seealso cref="FileBitstreamReaderAgent.initSubbandsFields">
		/// 
		/// </seealso>
		protected internal int[] gb = null;
		
		/// <summary> Dequantization parameters of all subbands and all components in the
		/// current tile. The value is actualized by the setTile method in
		/// FileBitstreamReaderAgent.
		/// 
		/// </summary>
		/// <seealso cref="FileBitstreamReaderAgent.initSubbandsFields">
		/// 
		/// </seealso>
		protected internal StdDequantizerParams[] params_Renamed = null;
		
		/// <summary>The prefix for bit stream reader options: 'B' </summary>
		public const char OPT_PREFIX = 'B';
		
		/// <summary>The list of parameters that is accepted by the bit stream
		/// readers. They start with 'B'. 
		/// </summary>
		//UPGRADE_NOTE: Final was removed from the declaration of 'pinfo'. "ms-help://MS.VSCC.v80/dv_commoner/local/redirect.htm?index='!DefaultContextWindowIndex'&keyword='jlca1003'"
		private static readonly System.String[][] pinfo = null;
		
		/// <summary> The maximum number of decompostion levels for each component of the
		/// current tile. It means that component c has mdl[c]+1 resolution levels
		/// (indexed from 0 to mdl[c])
		/// 
		/// </summary>
		protected internal int[] mdl;
		
		/// <summary>The number of components </summary>
		//UPGRADE_NOTE: Final was removed from the declaration of 'nc '. "ms-help://MS.VSCC.v80/dv_commoner/local/redirect.htm?index='!DefaultContextWindowIndex'&keyword='jlca1003'"
		protected internal int nc;
		
		/// <summary>Image resolution level to generate </summary>
		protected internal int targetRes;
		
		/// <summary> The subband trees for each component in the current tile. Each element
		/// in the array is the root element of the subband tree for a
		/// component. The number of magnitude bits in each subband (magBits member
		/// variable) is not initialized.
		/// 
		/// </summary>
		protected internal SubbandSyn[] subbTrees;
		
		/// <summary>The image width on the hi-res reference grid </summary>
		//UPGRADE_NOTE: Final was removed from the declaration of 'imgW '. "ms-help://MS.VSCC.v80/dv_commoner/local/redirect.htm?index='!DefaultContextWindowIndex'&keyword='jlca1003'"
		protected internal int imgW;
		
		/// <summary>The image width on the hi-res reference grid </summary>
		//UPGRADE_NOTE: Final was removed from the declaration of 'imgH '. "ms-help://MS.VSCC.v80/dv_commoner/local/redirect.htm?index='!DefaultContextWindowIndex'&keyword='jlca1003'"
		protected internal int imgH;
		
		/// <summary>The horizontal coordinate of the image origin in the canvas system, on
		/// the reference grid. 
		/// </summary>
		//UPGRADE_NOTE: Final was removed from the declaration of 'ax '. "ms-help://MS.VSCC.v80/dv_commoner/local/redirect.htm?index='!DefaultContextWindowIndex'&keyword='jlca1003'"
		protected internal int ax;
		
		/// <summary>The vertical coordinate of the image origin in the canvas system, on
		/// the reference grid. 
		/// </summary>
		//UPGRADE_NOTE: Final was removed from the declaration of 'ay '. "ms-help://MS.VSCC.v80/dv_commoner/local/redirect.htm?index='!DefaultContextWindowIndex'&keyword='jlca1003'"
		protected internal int ay;
		
		/// <summary>The horizontal coordinate of the tiling origin in the canvas system, on
		/// the reference grid. 
		/// </summary>
		//UPGRADE_NOTE: Final was removed from the declaration of 'px '. "ms-help://MS.VSCC.v80/dv_commoner/local/redirect.htm?index='!DefaultContextWindowIndex'&keyword='jlca1003'"
		protected internal int px;
		
		/// <summary>The vertical coordinate of the tiling origin in the canvas system, on
		/// the reference grid. 
		/// </summary>
		//UPGRADE_NOTE: Final was removed from the declaration of 'py '. "ms-help://MS.VSCC.v80/dv_commoner/local/redirect.htm?index='!DefaultContextWindowIndex'&keyword='jlca1003'"
		protected internal int py;
		
		/// <summary>The horizontal offsets of the upper-left corner of the current tile
		/// (not active tile) with respect to the canvas origin, in the component
		/// hi-res grid, for each component. 
		/// </summary>
		//UPGRADE_NOTE: Final was removed from the declaration of 'offX '. "ms-help://MS.VSCC.v80/dv_commoner/local/redirect.htm?index='!DefaultContextWindowIndex'&keyword='jlca1003'"
		protected internal int[] offX;
		
		/// <summary>The vertical offsets of the upper-left corner of the current tile (not
		/// active tile) with respect to the canvas origin, in the component hi-res
		/// grid, for each component. 
		/// </summary>
		//UPGRADE_NOTE: Final was removed from the declaration of 'offY '. "ms-help://MS.VSCC.v80/dv_commoner/local/redirect.htm?index='!DefaultContextWindowIndex'&keyword='jlca1003'"
		protected internal int[] offY;
		
		/// <summary>The horizontal coordinates of the upper-left corner of the active
		/// tile, with respect to the canvas origin, in the component hi-res grid,
		/// for each component. 
		/// </summary>
		//UPGRADE_NOTE: Final was removed from the declaration of 'culx '. "ms-help://MS.VSCC.v80/dv_commoner/local/redirect.htm?index='!DefaultContextWindowIndex'&keyword='jlca1003'"
		protected internal int[] culx;
		
		/// <summary>The vertical coordinates of the upper-left corner of the active tile,
		/// with respect to the canvas origin, in the component hi-res grid, for
		/// each component. 
		/// </summary>
		//UPGRADE_NOTE: Final was removed from the declaration of 'culy '. "ms-help://MS.VSCC.v80/dv_commoner/local/redirect.htm?index='!DefaultContextWindowIndex'&keyword='jlca1003'"
		protected internal int[] culy;
		
		/// <summary>The nominal tile width, in the hi-res reference grid </summary>
		//UPGRADE_NOTE: Final was removed from the declaration of 'ntW '. "ms-help://MS.VSCC.v80/dv_commoner/local/redirect.htm?index='!DefaultContextWindowIndex'&keyword='jlca1003'"
		protected internal int ntW;
		
		/// <summary>The nominal tile height, in the hi-res reference grid </summary>
		//UPGRADE_NOTE: Final was removed from the declaration of 'ntH '. "ms-help://MS.VSCC.v80/dv_commoner/local/redirect.htm?index='!DefaultContextWindowIndex'&keyword='jlca1003'"
		protected internal int ntH;
		
		/// <summary>The number of tile in the horizontal direction </summary>
		//UPGRADE_NOTE: Final was removed from the declaration of 'ntX '. "ms-help://MS.VSCC.v80/dv_commoner/local/redirect.htm?index='!DefaultContextWindowIndex'&keyword='jlca1003'"
		protected internal int ntX;
		
		/// <summary>The number of tiles in the vertical direction </summary>
		//UPGRADE_NOTE: Final was removed from the declaration of 'ntY '. "ms-help://MS.VSCC.v80/dv_commoner/local/redirect.htm?index='!DefaultContextWindowIndex'&keyword='jlca1003'"
		protected internal int ntY;
		
		/// <summary>The total number of tiles </summary>
		//UPGRADE_NOTE: Final was removed from the declaration of 'nt '. "ms-help://MS.VSCC.v80/dv_commoner/local/redirect.htm?index='!DefaultContextWindowIndex'&keyword='jlca1003'"
		protected internal int nt;
		
		/// <summary>The current tile horizontal index </summary>
		protected internal int ctX;
		
		/// <summary>The current tile vertical index </summary>
		protected internal int ctY;
		
		/// <summary>The decoded bit stream header </summary>
		//UPGRADE_NOTE: Final was removed from the declaration of 'hd '. "ms-help://MS.VSCC.v80/dv_commoner/local/redirect.htm?index='!DefaultContextWindowIndex'&keyword='jlca1003'"
		protected internal HeaderDecoder hd;
		
		/// <summary>Number of bytes targeted to be read </summary>
		protected internal int tnbytes;
		
		/// <summary>Actual number of read bytes </summary>
		protected internal int anbytes;
		
		/// <summary>Target decoding rate in bpp </summary>
		protected internal float trate;
		
		/// <summary>Actual decoding rate in bpp </summary>
		protected internal float arate;
		
		/// <summary> Initializes members of this class. This constructor takes a
		/// HeaderDecoder object. This object must be initialized by the
		/// constructor of the implementing class from the header of the bit
		/// stream.
		/// 
		/// </summary>
		/// <param name="hd">The decoded header of the bit stream from where to initialize
		/// the values.
		/// 
		/// </param>
		/// <param name="decSpec">The decoder specifications
		/// 
		/// </param>
		protected internal BitstreamReaderAgent(HeaderDecoder hd, DecoderSpecs decSpec)
		{
			Coord co;
			//int i, j, max;
			
			this.decSpec = decSpec;
			this.hd = hd;
			
			// Number of components
			nc = hd.NumComps;
			offX = new int[nc];
			offY = new int[nc];
			culx = new int[nc];
			culy = new int[nc];
			
			// Image size and origin
			imgW = hd.ImgWidth;
			imgH = hd.ImgHeight;
			ax = hd.ImgULX;
			ay = hd.ImgULY;
			
			// Tiles
			co = hd.getTilingOrigin(null);
			px = co.x;
			py = co.y;
			ntW = hd.NomTileWidth;
			ntH = hd.NomTileHeight;
			ntX = (ax + imgW - px + ntW - 1) / ntW;
			ntY = (ay + imgH - py + ntH - 1) / ntH;
			nt = ntX * ntY;
		}
		
		/// <summary> Returns the component subsampling factor in the horizontal direction,
		/// for the specified component. This is, approximately, the ratio of
		/// dimensions between the reference grid and the component itself, see the
		/// 'ImgData' interface desription for details.
		/// 
		/// </summary>
		/// <param name="c">The index of the component (between 0 and N-1)
		/// 
		/// </param>
		/// <returns> The horizontal subsampling factor of component 'c'
		/// 
		/// </returns>
		/// <seealso cref="jj2000.j2k.image.ImgData">
		/// 
		/// </seealso>
		public int getCompSubsX(int c)
		{
			return hd.getCompSubsX(c);
		}
		
		/// <summary> Returns the component subsampling factor in the vertical direction, for
		/// the specified component. This is, approximately, the ratio of
		/// dimensions between the reference grid and the component itself, see the
		/// 'ImgData' interface desription for details.
		/// 
		/// </summary>
		/// <param name="c">The index of the component (between 0 and C-1)
		/// 
		/// </param>
		/// <returns> The vertical subsampling factor of component 'c'
		/// 
		/// </returns>
		/// <seealso cref="jj2000.j2k.image.ImgData">
		/// 
		/// </seealso>
		public virtual int getCompSubsY(int c)
		{
			return hd.getCompSubsY(c);
		}
		
		/// <summary> Returns the overall width of the current tile in pixels for the given
		/// (tile) resolution level. This is the tile's width without accounting
		/// for any component subsampling.
		/// 
		/// <p>Note: Tile resolution level indexes may be different from
		/// tile-component resolution index. They are indeed indexed starting from
		/// the lowest number of decomposition levels of each component of the
		/// tile.</p>
		/// 
		/// <p>For an image (1 tile) with 2 components (component 0 having 2
		/// decomposition levels and component 1 having 3 decomposition levels),
		/// the first (tile-)component has 3 resolution levels and the second one
		/// has 4 resolution levels, whereas the tile has only 3 resolution levels
		/// available.</p>
		/// 
		/// </summary>
		/// <param name="rl">The (tile) resolution level.
		/// 
		/// </param>
		/// <returns> The current tile's width in pixels.
		/// 
		/// </returns>
		public virtual int getTileWidth(int rl)
		{
			// The minumum number of decomposition levels between all the
			// components
			int mindl = decSpec.dls.getMinInTile(TileIdx);
			if (rl > mindl)
			{
				throw new System.ArgumentException("Requested resolution level" + " is not available for, at " + "least, one component in " + "tile: " + ctX + "x" + ctY);
			}
			int ctulx, ntulx;
			int dl = mindl - rl; // Number of decomposition to obtain this
			// resolution
			
			// Calculate starting X of current tile at hi-res
			ctulx = (ctX == 0)?ax:px + ctX * ntW;
			// Calculate starting X of next tile X-wise at hi-res
			ntulx = (ctX < ntX - 1)?px + (ctX + 1) * ntW:ax + imgW;
			
			// The difference at the rl resolution level is the width
			return (ntulx + (1 << dl) - 1) / (1 << dl) - (ctulx + (1 << dl) - 1) / (1 << dl);
		}
		
		/// <summary> Returns the overall height of the current tile in pixels, for the given
		/// resolution level. This is the tile's height without accounting for any
		/// component subsampling.
		/// 
		/// <p>Note: Tile resolution level indexes may be different from
		/// tile-component resolution index. They are indeed indexed starting from
		/// the lowest number of decomposition levels of each component of the
		/// tile.</p>
		/// 
		/// <p>For an image (1 tile) with 2 components (component 0 having 2
		/// decomposition levels and component 1 having 3 decomposition levels),
		/// the first (tile-)component has 3 resolution levels and the second one
		/// has 4 resolution levels, whereas the tile has only 3 resolution levels
		/// available.</p>
		/// 
		/// </summary>
		/// <param name="rl">The (tile) resolution level.
		/// 
		/// </param>
		/// <returns> The total current tile's height in pixels.
		/// 
		/// </returns>
		public virtual int getTileHeight(int rl)
		{
			// The minumum number of decomposition levels between all the
			// components
			int mindl = decSpec.dls.getMinInTile(TileIdx);
			if (rl > mindl)
			{
				throw new System.ArgumentException("Requested resolution level" + " is not available for, at " + "least, one component in" + " tile: " + ctX + "x" + ctY);
			}
			
			int ctuly, ntuly;
			int dl = mindl - rl; // Number of decomposition to obtain this
			// resolution
			
			// Calculate starting Y of current tile at hi-res
			ctuly = (ctY == 0)?ay:py + ctY * ntH;
			// Calculate starting Y of next tile Y-wise at hi-res
			ntuly = (ctY < ntY - 1)?py + (ctY + 1) * ntH:ay + imgH;
			// The difference at the rl level is the height
			return (ntuly + (1 << dl) - 1) / (1 << dl) - (ctuly + (1 << dl) - 1) / (1 << dl);
		}
		
		/// <summary> Returns the overall width of the image in pixels, for the given (image)
		/// resolution level. This is the image's width without accounting for any
		/// component subsampling or tiling.
		/// 
		/// <p>Note: Image resolution level indexes may differ from tile-component
		/// resolution index. They are indeed indexed starting from the lowest
		/// number of decomposition levels of each component of each tile.</p>
		/// 
		/// <p>Example: For an image (1 tile) with 2 components (component 0 having
		/// 2 decomposition levels and component 1 having 3 decomposition levels),
		/// the first (tile-) component has 3 resolution levels and the second one
		/// has 4 resolution levels, whereas the image has only 3 resolution levels
		/// available.</p>
		/// 
		/// </summary>
		/// <param name="rl">The image resolution level.
		/// 
		/// </param>
		/// <returns> The total image's width in pixels.
		/// 
		/// </returns>
		public virtual int getImgWidth(int rl)
		{
			// The minimum number of decomposition levels of each
			// tile-component
			int mindl = decSpec.dls.Min;
			if (rl > mindl)
			{
				throw new System.ArgumentException("Requested resolution level" + " is not available for, at " + "least, one tile-component");
			}
			// Retrieve number of decomposition levels corresponding to
			// this resolution level
			int dl = mindl - rl;
			return (ax + imgW + (1 << dl) - 1) / (1 << dl) - (ax + (1 << dl) - 1) / (1 << dl);
		}
		
		/// <summary> Returns the overall height of the image in pixels, for the given
		/// resolution level. This is the image's height without accounting for any
		/// component subsampling or tiling.
		/// 
		/// <p>Note: Image resolution level indexes may differ from tile-component
		/// resolution index. They are indeed indexed starting from the lowest
		/// number of decomposition levels of each component of each tile.</p>
		/// 
		/// <p>Example: For an image (1 tile) with 2 components (component 0 having
		/// 2 decomposition levels and component 1 having 3 decomposition levels),
		/// the first (tile-) component has 3 resolution levels and the second one
		/// has 4 resolution levels, whereas the image has only 3 resolution levels
		/// available.</p>
		/// 
		/// </summary>
		/// <param name="rl">The image resolution level, from 0 to L.
		/// 
		/// </param>
		/// <returns> The total image's height in pixels.
		/// 
		/// </returns>
		public virtual int getImgHeight(int rl)
		{
			int mindl = decSpec.dls.Min;
			if (rl > mindl)
			{
				throw new System.ArgumentException("Requested resolution level" + " is not available for, at " + "least, one tile-component");
			}
			// Retrieve number of decomposition levels corresponding to this
			// resolution level
			int dl = mindl - rl;
			return (ay + imgH + (1 << dl) - 1) / (1 << dl) - (ay + (1 << dl) - 1) / (1 << dl);
		}
		
		/// <summary> Returns the horizontal coordinate of the image origin, the top-left
		/// corner, in the canvas system, on the reference grid at the specified
		/// resolution level.
		/// 
		/// <p>Note: Image resolution level indexes may differ from tile-component
		/// resolution index. They are indeed indexed starting from the lowest
		/// number of decomposition levels of each component of each tile.</p>
		/// 
		/// <p>Example: For an image (1 tile) with 2 components (component 0 having
		/// 2 decomposition levels and component 1 having 3 decomposition levels),
		/// the first (tile-) component has 3 resolution levels and the second one
		/// has 4 resolution levels, whereas the image has only 3 resolution levels
		/// available.</p>
		/// 
		/// </summary>
		/// <param name="rl">The resolution level, from 0 to L.
		/// 
		/// </param>
		/// <returns> The horizontal coordinate of the image origin in the canvas
		/// system, on the reference grid.
		/// 
		/// </returns>
		public virtual int getImgULX(int rl)
		{
			int mindl = decSpec.dls.Min;
			if (rl > mindl)
			{
				throw new System.ArgumentException("Requested resolution level" + " is not available for, at " + "least, one tile-component");
			}
			// Retrieve number of decomposition levels corresponding to this
			// resolution level
			int dl = mindl - rl;
			return (ax + (1 << dl) - 1) / (1 << dl);
		}
		
		/// <summary> Returns the vertical coordinate of the image origin, the top-left
		/// corner, in the canvas system, on the reference grid at the specified
		/// resolution level.
		/// 
		/// <p>Note: Image resolution level indexes may differ from tile-component
		/// resolution index. They are indeed indexed starting from the lowest
		/// number of decomposition levels of each component of each tile.</p>
		/// 
		/// <p>Example: For an image (1 tile) with 2 components (component 0 having
		/// 2 decomposition levels and component 1 having 3 decomposition levels),
		/// the first (tile-) component has 3 resolution levels and the second one
		/// has 4 resolution levels, whereas the image has only 3 resolution levels
		/// available.</p>
		/// 
		/// </summary>
		/// <param name="rl">The resolution level, from 0 to L.
		/// 
		/// </param>
		/// <returns> The vertical coordinate of the image origin in the canvas
		/// system, on the reference grid.
		/// 
		/// </returns>
		public virtual int getImgULY(int rl)
		{
			int mindl = decSpec.dls.Min;
			if (rl > mindl)
			{
				throw new System.ArgumentException("Requested resolution level" + " is not available for, at " + "least, one tile-component");
			}
			// Retrieve number of decomposition levels corresponding to this
			// resolution level
			int dl = mindl - rl;
			return (ay + (1 << dl) - 1) / (1 << dl);
		}
		
		/// <summary> Returns the width in pixels of the specified tile-component for the
		/// given (tile-component) resolution level.
		/// 
		/// </summary>
		/// <param name="t">The tile index
		/// 
		/// </param>
		/// <param name="c">The index of the component, from 0 to N-1.
		/// 
		/// </param>
		/// <param name="rl">The resolution level, from 0 to L.
		/// 
		/// </param>
		/// <returns> The width in pixels of component <tt>c</tt> in tile <tt>t</tt>
		/// for resolution level <tt>rl</tt>.
		/// 
		/// </returns>
		public int getTileCompWidth(int t, int c, int rl)
		{
			int tIdx = TileIdx;
			if (t != tIdx)
			{
				throw new System.ApplicationException("Asking the tile-component width of a tile " + "different  from the current one.");
			}
			int ntulx;
			int dl = mdl[c] - rl;
			// Calculate starting X of next tile X-wise at reference grid hi-res
			ntulx = (ctX < ntX - 1)?px + (ctX + 1) * ntW:ax + imgW;
			// Convert reference grid hi-res to component grid hi-res
			ntulx = (ntulx + hd.getCompSubsX(c) - 1) / hd.getCompSubsX(c);
			// Starting X of current tile at component grid hi-res is culx[c]
			// The difference at the rl level is the width
			return (ntulx + (1 << dl) - 1) / (1 << dl) - (culx[c] + (1 << dl) - 1) / (1 << dl);
		}
		
		/// <summary> Returns the height in pixels of the specified tile-component for the
		/// given (tile-component) resolution level.
		/// 
		/// </summary>
		/// <param name="t">The tile index.
		/// 
		/// </param>
		/// <param name="c">The index of the component, from 0 to N-1.
		/// 
		/// </param>
		/// <param name="rl">The resolution level, from 0 to L.
		/// 
		/// </param>
		/// <returns> The height in pixels of component <tt>c</tt> in the current
		/// tile.
		/// 
		/// </returns>
		public int getTileCompHeight(int t, int c, int rl)
		{
			int tIdx = TileIdx;
			if (t != tIdx)
			{
				throw new System.ApplicationException("Asking the tile-component width of a tile " + "different  from the current one.");
			}
			int ntuly;
			int dl = mdl[c] - rl; // Revert level indexation (0 is hi-res)
			// Calculate starting Y of next tile Y-wise at reference grid hi-res
			ntuly = (ctY < ntY - 1)?py + (ctY + 1) * ntH:ay + imgH;
			// Convert reference grid hi-res to component grid hi-res
			ntuly = (ntuly + hd.getCompSubsY(c) - 1) / hd.getCompSubsY(c);
			// Starting Y of current tile at component grid hi-res is culy[c]
			// The difference at the rl level is the height
			return (ntuly + (1 << dl) - 1) / (1 << dl) - (culy[c] + (1 << dl) - 1) / (1 << dl);
		}
		
		/// <summary> Returns the width in pixels of the specified component in the overall
		/// image, for the given (component) resolution level.
		/// 
		/// <p>Note: Component resolution level indexes may differ from
		/// tile-component resolution index. They are indeed indexed starting from
		/// the lowest number of decomposition levels of same component of each
		/// tile.</p>
		/// 
		/// <p>Example: For an image (2 tiles) with 1 component (tile 0 having 2
		/// decomposition levels and tile 1 having 3 decomposition levels), the
		/// first tile(-component) has 3 resolution levels and the second one has 4
		/// resolution levels, whereas the component has only 3 resolution levels
		/// available.</p>
		/// 
		/// </summary>
		/// <param name="c">The index of the component, from 0 to N-1.
		/// 
		/// </param>
		/// <param name="rl">The resolution level, from 0 to L.
		/// 
		/// </param>
		/// <returns> The width in pixels of component <tt>c</tt> in the overall
		/// image.
		/// 
		/// </returns>
		public int getCompImgWidth(int c, int rl)
		{
			int sx, ex;
			int dl = decSpec.dls.getMinInComp(c) - rl;
			// indexation (0 is hi-res)
			// Calculate image starting x at component hi-res grid
			sx = (ax + hd.getCompSubsX(c) - 1) / hd.getCompSubsX(c);
			// Calculate image ending (excluding) x at component hi-res grid
			ex = (ax + imgW + hd.getCompSubsX(c) - 1) / hd.getCompSubsX(c);
			// The difference at the rl level is the width
			return (ex + (1 << dl) - 1) / (1 << dl) - (sx + (1 << dl) - 1) / (1 << dl);
		}
		
		/// <summary> Returns the height in pixels of the specified component in the overall
		/// image, for the given (component) resolution level.
		/// 
		/// <p>Note: Component resolution level indexes may differ from
		/// tile-component resolution index. They are indeed indexed starting from
		/// the lowest number of decomposition levels of same component of each
		/// tile.</p>
		/// 
		/// <p>Example: For an image (2 tiles) with 1 component (tile 0 having 2
		/// decomposition levels and tile 1 having 3 decomposition levels), the
		/// first tile(-component) has 3 resolution levels and the second one has 4
		/// resolution levels, whereas the component has only 3 resolution levels
		/// available.</p>
		/// 
		/// </summary>
		/// <param name="c">The index of the component, from 0 to N-1.
		/// 
		/// </param>
		/// <param name="rl">The resolution level, from 0 to L.
		/// 
		/// </param>
		/// <returns> The height in pixels of component <tt>c</tt> in the overall
		/// image.
		/// 
		/// </returns>
		public int getCompImgHeight(int c, int rl)
		{
			int sy, ey;
			int dl = decSpec.dls.getMinInComp(c) - rl;
			// indexation (0 is hi-res)
			// Calculate image starting x at component hi-res grid
			sy = (ay + hd.getCompSubsY(c) - 1) / hd.getCompSubsY(c);
			// Calculate image ending (excluding) x at component hi-res grid
			ey = (ay + imgH + hd.getCompSubsY(c) - 1) / hd.getCompSubsY(c);
			// The difference at the rl level is the width
			return (ey + (1 << dl) - 1) / (1 << dl) - (sy + (1 << dl) - 1) / (1 << dl);
		}
		
		/// <summary> Changes the current tile, given the new indexes. An
		/// IllegalArgumentException is thrown if the indexes do not correspond to
		/// a valid tile.
		/// 
		/// </summary>
		/// <param name="x">The horizontal indexes the tile.
		/// 
		/// </param>
		/// <param name="y">The vertical indexes of the new tile.
		/// 
		/// </param>
		public abstract void  setTile(int x, int y);
		
		/// <summary> Advances to the next tile, in standard scan-line order (by rows then
		/// columns). An NoNextElementException is thrown if the current tile is
		/// the last one (i.e. there is no next tile).
		/// 
		/// </summary>
		public abstract void  nextTile();
		
		/// <summary> Returns the indexes of the current tile. These are the horizontal and
		/// vertical indexes of the current tile.
		/// 
		/// </summary>
		/// <param name="co">If not null this object is used to return the information. If
		/// null a new one is created and returned.
		/// 
		/// </param>
		/// <returns> The current tile's indexes (vertical and horizontal indexes).
		/// 
		/// </returns>
		public Coord getTile(Coord co)
		{
			if (co != null)
			{
				co.x = ctX;
				co.y = ctY;
				return co;
			}
			else
			{
				return new Coord(ctX, ctY);
			}
		}
		
		/// <summary> Returns the horizontal coordinate of the upper-left corner of the
		/// specified resolution in the given component of the current tile.
		/// 
		/// </summary>
		/// <param name="c">The component index.
		/// 
		/// </param>
		/// <param name="rl">The resolution level index.
		/// 
		/// </param>
		public int getResULX(int c, int rl)
		{
			int dl = mdl[c] - rl;
			if (dl < 0)
			{
				throw new System.ArgumentException("Requested resolution level" + " is not available for, at " + "least, one component in " + "tile: " + ctX + "x" + ctY);
			}
			int tx0 = (int) System.Math.Max(px + ctX * ntW, ax);
			//UPGRADE_WARNING: Data types in Visual C# might be different.  Verify the accuracy of narrowing conversions. "ms-help://MS.VSCC.v80/dv_commoner/local/redirect.htm?index='!DefaultContextWindowIndex'&keyword='jlca1042'"
			int tcx0 = (int) System.Math.Ceiling(tx0 / (double) getCompSubsX(c));
			//UPGRADE_WARNING: Data types in Visual C# might be different.  Verify the accuracy of narrowing conversions. "ms-help://MS.VSCC.v80/dv_commoner/local/redirect.htm?index='!DefaultContextWindowIndex'&keyword='jlca1042'"
			return (int) System.Math.Ceiling(tcx0 / (double) (1 << dl));
		}
		
		/// <summary> Returns the vertical coordinate of the upper-left corner of the
		/// specified component in the given component of the current tile.
		/// 
		/// </summary>
		/// <param name="c">The component index.
		/// 
		/// </param>
		/// <param name="rl">The resolution level index.
		/// 
		/// </param>
		public int getResULY(int c, int rl)
		{
			int dl = mdl[c] - rl;
			if (dl < 0)
			{
				throw new System.ArgumentException("Requested resolution level" + " is not available for, at " + "least, one component in " + "tile: " + ctX + "x" + ctY);
			}
			int ty0 = (int) System.Math.Max(py + ctY * ntH, ay);
			//UPGRADE_WARNING: Data types in Visual C# might be different.  Verify the accuracy of narrowing conversions. "ms-help://MS.VSCC.v80/dv_commoner/local/redirect.htm?index='!DefaultContextWindowIndex'&keyword='jlca1042'"
			int tcy0 = (int) System.Math.Ceiling(ty0 / (double) getCompSubsY(c));
			//UPGRADE_WARNING: Data types in Visual C# might be different.  Verify the accuracy of narrowing conversions. "ms-help://MS.VSCC.v80/dv_commoner/local/redirect.htm?index='!DefaultContextWindowIndex'&keyword='jlca1042'"
			return (int) System.Math.Ceiling(tcy0 / (double) (1 << dl));
		}
		
		/// <summary> Returns the number of tiles in the horizontal and vertical directions.
		/// 
		/// </summary>
		/// <param name="co">If not null this object is used to return the information. If
		/// null a new one is created and returned.
		/// 
		/// </param>
		/// <returns> The number of tiles in the horizontal (Coord.x) and vertical
		/// (Coord.y) directions.
		/// 
		/// </returns>
		public Coord getNumTiles(Coord co)
		{
			if (co != null)
			{
				co.x = ntX;
				co.y = ntY;
				return co;
			}
			else
			{
				return new Coord(ntX, ntY);
			}
		}
		
		/// <summary> Returns the total number of tiles in the image.
		/// 
		/// </summary>
		/// <returns> The total number of tiles in the image.
		/// 
		/// </returns>
		public int getNumTiles()
		{
			return ntX * ntY;
		}
		
		/// <summary> Returns the subband tree, for the specified tile-component. This method
		/// returns the root element of the subband tree structure, see Subband and
		/// SubbandSyn. The tree comprises all the available resolution levels.
		/// 
		/// <p>Note: this method is not able to return subband tree for a tile
		/// different than the current one.</p>
		/// 
		/// <p>The number of magnitude bits ('magBits' member variable) for each
		/// subband is not initialized.</p>
		/// 
		/// </summary>
		/// <param name="t">The tile index
		/// 
		/// </param>
		/// <param name="c">The index of the component, from 0 to C-1.
		/// 
		/// </param>
		/// <returns> The root of the tree structure.
		/// 
		/// </returns>
		public SubbandSyn getSynSubbandTree(int t, int c)
		{
			if (t != TileIdx)
			{
				throw new System.ArgumentException("Can not request subband" + " tree of a different tile" + " than the current one");
			}
			if (c < 0 || c >= nc)
			{
				throw new System.ArgumentException("Component index out of range");
			}
			return subbTrees[c];
		}
		
		/// <summary> Creates a bit stream reader of the correct type that works on the
		/// provided RandomAccessIO, with the special parameters from the parameter
		/// list.
		/// 
		/// </summary>
		/// <param name="in">The RandomAccessIO source from which to read the bit stream.
		/// 
		/// </param>
		/// <param name="hd">Header of the codestream.
		/// 
		/// </param>
		/// <param name="pl">The parameter list containing parameters applicable to the
		/// bit stream read (other parameters may also be present).
		/// 
		/// </param>
		/// <param name="decSpec">The decoder specifications
		/// 
		/// </param>
		/// <param name="cdstrInfo">Whether or not to print information found in
		/// codestream. 
		/// 
		/// </param>
		/// <param name="hi">Reference to the HeaderInfo instance.
		/// 
		/// </param>
		/// <exception cref="IOException">If an I/O error occurs while reading initial
		/// data from the bit stream.
		/// </exception>
		/// <exception cref="IllegalArgumentException">If an unrecognised bit stream
		/// reader option is present.
		/// 
		/// </exception>
		public static BitstreamReaderAgent createInstance(RandomAccessIO in_Renamed, HeaderDecoder hd, ParameterList pl, DecoderSpecs decSpec, bool cdstrInfo, HeaderInfo hi)
		{
			
			// Check parameters
			pl.checkList(BitstreamReaderAgent.OPT_PREFIX, CSJ2K.j2k.util.ParameterList.toNameArray(BitstreamReaderAgent.ParameterInfo));
			
			return new FileBitstreamReaderAgent(hd, in_Renamed, decSpec, pl, cdstrInfo, hi);
		}
		
		/// <summary> Returns the precinct partition width for the specified tile-component
		/// and (tile-component) resolution level.
		/// 
		/// </summary>
		/// <param name="t">the tile index
		/// 
		/// </param>
		/// <param name="c">The index of the component (between 0 and N-1)
		/// 
		/// </param>
		/// <param name="rl">The resolution level, from 0 to L.
		/// 
		/// </param>
		/// <returns> the precinct partition width for the specified component,
		/// resolution level and tile.
		/// 
		/// </returns>
		public int getPPX(int t, int c, int rl)
		{
			return decSpec.pss.getPPX(t, c, rl);
		}
		
		/// <summary> Returns the precinct partition height for the specified tile-component
		/// and (tile-component) resolution level.
		/// 
		/// </summary>
		/// <param name="t">The tile index
		/// 
		/// </param>
		/// <param name="c">The index of the component (between 0 and N-1)
		/// 
		/// </param>
		/// <param name="rl">The resolution level, from 0 to L.
		/// 
		/// </param>
		/// <returns> The precinct partition height in the specified component, for
		/// the specified resolution level, for the current tile.
		/// 
		/// </returns>
		public int getPPY(int t, int c, int rl)
		{
			return decSpec.pss.getPPY(t, c, rl);
		}
		
		/// <summary> Initialises subbands fields, such as number of code-blocks, code-blocks
		/// dimension and number of magnitude bits, in the subband tree. The
		/// nominal code-block width/height depends on the precincts dimensions if
		/// used. The way the number of magnitude bits is computed depends on the
		/// quantization type (reversible, derived, expounded).
		/// 
		/// </summary>
		/// <param name="c">The component index
		/// 
		/// </param>
		/// <param name="sb">The subband tree to be initialised.
		/// 
		/// </param>
		protected internal virtual void  initSubbandsFields(int c, SubbandSyn sb)
		{
			int t = TileIdx;
			int rl = sb.resLvl;
			int cbw, cbh;
			
			cbw = decSpec.cblks.getCBlkWidth(ModuleSpec.SPEC_TILE_COMP, t, c);
			cbh = decSpec.cblks.getCBlkHeight(ModuleSpec.SPEC_TILE_COMP, t, c);
			
			if (!sb.isNode)
			{
				// Code-block dimensions
				if (hd.precinctPartitionUsed())
				{
					// The precinct partition is used
					int ppxExp, ppyExp, cbwExp, cbhExp;
					
					// Get exponents
					ppxExp = MathUtil.log2(getPPX(t, c, rl));
					ppyExp = MathUtil.log2(getPPY(t, c, rl));
					cbwExp = MathUtil.log2(cbw);
					cbhExp = MathUtil.log2(cbh);
					
					switch (sb.resLvl)
					{
						
						case 0: 
							sb.nomCBlkW = (cbwExp < ppxExp?(1 << cbwExp):(1 << ppxExp));
							sb.nomCBlkH = (cbhExp < ppyExp?(1 << cbhExp):(1 << ppyExp));
							break;
						
						
						default: 
							sb.nomCBlkW = (cbwExp < ppxExp - 1?(1 << cbwExp):(1 << (ppxExp - 1)));
							sb.nomCBlkH = (cbhExp < ppyExp - 1?(1 << cbhExp):(1 << (ppyExp - 1)));
							break;
						
					}
				}
				else
				{
					sb.nomCBlkW = cbw;
					sb.nomCBlkH = cbh;
				}
				
				// Number of code-blocks
				if (sb.numCb == null)
					sb.numCb = new Coord();
				if (sb.w == 0 || sb.h == 0)
				{
					sb.numCb.x = 0;
					sb.numCb.y = 0;
				}
				else
				{
					int cb0x = CbULX;
					int cb0y = CbULY;
					int tmp;
					
					// Projects code-block partition origin to subband. Since the
					// origin is always 0 or 1, it projects to the low-pass side
					// (throught the ceil operator) as itself (i.e. no change) and
					// to the high-pass side (through the floor operator) as 0,
					// always.
					int acb0x = cb0x;
					int acb0y = cb0y;
					
					switch (sb.sbandIdx)
					{
						
						case Subband.WT_ORIENT_LL: 
							// No need to project since all low-pass => nothing to do
							break;
						
						case Subband.WT_ORIENT_HL: 
							acb0x = 0;
							break;
						
						case Subband.WT_ORIENT_LH: 
							acb0y = 0;
							break;
						
						case Subband.WT_ORIENT_HH: 
							acb0x = 0;
							acb0y = 0;
							break;
						
						default: 
							throw new System.ApplicationException("Internal JJ2000 error");
						
					}
					if (sb.ulcx - acb0x < 0 || sb.ulcy - acb0y < 0)
					{
						throw new System.ArgumentException("Invalid code-blocks " + "partition origin or " + "image offset in the " + "reference grid.");
					}
					
					// NOTE: when calculating "floor()" by integer division the
					// dividend and divisor must be positive, we ensure that by
					// adding the divisor to the dividend and then substracting 1
					// to the result of the division
					
					tmp = sb.ulcx - acb0x + sb.nomCBlkW;
					sb.numCb.x = (tmp + sb.w - 1) / sb.nomCBlkW - (tmp / sb.nomCBlkW - 1);
					
					tmp = sb.ulcy - acb0y + sb.nomCBlkH;
					sb.numCb.y = (tmp + sb.h - 1) / sb.nomCBlkH - (tmp / sb.nomCBlkH - 1);
				}
				
				// Number of magnitude bits
				if (derived[c])
				{
					sb.magbits = gb[c] + (params_Renamed[c].exp[0][0] - (mdl[c] - sb.level)) - 1;
				}
				else
				{
					sb.magbits = gb[c] + params_Renamed[c].exp[sb.resLvl][sb.sbandIdx] - 1;
				}
			}
			else
			{
				initSubbandsFields(c, (SubbandSyn) sb.LL);
				initSubbandsFields(c, (SubbandSyn) sb.HL);
				initSubbandsFields(c, (SubbandSyn) sb.LH);
				initSubbandsFields(c, (SubbandSyn) sb.HH);
			}
		}
		public abstract CSJ2K.j2k.entropy.decoder.DecLyrdCBlk getCodeBlock(int param1, int param2, int param3, CSJ2K.j2k.wavelet.synthesis.SubbandSyn param4, int param5, int param6, CSJ2K.j2k.entropy.decoder.DecLyrdCBlk param7);
	}
}