/*!
 * Stand Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU GPL License
 *
 * Stand Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 *  @file MathSet.h
 *  @author HAL@shurabaP
 */
#ifndef MATHSET_H
#define MATHSET_H

#include <math.h>

#include "SmootMatching.h"

namespace stand
{
namespace math
{
/// <summary>配列の中間値を線形補間します．</summary>
/// <param name="x">配列のインデックス．</param>
/// <param name="p">配列</param>
/// <returns>配列の中間値を返します．</returns>
double interpolateArray(double x, const double *p);

/*!
 *  @brief 自己相関関数を計算する．
 *  @param[out] a 計算結果．
 *  @param[in] aLen 自己相関関数の長さ
 *  @param[in] y 信号
 *  @param[in] yLen 信号長
 */
void autoCorrelation(double *a, int aLen, const double *y, int yLen);

/*!
 *  @brief 導関数を計算する．
 *  @param[in] src 導関数を求めたい関数
 *  @param[out] dst 導関数を保持する配列
 *  @param[in] length 配列の長さ
 */
void calculateDelta(double *dst, const double *src, int length);

/*!
 *  @brief 平均を求める
 *  @param[in] src ベクトル列
 *  @param[out] dst 平均ベクトル
 *  @param[in] n ベクトルの個数
 *  @param[in] dim ベクトルの次元
 *  @return 平均へのポインタ
 */
double *average(double *dst, double **src, int n, int dim);

/*!
 *  @brief 系列ごとの分散を求める
 *  @param[in] src ベクトル列
 *  @param[in] avr 平均ベクトル
 *  @param[out] dev 系列ごとの分散
 *  @param[in] n ベクトルの個数
 *  @param[in] dim ベクトルの次元
 *  @return 分散へのポインタ
 */
double *deviation(double *dst, double **src, int n, int dim, const double *avr = 0);

}
}

#endif // MATHSET_H
