//-----------------------------------------------------------------------------
// Aperiodicity based on TANDEM-STRAIGHT.
// This function would be changed in near future.
//-----------------------------------------------------------------------------
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "world.h"

namespace stand
{
namespace math
{
namespace dsp
{

#define MIN(a,b) (((a)<(b))?(a):(b))
#define MAX(a,b) (((a)<(b))?(b):(a))

const double NOMINALCUTOFF = 600.0;

// Design of QMF
void GetQMFpairOfFilters(int fs, double *hHP, double *hLP);

// Names of these variables are copied by the source code of Matlab.
// The developer does not know the meaning of these names.
typedef struct{
  int segmentLength;
  int nMargin;
  double **w;
  double *wsqrt;
  double **H;
  double **Hw;
  double **R;
  double **invR;

  double *Hwx;
  double *a;
  double *Ha;
  double *wxHa;
  double *wx;
} InternalParameters;

//-----------------------------------------------------------------------------
// SetInternalParameters() allocates the memory to the struct.
//-----------------------------------------------------------------------------
void SetInternalParameters(int segment_length, int n_margin, 
  InternalParameters* internal_parameters)
{
  internal_parameters->segmentLength = segment_length;
  internal_parameters->nMargin = n_margin;

  internal_parameters->w = (double **)malloc(sizeof(double *) * segment_length);
  for(int i = 0;i < segment_length;i++) internal_parameters->w[i] = (double *)malloc(sizeof(double) * segment_length);

  internal_parameters->wsqrt = (double *)malloc(sizeof(double)*segment_length);

  internal_parameters->H = (double **)malloc(sizeof(double *)*segment_length);
  for(int i = 0;i < segment_length;i++)
    internal_parameters->H[i] = (double *)malloc(sizeof(double)*n_margin*2);

  internal_parameters->Hw   = (double **)malloc(sizeof(double *)*n_margin*2);
  for(int i = 0;i < n_margin*2;i++)
    internal_parameters->Hw[i] = (double *)malloc(sizeof(double)*segment_length );

  internal_parameters->R    = (double **)malloc(sizeof(double *)*n_margin*2);
  for(int i = 0;i < n_margin*2;i++)
    internal_parameters->R[i] = (double *)malloc(sizeof(double)*n_margin*2 );

  internal_parameters->invR = (double **)malloc(sizeof(double *)*n_margin*2);
  for(int i = 0;i < n_margin*2;i++)
    internal_parameters->invR[i] = (double *)malloc(sizeof(double)*n_margin*2 );

  internal_parameters->Hwx   = (double *)malloc(sizeof(double) * n_margin*2);
  internal_parameters->a     = (double *)malloc(sizeof(double) * n_margin*2);
  internal_parameters->Ha    = (double *)malloc(sizeof(double) * segment_length);
  internal_parameters->wxHa  = (double *)malloc(sizeof(double) * segment_length);
  internal_parameters->wx    = (double *)malloc(sizeof(double) * segment_length);
}

//-----------------------------------------------------------------------------
// SetInternalParameters() frees the memory of the struct.
//-----------------------------------------------------------------------------
void DestroyInternalParameters(InternalParameters* internal_parameters)
{
  free(internal_parameters->wsqrt);
  free(internal_parameters->wx); 
  free(internal_parameters->wxHa); 
  free(internal_parameters->Ha); 
  free(internal_parameters->a); 
  free(internal_parameters->Hwx);

  for(int i = 0;i < internal_parameters->nMargin*2;i++)
    free(internal_parameters->invR[i]);  
  free(internal_parameters->invR);
  for(int i = 0;i < internal_parameters->nMargin*2;i++)
    free(internal_parameters->R[i]);  
  free(internal_parameters->R);
  for(int i = 0;i < internal_parameters->nMargin*2;i++)
    free(internal_parameters->Hw[i]); 
  free(internal_parameters->Hw);
  for(int i = 0;i < internal_parameters->segmentLength;i++)
    free(internal_parameters->H[i]);  
  free(internal_parameters->H);
  for(int i = 0;i < internal_parameters->segmentLength;i++) 
    free(internal_parameters->w[i]); 
  free(internal_parameters->w);
}

void GetH(double *x, int x_length, int segment_length, 
  int index_bias, int current_position_in_sample, int t0_in_samples, 
  double **H)
{
  int index;
  for(int i = -1;i < 2;i++)
  {
    for(int j = 0;j < segment_length;j++)
    {
      index = MAX(0, MIN(x_length-1, 
        i+current_position_in_sample-index_bias-t0_in_samples+j));
      H[j][i+1] = x[index];
      index = MAX(0, MIN(x_length-1, 
        i+current_position_in_sample-index_bias+t0_in_samples+j));
      H[j][i+4] = x[index];
    }
  }
}

void GetHw(double **H, int segment_length, int n_margin2, double **w,
  double **Hw)
{
  double tmp;
  for(int i = 0;i < n_margin2;i++)
  {
    for(int j = 0;j < segment_length;j++)
    {
      tmp = 0.0;
      for(int k = 0;k < segment_length;k++)
        tmp += H[k][i] * w[k][j];
      Hw[i][j] = tmp;
    }
  }
}

void GetR(double **Hw, int n_margin2, int segment_length, double **H,
  double **R)
{
  double tmp;
  for(int i = 0;i < n_margin2;i++)
  {
    for(int j = 0;j < n_margin2;j++)
    {
      tmp = 0.0;
      for(int k = 0;k < segment_length;k++)
        tmp += Hw[i][k] * H[k][j];
      R[i][j] = tmp;
    }
  }
}

void GetHwx(double **Hw, int n_margin2, int segment_length, double *x, 
  int origin, 
  double *Hwx)
{
  double tmp;
  for(int i = 0;i < n_margin2;i++)
  {
    tmp = 0.0;
    for(int j = 0;j < segment_length;j++)
      tmp += Hw[i][j]*x[origin+j];
    Hwx[i] = tmp;
  }
}

void Geta(double **invR, int n_margin2, double *Hwx,
  double *a)
{
  double tmp;
  for(int i = 0;i < n_margin2;i++)
  {
    tmp = 0.0;
    for(int j = 0;j < n_margin2;j++)
      tmp += invR[i][j]*Hwx[j];
    a[i] = tmp;
  }
}

void GetHa(double **H, int segment_length, int n_margin2, double *a,
  double *Ha)
{
  double tmp;
  for(int i = 0;i < segment_length;i++)
  {
    tmp = 0.0;
    for(int j = 0;j < n_margin2;j++)
      tmp += H[i][j]*a[j];
    Ha[i] = tmp;
  }
}

void GetW(int segment_length, 
  double **w)
{
  for(int i = 0;i < segment_length;i++)
    for(int j = 0;j < segment_length;j++)
      w[i][j] = 0.0;
  for(int i = 0;i < (segment_length-1)/2;i++)
  {
    w[i][i] = 0.5 - 0.5*cos(((double)i+1.0) / 
      (double)(segment_length+1)*2.0*PI);
    w[segment_length-i-1][segment_length-i-1] = w[i][i];
  }
  w[(segment_length-1)/2][(segment_length-1)/2] = 1.0;
}

double GetStdwxHa(double *wsqrt, int segment_length, double *x, 
  int origin, double *Ha, 
  double *wxHa)
{
  for(int i = 0;i < segment_length;i++)
    wxHa[i] = wsqrt[i]*(x[i+origin]-Ha[i]);
  return matlab_std(wxHa, segment_length);
}

double GetStdwx(double *wsqrt, int segment_length, double *x, int origin, 
  double *wx)
{
  for(int i = 0;i < segment_length;i++)
    wx[i] = wsqrt[i]*x[i+origin];
  return matlab_std(wx, segment_length);
}

void f0PredictionResidualFixSegmentW(double *x, int x_length, double fs, 
  double target_f0, double *temporalPositions, double *vuv, int f0_length, 
  double initial_time, int duration_ms,
  double *rms_residual)
{
  int n_margin = 3;
  int segment_length = (int)(fs*duration_ms/2000.0+0.5)*2 + 1;

  InternalParameters internal_parameters = {0};
  SetInternalParameters(segment_length, n_margin, &internal_parameters);

  GetW(segment_length, internal_parameters.w);
  for(int i = 0;i < segment_length;i++)
    internal_parameters.wsqrt[i] = sqrt(internal_parameters.w[i][i]);

  int t0_in_samples = (int)(fs/target_f0 + 0.5);
  int index_bias = (int)(fs/target_f0/2.0 + 0.5);

  int current_position_in_sample;
  int origin;
  for(int i = 0;i < f0_length;i++)
  {
    current_position_in_sample = 
      (int)(-initial_time+temporalPositions[i]*fs + 0.5)+1;
    if(vuv[i] != 0.0)
    {
      origin = MAX(0, MIN(x_length-1, current_position_in_sample-index_bias));
      GetH(x, x_length, segment_length, index_bias, 
        current_position_in_sample, t0_in_samples, internal_parameters.H);
      GetHw(internal_parameters.H, segment_length, n_margin*2, 
        internal_parameters.w, internal_parameters.Hw);
      GetR(internal_parameters.Hw, n_margin*2, segment_length, 
        internal_parameters.H, internal_parameters.R);
      GetHwx(internal_parameters.Hw, n_margin*2, segment_length, 
        x, origin, internal_parameters.Hwx);
      inv(internal_parameters.R, n_margin*2, internal_parameters.invR);
      Geta(internal_parameters.invR, n_margin*2, internal_parameters.Hwx, 
        internal_parameters.a);
      GetHa(internal_parameters.H, segment_length, n_margin*2, 
        internal_parameters.a, internal_parameters.Ha);
      rms_residual[i] = GetStdwxHa(internal_parameters.wsqrt, segment_length, 
        x, origin, internal_parameters.Ha, internal_parameters.wxHa) / 
        GetStdwx(internal_parameters.wsqrt, segment_length, x, origin, 
        internal_parameters.wx);
    }
    else // 無声音の場合，非周期性指標は意味を持たない．
      rms_residual[i] = 0.0;
  }
  DestroyInternalParameters(&internal_parameters);
}

//-----------------------------------------------------------------------------
// 帯域毎の非周期性指標を計算する．
// この関数は，QMFによるサブバンド分解を主に担当．
//-----------------------------------------------------------------------------
void BandwiseAperiodicity(double *x, int x_length, int fs, double *f0, 
  double *vuv, int f0_length, double *stretched_locations, 
  int window_length_ms,
  double **aperiodicity)
{
  double hHP[41], hLP[37];
  GetQMFpairOfFilters(fs, hHP, hLP);

  int number_of_bands = (int)(log((double)fs/NOMINALCUTOFF)/log(2.0));

  double *cutoff_list = (double *)malloc(sizeof(double) * number_of_bands);

  for(int i = 0;i < number_of_bands;i++)
    cutoff_list[i] = (double)fs / pow(2.0, (double)(i+2)); 

  int fft_size = (int)pow(2.0, 1.0 + (int)(log((double)(x_length+82) ) / 
    log(2.0))); // hHPの倍の長さ分余計に確保する必要がある

  double *whole_signal = (double *)malloc(sizeof(double) * fft_size);
  double *high_signal = (double *)malloc(sizeof(double) * fft_size);
  double *low_signal = (double *)malloc(sizeof(double) * fft_size);
  double *downsampled_high_signal =
    (double *)malloc(sizeof(double) * fft_size);

  int wLen = x_length+82;
  double *rms_residual = (double *)malloc(sizeof(double) * f0_length);

  for(int i = 0;i < x_length;i++) whole_signal[i] = x[i];
  for(int i = x_length;i < fft_size;i++) whole_signal[i] = 0.0;

  double tmp_fs;
  int j;
  for(int i = 0;i < number_of_bands-1;i++)
  {
    tmp_fs = cutoff_list[i]*2.0;
    fftfilt(whole_signal, wLen, hHP, 41, fft_size, high_signal);
    fftfilt(whole_signal, wLen, hLP, 37, fft_size, low_signal );
    for(j = 0;j < wLen;j+=2) downsampled_high_signal[j/2] = high_signal[j];

    f0PredictionResidualFixSegmentW(downsampled_high_signal, 
      (int)((double)wLen/2.0+0.5), tmp_fs, f0[0], stretched_locations, vuv, 
      f0_length, 41.0/2.0/tmp_fs, window_length_ms, rms_residual);

    // ある帯域の非周期性指標をコピー
    for(j = 0;j < f0_length;j++) 
      aperiodicity[j][number_of_bands-i-1] = rms_residual[j];

    // サブバンド分解 (?)
    for(j = 0;j < wLen; j+=2) whole_signal[j/2] = low_signal[j];
    wLen = (int)((double)wLen/2.0 + 0.5) + 82;
    fft_size = (int)pow(2.0, 1.0 + (int)(log((double)(wLen) ) / log(2.0)));
    for(j=j/2;j<fft_size;j++) whole_signal[j] = 0.0;
  }

  wLen = (wLen-82)*2;
  f0PredictionResidualFixSegmentW(whole_signal, (int)((double)wLen/2.0+0.5), 
    tmp_fs, f0[0], stretched_locations, vuv, f0_length, 41.0/2.0/tmp_fs, 
    window_length_ms, rms_residual);
  for(j = 0;j < f0_length;j++) aperiodicity[j][0] = rms_residual[j];

  free(rms_residual);
  free(downsampled_high_signal); 
  free(low_signal); 
  free(high_signal); 
  free(whole_signal);
  free(cutoff_list);
}

//-----------------------------------------------------------------------------
// 4倍にアップサンプリングするための関数
//-----------------------------------------------------------------------------
void GetInterpolatedSignal(const double *x, int x_length, double *interpolated_x)
{
  interpolated_x[0] = x[0] * 0.14644660940672621;
  interpolated_x[1] = x[0] * 0.49999999999999994;
  interpolated_x[2] = x[0] * 0.85355339059327373;
  for(int i = 0;i < x_length-1;i++)
  {
    interpolated_x[i*4 + 3] = x[i];
    interpolated_x[i*4 + 4] = x[i]*0.85355339059327373 + 
      x[i+1]*0.14644660940672621;
    interpolated_x[i*4 + 5] = x[i]*0.49999999999999994 + 
      x[i+1]*0.49999999999999994;
    interpolated_x[i*4 + 6] = x[i]*0.14644660940672621 + 
      x[i+1]*0.85355339059327373;
  }
  interpolated_x[(x_length-1)*4 + 3] = x[x_length-1];
  interpolated_x[(x_length-1)*4 + 4] = x[x_length-1] * 0.85355339059327373;
  interpolated_x[(x_length-1)*4 + 5] = x[x_length-1] * 0.49999999999999994;
  interpolated_x[(x_length-1)*4 + 6] = x[x_length-1] * 0.14644660940672621;
  interpolated_x[(x_length-1)*4 + 7] = 
    interpolated_x[(x_length-1)*4 + 8] = 
    interpolated_x[(x_length-1)*4 + 9] = 0.0;
  return;
}

//-----------------------------------------------------------------------------
// GetNormalizedSignal() calculates the signal that the f0 contour is constant.
//-----------------------------------------------------------------------------
int GetNormalizedSignal(const double *x, int x_length, int fs, double *f0,
  int f0_length, double frame_period, double target_f0, 
  double **stretched_signal, double **stretched_locations)
{
  int ix_length = x_length*4 + 6;

  double *interpolated_x = (double *)malloc(sizeof(double) * (ix_length+16));
  GetInterpolatedSignal(x, x_length, interpolated_x);

  double *original_signal_time_axis = 
    (double *)malloc(sizeof(double) * ix_length);

  for(int i = 0;i < ix_length;i++)
    original_signal_time_axis[i] = (double)i/( (double)fs * 4.0);

  double *base_f0 = (double *)malloc(sizeof(double) * (f0_length+1));
  double *base_time_axis = (double *)malloc(sizeof(double) * (f0_length+1));

  for(int i = 0;i < f0_length;i++)
  {
    base_f0[i]   = f0[i] == 0.0 ? target_f0 : f0[i];
    base_time_axis[i] = (double)i * frame_period;
  }
  base_f0[f0_length]   = target_f0;
  base_time_axis[f0_length] = (double)f0_length * frame_period;

  double *interpolated_f0 = (double *)malloc(sizeof(double) * ix_length);
  double *stretched_time_axis = (double *)malloc(sizeof(double) * ix_length);
  interp1(base_time_axis, base_f0, f0_length+1, 
    original_signal_time_axis, ix_length, interpolated_f0);

  double tmp = target_f0*( (double)fs*4.0);
  stretched_time_axis[0] = interpolated_f0[0]/tmp;
  for(int i = 1;i < ix_length;i++)
    stretched_time_axis[i] = stretched_time_axis[i-1] + 
    (interpolated_f0[i]/tmp);

  int stretched_signal_length = 
    (int)(stretched_time_axis[ix_length-1] * (double)fs * 4.0) + 1;
  double *tmp_time_axis = 
    (double *)malloc(sizeof(double) * stretched_signal_length);
  double *stretched_signal4 = 
    (double *)malloc(sizeof(double) * stretched_signal_length);

  for(int i = 0;i < stretched_signal_length;i++)
    tmp_time_axis[i] = (double)i / ( (double)fs*4.0);
  interp1(stretched_time_axis, interpolated_x, ix_length, 
    tmp_time_axis, stretched_signal_length, stretched_signal4);

  *stretched_locations = (double *)malloc(sizeof(double) * f0_length);
  interp1(original_signal_time_axis, stretched_time_axis, ix_length, 
    base_time_axis, f0_length, *stretched_locations);

  *stretched_signal = (double *)malloc(sizeof(double) * 
    ((int)((double)stretched_signal_length/4.0)+17) );
  decimate(stretched_signal4, stretched_signal_length, 4, *stretched_signal);

  // メモリの開放
  free(stretched_signal4);
  free(tmp_time_axis);
  free(stretched_time_axis);
  free(base_f0);
  free(base_time_axis);
  free(interpolated_f0);
  free(original_signal_time_axis);
  free(interpolated_x);

  return 1 + (int)((double)stretched_signal_length/4.0);
}

//-----------------------------------------------------------------------------
// 非周期性指標は，バンド毎に計算するため，
// 先にバンド数を計算する必要がある．
//-----------------------------------------------------------------------------
int GetBands_v3(int fs)
{
  return (int)(log((double)fs/NOMINALCUTOFF)/log(2.0));
}

//-----------------------------------------------------------------------------
// TANDEM最新版で採用されている非周期性指標の移植
// 一部簡略化したが，品質はほぼ等価 (だと嬉しい)．
//-----------------------------------------------------------------------------
void AperiodicityRatio_v3(const double *x, int x_length, int fs, double *f0,
  double frame_period, 
  double **aperiodicity, double *target_f0_output)
{
  int f0_length = (int)((double)x_length / (double)fs / (frame_period/1000.0) ) + 1;

  double max_f0 = 0.0;
  for(int i = 0;i < f0_length;i++)
    max_f0 = max_f0 > f0[i] ? max_f0 : f0[i];
  double target_f0 = MAX(32.0, MIN(200.0, max_f0));
  *target_f0_output = target_f0;

  // The number of two arraies are unknown.
  double *stretched_signal = NULL;
  double *stretched_locations = NULL;

  int sLen = GetNormalizedSignal(x, x_length, fs, f0, f0_length, 
    frame_period/1000.0, target_f0, &stretched_signal, &stretched_locations);

  double *stretched_f0 = (double *)malloc(sizeof(double) * f0_length);
  for(int i = 0;i < f0_length;i++) stretched_f0[i] = target_f0;

  BandwiseAperiodicity(stretched_signal, sLen, fs, stretched_f0, f0, 
    f0_length, stretched_locations, (int)(2000.0/target_f0 + 0.5),
    aperiodicity);

  free(stretched_f0);
  free(stretched_signal);
  free(stretched_locations);
}

//-----------------------------------------------------------------------------
// 非周期性指標から非周期性スペクトルの計算
//-----------------------------------------------------------------------------
void CalculateAperiodicity(double *aperiodicity, int number_of_bands, 
  int fft_size, double f0, int fs, double target_f0, 
  double *periodic_spec)
{
  if(f0 == 0)
  {
    for(int i = 0;i <= fft_size/2;i++) periodic_spec[i] = 0.0;
    return;
  }
  double *ap = (double *)malloc(sizeof(double) * (number_of_bands+1) );
  double *axis = (double *)malloc(sizeof(double) * (number_of_bands+1) );
  double *w = (double *)malloc(sizeof(double) * (fft_size/2+1) );
  double *tmp_ap = (double *)malloc(sizeof(double) * (fft_size/2+1) );

  double *cutoff_list = (double *)malloc(sizeof(double) * number_of_bands);
  for(int i = 0;i < number_of_bands;i++)
    cutoff_list[i] = (double)fs / pow(2.0, (double)(i+2));

  double stretching_factor = MAX(f0, target_f0)/target_f0;

  ap[0] = log(0.0000000005); // セーフガード，実際はまぁ，色々とある
  axis[0] = 0.0;
  for(int i = 0;i < number_of_bands-1;i++)
  {
    ap[i+1] = log(aperiodicity[i]);
    axis[i+1] = cutoff_list[number_of_bands-i-2];
  }
  ap[number_of_bands]   = log(aperiodicity[number_of_bands-1]);
  axis[number_of_bands] = (double)fs/2.0;

  for(int i = 0;i <= fft_size/2;i++)
    w[i] = (double)(i * fs)/ (double)fft_size;
  interp1(axis, ap, number_of_bands+1, w, fft_size/2+1, tmp_ap);
  for(int i = 0;i < number_of_bands-1;i++)
    axis[i+1] *= stretching_factor;
  axis[number_of_bands] = (double)fs/2.0 * stretching_factor;

  interp1(axis, ap, number_of_bands+1, w, fft_size/2+1, periodic_spec);

  for(int i = 0;i <= fft_size/2;i++)
    periodic_spec[i] = 1.0 - 
    MIN(exp(tmp_ap[i]*2.0), exp(periodic_spec[i]*2.0));

  free(tmp_ap);
  free(cutoff_list);
  free(w);
  free(axis);
  free(ap);
}

//-----------------------------------------------------------------------------
// hHP:41, hLP:37の要素を持つdouble型のQMFを設定．
// Matlab版もfsを引数にするが，結果はfsに依存せず一定．これって何だろう？
//-----------------------------------------------------------------------------
void GetQMFpairOfFilters(int fs, double *hHP, double *hLP)
{
  // hHP
  hHP[0]  =  0.00041447996898231424;
  hHP[1]  =  0.00078125051417292477;
  hHP[2]  = -0.0010917236836275842;
  hHP[3]  = -0.0019867925675967589;
  hHP[4]  =  0.0020903896961562292;
  hHP[5]  =  0.0040940570272849346;
  hHP[6]  = -0.0034025808529816698;
  hHP[7]  = -0.0074961541272056016;
  hHP[8]  =  0.0049722633399330637;
  hHP[9]  =  0.012738791249119802;
  hHP[10] = -0.0066960326895749113;
  hHP[11] = -0.020694051570247052;
  hHP[12] =  0.0084324365650413451;
  hHP[13] =  0.033074383758700532;
  hHP[14] = -0.010018936738799522;
  hHP[15] = -0.054231361405808247;
  hHP[16] =  0.011293988915051487;
  hHP[17] =  0.10020081367388213;
  hHP[18] = -0.012120546202484579;
  hHP[19] = -0.31630021039095702;
  hHP[20] =  0.51240682580627639;
  hHP[21] = -0.31630021039095702;
  hHP[22] = -0.012120546202484579;
  hHP[23] =  0.10020081367388213;
  hHP[24] =  0.011293988915051487;
  hHP[25] = -0.054231361405808247;
  hHP[26] = -0.010018936738799522;
  hHP[27] =  0.033074383758700532;
  hHP[28] =  0.0084324365650413451;
  hHP[29] = -0.020694051570247052;
  hHP[30] = -0.0066960326895749113;
  hHP[31] =  0.012738791249119802;
  hHP[32] =  0.0049722633399330637;
  hHP[33] = -0.0074961541272056016;
  hHP[34] = -0.0034025808529816698;
  hHP[35] =  0.0040940570272849346;
  hHP[36] =  0.0020903896961562292;
  hHP[37] = -0.0019867925675967589;
  hHP[38] = -0.0010917236836275842;
  hHP[39] =  0.00078125051417292477;
  hHP[40] =  0.00041447996898231424;

  // hLP
  hLP[0]  = -0.00065488170077483048;
  hLP[1]  =  0.00007561994958159384;
  hLP[2]  =  0.0020408456937895227;
  hLP[3]  = -0.00074680535322030437;
  hLP[4]  = -0.0043502235688264931;
  hLP[5]  =  0.0025966428382642732;
  hLP[6]  =  0.0076396022827566962;
  hLP[7]  = -0.0064904118901497852;
  hLP[8]  = -0.011765804538954506;
  hLP[9]  =  0.013649908479276255;
  hLP[10] =  0.01636866479016021;
  hLP[11] = -0.026075976030529347;
  hLP[12] = -0.020910294856659444;
  hLP[13] =  0.048260725032316647;
  hLP[14] =  0.024767846611048111;
  hLP[15] = -0.096178467583360641;
  hLP[16] = -0.027359756709866623;
  hLP[17] =  0.31488052161630042;
  hLP[18] =  0.52827343594055032;
  hLP[19] =  0.31488052161630042;
  hLP[20] = -0.027359756709866623;
  hLP[21] = -0.096178467583360641;
  hLP[22] =  0.024767846611048111;
  hLP[23] =  0.048260725032316647;
  hLP[24] = -0.020910294856659444;
  hLP[25] = -0.026075976030529347;
  hLP[26] =  0.01636866479016021;
  hLP[27] =  0.013649908479276255;
  hLP[28] = -0.011765804538954506;
  hLP[29] = -0.0064904118901497852;
  hLP[30] =  0.0076396022827566962;
  hLP[31] =  0.0025966428382642732;
  hLP[32] = -0.0043502235688264931;
  hLP[33] = -0.00074680535322030437;
  hLP[34] =  0.0020408456937895227;
  hLP[35] =  0.00007561994958159384;
  hLP[36] = -0.00065488170077483048;
}

}
}
}
