#include "SequenceXmlFactory.h"

#include "TreeItem.h"
#include "ControlItem.h"
#include "EventItem.h"
#include "TrackItem.h"
#include "MeasureItem.h"

#include <QDomNodeList>
#include <QPoint>

using namespace stand::model::xml;
using namespace stand::model;

SequenceXmlFactory::SequenceXmlFactory()
{
    _factories["item"] = &SequenceXmlFactory::_createItem;
    _factories["control"] = &SequenceXmlFactory::_createControl;
    _factories["track"] = &SequenceXmlFactory::_createTrack;
    _factories["measure"] = &SequenceXmlFactory::_createMeasure;
    _factories["event"] = &SequenceXmlFactory::_createEvent;
}

TreeItem *SequenceXmlFactory::createItem(const QDomElement &element, TreeItem *parent) const
{
    if(_factories.contains(element.tagName()))
    {
        TreeItem *item = (this->*(_factories[element.tagName()]))(element, parent);
        item->setFlags((Qt::ItemFlag)element.attribute("edit_flags").toInt());
        return item;
    }
    qDebug("SequenceXmlFactory::createItem(%s); // no known type", element.tagName().toUtf8().data());
    TreeItem *item = _createItem(element, parent);
    item->setFlags((Qt::ItemFlag)element.attribute("edit_flags").toInt());
    return item;
}

TreeItem *SequenceXmlFactory::_createControl(const QDomElement &element, TreeItem *parent) const
{
    double defaultValue = 0.0;
    QString attribute;
    if((attribute = element.attribute("default_value")) != QString::null)
    {
        defaultValue = attribute.toDouble();
    }
    ControlItem *item = new ControlItem(element.attribute("name"), defaultValue, parent);

    for(QDomElement ce = element.firstChildElement("point"); !ce.isNull(); ce = ce.nextSiblingElement("point"))
    {
        stand::utility::ControlPoint p = {ce.attribute("tick").toInt(), ce.attribute("value").toDouble()};
        item->appendPoint(p);
    }
    return item;
}

TreeItem *SequenceXmlFactory::_createEvent(const QDomElement &element, TreeItem *parent) const
{
    int note = element.attribute("note").toInt();
    int length = element.attribute("length").toInt();
    int velocity = element.attribute("velocity").toInt();
    int tick = element.attribute("tick").toInt();
    EventItem *item = new EventItem(element.attribute("name"), parent);
    item->setData(EventItem::TickData, tick, 0);
    item->setData(EventItem::LengthData, length, 0);
    item->setData(EventItem::NoteData, note, 0);
    item->setData(EventItem::VelocityData, velocity, 0);

    QDomElement pr = element.firstChildElement("pronounce");
    item->setData(EventItem::PronounceData, pr.text(), 0);
    QDomElement ly = element.firstChildElement("lyric");
    item->setData(EventItem::LyricData, ly.text(), 0);

    return item;
}

TreeItem *SequenceXmlFactory::_createMeasure(const QDomElement &element, TreeItem *parent) const
{
    int num = element.attribute("default_numerator").toInt();
    int den = element.attribute("default_denominator").toInt();
    MeasureItem *item = new MeasureItem(element.attribute("name"), num, den, parent);

    for(QDomElement ce = element.firstChildElement("beat"); !ce.isNull(); ce = ce.nextSiblingElement("beat"))
    {
        int measure = ce.attribute("measure").toInt();
        int numerator = ce.attribute("numerator").toInt();
        int denominator = ce.attribute("denominator").toInt();
        item->setData(measure, QPoint(numerator, denominator), 0);
    }

    return item;
}

TreeItem *SequenceXmlFactory::_createItem(const QDomElement &element, TreeItem *parent) const
{
    TreeItem *item = new TreeItem(element.attribute("name"), parent);

    QDomNodeList nl = element.childNodes();
    for(int i = 0; i < nl.size(); i++)
    {
        QDomElement ce = nl.at(i).toElement();
        if(ce.tagName() == "data")
        {
            int column = ce.attribute("column").toInt();
            //int type = ce.attribute("type").toInt();
            QVariant var(ce.text());
            item->setData(column, var, 0);
        }
        else
        {
            item->appendChild(createItem(ce, item));
        }
    }
    return item;
}

TreeItem *SequenceXmlFactory::_createTrack(const QDomElement &element, TreeItem *parent) const
{
    TrackItem *item = new TrackItem(element.attribute("name"), parent);
    while(item->child(0))
    {
        item->removeRow(0);
    }

    QDomNodeList nl = element.childNodes();
    for(int i = 0; i < nl.size(); i++)
    {
        QDomElement ce = nl.at(i).toElement();
        if(ce.tagName() == "data")
        {
            int column = ce.attribute("column").toInt();
            int type = ce.attribute("type").toInt();
            QVariant var(ce.text());
            item->setData(column, var, 0);
        }
        else
        {
            TreeItem *ci = createItem(ce, item);
            if(ce.attribute("name") == "Events")
            {
                if(!item->insertRow(1, ci))
                {
                    item->appendChild(ci);
                }
            }
            else
            {
                item->appendChild(ci);
            }
        }
    }
    return item;
}

