#include "Utility.h"
#include "../configure.h"

#include "WaveBuffer.h"
#include "UtauPhoneme.h"

#include <QMessageBox>
#include <QHash>

#include <math.h>

namespace stand
{
namespace utility
{

// 波形の正規化に使う定数群
const static unsigned int NUM_NORMALIZE = 2048;
const static double VOL_NORMALIZE = 0.06;

}
}
int stand::utility::FromBigEndian(int i)
{
    char *p = (char *)(&i), tmp;
    tmp = p[3];
    p[3] = p[0];
    p[0] = tmp;
    tmp = p[2];
    p[2] = p[1];
    p[1] = tmp;
    return i;
}

short stand::utility::FromBigEndian(short s)
{
    char *p = (char *)(&s), tmp;
    tmp = p[1];
    p[1] = p[0];
    p[0] = tmp;
    return s;
}

bool stand::utility::makeDirectory(QWidget *w, const QDir &dir, bool confirm)
{
    // 存在する場合は真を返して終了．
    if(dir.exists())
    {
        return true;
    }
    // 確認する場合
    if(confirm)
    {
        int val = QMessageBox::warning(w,
                                       QObject::tr("Confirm"),
                                       dir.path() + QObject::tr("\ndoes not exist. Do you want to create the directory"),
                                       QMessageBox::Yes, QMessageBox::No);
        if(val == QMessageBox::No)
        {
            return false;
        }
    }
    // ディレクトリを作成．
    if(!dir.mkpath(dir.absolutePath()))
    {
        QMessageBox::critical(w, QObject::tr("Error"), dir.path() + QObject::tr("\nCould not mkdir."));
        return false;
    }
    return true;
}

int stand::utility::lengthForUtauSetting(int waveLength, int samplingFrequency, float leftBlank, float rightBlank)
{
    int ret = 0;
    // 負値の右ブランクは音素長を示す．
    if(rightBlank < 0)
    {
        ret = -rightBlank / 1000.0 * samplingFrequency;
    }
    // 正値の右ブランクは音素終了位置からのブランク値
    else
    {
        ret = waveLength - (leftBlank + rightBlank) / 1000.0 * samplingFrequency;
    }
    return ret;
}

void stand::utility::normalizeByAnchor(double *x, unsigned int length)
{
    double sumBegin = 0, sumEnd = 0, maxAmp, c;
    // 開始・終了位置から NUM_NORMALIZE 点の二乗振幅平均のうち大きい方を使って正規化する．
    for(unsigned i = 0; i < NUM_NORMALIZE; i++)
    {
        sumBegin += x[i] * x[i];
        sumEnd += x[length - i - 1] * x[length - i - 1];
    }
    maxAmp = (sumBegin > sumEnd) ? sumBegin : sumEnd;
    maxAmp = sqrt(maxAmp / (double)NUM_NORMALIZE);

    // 完全に無音の場合はゼロ除算が出るので避ける．
    if(maxAmp == 0.0)
    {
        return;
    }
    c = VOL_NORMALIZE / maxAmp;
    for(unsigned int i = 0; i < length; i++)
    {
        x[i] *= c;
    }
}

bool stand::utility::IsBlackKey(unsigned char note)
{
    return stand::sequence::IsBlackKey[note % 12];
}

QString stand::utility::NoteName(unsigned char note)
{
    int octave = note / 12;
    return QString(stand::sequence::KeyName[note % 12]) + QString::number(octave);
}

int stand::utility::NoteNumber(const QString &name)
{
    for(int i = 0; i < 128; i++)
    {
        if(NoteName(i) == name)
        {
            return i;
        }
    }
    return 0;
}

double stand::utility::NoteFrequency(int note)
{
    return stand::A4Frequency * pow(2.0, (note - stand::A4NoteNumber) / 12.0);
}

double stand::utility::NoteFrequency(double note)
{
    return stand::A4Frequency * pow(2.0, (note - stand::A4NoteNumber) / 12.0);
}

double stand::utility::NoteAt(double frequency)
{
    return log(frequency / stand::A4Frequency) / log(2.0) * 12.0 + stand::A4NoteNumber;
}

int stand::utility::NoteFromString(const QString &name)
{
    static QHash<QString, int> dictionary;
    if(dictionary.empty())
    {
        for(int i = 0; i < 128; i++)
        {
            dictionary[NoteName(i)] = i;
        }
    }
    int retval = -1;
    if(dictionary.contains(name))
    {
        retval = dictionary[name];
    }
    return retval;
}

double stand::utility::msLengthForUtauPhoneme(const char *path, const stand::io::utau::UtauPhoneme *p)
{
    if(p->rightBlank < 0)
    {
        return -p->rightBlank;
    }

    stand::io::WaveBuffer wave;
    if(!wave.read(path) || wave.empty())
    {
        return 0.0;
    }
    double ms = wave.length() * 1000.0 / (double)wave.header()->samplesPerSecond;
    ms = ms - p->leftBlank - p->rightBlank;

    return ms;
}

double stand::utility::MsFromTick(int tick, double bpm)
{
    return (double)tick / (double)stand::sequence::DefaultTicksPerBeat * 60.0 / bpm * 1000.0;
}

double stand::utility::TicksFromMs(double ms, double bpm)
{
    return ms / 1000.0 / 60.0 * bpm * stand::sequence::DefaultTicksPerBeat;
}

