#include "Envelope.h"

#include <qglobal.h>
#include <stdio.h>
#include <string.h>

using namespace stand::utility::envelope;

Envelope::Envelope(double *data, int tLen, double framePeriod, bool autodelete)
{
    _data = NULL;
    _autodelete = false;
    _defaultValue = 0.0;
    if(data)
    {
        set(data, tLen, framePeriod, autodelete);
    }
    else
    {
        _framePeriod = framePeriod;
        if(tLen)
        {
            set(new double[tLen], tLen, framePeriod, true);
        }
    }
}

Envelope::~Envelope()
{
    _destroy();
}

void Envelope::_destroy()
{
    if(_autodelete)
    {
        delete _data;
    }
    _data = NULL;
    _autodelete = false;
}

void Envelope::set(double *data, int tLen, double framePeriod, bool autodelete)
{
    if(!data)
    {
        qDebug("Envelope::set(NULL); // Invalid args.");
        return;
    }
    _destroy();
    _data = data;
    _tLen = tLen;
    _framePeriod = framePeriod;
    _autodelete = autodelete;
}

bool Envelope::write(const char *path) const
{
    if(!_data)
    {
        qDebug("Envelope::write(%s); // envelope is empty.", path);
        return false;
    }
    if(!path)
    {
        qDebug("Envelope::write(NULL); // invalid arg");
        return false;
    }
    FILE *fp = fopen(path, "wb");
    if(!fp)
    {
        qDebug("Envelope::write(%s); // File could not be opened.", path);
        return false;
    }

    fprintf(fp, "Stnd");
    size_t size = sizeof(double) * _tLen + sizeof(size_t) + 4 + sizeof(int) + sizeof(double);
    fwrite(&size, sizeof(size_t), 1, fp);
    fprintf(fp, "Evlp");
    fwrite(&_tLen, sizeof(int), 1, fp);
    fwrite(&_framePeriod, sizeof(double), 1, fp);
    fwrite(_data, sizeof(double), _tLen, fp);

    fclose(fp);

    return true;
}

bool Envelope::read(const char *path)
{
    if(!path)
    {
        qDebug("Envelope::read(NULL); // invalid arg");
        return false;
    }
    FILE *fp = fopen(path, "rb");
    if(!fp)
    {
        qDebug("Envelope::read(%s); // File could not be opened.", path);
        return false;
    }

    char header[5];
    header[4] = '\0';
    fread(header, sizeof(char), 4, fp);
    if(strcmp(header, "Stnd"))
    {
        fclose(fp);
        qDebug("Envelope::read(%s); // This file is not stand envelope.", path);
        return false;
    }
    size_t size;
    fread(&size, sizeof(size_t), 1, fp);
    if(size <= 0)
    {
        fclose(fp);
        qDebug("Envelope::read(%s); // File size is invalid.", path);
        return false;
    }
    size_t c;
    char *buf = new char[size];
    c = fread(buf, sizeof(char), size, fp);
    fclose(fp);
    if(c != size || size < sizeof(double) * 2 + sizeof(int) + 4)
    {
        delete[] buf;
        qDebug("Envelope::read(%s); // Invalid file size.", path);
        return false;
    }
    memcpy(header, buf, 4);
    if(strcmp(header, "Evlp"))
    {
        delete[] buf;
        qDebug("Envelope::read(%s); // This file is not a envelope.", path);
        return false;
    }

    int tLen;
    memcpy(&tLen, buf + 4, sizeof(int));
    double framePeriod;
    memcpy(&framePeriod, buf + sizeof(int) + 4, sizeof(double));
    double *data = new double[tLen];
    memcpy(&data, buf + sizeof(int) + sizeof(double) + 4, sizeof(double) * tLen);
    set(data, tLen, framePeriod, true);

    delete[] buf;

    return false;
}

double Envelope::at(double ms) const
{
    int index = ms / _framePeriod;
    return at(index);
}

double Envelope::at(int index) const
{
    if(!_data)
    {
        return 0.0;
    }
    if(index < 0)
    {
        index = 0;
    }
    else if(index >= _tLen)
    {
        index = _tLen - 1;
    }
    return _data[index];
}

double *Envelope::data() const
{
    return _data;
}

double Envelope::framePeriod() const
{
    return _framePeriod;
}

int Envelope::tLen() const
{
    return _tLen;
}

double Envelope::length() const
{
    return _tLen * _framePeriod;
}
