##############################################################################
#    pvm
#    software development environment
#
#    Copyright (C) 1997  Andrew Guryanov
#    andrew-guryanov@usa.net
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
##############################################################################

#===========================================
#	Global variables and initialization
#===========================================

main:RegisterDocType "doc-project"

global docdata


#===========================================
#	Project document
#===========================================

proc doc-project:MatchDocType {filename} {
#---------------------------------------------------
    global docdata
    set ext [file extension $filename]
    if {$ext == $docdata(projectdef,type_make)} {
        return 10
    }
    set tail [file tail $filename]
    if {[string tolower $tail] == "makefile"} {
        return 10
    }
    return 0
}

proc doc-project:OnCreateClient {this} {
#---------------------------------
    listbox $this.list -relief sunken  -borderwidth 1 -selectmode extended  -width 1 -height 1 \
        -yscrollcommand "$this.yscroll set"
    scrollbar $this.yscroll -orient vertical -relief ridge -command "$this.list yview"
    pack $this.yscroll -side right -fill y
    pack $this.list -expand 1 -fill both -side bottom

    bind $this.list <Return> {doc-project:OnListDblClk %W}
    bind $this.list <Double-Button-1> {doc-project:OnListDblClk %W}
    bind $this.list <Button-1> {doc-project:OnListSelect %W [%W index @%x,%y]}
    bind $this.list <KeyRelease> {doc-project:OnListSelect %W active}
    bind $this.list <FocusOut> {main:SetStatusBarText ""}
}

proc doc-project:RenameDocument {this filename} {
#--------------------------------------------
    global docdata
    doc:RenameDocument $this $filename
    set id [main:GetWindowId $this]
    if {$filename != ""} {
        set docdata($id,title) [file tail $filename]
    } else {
        set docdata($id,title) "Project"
    }
    if {$docdata($id,modified)} {
        set docdata($id,title) "*$docdata($id,title)"
    }
}

proc doc-project:OnChangeAppearance {this} {
#-----------------------------------------
    $this.list configure \
        -font [main:GetMainData font def] \
        -fg [main:GetMainData color projectfg] \
        -bg [main:GetMainData color projectbg] \
        -selectforeground [main:GetMainData color projectselfg] \
        -selectbackground [main:GetMainData color projectselbg]
}

proc doc-project:InitDocument {this} {
#--------------------------------------------
    global docdata
    set id [main:GetWindowId $this]

    set docdata($id,contents)    ""
    set docdata($id,alldepend)   ""

    set docdata($id,glob_stdlib)  ""
    set docdata($id,glob_as)      ""
    set docdata($id,glob_cc)      ""
    set docdata($id,glob_ar)      ""
    set docdata($id,glob_link)    ""
    set docdata($id,glob_depgen)  $docdata(projectdef,glob_depgen)
    set docdata($id,glob_pathmode)  "1"

    set docdata($id,deb_name)    $docdata(projectdef,deb_name)
    set docdata($id,deb_out)     $docdata(projectdef,deb_out)
    set docdata($id,deb_obj)     $docdata(projectdef,deb_obj)
    set docdata($id,deb_as)      $docdata(projectdef,deb_as)
    set docdata($id,deb_cc)      $docdata(projectdef,deb_cc)
    set docdata($id,deb_ar)      $docdata(projectdef,deb_ar)
    set docdata($id,deb_link)    $docdata(projectdef,deb_link)

    set docdata($id,rel_name)    $docdata(projectdef,rel_name)
    set docdata($id,rel_out)     $docdata(projectdef,rel_out)
    set docdata($id,rel_obj)     $docdata(projectdef,rel_obj)
    set docdata($id,rel_as)      $docdata(projectdef,rel_as)
    set docdata($id,rel_cc)      $docdata(projectdef,rel_cc)
    set docdata($id,rel_ar)      $docdata(projectdef,rel_ar)
    set docdata($id,rel_link)    $docdata(projectdef,rel_link)

    set docdata($id,debug_dbg)   ""
    set docdata($id,debug_exe)   ""
    set docdata($id,debug_start) ""
    set docdata($id,debug_args)  ""

    set docdata($id,as)          $docdata(projectdef,as)
    set docdata($id,cc)          $docdata(projectdef,cc)
    set docdata($id,ar)          $docdata(projectdef,ar)
    set docdata($id,link)        $docdata(projectdef,link)

    set docdata($id,type_as)     $docdata(projectdef,type_as)
    set docdata($id,type_cc)     $docdata(projectdef,type_cc)
    set docdata($id,type_ar)     $docdata(projectdef,type_ar)
    set docdata($id,type_obj)    $docdata(projectdef,type_obj)
}

proc doc-project:OnNewDocument {this} {
#--------------------------------------------
    doc:OnNewDocument $this
    doc-project:InitDocument $this
}

proc doc-project:OnOpenDocument {this filename} {
#-------------------------------------------------
    global docdata
    set id [main:GetWindowId $this]
    doc:OnOpenDocument $this $filename
    doc-project:InitDocument $this
    set f [open $filename "r"]
    set ext [file extension $filename]
    if {$f != ""} {
        doc-project:DoOpenMake $this $f $filename
        close $f
    }
    doc-project:ShowDocData $this $filename
}

proc doc-project:ShowDocData {this {projfile ""}} {
#------------------------------------------------
    global docdata
    set id [main:GetWindowId $this]
    if {$projfile == ""} {
        set projfile [main:DocDispatch GetDocData $this path]
    }
    $this.list delete 0 end
    foreach name $docdata($id,contents) {
        $this.list insert end [doc-project:CreateRelativePath $projfile $name]
    }
    foreach name $docdata($id,alldepend) {
        $this.list insert end $name
    }
}

proc doc-project:CreateRelativePath {base abspath} {
#-------------------------------------------------------
    set base_loc [split $base "/"]
    set base_len [llength $base_loc]
    set path_loc [split $abspath "/"]
    set path_len [llength $path_loc]

    for {set i 0} {$i < $base_len && $i < $path_len} {incr i} {
        if {[lindex $base_loc $i] != [lindex $path_loc $i]} {
            break;
        }
    }
    set drop [expr $base_len - $i - 1]
    set relpath ""
    for {set l 0} {$l < $drop} {incr l} {
        set relpath "${relpath}../"
    }
    for {set l $i} {$l < $path_len} {incr l} {
        set add [lindex $path_loc $l]
        set relpath "${relpath}${add}/"
    }
    return [string trimright $relpath "/"]
}

proc doc-project:CreateAbsolutePath {base relpath} {
#-------------------------------------------------------
    set base_loc [split $base "/"]
    set base_len [llength $base_loc]
    set path_loc [split $relpath "/"]
    set path_len [llength $path_loc]

    for {set i 0} {$i < $path_len} {incr i} {
        if {[lindex $path_loc $i] != ".."} {
            break;
        }
    }
    set keep [expr $base_len -$i -1]
    set abspath ""
    for {set l 0} {$l < $keep} {incr l} {
        set add [lindex $base_loc $l]
        set abspath "${abspath}${add}/"
    }
    for {set l $i} {$l < $path_len} {incr l} {
        set add [lindex $path_loc $l]
        set abspath "${abspath}${add}/"
    }
    return [string trimright $abspath "/"]
}

proc doc-project:OnSaveDocument {this filename} {
#------------------------------------------------
    main:DocDispatch DoSaveDocument $this $filename "rel"
    doc:SetModified $this false
    doc-project:ShowDocData $this $filename
}

proc doc-project:DoSaveDocument {this filename type} {
#----------------------------------------------------
    global docdata
    set id [main:GetWindowId $this]
    set f [open $filename "w"]
    if {$f != ""} {
        doc-project:DoSaveMake $this $f $filename $type
        close $f
    }
}

proc doc-project:OnProjectEdit {this} {
#--------------------------------------
    set dlg [dialog-project:CreateDialog [main:GetMainData app frame] $this]
    set ret [main:DlgDispatch DoModal $dlg]
    if {$ret == "ok"} {
        main:DocDispatch OnDocChanged $this
    }
}

proc doc-project:GetOutputChannel {} {
#-----------------------------------------------------------------
    set fr_out [mdiclient:GetFrameByType "doc-text-output"]
    if {$fr_out == ""} {
        set fr_out [main:DoOpenFile "" "doc-text-output"]
    }
    if {$fr_out == ""} {
        puts "Cannot create Output window"
        return ""
    }
    set out_ch [main:DocDispatch GetWriteChannel $fr_out 1]
    if {$out_ch == ""} {
        puts "Cannot write to Output window"
    }
    return $out_ch
}

proc doc-project:OnProjectBuild {this args} {
#-----------------------------------------------------------------
    global docdata
    set id [main:GetWindowId $this]
    set cursel [$this.list curselection]

    set active [mdiclient:GetActiveFrame]

    set out_ch [doc-project:GetOutputChannel]
    if {$out_ch == ""} {
        return
    }

    set projfile [main:DocDispatch GetDocData $this path]
    if {$projfile ==""} {
        $out_ch insert end "Cannot build UNTITLED project\n"; $out_ch see end
        return
    }

    if {[mdiclient:SaveAllModified] == "false"} {
        $out_ch insert end "Build canceled\n"; $out_ch see end
        return
    }

#--- build type
    if {[llength $args] == 0} {
        set type deb
    } else {
        set type [lindex $args 0]
        if {$type != "rel"} {
            set type deb
        }
        set args [lreplace $args 0 0]
    }

#--- create targets list
    set seltype [lindex $args 0]
    if {[llength $args] == 0} {
        set targets all
    } else {
        if {$seltype != "select" && $seltype != "rebuildselect"} {
            set targets $args
        } else {
            set targets ""
            set list ""
#--- compile active doc
            if {$active != $this} {
                set name [main:DocDispatch GetDocData $active path]
                if {$name == "" || [lsearch $docdata($id,contents) $name] < 0} {
                    $out_ch insert end "Do not know how to build $name\n"; $out_ch see end
                    return
                }
                set list $name
            } else {
#--- compile selected docs
#                set cursel [$this.list curselection]
                if {[llength $cursel] == 0} {
                    set cursel active
                }
                foreach index $cursel {
                    set item [$this.list get $index]
                    if {$item != ""} {
                        lappend list $item
                    }
                }
            }
            foreach name $list {
                set filename [file tail $name]
                set ext      [file extension $filename]
                set root     [file rootname  $filename]
                if {$seltype == "rebuildselect"} {
                    set trg "${root}$docdata(projectdef,type_clean)"
                    lappend targets $trg
                }
                if {$ext == $docdata($id,type_as) || $ext == $docdata($id,type_cc)} {
                    set trg ""
                    if {$docdata($id,${type}_obj) != ""} {
                        set trg "$docdata($id,${type}_obj)/"
                    }
                    set trg "$trg${root}$docdata($id,type_obj)"
                    lappend targets $trg
                } elseif {$ext == $docdata(projectdef,type_make)} {
                    set trg "${root}$docdata(projectdef,type_all)"
                    lappend targets $trg
                } else {
                    $out_ch insert end "Do not know how to build $name\n"; $out_ch see end
#                    lappend targets $name
                }
            }
        }
    }
    if {[llength $targets] == 0} {
        $out_ch insert end "There is nothing to build\n"; $out_ch see end
        return
    }

#--- save tmp
    set tmpfile [file rootname $projfile]
    set tmpfile ${tmpfile}$docdata(projectdef,type_tmp)
    main:DocDispatch DoSaveDocument $this $tmpfile $type

#--- do build
    set docdata($id,no_stop) true
    foreach target $targets {
        if {$docdata($id,no_stop)} {
            doc-project:DoBuild $this $out_ch $target $tmpfile
        } else {
            break
        }
        $out_ch see end
        update
    }
    if {$docdata($id,no_stop)} {
        $out_ch insert end "Done\n"
    } else {
        $out_ch insert end "Build aborted\n"
    }
}

proc doc-project:OnStopBuild {this} {
#--------------------------------------------------------
    global docdata
    set id [main:GetWindowId $this]
    if {[main:AbortExternalProcess]} {
        set docdata($id,no_stop) false
    }
}

proc doc-project:DoBuild {this out_channel target makefile} {
#--------------------------------------------------------
    global docdata
    set id [main:GetWindowId $this]

    set prev [pwd]
    set filename [file tail $makefile]
    cd [file dirname $makefile]
# puts "$docdata(projectdef,make) -f $filename $target"
    set res [main:StartExternalProcess $out_channel true true $docdata(projectdef,make) -f $filename $target]
    if {$res != ""} {
        main:WaitExternalProcess
    }
    cd $prev
}

proc doc-project:OnProjectDependencies {this} {
#------------------------------------------------
    global docdata
    set id [main:GetWindowId $this]
    set out_ch [doc-project:GetOutputChannel]
    if {$out_ch == ""} {
        return
    }
    if {$docdata($id,glob_depgen) == ""} {
        $out_ch insert end "Do not know how to generate dependencies\n"; $out_ch see end
        return
    }

    set depgen $docdata($id,glob_depgen)
    set docdata(global,type_obj) $docdata($id,type_obj)
    set docdata(global,name) ""
    set docdata(global,list) ""
    set docdata(global,output) ""
    set docdata($id,alldepend) ""
    set prev [pwd]
    set projpath [main:DocDispatch GetDocData $this path]
    if {[file isfile $projpath]} {
        cd [file dirname $projpath]
    } else {
        $out_ch insert end "Cannot generate dependencies for UNTITLED project\n"; $out_ch see end
        return
    }
    set docdata($id,no_stop) true
    foreach name $docdata($id,contents) {
        if {$docdata($id,no_stop) == "false"} {
            break
        }
        set ext [file extension $name]
        if {$ext != $docdata($id,type_cc)} {
            $out_ch insert end "skipping ${name}\n"; $out_ch see end
            continue
        }
        set docdata(global,name) [doc-project:CreateRelativePath $projpath $name]
        $out_ch insert end "scanning ${name}\n"; $out_ch see end
        set success 1
        set res [main:StartExternalProcess doc-project:ReadDependencies false true $depgen $docdata(global,name)]
        if {$res != ""} {
            set success [main:WaitExternalProcess]
        } else {
            $out_ch insert end "Cannot execute ${depgen} ${name}\n"; $out_ch see end
            set docdata($id,no_stop) false
            break
        }
        set docdata($id,depend,$name) ""
        if {!$success} {
            $out_ch insert end $docdata(global,output); $out_ch see end
            set docdata($id,no_stop) false
            break
        }
        foreach entry $docdata(global,list) {
            if {[file pathtype $entry] != "absolute"} {
                set entry [doc-project:CreateAbsolutePath $projpath $entry]
            }
            lappend docdata($id,depend,$name) $entry
            if {[lsearch -exact $docdata($id,alldepend) $entry] < 0} {
                lappend docdata($id,alldepend) $entry
            }
        }
        set docdata(global,list) ""
        set docdata(global,output) ""
    }
    set docdata($id,alldepend) [lsort $docdata($id,alldepend)]
    if {$docdata($id,no_stop)} {
        $out_ch insert end "Done\n"
    } else {
        $out_ch insert end "Process aborted\n"
    }
    $out_ch see end
    cd $prev
    unset docdata(global,type_obj)
    unset docdata(global,name)
    unset docdata(global,list)
    unset docdata(global,output)
    doc:SetModified $this
    doc-project:ShowDocData $this
}

proc doc-project:ReadDependencies {args} {
#------------------------------------------
    global docdata
    set lst [split $args]
    if {[lindex $lst 0] == "insert"} {
        set num [llength $lst]
        for {set i 2} {$i < $num} {incr i} {
            set entries [lindex $lst $i]
            foreach entry [split $entries ":"] {
                set entry [string trimleft $entry "{"]
                while {[string range $entry 0 0] == "\\" && [string range $entry 1 1] == "t"} {
                    set entry [string range $entry 2 end]
                }
                set entry [string trimleft $entry]
                set entry [string trimright $entry "}"]
                set entry [string trimright $entry "\\"]
                set entry [string trimright $entry]
                set ext [file extension $entry]
                if {$ext == $docdata(global,type_obj)} {
                    continue
                }
                set docdata(global,output) "$docdata(global,output) ${entry}"
                if {$entry != $docdata(global,name) && [file isfile $entry] && [lsearch -exact $docdata(global,list) $entry] < 0} {
                    lappend docdata(global,list) $entry
                }
            }
        }
        set docdata(global,output) "$docdata(global,output)\n"
    }
}

proc doc-project:OnProjectDebug {this} {
#--------------------------------------
    global docdata
    set id [main:GetWindowId $this]
    set prev [pwd]
    if {[file isdirectory $docdata($id,debug_start)]} {
        cd $docdata($id,debug_start)
    }
    if {$docdata($id,debug_dbg) != ""} {
        exec -- $docdata($id,debug_dbg) $docdata($id,debug_exe) $docdata($id,debug_args) &
    } elseif {$docdata($id,debug_exe) != ""} {
        exec -- $docdata($id,debug_exe) $docdata($id,debug_args) &
    }
    cd $prev
}

proc doc-project:OnProjectRun {this} {
#--------------------------------------
    global docdata
    set id [main:GetWindowId $this]
    set prev [pwd]
    if {"xterm" != ""} {
        if {[file isdirectory $docdata($id,debug_start)]} {
            cd $docdata($id,debug_start)
        }
        exec -- xterm -e ./first.exe &
    }
    cd $prev
}

proc doc-project:OnDocChanged {this} {
#--------------------------------------
    doc-project:ShowDocData $this
    doc:SetModified $this
}

proc doc-project:OnListDblClk {W} {
#------------------------------------
    global docdata
    set this [mdiclient:GetDocFrame $W]
    set id [main:GetWindowId $this]
    set projpath [main:DocDispatch GetDocData $this path]
    set filename [$W get active]
    if {[file pathtype $filename] != "absolute"} {
        set filename [doc-project:CreateAbsolutePath $projpath $filename]
    }
    main:DoOpenFile $filename
}

proc doc-project:OnListSelect {W index} {
#---------------------------------------
    global docdata
    set this [mdiclient:GetDocFrame $W]
    set id [main:GetWindowId $this]
    set projpath [main:DocDispatch GetDocData $this path]
    set file [$W get $index]
    if {[file pathtype $file] != "absolute"} {
        set file [doc-project:CreateAbsolutePath $projpath $file]
    }
    if {[file exists $file]} {
        set size  [file size $file]
        set mtime [clock format [file mtime $file] -format "%d-%b-%Y  %H:%M:%S"]
        set text "$file : $size bytes,  $mtime"
    } else {
        set text "$file : NOT FOUND"
    }
    main:SetStatusBarText $text
}


proc doc-project:DoOpenMake {this f filename} {
#---------------------------------------------------
    global docdata
    set id [main:GetWindowId $this]
    set docdata($id,contents)  ""
    set projpath [file dirname $filename]
    set pvm_version 1.0
    while {![eof $f]} {
        if {[gets $f line] > 0} {
            set ll [split $line "="]
            if {[llength $ll] >= 2} {
                set l0 [lindex $ll 0]

                set l1 [lindex $ll 1]
                set count [llength $ll]
                if {$count > 2} {
                    for {set i 2} {$i < $count} {incr i} {
                        set l1 "$l1=[lindex $ll $i]"
                    }
                }

                switch $l0 {

                PVM_VERSION     {set pvm_version $l1}
                ASSEMBLER       {set docdata($id,as)       $l1}
                COMPILER        {set docdata($id,cc)       $l1}
                ARCHIVER        {set docdata($id,ar)       $l1}
                LINKER          {set docdata($id,link)     $l1}
                TYPE_ASSEMBLER  {set docdata($id,type_as)  $l1}
                TYPE_COMPILER   {set docdata($id,type_cc)  $l1}
                TYPE_ARCHIVER   {set docdata($id,type_ar)  $l1}
                TYPE_OBJECT     {set docdata($id,type_obj) $l1}

                PROJECT_PATH    {set projpath $l1}

                STD_LIBS        {set docdata($id,glob_stdlib)  $l1}
                _GLOB_ASF_      {set docdata($id,glob_as)      $l1}
                _GLOB_CCF_      {set docdata($id,glob_cc)      $l1}
                _GLOB_ARF_      {set docdata($id,glob_ar)      $l1}
                _GLOB_LINKF_    {set docdata($id,glob_link)    $l1}
                _GLOB_DEPGEN_   {set docdata($id,glob_depgen)  $l1}
                SAVE_PATH_MODE  {set docdata($id,glob_pathmode)  $l1}

                DEB_PROJECT_RESULT  {set docdata($id,deb_name) $l1}
                DEB_OUT_PATH        {set docdata($id,deb_out)  $l1}
                DEB_OBJECT_PATH     {set docdata($id,deb_obj)  $l1}
                DEB_ASSEMBLER_FLAGS {set docdata($id,deb_as)   $l1}
                DEB_COMPILER_FLAGS  {set docdata($id,deb_cc)   $l1}
                DEB_ARCHIVER_FLAGS  {set docdata($id,deb_ar)   $l1}
                DEB_LINKER_FLAGS    {set docdata($id,deb_link) $l1}

                REL_PROJECT_RESULT  {set docdata($id,rel_name) $l1}
                REL_OUT_PATH        {set docdata($id,rel_out)  $l1}
                REL_OBJECT_PATH     {set docdata($id,rel_obj)  $l1}
                REL_ASSEMBLER_FLAGS {set docdata($id,rel_as)   $l1}
                REL_COMPILER_FLAGS  {set docdata($id,rel_cc)   $l1}
                REL_ARCHIVER_FLAGS  {set docdata($id,rel_ar)   $l1}
                REL_LINKER_FLAGS    {set docdata($id,rel_link) $l1}

                PROJECT_RESULT    {}
                OUT_PATH          {}
                OBJECT_PATH       {}
                ASSEMBLER_FLAGS   {}
                COMPILER_FLAGS    {}
                ARCHIVER_FLAGS    {}
                LINKER_FLAGS      {}

                DEBUG_DBG       {set docdata($id,debug_dbg)   $l1}
                DEBUG_EXE       {set docdata($id,debug_exe)   $l1}
                DEBUG_START     {
                                    if {[file pathtype $l1] != "absolute"} {
                                        set l1 [doc-project:CreateAbsolutePath "$projpath/x" $l1]
                                    }
                                    set docdata($id,debug_start) $l1
                                }
                DEBUG_ARGS      {set docdata($id,debug_args)  $l1}

                OBJECT_FILES    {break}
                OBJECT_CONTENTS {break}
                MAKE_FILES      {break}
                MAKE_TARGETS    {break}
                ARC_FILES       {break}

                default {
                    set cont [split $l0 "_"]
                    if {[llength $cont] == 3} {
                        set cont0 [lindex $cont 0]
                        set cont1 [lindex $cont 1]
                        if {$cont0 == "" && $cont1 == "file"} {
                            set fname $l1
                            gets $f line
                            set dirname [lindex [split $line "="] 1]
                            if {$dirname != "" && $dirname != "."} {
                                set item "$dirname/$fname"
                            } else {
                                set item "$fname"
                            }
                            if {[file pathtype $item] != "absolute"} {
                                set item [doc-project:CreateAbsolutePath "$projpath/x" $item]
                            }
                            lappend docdata($id,contents) $item

                            set docdata($id,depend,$item) ""
                            set prevpos [tell $f]
                            gets $f line
                            set cont [split $line "_"]
                            if {[llength $cont] == 3 && [lindex $cont 1] == "depend"} {
                                while {1} {
                                    set prevpos [tell $f]
                                    gets $f line
                                    set line [string trimleft $line]
                                    set line [string trimright $line "\\"]
                                    set line [string trimright $line]
                                    if {$line == "" || [string range $line 0 0] == "_"} {
                                        seek $f $prevpos start
                                        break
                                    }
                                    foreach dep [split $line] {
                                        if {[file pathtype $dep] != "absolute"} {
                                            set dep [doc-project:CreateAbsolutePath "$projpath/x" $dep]
                                        }
                                        lappend docdata($id,depend,$item) $dep
                                        if {[lsearch -exact $docdata($id,alldepend) $dep] < 0} {
                                            lappend docdata($id,alldepend) $dep
                                        }
                                    }
                                }
                            } else {
                                seek $f $prevpos start
                            }
                        }
                    }
                }
                }
            }
        }
    }
    set docdata($id,alldepend) [lsort $docdata($id,alldepend)]
    set currpath [file dirname $filename]
    if {$currpath != $projpath} {
        set out_ch [doc-project:GetOutputChannel]
        if {$out_ch!=""} {$out_ch insert end "WARNING: project $filename was moved from $projpath\n"}
        if {$out_ch!=""} {$out_ch insert end "Looking for project files:\n"}
        set contents ""
        foreach name $docdata($id,contents) {
            set filename [file tail $name]
            set new_loc [doc-project:FindNewPath $projpath [file dirname $name] $currpath]
            set new_loc "${new_loc}/${filename}"
            if {[file isfile $new_loc]} {
                lappend contents $new_loc
                if {$out_ch!=""} {$out_ch insert end "$name is found as $new_loc\n"}
            } else {
                lappend contents $name
                if {[file isfile $name]} {
                    if {$out_ch!=""} {$out_ch insert end "NOT MOVED: $name\n"}
                } else {
                    if {$out_ch!=""} {$out_ch insert end "NOT FOUND: $name\n"}
                }
            }
        }
        set docdata($id,contents) $contents
        if {$docdata($id,debug_start) != ""} {
            set new_loc [doc-project:FindNewPath $projpath $docdata($id,debug_start) $currpath]
        } else {
            set new_loc $currpath
        }
        if {$out_ch!=""} {$out_ch insert end "Debug start-up directory $docdata($id,debug_start) is changed to $new_loc\n"}
        set docdata($id,debug_start) $new_loc
        doc:SetModified $this
        if {$out_ch!=""} {$out_ch insert end "NOTE: regenerate all dependencies, please\n"}
        set docdata($id,alldepend) ""
    }
}

proc doc-project:FindNewPath {projpath filepath currpath} {
#--------------------------------------------------------------
    set curr_loc [split $currpath "/"]
    set curr_len [llength $curr_loc]
    set proj_loc [split $projpath "/"]
    set proj_len [llength $proj_loc]
    set file_loc [split $filepath "/"]
    set file_len [llength $file_loc]

    for {set i 0} {$i < $proj_len && $i < $file_len} {incr i} {
        if {[lindex $proj_loc $i] != [lindex $file_loc $i]} {
            break;
        }
    }
    set drop [expr $proj_len - $i]
    set keep [expr $curr_len - $drop]
    set new_loc ""
    for {set l 0} {$l < $keep} {incr l} {
        set add [lindex $curr_loc $l]
        set new_loc "${new_loc}${add}/"
    }
    for {set l $i} {$l < $file_len} {incr l} {
        set add [lindex $file_loc $l]
        set new_loc "${new_loc}${add}/"
    }
    regsub -all "//" $new_loc "/" new_loc
    return [string trimright $new_loc "/"]
}

proc doc-project:DoSaveMake {this f filename save_type} {
#--------------------------------------------------------------
    global main
    global docdata
    set id [main:GetWindowId $this]

    set file_t    "_file_"
    set path_t    "_path_"
    set depend_t  "_depend_"
    set object_t  $docdata($id,type_obj)
    set type_make $docdata(projectdef,type_make)
    set all_t     $docdata(projectdef,type_all)
    set clean_t   $docdata(projectdef,type_clean)

    puts $f {#}
    puts $f "#    $main(app,title)"
    puts $f "#    software development environment"
    puts $f {#    Copyright (C) 1997  Andrew Guryanov}
    puts $f {#    auto-generated makefile, Please do not modify}
    puts $f {#}
    puts $f ""
    puts $f {#------ General settings ------}
    puts $f {SHELL=/bin/sh}
    puts $f "PVM_VERSION=$main(app,version)"
    puts $f "SAVE_PATH_MODE=$docdata($id,glob_pathmode)"
    puts $f "ASSEMBLER=$docdata($id,as)"
    puts $f "COMPILER=$docdata($id,cc)"
    puts $f "ARCHIVER=$docdata($id,ar)"
    puts $f "LINKER=$docdata($id,link)"
    puts $f "TYPE_ASSEMBLER=$docdata($id,type_as)"
    puts $f "TYPE_COMPILER=$docdata($id,type_cc)"
    puts $f "TYPE_ARCHIVER=$docdata($id,type_ar)"
    puts $f "TYPE_OBJECT=$docdata($id,type_obj)"
    puts $f ""
    puts $f {#------ Project settings ------}
    set projpath [file dirname $filename]
    puts $f "PROJECT_PATH=$projpath"
    puts $f ""
    puts $f "STD_LIBS=$docdata($id,glob_stdlib)"
    puts $f "_GLOB_ASF_=$docdata($id,glob_as)"
    puts $f "_GLOB_CCF_=$docdata($id,glob_cc)"
    puts $f "_GLOB_ARF_=$docdata($id,glob_ar)"
    puts $f "_GLOB_LINKF_=$docdata($id,glob_link)"
    puts $f "_GLOB_DEPGEN_=$docdata($id,glob_depgen)"
    puts $f ""
    puts $f "DEB_PROJECT_RESULT=$docdata($id,deb_name)"
    puts $f "DEB_OUT_PATH=$docdata($id,deb_out)"
    puts $f "DEB_OBJECT_PATH=$docdata($id,deb_obj)"
    puts $f "DEB_ASSEMBLER_FLAGS=$docdata($id,deb_as)"
    puts $f "DEB_COMPILER_FLAGS=$docdata($id,deb_cc)"
    puts $f "DEB_ARCHIVER_FLAGS=$docdata($id,deb_ar)"
    puts $f "DEB_LINKER_FLAGS=$docdata($id,deb_link)"
    puts $f ""
    puts $f "REL_PROJECT_RESULT=$docdata($id,rel_name)"
    puts $f "REL_OUT_PATH=$docdata($id,rel_out)"
    puts $f "REL_OBJECT_PATH=$docdata($id,rel_obj)"
    puts $f "REL_ASSEMBLER_FLAGS=$docdata($id,rel_as)"
    puts $f "REL_COMPILER_FLAGS=$docdata($id,rel_cc)"
    puts $f "REL_ARCHIVER_FLAGS=$docdata($id,rel_ar)"
    puts $f "REL_LINKER_FLAGS=$docdata($id,rel_link)"
    puts $f ""
    puts $f {#------ Build settings ------}
    if {$save_type == "deb"} {
        set build_t "DEB"
    } else {
        set build_t "REL"
    }
    puts $f "PROJECT_RESULT=\$(${build_t}_PROJECT_RESULT)"
    puts $f "OUT_PATH=\$(${build_t}_OUT_PATH)"
    puts $f "OBJECT_PATH=\$(${build_t}_OBJECT_PATH)"
    puts $f "ASSEMBLER_FLAGS=\$(${build_t}_ASSEMBLER_FLAGS)"
    puts $f "COMPILER_FLAGS=\$(${build_t}_COMPILER_FLAGS)"
    puts $f "ARCHIVER_FLAGS=\$(${build_t}_ARCHIVER_FLAGS)"
    puts $f "LINKER_FLAGS=\$(${build_t}_LINKER_FLAGS)"
    puts $f ""
    puts $f {#------ Debug settings ------}
    puts $f "DEBUG_DBG=$docdata($id,debug_dbg)"
    puts $f "DEBUG_EXE=$docdata($id,debug_exe)"
    set debug_start $docdata($id,debug_start)
    if {[file pathtype $debug_start] != "absolute"} {
        set debug_start [doc-project:CreateAbsolutePath $filename $debug_start]
    }
    puts $f "DEBUG_START=$debug_start"
    puts $f "DEBUG_ARGS=$docdata($id,debug_args)"
    puts $f ""

    puts $f {#------ Project contents ------}
    set filecount [llength $docdata($id,contents)]
    if {$filecount < 1} {
        return
    }
    set i 0
    set n_src 0
    set n_obj 0
    set n_make 0
    set n_lib 0
    foreach name $docdata($id,contents) {
        incr i
        set dirname  [file dirname $name]
        set tailname [file tail $name]
        puts $f "$file_t$i=$tailname"
        if {!$docdata($id,glob_pathmode)} {
            set dirname [doc-project:CreateRelativePath $filename $dirname]
        }
        if {$dirname != ""} {
            puts $f "$path_t$i=$dirname"
        } else {
            puts $f "$path_t$i=."
        }
        if {[info exists docdata($id,depend,$name)]} {
            set deplen [llength $docdata($id,depend,$name)]
            if {$deplen > 0} {
                puts $f "$depend_t$i=\\"
                set dd 0
                foreach dep $docdata($id,depend,$name) {
                    incr dd
                    if {!$docdata($id,glob_pathmode)} {
                        set dep [doc-project:CreateRelativePath $filename $dep]
                    }
                    puts -nonewline $f "\t${dep}"
                    if {$dd < $deplen} {
                        puts $f "\\"
                    } else {
                        puts $f ""
                    }
                }
            }
        }
        set ext [file extension $name]
#--- if the file needs compilation
        if {$ext == $docdata($id,type_as) || $ext == $docdata($id,type_cc)} {
            incr n_src
        } else {
            if {$ext == $docdata($id,type_obj)} {
                incr n_obj
            }
            if {$ext == $docdata($id,type_ar)} {
                incr n_lib
            }
            if {$ext == $type_make} {
                incr n_make
            }
        }
    }
    puts $f ""

    if {$n_src > 0} {
        puts $f "OBJECT_FILES=\\"
        set i 0
        set i_src 0
        foreach name $docdata($id,contents) {
            incr i
            set ext [file extension $name]
#--- if the file needs compilation
            if {$ext == $docdata($id,type_as) || $ext == $docdata($id,type_cc)} {
                incr i_src
                puts -nonewline $f "\t"
                if {$docdata($id,${save_type}_obj) != ""} {
                    puts -nonewline $f "\$(OBJECT_PATH)/"
                }
                puts -nonewline $f "\$(${file_t}${i}:${ext}=${object_t})"
                if {$i_src < $n_src} {
                    puts $f "\\"
                } else {
                    puts $f ""
                }
            }
        }
        puts $f ""
    }

    if {$n_obj > 0} {
        puts $f "OBJECT_CONTENTS=\\"
        set i 0
        set i_obj 0
        foreach name $docdata($id,contents) {
            incr i
            set ext [file extension $name]
            if {$ext == $docdata($id,type_obj)} {
                incr i_obj
                puts -nonewline $f "\t\$(${path_t}${i})/\$(${file_t}${i})"
                if {$i_obj < $n_obj} {
                    puts $f "\\"
                } else {
                    puts $f ""
                }
            }
        }
        puts $f ""
    }

    if {$n_make > 0} {
        puts $f "MAKE_FILES=\\"
        set i 0
        set i_make 0
        foreach name $docdata($id,contents) {
            incr i
            set ext [file extension $name]
            if {$ext == $type_make} {
                incr i_make
                puts -nonewline $f "\t\$(${path_t}${i})/\$(${file_t}${i})"
                if {$i_make < $n_make} {
                    puts $f "\\"
                } else {
                    puts $f ""
                }
            }
        }
        puts $f ""
    }

    if {$n_make > 0} {
        puts $f "MAKE_TARGETS=\\"
        set i 0
        set i_make 0
        foreach name $docdata($id,contents) {
            incr i
            set ext [file extension $name]
            if {$ext == $type_make} {
                incr i_make
                puts -nonewline $f "\t\$(${file_t}${i}:${ext}=${all_t})"
                if {$i_make < $n_make} {
                    puts $f "\\"
                } else {
                    puts $f ""
                }
            }
        }
        puts $f ""
    }

    if {$n_lib > 0} {
        puts $f "ARC_FILES=\\"
        set i 0
        set i_lib 0
        foreach name $docdata($id,contents) {
            incr i
            set ext [file extension $name]
            if {$ext == $docdata($id,type_ar)} {
                incr i_lib
                puts -nonewline $f "\t\$(${path_t}${i})/\$(${file_t}${i})"
                if {$i_lib < $n_lib} {
                    puts $f "\\"
                } else {
                    puts $f ""
                }
            }
        }
        puts $f ""
    }

    puts $f {#------ Targets and build rules ------}
    puts -nonewline $f "all :"
#--- is there a need to create OBJECT_PATH ?
#    if {$docdata($id,${save_type}_obj) != ""} {
#        puts -nonewline $f " \$(OBJECT_PATH)"
#    }
#--- is there a need to create OUT_PATH ?
#    if {$docdata($id,${save_type}_out) != "" && \
#        $docdata($id,${save_type}_out) != $docdata($id,${save_type}_obj)} {
#        puts -nonewline $f " \$(OUT_PATH)"
#    }
    if {$n_src  > 0} { puts -nonewline $f " \$(OBJECT_FILES)"}
    if {$n_obj  > 0} { puts -nonewline $f " \$(OBJECT_CONTENTS)"}
    if {$n_make > 0} { puts -nonewline $f " \$(MAKE_TARGETS)"}
    if {$n_lib  > 0} { puts -nonewline $f " \$(ARC_FILES)"}

    if {$docdata($id,${save_type}_name) != ""} {
        set target_type [file extension $docdata($id,${save_type}_name)]
        if {$target_type == $docdata($id,type_ar)} {
            if {$n_src > 0 || $n_obj  > 0} {
                puts $f ""
                if {$docdata($id,${save_type}_out) != ""} {
                    puts $f "\t@if (test ! -d \$(OUT_PATH)) then (mkdir -p \$(OUT_PATH)) fi"
                }
                puts -nonewline $f "\t\$(ARCHIVER) \$(_GLOB_ARF_) \$(ARCHIVER_FLAGS) "
                if {$docdata($id,${save_type}_out) != ""} {
                    puts -nonewline $f "\$(OUT_PATH)/"
                }
                puts -nonewline $f "\$(PROJECT_RESULT)"
            }
            if {$n_src  > 0} { puts -nonewline $f " \$(OBJECT_FILES)"}
            if {$n_obj  > 0} { puts -nonewline $f " \$(OBJECT_CONTENTS)"}
        } else {
            puts -nonewline $f " "
            if {$docdata($id,${save_type}_out) != ""} {
                puts -nonewline $f "\$(OUT_PATH)/"
            }
            puts -nonewline $f "\$(PROJECT_RESULT)"
        }
    }
    puts $f ""
    puts $f ""

    puts $f "clean :"
    if {$n_src > 0 || $n_make > 0} {
        if {$n_src > 0} {
            puts $f "\t@rm -f \$(OBJECT_FILES)"
        }
        if {$n_make > 0} {
            puts $f "\t@for name in \$(MAKE_FILES); do \\"
            puts -nonewline $f "\t\t(cd `dirname \$\$name`;"
            puts -nonewline $f " echo \"$docdata(projectdef,make) clean \$\$name\";"
            puts $f " $docdata(projectdef,make) -f \$\$name $@) \\"
            puts $f "\tdone;"
        }
    } else {
        puts $f "\t@echo "
    }
    puts $f ""

#    if {$docdata($id,${save_type}_obj) != ""} {
#        puts $f "\$(OBJECT_PATH) :"
#        puts $f "\t@mkdir -p \$(OBJECT_PATH)"
#        puts $f ""
#    }
#    if {$docdata($id,${save_type}_out) != "" && \
#        $docdata($id,${save_type}_out) != $docdata($id,${save_type}_obj)} {
#        puts $f "\$(OUT_PATH) :"
#        puts $f "\t@mkdir -p \$(OUT_PATH)"
#        puts $f ""
#    }

    if {$docdata($id,${save_type}_name) != ""} {
        set target_type [file extension $docdata($id,${save_type}_name)]
        if {$target_type != $docdata($id,type_ar)} {
            if {$docdata($id,${save_type}_out) != ""} {
                puts -nonewline $f "\$(OUT_PATH)/"
            }
            puts -nonewline $f "\$(PROJECT_RESULT) :"
            if {$n_src  > 0} { puts -nonewline $f " \$(OBJECT_FILES)"}
            if {$n_obj  > 0} { puts -nonewline $f " \$(OBJECT_CONTENTS)"}
#            if {$n_make > 0} { puts -nonewline $f " \$(MAKE_TARGETS)"}
            if {$n_lib  > 0} { puts -nonewline $f " \$(ARC_FILES)"}
            puts $f ""
            if {$docdata($id,${save_type}_out) != ""} {
                puts $f "\t@if (test ! -d \$(OUT_PATH)) then (mkdir -p \$(OUT_PATH)) fi"
            }
            if {$n_src > 0 || $n_obj > 0 || $n_lib  > 0} {
                puts -nonewline $f "\t\$(LINKER) \$(_GLOB_LINKF_) \$(LINKER_FLAGS)"
            }
            if {$n_src  > 0} { puts -nonewline $f " \$(OBJECT_FILES)"}
            if {$n_obj  > 0} { puts -nonewline $f " \$(OBJECT_CONTENTS)"}
            if {$n_lib  > 0} { puts -nonewline $f " \$(ARC_FILES)"}
            puts $f " \$(STD_LIBS)"
            puts $f ""
        }
    }

    set i 0
    foreach name $docdata($id,contents) {
        incr i
        set ext [file extension $name]
        if {$ext == $docdata($id,type_as) || $ext == $docdata($id,type_cc)} {
            if {$docdata($id,${save_type}_obj) != ""} {
                puts -nonewline $f "\$(OBJECT_PATH)/"
            }
            puts -nonewline $f "\$(${file_t}${i}:${ext}=${object_t}) :"
            puts -nonewline $f " \$(${path_t}${i})/\$(${file_t}${i})"
            if {[info exists docdata($id,depend,$name)] && [llength $docdata($id,depend,$name)] > 0} {
                puts -nonewline $f " \$(${depend_t}${i})"
            }
            puts $f ""
            if {$docdata($id,${save_type}_obj) != ""} {
                puts $f "\t@if (test ! -d \$(OBJECT_PATH)) then (mkdir -p \$(OBJECT_PATH)) fi"
            }
            if {$ext == $docdata($id,type_cc)} {
                puts -nonewline $f "\t\$(COMPILER) \$(_GLOB_CCF_) \$(COMPILER_FLAGS)"
            } else {
                puts -nonewline $f "\t\$(ASSEMBLER) \$(_GLOB_ASF_) \$(ASSEMBLER_FLAGS)"
            }
            puts $f " \$(${path_t}${i})/\$(${file_t}${i})"
            puts $f ""

            puts $f "\$(${file_t}${i}:${ext}=${clean_t}) :"
            puts -nonewline $f "\t@rm -f "
            if {$docdata($id,${save_type}_obj) != ""} {
                puts -nonewline $f "\$(OBJECT_PATH)/"
            }
            puts $f "\$(${file_t}${i}:${ext}=${object_t})"
            puts $f ""
        }
        if {$ext == $type_make} {
            puts $f "\$(${file_t}${i}:${ext}=${all_t}) :"
            puts $f "\t@(prev=`pwd`; cd \$(${path_t}${i}); echo \"make all \$(${path_t}${i})/\$(${file_t}${i})\"; \\"
            puts $f "\tmake _GLOB_ASF_=\"\$(_GLOB_ASF_)\"  _GLOB_CCF_=\"\$(_GLOB_CCF_)\" \\"
            puts $f "\t\t_GLOB_ARF_=\"\$(_GLOB_ARF_)\" _GLOB_LINKF_=\"\$(_GLOB_LINKF_)\" -f \$(${file_t}${i}) all; \\"
            puts $f "\tcd \$\$prev)"
            puts $f ""

            puts $f "\$(${file_t}${i}:${ext}=${clean_t}) :"
            puts $f "\t@(prev=`pwd`; cd \$(${path_t}${i}); echo \"make clean \$(${path_t}${i})/\$(${file_t}${i})\"; \\"
            puts $f "\tmake -f \$(${file_t}${i}) clean; \\"
            puts $f "\tcd \$\$prev)"
            puts $f ""
        }
    }
}
