##############################################################################
#    pvm
#    software development environment
#
#    Copyright (C) 1997  Andrew Guryanov
#    andrew-guryanov@usa.net
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
##############################################################################

#===========================================
#	Global variables and initialization
#===========================================

set withundo 1

bind Entry <Control-x>   {tk_textCut %W}
bind Entry <Control-X>   {tk_textCut %W}
bind Entry <Control-c>   {tk_textCopy %W}
bind Entry <Control-C>   {tk_textCopy %W}
bind Entry <Control-v>   {tk_textPaste %W}
bind Entry <Control-V>   {tk_textPaste %W}

main:RegisterDocType "doc-text"
bind Text <Control-v>   {}
bind Text <Control-f>   {main:OnEditFind}
bind Text <Control-F>   {main:OnEditFind}
bind Text <Control-x>   {main:DocDispatch OnEditCut}
bind Text <Control-X>   {main:DocDispatch OnEditCut}
bind Text <Control-c>   {main:DocDispatch OnEditCopy}
bind Text <Control-C>   {main:DocDispatch OnEditCopy}
bind Text <Control-v>   {main:DocDispatch OnEditPaste}
bind Text <Control-V>   {main:DocDispatch OnEditPaste}
bind Text <Control-g>   {main:DocDispatch OnEditGoto}
bind Text <Control-G>   {main:DocDispatch OnEditGoto}
bind Text <Control-F3>  {main:DocDispatch OnEditFindSelected}
bind Text <F3>          {main:DocDispatch OnEditFindAgain}

bind Text <Delete>      {main:DocDispatch OnEditDelete}
bind Text <BackSpace>   {main:DocDispatch OnEditBackspace}
bind Text <KeyPress>    {main:DocDispatch OnKeyPress "" %K %A}
bind Text <Return>      {main:DocDispatch OnKeyPress "" %K \n}
bind Text <KeyRelease>  {+main:DocDispatch ShowPosition}



#===========================================
#	Text document
#===========================================


proc doc-text:GetHullPath {frame} {
    return $frame.hull
}

proc doc-text:MatchDocType {filename} {
#---------------------------------------------------
    return 5
}

proc doc-text:OnCreateClient {this} {
#---------------------------------
    global docdata
    set id [main:GetWindowId $this]
    set docdata($id,lastfind) ""
    undo:init $id

    set hull [doc-text:GetHullPath $this]
    frame $hull
    text $hull.text -relief sunken  -borderwidth 1 -wrap none \
        -yscrollcommand "$hull.yscroll set" \
        -xscrollcommand "$hull.xscroll set"
    scrollbar $hull.yscroll -orient vertical -relief ridge \
        -command "$hull.text yview"
    scrollbar $hull.xscroll -orient horizontal -relief ridge \
        -command "$hull.text xview"

#    pack $hull.yscroll -side right -fill y
#    pack $hull.xscroll -side bottom -fill x
#    pack $hull.text -expand 1 -fill both -side bottom

    pack $hull -fill both -side bottom -expand 1

    grid $hull.text    -row 0 -column 0 -sticky nsew
    grid $hull.yscroll -row 0 -column 1 -sticky ns
    grid $hull.xscroll -row 1 -column 0 -sticky ew
    grid columnconfigure $hull 0 -weight 1
    grid rowconfigure $hull 0 -weight 1


    bind $hull.text <ButtonRelease-1>    {+main:DocDispatch ShowPosition}
    bind $hull.text <FocusIn>     "+main:DocDispatch ShowPosition $this"
    bind $hull.text <FocusOut>    {+main:SetStatusBarText "" 1}
}

proc doc-text:OnOpenDocument {this filename} {
#--------------------------------------------
    doc:OnOpenDocument $this $filename
    set hull [doc-text:GetHullPath $this]
    set f [open $filename "r"]
    if {$f != ""} {
        while {![eof $f]} {
            $hull.text insert end [read $f 1000]
        }
        close $f
        $hull.text mark set insert 1.0
        $hull.text see insert
    }
}

proc doc-text:OnSaveDocument {this filename} {
#--------------------------------------------
    set hull [doc-text:GetHullPath $this]
    set f [open $filename "w"]
    if {$f != ""} {
        for {set istart 1.0; set iend [$hull.text index "$istart+1024c"]; set iend [$hull.text index "$iend-1c"]} \
            {$istart < $iend} \
            {set istart $iend; set iend [$hull.text index "$istart+1024c"]; set iend [$hull.text index "$iend-1c"]} {
            puts -nonewline $f [$hull.text get $istart $iend]
        }
        doc:SetModified $this false
        close $f
    }
}

proc doc-text:OnCloseDocument {this} {
#--------------------------------------
    main:SetStatusBarText "" 1
    doc:OnCloseDocument $this
}

proc doc-text:OnChangeAppearance {this} {
#-----------------------------------------
    set hull [doc-text:GetHullPath $this]
    $hull.text configure \
        -font [main:GetMainData font def] \
        -fg [main:GetMainData color textfg] \
        -bg [main:GetMainData color textbg] \
        -selectforeground [main:GetMainData color textselfg] \
        -selectbackground [main:GetMainData color textselbg] \
        -insertbackground [main:GetMainData color textfg]
}

proc doc-text:cut {this} {
    set hull [doc-text:GetHullPath $this]
    if {![catch {$hull.text get sel.first sel.last} data]} {
        doc-text:DeleteRegion $this sel.first sel.last
        clipboard clear -displayof $hull.text
        clipboard append -displayof $hull.text $data
    }
}

proc doc-text:OnEditCut {this} {
#-----------------------------------------
    set hull [doc-text:GetHullPath $this]
    doc-text:cut $this
    $hull.text see insert
    doc:SetModified $this
}

proc doc-text:OnEditCopy {this} {
#-----------------------------------------
    set hull [doc-text:GetHullPath $this]
    tk_textCopy $hull.text
}

proc doc-text:paste {this} {
    global tcl_platform
    set hull [doc-text:GetHullPath $this]
    set w $hull.text
    catch {
        if {"$tcl_platform(platform)" != "unix"} {
            if ![catch {$w get sel.first sel.last}] {
                doc-text:DeleteRegion $this sel.first sel.last
                $w delete sel.first sel.last }] {
            }
        }
        doc-text:PutText $this [selection get -displayof $w -selection CLIPBOARD]
    }
}

proc doc-text:OnEditPaste {this} {
#-----------------------------------------
    set hull [doc-text:GetHullPath $this]
    doc-text:paste $this
    $hull.text see insert
    doc:SetModified $this
}

proc doc-text:OnEditDelete {this} {
#-----------------------------------------
    set hull [doc-text:GetHullPath $this]
    if {[$hull.text tag nextrange sel 1.0 end] != ""} {
        doc-text:DeleteRegion $this sel.first sel.last
        $hull.text see insert
    } else {
        doc-text:DeleteRegion $this insert {insert+1c}
        $hull.text see insert
    }
    doc:SetModified $this
}

proc doc-text:OnEditBackspace {this} {
#-----------------------------------------
    set hull [doc-text:GetHullPath $this]
    if {[$hull.text tag nextrange sel 1.0 end] != ""} {
        doc-text:DeleteRegion $this sel.first sel.last
        $hull.text see insert
    } elseif [$hull.text compare insert != 1.0] {
        doc-text:DeleteRegion $this {insert-1c} insert
        $hull.text see insert
    }
    doc:SetModified $this
}

proc doc-text:OnEditSelectAll {this} {
#-----------------------------------------
    set hull [doc-text:GetHullPath $this]
    set iend [$hull.text index end-1c]
    $hull.text tag remove 1.0 $iend
    $hull.text tag add sel 1.0 $iend
    $hull.text mark set anchor insert
}

proc doc-text:OnEditGoto {this} {
#-----------------------------------------
    set hull [doc-text:GetHullPath $this]
    set dlg [dialog-entry:CreateDialog $this "Go to" "Line number"]
    set res [main:DlgDispatch DoModal $dlg]
    if {$res != ""} {
        $hull.text tag remove sel 0.0 end
        $hull.text mark set insert ${res}.0
        $hull.text mark set anchor insert
        $hull.text see insert
    }
}

proc doc-text:OnEditFindSelected {this} {
#-----------------------------------------
    set hull [doc-text:GetHullPath $this]
    if {[$hull.text tag nextrange sel 1.0 end] != ""} {
        set find [$hull.text get sel.first sel.last]
        if {![doc-text:OnFindText $this $find]} {
            bell
        }
    }
}

proc doc-text:OnEditFindAgain {this} {
#-----------------------------------------
    global docdata
    set id [main:GetWindowId $this]
    if {$docdata($id,lastfind) != ""} {
        if {![doc-text:OnFindText $this $docdata($id,lastfind)]} {
            bell
        }
    }
}

proc doc-text:OnKeyPress {this key ascii} {
#--------------------------------------------
    if {$ascii != "" && $key != "Escape" && $key != "Delete"} {
        doc-text:PutText $this $ascii
        doc:SetModified $this
    }
    doc-text:ShowPosition $this
}

proc doc-text:PutText {this text} {
#-----------------------------------------
    global withundo

    set hull [doc-text:GetHullPath $this]
    set i1 [$hull.text index insert]
    $hull.text insert insert $text
    $hull.text see insert
    set i2 [$hull.text index insert]
    if {$withundo} {
        undo:append $this 0 $i1 $i2 $text
    }
}

proc doc-text:ShowPosition {this} {
#-----------------------------------------
    set hull [doc-text:GetHullPath $this]
    set pos [$hull.text index insert]
    scan $pos %d.%d line col
    incr col
    main:SetStatusBarText [format "%d, %d" $line $col] 1
}

proc doc-text:OnFindText {this find {direction next} {case 0} {circle 1} {regexp 0}} {
#------------------------------------------------------------------------------------
    global docdata
    set hull [doc-text:GetHullPath $this]

    if {$regexp} {
        set mode "-regexp"
    } else {
        set mode "-exact"
    }
    if {$direction == "next"} {
        set dir "-forwards"
        set pos [$hull.text index insert+1c]
        set endpos end
    } else {
        set dir "-backwards"
        set pos [$hull.text index insert-1c]
        set endpos 1.0
    }
    if {$case} {
        if {$circle} {
            set first [$hull.text search $dir $mode -count cnt -- "$find" $pos]
        } else {
            set first [$hull.text search $dir $mode -count cnt -- "$find" $pos $endpos]
        }
    } else {
        if {$circle} {
            set first [$hull.text search $dir $mode -nocase -count cnt -- "$find" $pos]
        } else {
            set first [$hull.text search $dir $mode -nocase -count cnt -- "$find" $pos $endpos]
        }
    }
    if {$first != ""} {
        set last "${first}+${cnt}c"
        $hull.text tag remove sel 0.0 $first
        $hull.text tag add sel $first $last
        $hull.text tag remove sel $last end
        $hull.text mark set insert $first
        $hull.text mark set anchor $last
        $hull.text see insert
        set id [main:GetWindowId $this]
        set docdata($id,lastfind) "$find"
        return 1
    }
    return 0
}

proc doc-text:OnReplaceText {this replace} {
#----------------------------------------------------------
    main:DocDispatch OnEditDelete $this
    set hull [doc-text:GetHullPath $this]
    set first [$hull.text index insert]
    set last  "${first}+[string length $replace]c"
    $hull.text insert insert "$replace"

    $hull.text tag remove sel 0.0 $first
    $hull.text tag add sel $first $last
    $hull.text tag remove sel $last end
    $hull.text mark set insert $first
    $hull.text see insert
}

proc doc-text:SetCursor {this pos} {
#----------------------------------------------------------
    set hull [doc-text:GetHullPath $this]
    $hull.text mark set insert $pos
    $hull.text see insert
}

proc doc-text:DeleteRegion {this pos1 pos2} {
    global withundo

    doc:SetModified $this
    set hull [doc-text:GetHullPath $this]
    set ch [$hull.text get $pos1 $pos2]
    set p1 [$hull.text index $pos1]
    set p2 [$hull.text index $pos2]
    if {$withundo} {
        undo:append $this 1 $p1 $p2 $ch
    }
    $hull.text delete $pos1 $pos2
}

proc doc-text:OnEditUndo {this} {
    undo:do $this
}

proc doc-text:OnEditRedo {this} {
    undo:redo $this
}

proc undo:set {} {
    global undo docdata

    set id [main:GetWindowId [mdiclient:GetActiveFrame]]
    set undo($id,$docdata($id,uncount)) [list $docdata($id,undel) $docdata($id,unbegin) $docdata($id,unend) $docdata($id,untext)]
}

proc undo:append {this del begin end text} {
    global undo docdata

    set id [main:GetWindowId $this]

    if {$docdata($id,undel) != $del} {
        undo:new $id $del $begin $end $text
    } else {
        if {$docdata($id,undel)} {
            if {$docdata($id,unbegin) == $begin} {
                append docdata($id,untext) $text
                set docdata($id,unend) $end
            } elseif {$docdata($id,unbegin) == $end} {
                set docdata($id,untext) [append text $docdata($id,untext)]
                set docdata($id,unbegin) $begin
            } else {
                undo:new $id $del $begin $end $text
            }
        } else {
            if {$docdata($id,unend) == $begin} {
                append docdata($id,untext) $text
                set docdata($id,unend) $end
            } else {
                undo:new $id $del $begin $end $text
            }
        }
    }
}

proc undo:do {this} {
    global undo docdata withundo

    set id [main:GetWindowId $this]
    if {$docdata($id,undel) != -1} {
        set w [mdiclient:GetActiveFrame]
        if {$docdata($id,uncount) == $docdata($id,unmax)} {
            undo:set
        }
        set withundo 0
        if {$docdata($id,undel)} {
            doc-text:SetCursor $w $docdata($id,unbegin)
            doc-text:PutText $w $docdata($id,untext)
        } else {
            doc-text:DeleteRegion $w $docdata($id,unbegin) $docdata($id,unend)
        }
        doc-text:SetCursor $w $docdata($id,unbegin)
        set withundo 1
        incr docdata($id,uncount) -1
        if {$docdata($id,uncount) > 0} {
            set u $undo($id,$docdata($id,uncount))
            set docdata($id,undel)   [lindex $u 0]
            set docdata($id,unbegin) [lindex $u 1]
            set docdata($id,unend)   [lindex $u 2]
            set docdata($id,untext)  [lindex $u 3]
        } else {
            set docdata($id,undel) -1
            doc:SetModified $w false
        }
    } else {
        main:SetStatusBarText "There is nothing to undo!"
    }
}

proc undo:init {id} {
    global undo docdata

    set undo($id,0)         {}
    set docdata($id,uncount) 0
    set docdata($id,unbegin) 1.0
    set docdata($id,unend)   1.0
    set docdata($id,untext)  {}
    set docdata($id,undel)   -1
    set docdata($id,unmax)   0
}

proc undo:new {this del begin end text} {
    global undo docdata

    undo:set
    set id $this

    set docdata($id,unmax)   [incr docdata($id,uncount)]
    set docdata($id,undel)   $del
    set docdata($id,unbegin) $begin
    set docdata($id,unend)   $end
    set docdata($id,untext)  $text
}

proc undo:redo {this} {
    global undo docdata withundo

    set id [main:GetWindowId $this]
    if {$docdata($id,uncount) < $docdata($id,unmax)} {
        set u $undo($id,[incr docdata($id,uncount)])
        set docdata($id,undel)   [lindex $u 0]
        set docdata($id,unbegin) [lindex $u 1]
        set docdata($id,unend)   [lindex $u 2]
        set docdata($id,untext)  [lindex $u 3]
        set withundo 0
        if {$docdata($id,undel)} {
            doc-text:DeleteRegion $this $docdata($id,unbegin) "$docdata($id,unbegin)+[string length $docdata($id,untext)]c"
            doc-text:SetCursor $this $docdata($id,unbegin)
        } else {
            doc-text:SetCursor $this $docdata($id,unbegin)
            doc-text:PutText $this $docdata($id,untext)
        }
        set withundo 1
    } else {
        main:SetStatusBarText "There is nothing to redo!"
    }
}
