<?php
// vim: foldmethod=marker
/**
 *  Ethna_DB_PgSQL.php
 *
 *  @author     Yoshinari Takaoka <takaoka@beatcraft.com>
 *  @license    http://www.opensource.org/licenses/bsd-license.php The BSD License
 *  @package    Ethna
 *  @version    $Id: 35a17a036c151dfdc3e0792e146503c605631399 $
 */

// {{{ Ethna_DB_PgSQL
/**
 *  PostgreSQL用、データベースアクセスドライバ
 *
 *  @author     Yoshinari Takaoka <takaoka@beatcraft.com>
 *  @access     public
 *  @package    Ethna
 */
class Ethna_DB_PgSQL extends Ethna_DB
{
    /**
     *  コンストラクタ
     *
     *  @access public
     *  @param  string  $dsn DSN
     *  @param  bool    $persistent 持続接続設定
     */
    function Ethna_DB_PgSQL($dsn, $persistent)
    {
        parent::Ethna_DB($dsn, $persistent);
    }

    /**
     *  DBに接続する
     *
     *  @access public
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function connect()
    {
        $con_spec = array(
            'host' => $this->_dsninfo['hostspec'],
            'port' => $this->_dsninfo['port'],
            'dbname' => $this->_dsninfo['database'],
            'user' => $this->_dsninfo['username'],
            'password' => $this->_dsninfo['password'],
        );
        $con_str = '';
        foreach ($con_spec as $key => $val) {
            if (empty($val)) {
                $val = "''";
                continue;
            }
            $val = preg_replace("#'#", "\\'", $val);
            if (strpos($val, ' ') !== false) {
                $val = "'$val'";
            }
            $con_str .= " $key=$val"; 
        }

        //  finnally connect
        $con_str = ltrim($con_str);
        if ($this->_persistent) {
            $this->_db = pg_pconnect($con_str);
        } else {
            $this->_db = pg_connect($con_str);
        }
          
        if ($this->_db === false) {
            return Ethna::raiseError(
                       "could not connect database: $con_str"
                   );
        }
        return 0;
    }

    /**
     *  DB接続を切断する
     *
     *  @access public
     */
    function disconnect()
    {
        if ($this->_persistent) {
            return 0;  //  持続的接続は閉じない
        } 
        $result = pg_close($this->_db);
        if ($result === false) {
            return Ethna::raiseError(
                "could not close Connection!: "
              . $this->_dsn
            );
        }
        $this->_db = NULL;
        return 0;
    }

    /**
     *  DBトランザクションを開始する
     *
     *  @access public
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function begin()
    {
        return $this->__transact_query('begin');
    }

    /**
     *  DBトランザクションを中断する
     *
     *  @access public
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function rollback()
    {
        return $this->__transact_query('rollback');
    }

    /**
     *  DBトランザクションを終了する
     *
     *  @access public
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function commit()
    {
        return $this->__transact_query('commit');
    }

    /**
     *  SQL ステートメントを実行する準備を行い、
     *  Ethna_Statementオブジェクトを返します。
     *
     *  Ethna_Statement::execute() メソッドによ
     *  って実行される SQL ステートメントを準備
     *  します。 SQL ステートメントは、文が実行
     *  されるときに実際の値に置き換えられます。
     *
     *  (ステートメントにパラメータを指定する場合、
     *   ? で指定すること)
     *  @access public
     *  @param  $sql  実行を準備するSQL
     *  @return mixed 成功時にEthna_DB_Statement
     *                Ethna_Error:エラー
     */
    function prepare($sql)
    {
        $res = NULL;
        $emulation_mode = true;
        $stmt = new Ethna_DB_PgSQL_Statement($this);
        $stmt->setLogger($this->_logger);

        if (version_compare(PHP_VERSION, '5.1.0') == 1) {
            $emulation_mode = false;
            $sql_parts = explode('?', $sql);
            $prepare_sql = '';
            if (count($sql_parts) > 1) {
                $count = 0;
                foreach ($sql_parts as $part) {
                    if (empty($part)) {
                        continue;
                    }
                    if (count($sql_parts) == ($count + 1)) {
                        $prepare_sql .= $part;
                    } else {
                        $prepare_sql .= ($part . '$'
                                       . ($count + 1) . ' '
                                        );
                    }
                    $count++;
                }
            } else {
                $prepare_sql = $sql;
            }
            $res = pg_prepare($this->_db, '', $prepare_sql);
            if ($res === false) {
                return Ethna::raiseError(
                    "could not prepare your sql: "
                  . $this->_dsn
                  . " query: $prepare_sql"
                  . " detail: " . pg_last_error($this->_db),
                    E_DB_QUERY 
                );
            }
            $stmt->setOption('__sql', trim($prepare_sql));
        } else {
            $stmt->setOption('__sql', trim($sql));
        }

        $stmt->setOption('__emulation_mode', $emulation_mode);
        return $stmt; 
    }

    /**
     *  SQL ステートメントを実行し、作用した行数を返します。
     *  SELECT 文からは結果を返しません。SELECT 文を指定した
     *  場合は常に0が返ります。
     *
     *  @access public
     *  @param  $sql    実行するSQL
     *  @return mixed   0以上の数:作用した行数 Ethna_Error:エラー
     */
    function exec($sql)
    {
        $sql = ltrim($sql);
        if (preg_match('/^SELECT/i', $sql)) {
            return 0;
        }
        $this->_logger->log(LOG_DEBUG, $sql);
        $result = pg_query($this->_db, $sql);
        if ($result === false) {
             return Ethna::raiseError(
                "could not exec your sql: "
              . $this->_dsn
              . " query: $sql",
                E_DB_QUERY 
            );
        }
        return pg_affected_rows($result); 
    }

    /**
     *  直前のINSERTによるIDを取得する
     *  PostgreSQL では、$name パラメーターにシ
     *  ーケンス名を指定する必要があります。
     *
     *  @access public
     *  @param  string  $name データベース依存のパラメータ
     *  @return mixed   int:直近のINSERTにより生成されたID null:エラー
     */
    function getInsertId($name = NULL)
    {
        if (empty($name)) {
            return NULL;
        }

        //   シーケンス名と見なし、検索する
        //   結果取得を高速にするため、Ethna_DB_Statement
        //   ではラップしない
        $query = "SELECT last_value FROM $name";
        $this->_logger->log(LOG_DEBUG, $query);
        $result = pg_query($this->_db, $query);
        if ($result === false) {
            Ethna::raiseWarning(
                "could not get last value from sequence: "
              . $this->_dsn
              . " query: $query",
                E_DB_QUERY 
            );
            return NULL;
        }

        $row = pg_fetch_row($result, 0);
        if ($row == false) {
            Ethna::raiseWarning(
                "could not fetch row",
                E_DB_GENERAL 
            );
            return NULL;
        }
        $id = (isset($row[0])) ? $row[0] : NULL;
        pg_free_result($result);
        return $id;
    }

    /**
     *  文字列をエスケープする 
     *
     *  @access public
     *  @param  string  $value  取得対象のテーブル名
     *  @return string  エスケープ済みの値
     */
    function escape($value)
    {
        return pg_escape_string($this->_db, $value);
    }

    /**
     *  バイナリデータをデータベースに保存できる形式に
     *  エンコードします
     *
     *  @access public
     *  @param  string  $value  エスケープ対象の値 
     *  @return string  エスケープ済みの値
     */
    function blobEncode($value)
    {
        if (version_compare(PHP_VERSION, '5.2.0') === 1) {
            return pg_escape_bytea($this->_db, $value);
        }
        return pg_escape_bytea($value);
    }

    /**
     *  データベースに保存されたエスケープ済みバイナリ
     *  データを元のデータに復元します。
     *
     *  @access public
     *  @param  string  $value  復元対象の値 
     *  @return string  復元済みの値
     */
    function blobDecode($value)
    {
        return pg_unescape_bytea($value);
    }

    /**
     *  バイナリデータを指定したテーブルのカラムに保存します。
     *  BLOBデータを保存する場合は、該当カラムにNULLをINSERT
     *  してからこのメソッドを使用してください。
     *
     *  @access public
     *  @param  string  $table    保存対象のテーブル名  
     *  @param  string  $column   保存対象のカラム名  
     *  @param  string  $value    保存する値  
     *  @param  string  $where    保存するための条件  
     *  @param  string  $blobtype BLOBのタイプ(CLOB|BLOB)
     *  @return mixed   0以上の数:作用した行数 Ethna_Error:エラー
     */
    function updateBlob($table, $column, $value, $where, $blobtype='BLOB')
    {
        $query = '';
        if ($blobtype == 'CLOB') {
            $query = "UPDATE $table SET $column='"
                   . $this->escape($value) . "' WHERE $where";
        } else {
            //    BLOB 型の場合、PostgreSQL の
            //    全てのバージョンに対応させるためにはキャストが必要
            //    @see http://www.php.net/manual/en/function.pg-escape-bytea.php
            $query = "UPDATE $table SET $column='"
                   . $this->blobEncode($value) . "'::bytea WHERE $where";
        }
        return $this->exec($query);
    }

    /**
     *  テーブル定義情報を取得する
     *
     *  <code>
     *    'colname' => array(
     *                     'required' => [true|false] // NOT NULLかどうか
     *                     'type' => ...    //  データベースの型名
     *                     'unique' => [true|false] //  unique 制約があるか
     *                     'primary' => [true|false] // primary keyか否か 
     *                     'seq' => [true|false] // seq(auto increment) か否か 
     *                 )
     *  </code>
     *  @access public
     *  @param  string  $table_name テーブル名 
     *  @return mixed   array: PEAR::DBに準じたメタデータ
     *                  Ethna_Error::エラー
     */
    function getMetaData($table_name)
    {
        //   メタデータ取得SQL
        //   pg_meta_data 関数では十分な情報を取得できないので、
        //   自前でSQLを流し、ネイティブ型もこちらの型名に置き換えておく
        $sql = 'SELECT '
             . '  att.attname AS colname, '
             . "  CASE att.attnotnull WHEN 't' THEN 1 ELSE 0 END AS required, "
             . "  CASE type.typname " 
             . "    WHEN '_bpchar' THEN 'string' WHEN '_varchar' THEN 'string' "
             . "    WHEN '_date' THEN 'date' WHEN '_float8' THEN 'float' WHEN '_int4' "
             . "    THEN 'integer' WHEN '_interval' THEN 'interval' WHEN '_numeric' THEN "
             . "    'decimal' WHEN '_float4' THEN 'float' WHEN '_int2' THEN 'integer' " 
             . "    WHEN '_text' THEN 'text' WHEN '_time' THEN 'time' WHEN '_timestamp' "
             . "    THEN 'timestamp' WHEN '_bool' THEN 'boolean' WHEN '_bytea' THEN 'binary' "
             . "  END AS type, "
             . "  CASE con_u.contype WHEN 'u' THEN 1 ELSE 0 END AS unique, "
             . "  CASE con_p.contype WHEN 'p' THEN 1 ELSE 0 END AS primary, "
             . "  CASE substr(attdef.adsrc, 0, 8) WHEN 'nextval' THEN 1 ELSE 0 END AS seq "
             . 'FROM pg_stat_user_tables stat '
             . '  INNER JOIN pg_attribute att ON att.attrelid = stat.relid '
             . '  INNER JOIN pg_type type ON att.atttypid = type.typelem '
             . '  INNER JOIN pg_class class ON class.relname = stat.relname '
             . '  LEFT JOIN pg_attrdef attdef ON att.attrelid = attdef.adrelid '
             . '       AND att.attnum = attdef.adnum '
             . '  LEFT JOIN pg_constraint con_u ON con_u.conkey[1] = att.attnum '
             . "       AND con_u.contype = 'u' AND con_u.conrelid = class.relfilenode "
             . '  LEFT JOIN pg_constraint con_p ON att.attnum = ANY (con_p.conkey) '
             . "       AND con_p.contype = 'p' AND con_p.conrelid = class.relfilenode "
             . "WHERE stat.schemaname = 'public' "
             . '  AND att.attnum > 0 '
             . "  AND substr(type.typname,1,1) = '_' "
             . '  AND stat.relname = ?';
        $result = $this->query($sql, array($table_name));
        if (Ethna::isError($result)) {
            return $result;
        }

        $meta = $result->fetchAll();
        $result = array();
        if (!empty($meta)) {
            foreach ($meta as $row) {
                $colname = $row['colname'];
                unset($row['colname']);
                $result[$colname] = $row;
            }
        }
        return $result;
    }

    /**
     *  データベースのタイプを取得する
     *
     *  @access public
     *  @return string  データベースのタイプ
     */
    function getType()
    {
        return 'pgsql';
    }

    /**
     *  DBトランザクション関連のクエリを送信
     *  する内部関数です。
     *
     *  @access private 
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function __transact_query($query)
    {
        if (!$this->isValid()) {
             return Ethna::raiseError(
                "Not Connected: "
              . $this->_dsn
            );
        }
        $this->_logger->log(LOG_DEBUG, $query);
        $result = pg_query($this->_db, $query);
        if ($result === false) {
            $error = pg_last_error($this->_db);
              return Ethna::raiseError(
                "$query transation failed: "
              . $this->_dsn
              . " $error",
                E_DB_QUERY 
            );
        }
        return 0; 
    }
}
// }}}

// {{{ Ethna_DB_PgSQL_Statement
/**
 *  PostgreSQL ドライバの一部
 *
 *  実行前にはプリペアドステートメント。
 *  実行後には関連する結果セットを表す
 *
 *  @author     Yoshinari Takaoka <takaoka@beatcraft.com>
 *  @access     public
 *  @package    Ethna
 */
class Ethna_DB_PgSQL_Statement extends Ethna_DB_Statement
{
    /**
     *  コンストラクタ
     *
     *  @access public
     *  @param  Ethna_DB $db データベース接続 
     */
    function Ethna_DB_PgSQL_Statement(&$db)
    {
        parent::Ethna_DB_Statement($db);
    } 

    /**
     *  exec メソッドの実際の実行ロジックです。
     *  プリペアドステートメントが有効であった場合のみ実行されます。 
     *  
     *  @access protected
     *  @param  string $sql 実行するSQL (パラメータは ? で指定)
     *  @param  mixed  $param  パラメータの配列
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function __exec($sql, $params = array())
    {
        //
        //  $sql param is ignored.
        //  we  use prepare'ed sql query.
        //

        $prepared_query = $this->_options['__sql'];
        $this->_result = pg_query_params($this->_db, $prepared_query, $params);
        $this->_logger->log(LOG_DEBUG, $prepared_query);
        if ($this->_result === false) {
            return Ethna::raiseError(
                "could not exec sql : $prepared_query",
                E_DB_QUERY 
            );
        }
        return 0;
    }
    
    /**
     *  プリペアドステートメント実行結果から
     *  次の行を取得します。 
     *
     *  @access public
     *  @return mixed   結果がある場合は配列。ない場合はfalse 
     */
    function fetchRow()
    {
        if (empty($this->_result)
         || is_resource($this->_result) == false) {
            return false;
        }
        switch ($this->_fetch_mode) {
        case DB_FETCHMODE_NUM:
            return pg_fetch_row($this->_result);
        case DB_FETCHMODE_ASSOC:
            return pg_fetch_assoc($this->_result);
        default:
            Ethna::raiseWarning(
               'Unknown fetch mode: ' . $this->fetch_mode,
               E_DB_GENERAL
            );
            return false;
        } 
    }
     
    /**
     *  プリペアドステートメント実行結果の
     *  結果セットを「全て」配列で返します
     *
     *  @access public
     *  @return mixed   結果がある場合は配列。ない場合はfalse
     */
    function fetchAll()
    {
        if (empty($this->_result)
         || is_resource($this->_result) == false) {
            return false;
        }
        $allresult = array();
        switch ($this->_fetch_mode) {
        case DB_FETCHMODE_NUM:
            while ($row = pg_fetch_row($this->_result)) {
                array_push($allresult, $row);
            }
            break;
        case DB_FETCHMODE_ASSOC:
            $allresult = pg_fetch_all($this->_result);
            break;
        default:
            Ethna::raiseWarning(
               'Unknown fetch mode: ' . $this->fetch_mode,
               E_DB_GENERAL
            );
            return false;
        } 
        return $allresult;
    }

    /**
     *  直近の DELETE, INSERT, UPDATE 文によっ
     *  て作用した行数を返します。 
     *
     *  @access public
     *  @return int  作用した行数
     *               SELECT の場合は 0
     */
    function affectedRows()
    {
        if (empty($this->_result)
         || is_resource($this->_result) == false) {
            return 0;
        }
        return pg_affected_rows($this->_result);
    }

    /**
     *  プリペアドステートメントのエミュレー
     *  ションロジックです。
     *
     *  @access protected
     *  @param  string $sql 実行するSQL (パラメータは ? として指定)
     *  @param  mixed  $param  パラメータの配列
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function __emulatePstmt($sql, $param = array())
    {
        $prepare_sql = $this->__getEmulatePstmtSQL($sql, $param);
        $result = pg_query($this->_db, $prepare_sql);
        if ($result === false) {
            return Ethna::raiseError(
                "could not exec your sql: "
              . " query: $prepare_sql",
                E_DB_QUERY 
            );
        }
        $this->_result = $result;
        return 0;                
    }
}
// }}}

?>
