﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Drawing;
using System.Windows.Forms;
using System.Xml;
using System.IO;

namespace FeedGenerator.lib
{
    public class Config
    {
        public Font contentFont;
        public bool isIdGenerate;
        public bool isUrlGenerate;
        public bool isUpdateGenerate;
        public bool isInsertBrTag;

        /// <summary>
        /// 各種初期化を行います。必要に応じてファイルから設定を読み込みます。
        /// </summary>
        public Config()
        {
            isIdGenerate = true;
            isUrlGenerate = true;
            isUpdateGenerate = true;
            isInsertBrTag = false;
            contentFont = new Font("メイリオ", 9.75f);

            try
            {
                // 設定ファイルを読み込む
                readConfigFromFile();
            }
            catch (FileNotFoundException)
            {
                // 設定ファイルが存在しない(＝初回起動)の場合は規定設定でファイルを生成する
                writeConfigFormFile();
            }
            catch (Exception)
            {
                // エラー時はディフォルト設定を適用
                MessageBox.Show("設定ファイルが異常値を示しました。ディフォルトの設定を適用します。", "ERROR", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /// <summary>
        /// 設定ファイルから設定を読み込みます
        /// </summary>
        public void readConfigFromFile()
        {
            XmlTextReader reader = new XmlTextReader(Application.StartupPath + "\\config.xml");

            string fontFamily = "";
            float emSize = 0.0f;

            while (reader.Read())
            {
                if (reader.NodeType == XmlNodeType.Element)
                {
                    switch (reader.LocalName)
                    {
                        case "fontfamily":
                            fontFamily = reader.ReadString();
                            break;

                        case "emsize":
                            emSize = float.Parse(reader.ReadString());
                            break;

                        case "isidgenerate":
                            isIdGenerate = bool.Parse(reader.ReadString());
                            break;

                        case "isurlgenerate":
                            isUrlGenerate = bool.Parse(reader.ReadString());
                            break;

                        case "isupdategenerate":
                            isUpdateGenerate = bool.Parse(reader.ReadString());
                            break;

                        case "isinsertbrtag":
                            isInsertBrTag = bool.Parse(reader.ReadString());
                            break;
                    }
                }
            }

            contentFont = new Font(fontFamily, emSize);
        }

        /// <summary>
        /// 設定ファイルに設定を書き込みます
        /// </summary>
        public void writeConfigFormFile()
        {
            StringBuilder buffer = new StringBuilder();
            buffer.AppendLine("<?xml version=\"1.0\" encoding=\"utf-8\"?>");
            buffer.AppendLine("<config>");
            buffer.AppendLine("<font>");
            buffer.AppendLine("\t<fontfamily>" + contentFont.FontFamily.Name + "</fontfamily>");
            buffer.AppendLine("\t<emsize>" + contentFont.Size + "</emsize>");
            buffer.AppendLine("</font>");
            buffer.AppendLine("<feedconfig>");
            buffer.AppendLine("\t<isidgenerate>" + isIdGenerate.ToString() + "</isidgenerate>");
            buffer.AppendLine("\t<isurlgenerate>" + isUrlGenerate.ToString() + "</isurlgenerate>");
            buffer.AppendLine("\t<isupdategenerate>" + isUpdateGenerate.ToString() + "</isupdategenerate>");
            buffer.AppendLine("\t<isinsertbrtag>" + isInsertBrTag.ToString() + "</isinsertbrtag>");
            buffer.AppendLine("</feedconfig>");
            buffer.AppendLine("</config>");

            try
            {
                StreamWriter writer = new StreamWriter(Application.StartupPath + "\\config.xml", false, Encoding.UTF8);
                writer.Write(buffer.ToString());
                writer.Close();
            }
            catch (Exception)
            {
                MessageBox.Show("ファイルの書き込み中にエラーが発生しました。", "ERROR", MessageBoxButtons.OK, MessageBoxIcon.Error); 
            }
        }
    }
}
