﻿using System;
using System.Windows.Forms;
using System.Xml;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;

namespace FeedGenerator.lib
{
    class FeedInfo
    {
        private Dictionary<string, string> feedattr;
        private string self;
        private string title;
        private string subtitle;
        private string link;
        private DateTime updated;
        private string id;
        private string rights;
        private string authorname;
        private string authoremail;
        private List<string> others;
        private List<string> error;

        // プロパティ
        public Dictionary<string, string> Feedattr
        {
            set { this.feedattr = value; }
            get { return this.feedattr; }
        }
        public string Self
        {
            set { this.self = value; }
            get { return this.self; }
        }
        public string Title
        {
            set { this.title = value; }
            get { return this.title; }
        }
        public string Subtitle
        {
            set { this.subtitle = value; }
            get { return this.subtitle; }
        }
        public string Link
        {
            set { this.link = value; }
            get { return this.link; }
        }
        public DateTime Updated
        {
            set { this.updated = value; }
            get { return this.updated; }
        }
        public string Id
        {
            set { this.id = value; }
            get { return this.id; }
        }
        public string Rights
        {
            set { this.rights = value; }
            get { return this.rights; }
        }
        public string Authorname
        {
            set { this.authorname = value; }
            get { return this.authorname; }
        }
        public string Authoremail
        {
            set { this.authoremail = value; }
            get { return this.authoremail; }
        }
        public List<string> Others
        {
            set { this.others = value; }
            get { return this.others; }
        }
        public List<string> Error
        {
            set { this.error = value; }
            get { return this.error; }
        }

        public FeedInfo()
        {
            feedattr = new Dictionary<string, string>();
            others = new List<string>();
            error = new List<string>();
        }

        /// <summary>
        /// XMLを読み込んだ後に解析して、オブジェクトに格納するメソッド
        /// </summary>
        /// <param name="filePath">XMLのファイルフルパス</param>
        public void readXml(string filePath)
        {
            // 実際にXMLを読み込んで解析する
            XmlTextReader reader = new XmlTextReader(filePath);
            // 頭が悪いReadOuterXMLのための制御用変数
            bool isActiveReadOuterXmlFlag = false;

            while (isActiveReadOuterXmlFlag || reader.Read())
            {
                isActiveReadOuterXmlFlag = false;

                if (reader.NodeType == XmlNodeType.Element)
                {
                    // <entry>要素以降は日記ファイルなのでループを抜ける
                    if (reader.LocalName == "entry")
                        break;

                    try
                    {
                        switch (reader.LocalName)
                        {
                            case "feed":
                                if (reader.HasAttributes)
                                {
                                    while (reader.MoveToNextAttribute())
                                    {
                                        feedattr[reader.Name] = reader.Value;
                                    }
                                }
                                break;

                            case "title":
                                title = reader.ReadString();
                                break;

                            case "subtitle":
                                subtitle = reader.ReadString();
                                break;

                            case "link":
                                if (reader.HasAttributes)
                                {
                                    reader.MoveToAttribute("rel");
                                    string rel = reader.Value;
                                    reader.MoveToAttribute("href");
                                    if (rel == "alternate")
                                        link = reader.Value;
                                    else if (rel == "self")
                                        self = reader.Value;
                                }
                                break;

                            case "updated":
                                updated = RFC3339.getDateTime(reader.ReadString());
                                break;

                            case "id":
                                id = reader.ReadString();
                                break;

                            case "rights":
                                rights = reader.ReadString();
                                break;

                            case "name":
                                authorname = reader.ReadString();
                                break;

                            case "email":
                                authoremail = reader.ReadString();
                                break;

                            default:
                                if (reader.LocalName == string.Empty || reader.LocalName == "feed" || reader.LocalName == "author")
                                    break;

                                // 未知の定義要素を読み取る
                                isActiveReadOuterXmlFlag = true;
                                // 自動挿入される不要な属性を削除
                                others.Add(reader.ReadOuterXml().Replace(" xmlns=\"http://www.w3.org/2005/Atom\"", ""));

                                break;
                        }
                    }
                    catch (Exception)
                    {
                        // XML形式エラー出力用
                        error.Add("WARNING : LINE " + reader.LineNumber.ToString());
                    }
                }
            }
            // リーダーをクローズする
            reader.Close();
        }

        /// <summary>
        /// オーバーライドされたメソッドです。Feed形式のヘッダを返却します。
        /// </summary>
        /// <returns></returns>
        public override string ToString()
        {
            StringBuilder buffer = new StringBuilder();
            buffer.Append("<feed ");
            foreach (string key in feedattr.Keys)
            {
                buffer.Append(key + "=\"" + feedattr[key] + "\" ");
            }
            buffer.AppendLine(">");
            buffer.AppendLine("<title type=\"text\">" + this.title + "</title>");
            buffer.AppendLine("<subtitle type=\"text\">" + this.subtitle + "</subtitle>");
            buffer.AppendLine("<link rel=\"self\" type=\"application/atom+xml\" href=\"" + this.self + "\" />");
            buffer.AppendLine("<link rel=\"alternate\" type=\"text/html\" href=\"" + this.link + "\" />");
            buffer.AppendLine("<updated>" + DateTime.Now.ToString("s") + "+09:00</updated>");
            buffer.AppendLine("<id>" + this.id + "</id>");
            buffer.AppendLine("<rights type=\"text\">" + this.rights + "</rights>");
            // 挿入されていたほかの要素を挿入
            foreach (string element in others)
            {
                buffer.AppendLine(element);
            }
            buffer.AppendLine("<author>");
            buffer.AppendLine("\t<name>" + this.authorname + "</name>");
            buffer.AppendLine("\t<email>" + this.authoremail + "</email>");
            buffer.AppendLine("</author>");

            return buffer.ToString();
        }
    }
}
