/*
 * Copyright 2009-2009 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.fess.solr;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.annotation.Resource;

import jp.sf.fess.Constants;
import jp.sf.fess.solr.response.ReplicationResponse;
import jp.sf.fess.util.FessProperties;

import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.SolrRequest.METHOD;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.client.solrj.response.SolrPingResponse;
import org.apache.solr.client.solrj.response.UpdateResponse;
import org.apache.solr.common.SolrInputDocument;
import org.apache.solr.common.params.SolrParams;
import org.seasar.framework.util.StringUtil;

public class SolrServerGroup {
    private static final int MAX_LOAD_COUNT = Integer.MAX_VALUE / 2;

    protected Map<String, SolrServer> solrServerMap = new LinkedHashMap<String, SolrServer>();

    /** a max error count */
    public int maxErrorCount = 3;

    /** the number of minimum active servers */
    public int minActiveServer = 1;

    protected String groupName;

    @Resource
    protected FessProperties solrServerProperties;

    private volatile Map<String, Integer> solrServerCountMap = new ConcurrentHashMap<String, Integer>();

    public void addServer(String name, SolrServer solrServer) {
        solrServerMap.put(name, solrServer);
    }

    public Collection<UpdateResponse> add(
            final Collection<SolrInputDocument> docs) {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<UpdateResponse>() {
            public UpdateResponse callback(SolrServer solrServer)
                    throws Exception {
                return solrServer.add(docs);
            }
        });
    }

    public Collection<UpdateResponse> add(final SolrInputDocument doc) {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<UpdateResponse>() {
            public UpdateResponse callback(SolrServer solrServer)
                    throws Exception {
                return solrServer.add(doc);
            }
        });
    }

    public Collection<UpdateResponse> addBean(final Object obj) {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<UpdateResponse>() {
            public UpdateResponse callback(SolrServer solrServer)
                    throws Exception {
                return solrServer.addBean(obj);
            }
        });
    }

    public Collection<UpdateResponse> addBean(final Collection<?> beans) {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<UpdateResponse>() {
            public UpdateResponse callback(SolrServer solrServer)
                    throws Exception {
                return solrServer.addBeans(beans);
            }
        });
    }

    public Collection<UpdateResponse> commit() {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<UpdateResponse>() {
            public UpdateResponse callback(SolrServer solrServer)
                    throws Exception {
                return solrServer.commit();
            }
        }, true);
    }

    public Collection<UpdateResponse> commit(final boolean waitFlush,
            final boolean waitSearcher) {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<UpdateResponse>() {
            public UpdateResponse callback(SolrServer solrServer)
                    throws Exception {
                return solrServer.commit(waitFlush, waitSearcher);
            }
        }, true);
    }

    public Collection<ReplicationResponse> replicate(final File snapshotDir) {
        // check this group status
        checkStatus();

        return updateQueryCallback(
                new UpdateProcessCallback<ReplicationResponse>() {
                    public ReplicationResponse callback(SolrServer solrServer)
                            throws Exception {
                        if (solrServer instanceof FessSolrServer) {
                            return ((FessSolrServer) solrServer)
                                    .replicate(snapshotDir);
                        }
                        throw new FessSolrException(solrServer.getClass()
                                .getCanonicalName()
                                + " is not supported for a replication.");
                    }
                }, true);
    }

    public Collection<UpdateResponse> deleteByQuery(final String query) {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<UpdateResponse>() {
            public UpdateResponse callback(SolrServer solrServer)
                    throws Exception {
                return solrServer.deleteByQuery(query);
            }
        });
    }

    public Collection<UpdateResponse> deleteById(final String id) {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<UpdateResponse>() {
            public UpdateResponse callback(SolrServer solrServer)
                    throws Exception {
                return solrServer.deleteById(id);
            }
        });
    }

    public Collection<UpdateResponse> optimize() {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<UpdateResponse>() {
            public UpdateResponse callback(SolrServer solrServer)
                    throws Exception {
                return solrServer.optimize();
            }
        });
    }

    public Collection<UpdateResponse> optimize(final boolean waitFlush,
            final boolean waitSearcher) {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<UpdateResponse>() {
            public UpdateResponse callback(SolrServer solrServer)
                    throws Exception {
                return solrServer.optimize(waitFlush, waitSearcher);
            }
        });
    }

    public Collection<UpdateResponse> optimize(final boolean waitFlush,
            final boolean waitSearcher, final int maxSegments) {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<UpdateResponse>() {
            public UpdateResponse callback(SolrServer solrServer)
                    throws Exception {
                return solrServer
                        .optimize(waitFlush, waitSearcher, maxSegments);
            }
        });
    }

    public Collection<SolrPingResponse> ping() {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<SolrPingResponse>() {
            public SolrPingResponse callback(SolrServer solrServer)
                    throws Exception {
                return solrServer.ping();
            }
        });
    }

    public QueryResponse query(SolrParams params) {
        // check this group status
        checkStatus();

        try {
            return getActiveSolrServer().query(params);
        } catch (SolrServerException e) {
            throw new FessSolrException("An exception occurs at SolrServer.", e);
        }
    }

    public QueryResponse query(SolrParams params, METHOD method) {
        // check this group status
        checkStatus();

        try {
            return getActiveSolrServer().query(params, method);
        } catch (SolrServerException e) {
            throw new FessSolrException("An exception occurs at SolrServer.", e);
        }
    }

    protected SolrServer getActiveSolrServer() {
        String name = null;
        SolrServer solrServer = null;
        int minValue = Integer.MAX_VALUE;
        for (Map.Entry<String, SolrServer> entry : solrServerMap.entrySet()) {
            Integer count = solrServerCountMap.get(entry.getKey());
            if (count == null) {
                // set count
                count = 0;
                solrServerCountMap.put(entry.getKey(), count);
            }
            if (count < minValue || solrServer == null) {
                String key = groupName + "." + entry.getKey() + ".status";
                String status = solrServerProperties.getProperty(key);
                if (status == null || status.equals(Constants.ACTIVE)) {
                    name = entry.getKey();
                    solrServer = entry.getValue();
                }
            }
        }

        if (solrServer == null) {
            throw new FessSolrException("No active solr server.");
        }

        // update count
        if (minValue > MAX_LOAD_COUNT) {
            solrServerCountMap.clear();
        } else {
            solrServerCountMap.put(name, minValue + 1);
        }

        return solrServer;
    }

    protected <T> Collection<T> updateQueryCallback(UpdateProcessCallback<T> upc) {
        return updateQueryCallback(upc, false);
    }

    protected <T> Collection<T> updateQueryCallback(
            UpdateProcessCallback<T> upc, boolean inactiveOnError) {
        boolean isPropertyUpdate = false;
        List<T> resultList = new ArrayList<T>();
        synchronized (solrServerProperties) {
            for (Map.Entry<String, SolrServer> entry : solrServerMap.entrySet()) {
                String key = groupName + "." + entry.getKey() + ".status";
                String status = solrServerProperties.getProperty(key);
                try {
                    if (StringUtil.isBlank(status)) {
                        resultList.add(upc.callback(entry.getValue()));

                        solrServerProperties.setProperty(key, Constants.ACTIVE);
                        solrServerProperties.store();
                        isPropertyUpdate = true;
                    } else if (status.equals(Constants.ACTIVE)) {
                        resultList.add(upc.callback(entry.getValue()));
                    }
                } catch (Exception e) {
                    if (inactiveOnError) {
                        solrServerProperties.setProperty(key,
                                Constants.INACTIVE);
                        isPropertyUpdate = true;
                    } else {
                        Integer count = solrServerCountMap.get(entry.getKey());
                        if (count == null) {
                            count = 1;
                        } else {
                            count++;
                        }
                        if (count > maxErrorCount) {
                            solrServerProperties.setProperty(key,
                                    Constants.INACTIVE);
                            isPropertyUpdate = true;
                        }
                        solrServerCountMap.put(entry.getKey(), count);
                    }
                }
            }

            if (isPropertyUpdate) {
                solrServerProperties.store();
            }
        }
        return resultList;
    }

    protected void checkStatus() {
        if (!isActive()) {
            throw new FessSolrException("SolrGroup(" + groupName
                    + ") is not available.");
        }
    }

    protected boolean isActive() {
        // TODO performance

        int numOfActive = 0;
        for (String name : solrServerMap.keySet()) {
            String key = groupName + "." + name + ".status";
            String status = solrServerProperties.getProperty(key);
            if (status == null || status.equals(Constants.ACTIVE)) {
                numOfActive++;
            }
        }

        String groupStatusKey = groupName + ".status";
        String groupStatus = solrServerProperties.getProperty(groupStatusKey,
                Constants.INACTIVE);
        if (numOfActive >= minActiveServer) {
            if (groupStatus.equals(Constants.INACTIVE)) {
                synchronized (solrServerProperties) {
                    solrServerProperties.setProperty(groupStatusKey,
                            Constants.ACTIVE);
                    solrServerProperties.store();
                }
            }
            return true;
        }

        if (!groupStatus.equals(Constants.INACTIVE)) {
            synchronized (solrServerProperties) {
                solrServerProperties.setProperty(groupStatusKey,
                        Constants.INACTIVE);
                solrServerProperties.store();
            }
        }
        return false;
    }

    public String getGroupName() {
        return groupName;
    }

    public void setGroupName(String groupName) {
        this.groupName = groupName;
    }

    protected static interface UpdateProcessCallback<V> {
        public abstract V callback(SolrServer solrServer) throws Exception;
    }

    /*
    public String[] getSolrServerNames() {
        return solrServerMap.keySet().toArray(new String[solrServerMap.size()]);
    }

    public SolrServer getActiveSolrServer(String name) {
        String key = groupName + "." + name + ".status";
        String status = solrServerProperties.getProperty(key);
        if (status == null) {
            synchronized (solrServerProperties) {
                solrServerProperties.setProperty(key, Constants.ACTIVE);
                solrServerProperties.store();
            }
        } else if (status.equals(Constants.ACTIVE)) {
            return solrServerMap.get(name);
        }
        return null;
    }

    protected void setSolrServerStatus(String name, String status) {
        String key = groupName + "." + name + ".status";
        synchronized (solrServerProperties) {
            solrServerProperties.setProperty(key, status);
            solrServerProperties.store();
        }
    }
    */
}
