package jp.sf.fess.ds.impl;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import jp.sf.fess.FessSystemException;
import jp.sf.fess.ds.IndexUpdateCallback;

import org.seasar.framework.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DatabaseDataStoreImpl extends AbstractDataStoreImpl {
    private static final Logger logger = LoggerFactory
            .getLogger(DatabaseDataStoreImpl.class);

    protected String getDriverClass(Map<String, String> paramMap) {
        String driverName = paramMap.get("driver");
        if (StringUtil.isBlank(driverName)) {
            throw new FessSystemException("JDBC driver is null");
        }
        return driverName;
    }

    protected String getUsername(Map<String, String> paramMap) {
        return paramMap.get("username");
    }

    protected String getPassword(Map<String, String> paramMap) {
        return paramMap.get("password");
    }

    protected String getUrl(Map<String, String> paramMap) {
        return paramMap.get("url");
    }

    protected String getSql(Map<String, String> paramMap) {
        String sql = paramMap.get("sql");
        if (StringUtil.isBlank(sql)) {
            throw new FessSystemException("sql is null");
        }
        return sql;
    }

    protected void storeData(IndexUpdateCallback callback,
            Map<String, String> paramMap, Map<String, String> scriptMap,
            Map<String, Object> defaultDataMap) {
        Connection con = null;
        Statement stmt = null;
        ResultSet rs = null;
        try {
            Class.forName(getDriverClass(paramMap)); // TODO not needed on java6?

            String url = getUrl(paramMap);
            String username = getUsername(paramMap);
            String password = getPassword(paramMap);
            if (StringUtil.isNotEmpty(username)) {
                con = DriverManager.getConnection(url, username, password);
            } else {
                con = DriverManager.getConnection(url);
            }

            stmt = con.createStatement();
            rs = stmt.executeQuery(getSql(paramMap));
            boolean loop = true;
            while (rs.next() && loop && alive) {
                Map<String, Object> dataMap = new HashMap<String, Object>();
                dataMap.putAll(defaultDataMap);
                for (Map.Entry<String, String> entry : scriptMap.entrySet()) {
                    dataMap.put(entry.getKey(), convertValue(entry.getValue(),
                            rs, paramMap));
                }

                try {
                    loop = callback.store(dataMap);
                } catch (Exception e) {
                    logger.warn("Failed to store data: " + dataMap, e);
                }
            }
        } catch (Exception e) {
            throw new FessSystemException("Failed to crawl data in DB.", e);
        } finally {
            if (rs != null)
                try {
                    rs.close();
                } catch (SQLException e) {
                    throw new FessSystemException(e.getMessage(), e);
                } finally {
                    if (stmt != null)
                        try {
                            stmt.close();
                        } catch (SQLException e) {
                            throw new FessSystemException(e.getMessage(), e);
                        } finally {
                            if (con != null)
                                try {
                                    con.close();
                                } catch (SQLException e) {
                                    throw new FessSystemException(e
                                            .getMessage(), e);
                                }

                        }
                }

        }
    }

    protected String convertValue(String template, ResultSet rs,
            Map<String, String> paramMap) {
        return convertValue(template, new ResultSetParamMap<String, String>(rs,
                paramMap));
    }

    protected static class ResultSetParamMap<K, V> implements Map<K, V> {
        private Map<K, V> paramMap = new HashMap<K, V>();

        public ResultSetParamMap(ResultSet resultSet, Map<K, V> paramMap) {
            this.paramMap.putAll(paramMap);

            try {
                ResultSetMetaData metaData = resultSet.getMetaData();
                int columnCount = metaData.getColumnCount();
                for (int i = 0; i < columnCount; i++) {
                    try {
                        String label = metaData.getColumnLabel(i + 1);
                        String value = resultSet.getString(i + 1);
                        this.paramMap.put((K) label, (V) value);
                    } catch (SQLException e) {
                        logger.warn(
                                "Failed to parse data in a result set. The column is "
                                        + (i + 1) + ".", e);
                    }
                }
            } catch (Exception e) {
                throw new FessSystemException("Failed to access meta data.", e);
            }

        }

        @Override
        public void clear() {
            paramMap.clear();
        }

        @Override
        public boolean containsKey(Object key) {
            return paramMap.containsKey(key);
        }

        @Override
        public boolean containsValue(Object value) {
            return paramMap.containsValue(value);
        }

        @Override
        public Set<java.util.Map.Entry<K, V>> entrySet() {
            return paramMap.entrySet();
        }

        @Override
        public V get(Object key) {
            return paramMap.get(key);
        }

        @Override
        public boolean isEmpty() {
            return paramMap.isEmpty();
        }

        @Override
        public Set<K> keySet() {
            return paramMap.keySet();
        }

        @Override
        public V put(K key, V value) {
            return paramMap.put(key, value);
        }

        @Override
        public void putAll(Map<? extends K, ? extends V> m) {
            paramMap.putAll(m);
        }

        @Override
        public V remove(Object key) {
            return paramMap.remove(key);
        }

        @Override
        public int size() {
            return paramMap.size();
        }

        @Override
        public Collection<V> values() {
            return paramMap.values();
        }

    }

}
