//
// Copyright 2009-2010 Fess Project
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

#import "SearchResultModel.h"
#import "SearchResult.h"
#import <extThree20JSON/extThree20JSON.h>

@implementation SearchResultModel

@synthesize searchQuery = _searchQuery;
@synthesize results = _results;
@synthesize offset = _offset;
@synthesize pageSize = _pageSize;
@synthesize recordCount = _recordCount;
@synthesize maxPageNum = _maxPageNum;

- (id)initWithSearchQuery:(NSString*)searchQuery {
	if (self = [super init]) {
		_searchQuery = searchQuery;
		_offset = 0;
		_recordCount = 0;
		_maxPageNum = 1;
		_pageSize = DEFAULT_PAGE_SIZE;
	}

	return self;
}

- (void) dealloc {
	TT_RELEASE_SAFELY(_searchQuery);
	TT_RELEASE_SAFELY(_results);
	[super dealloc];
}

- (void)load:(TTURLRequestCachePolicy)cachePolicy more:(BOOL)more {
	if (!self.isLoading && TTIsStringWithAnyText(_searchQuery)) {
		NSUserDefaults* defaults = [NSUserDefaults standardUserDefaults];

		// page size
		NSString *pageSizeStr = [defaults stringForKey:FESS_PAGE_SIZE_KEY];
		if (pageSizeStr != nil) {
			_pageSize = [pageSizeStr intValue];
		} else {
			_pageSize = DEFAULT_PAGE_SIZE;
		}

		if (more) {
			_offset += _pageSize;
		} else {
			TT_RELEASE_SAFELY(_results);
			_results = [[NSMutableArray alloc] initWithCapacity:_pageSize];
			_offset = 0;
		}
		
		// fess url
        NSString* baseUrl = [defaults stringForKey:FESS_URL_KEY];
		if (baseUrl == nil) {
			baseUrl = DEFAULT_FESS_URL;
			[defaults setObject:baseUrl forKey:FESS_URL_KEY];
		}
		NSString *queryUrl = DEFAULT_FESS_QUERY;
		NSString *fessUrl = [NSString stringWithFormat:@"%@%@", baseUrl, queryUrl];
		
		// ecode query
		NSString *query = [CFURLCreateStringByAddingPercentEscapes(NULL, _searchQuery, NULL, CFSTR (";,/?:@&=+$#"), kCFStringEncodingUTF8) autorelease];
		
		NSString* url = [NSString stringWithFormat:fessUrl, query, _offset, _pageSize];

		TTDPRINT(@"url=%@", url);
		
		TTURLRequest* request = [TTURLRequest
								 requestWithURL: url
								 delegate: self];

		request.cachePolicy = cachePolicy;
		// request.cacheExpirationAge = TT_CACHE_EXPIRATION_AGE_NEVER;
		request.cacheExpirationAge = 10; // 10sec

		TTURLJSONResponse* response = [[TTURLJSONResponse alloc] init];
		request.response = response;
		TT_RELEASE_SAFELY(response);

		[request send];
	}
}

- (void)requestDidFinishLoad:(TTURLRequest*)request {
	TTURLJSONResponse* response = request.response;
	if (![response.rootObject isKindOfClass:[NSDictionary class]]) {
		// invalid json format
		_maxPageNum = 1;
		_recordCount = 0;
		[super requestDidFinishLoad:request];
		return;
	}

	NSDictionary* rootObj = response.rootObject;
	TTDASSERT([[rootObj objectForKey:@"response"] isKindOfClass:[NSDictionary class]]);
	NSDictionary* fessResponse = [rootObj objectForKey:@"response"];

	_maxPageNum = [[fessResponse objectForKey:@"pageCount"] doubleValue];
	_recordCount = [[fessResponse objectForKey:@"recordCount"] doubleValue];

	if (_recordCount > 0 && _offset < _recordCount) {
		TTDASSERT([[fessResponse objectForKey:@"result"] isKindOfClass:[NSArray class]]);
		NSArray* entries = [fessResponse objectForKey:@"result"];

		for (NSDictionary* entry in entries) {
			SearchResult* result = [[SearchResult alloc] init];
			result.urlLink = [entry objectForKey:@"urlLink"];
			result.contentTitle = [entry objectForKey:@"contentTitle"];
			result.contentDescription = [entry objectForKey:@"contentDescription"];

			TTDPRINT(@"url=%@", result.urlLink);
			TTDPRINT(@"title=%@", result.contentTitle);
			[_results addObject:result];
			TT_RELEASE_SAFELY(result);
		}
	}

	[super requestDidFinishLoad:request];
}

- (void)setSearchQuery:(NSString*)text {
//	if( ![_searchQuery isEqualToString:text] ) {
		NSString* copy = [text copy];
		[_searchQuery release];
		_searchQuery = copy;
		[self load:TTURLRequestCachePolicyNetwork more:NO];
//	}
}

@end

