/*
 * Copyright 2007-2008 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.seasar.chronos.core.task.handler.impl.property.read;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

import org.aopalliance.intercept.MethodInvocation;
import org.seasar.chronos.core.TaskTrigger;
import org.seasar.chronos.core.task.TaskAnnotationReader;
import org.seasar.chronos.core.task.TaskAnnotationReader.TriggerAnnotationHandler;
import org.seasar.chronos.core.task.TaskPropertyWriter;
import org.seasar.chronos.core.task.handler.impl.AbstractTaskPropertyHandler;
import org.seasar.framework.beans.BeanDesc;
import org.seasar.framework.beans.PropertyDesc;
import org.seasar.framework.beans.factory.BeanDescFactory;
import org.seasar.framework.log.Logger;
import org.seasar.framework.util.StringUtil;
import org.seasar.framework.util.tiger.ReflectionUtil;

public class TaskIsStartTaskPropertyReadHandlerImpl extends
        AbstractTaskPropertyHandler {

    @SuppressWarnings("unused")
    private static Logger log = Logger
            .getLogger(TaskIsStartTaskPropertyReadHandlerImpl.class);

    private static final String YYYY_MM_DD_HH_MM_SS = "yyyy/MM/dd HH:mm:ss";

    private TaskAnnotationReader taskAnnotationReader;

    private TaskPropertyWriter taskPropertyWriter;

    public Object execute(final MethodInvocation methodInvocation)
            throws Throwable {
        boolean start = (Boolean) methodInvocation.proceed();
        TaskTrigger taskTrigger = getTaskPropertyReader(methodInvocation)
                .getTrigger(null);
        if (taskTrigger == null) {
            taskAnnotationReader.setup(getTaskPropertyReader(methodInvocation)
                    .getTaskClass());
            taskTrigger = taskAnnotationReader
                    .getTriggerAnnotationClass(new TriggerAnnotationHandler() {
                        public TaskTrigger process(final Annotation annotation,
                                final Class<?> triggerAnnotationClass) {
                            if (triggerAnnotationClass != null) {
                                try {
                                    final TaskTrigger taskTrigger = (TaskTrigger) ReflectionUtil
                                            .newInstance(triggerAnnotationClass);
                                    build(taskTrigger, annotation,
                                            triggerAnnotationClass);
                                    return taskTrigger;
                                } catch (final Exception e) {
                                    log.warn("Failed to build taskTrigger.", e);
                                }
                            }
                            return null;
                        }

                        private void build(final TaskTrigger taskTrigger,
                                final Annotation annotaion,
                                final Class<?> triggerAnnotationClass) {
                            final BeanDesc beanDesc = BeanDescFactory
                                    .getBeanDesc(triggerAnnotationClass);
                            final Method[] mis = annotaion.annotationType()
                                    .getMethods();
                            for (final Method m : mis) {
                                final String methodName = m.getName();
                                if (beanDesc.hasPropertyDesc(methodName)) {
                                    final PropertyDesc pd = beanDesc
                                            .getPropertyDesc(methodName);
                                    final Object value = ReflectionUtil.invoke(
                                            m, annotaion, new Object[] {});
                                    convertProperty(taskTrigger, m, pd, value);
                                }
                            }

                        }

                        private void convertProperty(
                                final TaskTrigger taskTrigger, final Method m,
                                final PropertyDesc pd, final Object value) {
                            if (m.getReturnType().equals(pd.getPropertyType())) {
                                pd.setValue(taskTrigger, value);
                            } else {
                                // StringからDateに変換する処理
                                if (pd.getPropertyType().equals(Date.class)
                                        && m.getReturnType().equals(
                                                String.class)
                                        && !StringUtil.isEmpty((String) value)) {
                                    final SimpleDateFormat sdf = new SimpleDateFormat(
                                            YYYY_MM_DD_HH_MM_SS);

                                    Date date = null;
                                    try {
                                        date = sdf.parse((String) value);
                                    } catch (final ParseException e) {
                                        ;
                                    }
                                    pd.setValue(taskTrigger, date);
                                }
                            }
                        }
                    });

            // PropertyWriterに書き込むことによってキャッシュを保持できるようにする．
            taskPropertyWriter.setup(getTaskPropertyReader(methodInvocation)
                    .getTask(), getTaskPropertyReader(methodInvocation)
                    .getBeanDesc());
            taskPropertyWriter.setTrigger(taskTrigger);
        }
        if (taskTrigger != null) {
            start = start || taskTrigger.isStartTask();
        }
        return start;
    }

    public void setTaskAnnotationReader(
            final TaskAnnotationReader taskAnnotationReader) {
        this.taskAnnotationReader = taskAnnotationReader;
    }

    public void setTaskPropertyWriter(
            final TaskPropertyWriter taskPropertyWriter) {
        this.taskPropertyWriter = taskPropertyWriter;
    }
}
