/*
 * Copyright 2009-2010 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package jp.sf.fess.crud.action.admin;

import java.io.Serializable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import jp.sf.fess.crud.CommonConstants;
import jp.sf.fess.crud.CrudMessageException;
import jp.sf.fess.crud.util.SAStrutsUtil;
import jp.sf.fess.db.exentity.WebAuthentication;
import jp.sf.fess.form.admin.WebAuthenticationForm;
import jp.sf.fess.pager.WebAuthenticationPager;
import jp.sf.fess.service.WebAuthenticationService;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.seasar.framework.beans.util.Beans;
import org.seasar.framework.util.StringUtil;
import org.seasar.struts.annotation.ActionForm;
import org.seasar.struts.annotation.Execute;
import org.seasar.struts.exception.ActionMessagesException;

public class BsWebAuthenticationAction implements Serializable {

    private static final long serialVersionUID = 1L;

    private static final Log log = LogFactory
            .getLog(BsWebAuthenticationAction.class);

    // for list

    public List<WebAuthentication> webAuthenticationItems;

    // for edit/confirm/delete

    @ActionForm
    @Resource
    protected WebAuthenticationForm webAuthenticationForm;

    @Resource
    protected WebAuthenticationService webAuthenticationService;

    @Resource
    protected WebAuthenticationPager webAuthenticationPager;

    protected String displayList(boolean redirect) {
        // page navi
        webAuthenticationItems = webAuthenticationService
                .getWebAuthenticationList(webAuthenticationPager);

        // restore from pager
        Beans.copy(webAuthenticationPager, webAuthenticationForm.searchParams)
                .excludes(CommonConstants.PAGER_CONVERSION_RULE).execute();

        if (redirect) {
            return "index?redirect=true";
        } else {
            return "index.jsp";
        }
    }

    @Execute(validator = false, input = "error.jsp")
    public String index() {
        return displayList(false);
    }

    @Execute(validator = false, input = "error.jsp", urlPattern = "list/{pageNumber}")
    public String list() {
        // page navi
        if (StringUtil.isNotBlank(webAuthenticationForm.pageNumber)) {
            try {
                webAuthenticationPager.setCurrentPageNumber(Integer
                        .parseInt(webAuthenticationForm.pageNumber));
            } catch (NumberFormatException e) {
                if (log.isDebugEnabled()) {
                    log.debug("Invalid value: "
                            + webAuthenticationForm.pageNumber, e);
                }
            }
        }

        return displayList(false);
    }

    @Execute(validator = false, input = "error.jsp")
    public String search() {
        Beans.copy(webAuthenticationForm.searchParams, webAuthenticationPager)
                .excludes(CommonConstants.PAGER_CONVERSION_RULE).execute();

        return displayList(false);
    }

    @Execute(validator = false, input = "error.jsp")
    public String reset() {
        webAuthenticationPager.clear();

        return displayList(false);
    }

    @Execute(validator = false, input = "error.jsp")
    public String back() {
        return displayList(false);
    }

    @Execute(validator = false, input = "error.jsp")
    public String editagain() {
        return "edit.jsp";
    }

    @Execute(validator = false, input = "error.jsp", urlPattern = "confirmpage/{crudMode}/{id}")
    public String confirmpage() {
        if (webAuthenticationForm.crudMode != CommonConstants.CONFIRM_MODE) {
            throw new ActionMessagesException("errors.crud_invalid_mode",
                    new Object[] { CommonConstants.CONFIRM_MODE,
                            webAuthenticationForm.crudMode });
        }

        loadWebAuthentication();

        return "confirm.jsp";
    }

    @Execute(validator = false, input = "error.jsp")
    public String createpage() {
        // page navi
        webAuthenticationForm.initialize();
        webAuthenticationForm.crudMode = CommonConstants.CREATE_MODE;

        return "edit.jsp";
    }

    @Execute(validator = false, input = "error.jsp", urlPattern = "editpage/{crudMode}/{id}")
    public String editpage() {
        if (webAuthenticationForm.crudMode != CommonConstants.EDIT_MODE) {
            throw new ActionMessagesException("errors.crud_invalid_mode",
                    new Object[] { CommonConstants.EDIT_MODE,
                            webAuthenticationForm.crudMode });
        }

        loadWebAuthentication();

        return "edit.jsp";
    }

    @Execute(validator = false, input = "error.jsp")
    public String editfromconfirm() {
        webAuthenticationForm.crudMode = CommonConstants.EDIT_MODE;

        loadWebAuthentication();

        return "edit.jsp";
    }

    @Execute(validator = true, input = "edit.jsp")
    public String confirmfromcreate() {
        return "confirm.jsp";
    }

    @Execute(validator = true, input = "edit.jsp")
    public String confirmfromupdate() {
        return "confirm.jsp";
    }

    @Execute(validator = false, input = "error.jsp", urlPattern = "deletepage/{crudMode}/{id}")
    public String deletepage() {
        if (webAuthenticationForm.crudMode != CommonConstants.DELETE_MODE) {
            throw new ActionMessagesException("errors.crud_invalid_mode",
                    new Object[] { CommonConstants.DELETE_MODE,
                            webAuthenticationForm.crudMode });
        }

        loadWebAuthentication();

        return "confirm.jsp";
    }

    @Execute(validator = false, input = "error.jsp")
    public String deletefromconfirm() {
        webAuthenticationForm.crudMode = CommonConstants.DELETE_MODE;

        loadWebAuthentication();

        return "confirm.jsp";
    }

    @Execute(validator = true, input = "edit.jsp")
    public String create() {
        try {
            WebAuthentication webAuthentication = createWebAuthentication();
            webAuthenticationService.store(webAuthentication);
            SAStrutsUtil.addSessionMessage("success.crud_create_crud_table");

            return displayList(true);
        } catch (ActionMessagesException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (CrudMessageException e) {
            log.error(e.getMessage(), e);
            throw new ActionMessagesException(e.getMessageId(), e.getArgs());
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new ActionMessagesException(
                    "errors.crud_failed_to_create_crud_table");
        }
    }

    @Execute(validator = true, input = "edit.jsp")
    public String update() {
        try {
            WebAuthentication webAuthentication = createWebAuthentication();
            webAuthenticationService.store(webAuthentication);
            SAStrutsUtil.addSessionMessage("success.crud_update_crud_table");

            return displayList(true);
        } catch (ActionMessagesException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (CrudMessageException e) {
            log.error(e.getMessage(), e);
            throw new ActionMessagesException(e.getMessageId(), e.getArgs());
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new ActionMessagesException(
                    "errors.crud_failed_to_update_crud_table");
        }
    }

    @Execute(validator = false, input = "error.jsp")
    public String delete() {
        if (webAuthenticationForm.crudMode != CommonConstants.DELETE_MODE) {
            throw new ActionMessagesException("errors.crud_invalid_mode",
                    new Object[] { CommonConstants.DELETE_MODE,
                            webAuthenticationForm.crudMode });
        }

        try {
            WebAuthentication webAuthentication = webAuthenticationService
                    .getWebAuthentication(createKeyMap());
            if (webAuthentication == null) {
                // throw an exception
                throw new ActionMessagesException(
                        "errors.crud_could_not_find_crud_table",

                        new Object[] { webAuthenticationForm.id });

            }

            webAuthenticationService.delete(webAuthentication);
            SAStrutsUtil.addSessionMessage("success.crud_delete_crud_table");

            return displayList(true);
        } catch (ActionMessagesException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (CrudMessageException e) {
            log.error(e.getMessage(), e);
            throw new ActionMessagesException(e.getMessageId(), e.getArgs());
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new ActionMessagesException(
                    "errors.crud_failed_to_delete_crud_table");
        }
    }

    protected void loadWebAuthentication() {

        WebAuthentication webAuthentication = webAuthenticationService
                .getWebAuthentication(createKeyMap());
        if (webAuthentication == null) {
            // throw an exception
            throw new ActionMessagesException(
                    "errors.crud_could_not_find_crud_table",

                    new Object[] { webAuthenticationForm.id });

        }

        Beans.copy(webAuthentication, webAuthenticationForm).excludes(
                "searchParams", "mode").execute();
    }

    protected WebAuthentication createWebAuthentication() {
        WebAuthentication webAuthentication;
        if (webAuthenticationForm.crudMode == CommonConstants.EDIT_MODE) {
            webAuthentication = webAuthenticationService
                    .getWebAuthentication(createKeyMap());
            if (webAuthentication == null) {
                // throw an exception
                throw new ActionMessagesException(
                        "errors.crud_could_not_find_crud_table",

                        new Object[] { webAuthenticationForm.id });

            }
        } else {
            webAuthentication = new WebAuthentication();
        }
        Beans.copy(webAuthenticationForm, webAuthentication).excludes(
                "searchParams", "mode").execute();

        return webAuthentication;
    }

    protected Map<String, String> createKeyMap() {
        Map<String, String> keys = new HashMap<String, String>();

        keys.put("id", webAuthenticationForm.id);

        return keys;
    }
}
