/*
 * Copyright 2007-2008 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.seasar.chronos.core.impl;

import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import org.seasar.chronos.core.SchedulerConfiguration;
import org.seasar.chronos.core.SchedulerEventListener;
import org.seasar.chronos.core.TaskScheduleEntry;
import org.seasar.chronos.core.event.SchedulerEventHandler;
import org.seasar.chronos.core.exception.CancellationRuntimeException;
import org.seasar.chronos.core.exception.ExecutionRuntimeException;
import org.seasar.chronos.core.exception.InterruptedRuntimeException;
import org.seasar.chronos.core.executor.ExecutorServiceFactory;
import org.seasar.chronos.core.handler.ScheduleExecuteHandler;
import org.seasar.chronos.core.schedule.TaskScheduleEntryManager;
import org.seasar.chronos.core.schedule.TaskScheduleEntryManager.TaskScheduleEntryHanlder;
import org.seasar.framework.container.ComponentDef;
import org.seasar.framework.container.S2Container;
import org.seasar.framework.util.tiger.CollectionsUtil;

public class SchedulerImpl extends AbstractScheduler {

    public static final int SHUTDOWN_AWAIT_TIME = 10;

    public static final TimeUnit SHUTDOWN_AWAIT_TIMEUNIT = TimeUnit.MILLISECONDS;

    private static final SchedulerConfiguration defaultConfiguration = new SchedulerConfiguration();

    private final SchedulerEventHandler schedulerEventHandler = new SchedulerEventHandler();

    private final AtomicBoolean pause = new AtomicBoolean();

    private ExecutorService executorService;

    private Future<Void> schedulerTaskFuture;

    private final TaskScheduleEntryManager taskScheduleEntryManager = TaskScheduleEntryManager
            .getInstance();

    private SchedulerConfiguration configuration = defaultConfiguration;

    private final List<ScheduleExecuteHandler> scheduleExecuteHandlerList = CollectionsUtil
            .newArrayList();

    private ExecutorServiceFactory executorServiceFactory;

    private long finishStartTime = 0;

    private boolean initialized;

    public SchedulerImpl() {
        schedulerEventHandler.setScheduler(this);
    }

    public void addScheduleExecuteHandler(
            final ScheduleExecuteHandler scheduleExecuteHandler) {
        scheduleExecuteHandlerList.add(scheduleExecuteHandler);
    }

    private synchronized void initialize() {
        if (initialized) {
            return;
        }
        executorService = executorServiceFactory.create(
                configuration.getThreadPoolType(),
                configuration.getThreadPoolSize(), configuration.isDaemon());
        schedulerEventHandler.setExecutorServiceFacotry(executorServiceFactory);
        initialized = true;
    }

    /**
     * リスナーを追加します．
     * 
     * @param listener
     *            リスナー
     */
    public boolean addListener(final SchedulerEventListener listener) {
        return schedulerEventHandler.add(listener);
    }

    /**
     * タスクを追加します．
     * 
     * @param taskComponentClass
     *            タスクコンポーネント
     */
    public synchronized boolean addTask(final Class<?> taskComponentClass) {
        final S2Container rootS2Container = s2container.getRoot();
        boolean result = false;
        result = registerChildTaskComponentByTarget(rootS2Container,
                taskComponentClass);
        result = result
                || registerTaskFromS2ContainerOnSmartDeployByTarget(
                        rootS2Container, taskComponentClass);
        return result;
    }

    /**
     * {@link SchedulerConfiguration}を返します．
     * 
     * @return {@link SchedulerConfiguration}
     */
    public SchedulerConfiguration getSchedulerConfiguration() {
        return configuration;
    }

    /**
     * スケジューラの終了条件を返します．
     * 
     * @return trueなら終了，falseなら終了しない
     */
    private boolean getSchedulerFinish() {
        if (finishStartTime != 0
                && taskScheduleEntryManager.size(TaskStateType.SCHEDULED) > 0) {
            finishStartTime = 0;
            return false;
        }
        if (finishStartTime != 0
                && System.currentTimeMillis() - finishStartTime >= configuration
                        .getAutoFinishTimeLimit()) {
            finishStartTime = 0;
            return true;
        }
        if (schedulerTaskFuture != null
                && taskScheduleEntryManager.size(TaskStateType.SCHEDULED) == 0
                && taskScheduleEntryManager.size(TaskStateType.RUNNING) == 0
                && configuration.isAutoFinish() && finishStartTime == 0) {
            finishStartTime = System.currentTimeMillis();
        }
        return false;
    }

    /**
     * 一時停止状態かどうかを返します．
     * 
     * @return 一時停止中ならtrue,それ以外ならfalse
     */
    public boolean isPaused() {
        return pause.get();
    }

    /**
     * スケジューラの終了を待機します．
     */
    public void join() {
        if (configuration.isDaemon()) {
            return;
        }
        log.log("DCHRONOS0016", null);
        try {
            schedulerTaskFuture.get();
        } catch (final CancellationException e) {
            throw new CancellationRuntimeException(e);
        } catch (final ExecutionException e) {
            if (e.getCause() instanceof InterruptedException) {
                throw new InterruptedRuntimeException(e.getCause());
            }
            throw new ExecutionRuntimeException(e);
        } catch (final InterruptedException e) {
            throw new InterruptedRuntimeException(e);
        } finally {
            if (!schedulerTaskFuture.isCancelled()) {
                schedulerEventHandler.fireEndScheduler();
            }
        }
        log.log("DCHRONOS0017", null);
    }

    /**
     * スケジューラを一時停止/再開させます．
     * 
     */
    public synchronized void pause() {
        if (!pause.get()) {
            log.log("DCHRONOS0018", null);
        } else {
            log.log("DCHRONOS0021", null);
        }
        pause.set(!pause.get());
        notify();
    }

    /**
     * S2コンテナ上のコンポーネントを検索し，スケジューラに登録します．
     */
    @Override
    protected void registerTaskFromS2Container() {
        final S2Container target = s2container.getRoot();
        registerTaskFromS2ContainerOnSmartDeploy(target);
        registerChildTaskComponent(target);
        registerJavascriptTaskComponent();
    }

    /**
     * リスナーを削除します．
     * 
     * @param listener
     *            リスナー
     */
    public boolean removeListener(final SchedulerEventListener listener) {
        return schedulerEventHandler.remove(listener);
    }

    /**
     * タスクを削除します．
     * 
     * @param taskClass
     *            タスククラス
     */
    public synchronized boolean removeTask(final Class<?> taskClass) {
        final TaskScheduleEntry taskScheduleEntry = taskScheduleEntryManager
                .getTaskScheduleEntry(taskClass);
        return taskScheduleEntryManager
                .removeTaskScheduleEntry(taskScheduleEntry);
    }

    protected TaskScheduleEntry unscheduleTask(final ComponentDef componentDef) {
        final TaskScheduleEntry target = (TaskScheduleEntry) taskScheduleEntryManager
                .forEach(new TaskScheduleEntryHanlder() {
                    public Object processTaskScheduleEntry(
                            final TaskScheduleEntry scheduleEntry) {
                        if (componentDef.equals(scheduleEntry.getComponentDef())) {
                            return scheduleEntry;
                        }
                        return null;
                    }
                });
        if (taskScheduleEntryManager.removeTaskScheduleEntry(target)) {
            return target;
        }
        return null;
    }

    @Override
    protected TaskScheduleEntry scheduleTask(final ComponentDef taskComponentDef) {
        return this.scheduleTask(taskComponentDef, false);
    }

    @Override
    protected TaskScheduleEntry scheduleTask(
            final ComponentDef taskComponentDef, final boolean force) {
        final boolean contains = taskScheduleEntryManager
                .contains(taskComponentDef.getComponentClass());
        if (contains) {
            return null;
        }
        final TaskScheduleEntry taskScheduleEntry = super.scheduleTask(
                taskComponentDef, force);
        if (taskScheduleEntry == null) {
            return null;
        }
        taskScheduleEntryManager.addTaskScheduleEntry(TaskStateType.SCHEDULED,
                taskScheduleEntry);
        schedulerEventHandler.fireAddTaskScheduleEntry(TaskStateType.SCHEDULED,
                taskScheduleEntry);
        return taskScheduleEntry;
    }

    public void setSchedulerConfiguration(
            final SchedulerConfiguration schedulerConfiguration) {
        configuration = schedulerConfiguration;
    }

    /**
     * スケジューラ実行ハンドラーをセットアップします．
     * 
     * @return スケジューラ実行ハンドラーの配列
     */
    private void setupHandler() {
        for (final ScheduleExecuteHandler seh : scheduleExecuteHandlerList) {
            seh.setExecutorService(executorService);
            seh.setPause(pause);
            seh.setSchedulerEventHandler(schedulerEventHandler);
        }
    }

    public void shutdown() {
        log.log("DCHRONOS0013", null);
        taskScheduleEntryManager.forEach(TaskStateType.RUNNING,
                new TaskScheduleEntryManager.TaskScheduleEntryHanlder() {
                    public Object processTaskScheduleEntry(
                            final TaskScheduleEntry taskScheduleEntry) {
                        taskScheduleEntry.getTaskExecutorService().cancel();
                        try {
                            while (!taskScheduleEntry.getTaskExecutorService()
                                    .await(SHUTDOWN_AWAIT_TIME,
                                            SHUTDOWN_AWAIT_TIMEUNIT)) {
                                final String taskName = taskScheduleEntry
                                        .getTaskExecutorService()
                                        .getTaskPropertyReader()
                                        .getTaskName(null);
                                log.log("DCHRONOS0014",
                                        new Object[] { taskName });
                            }
                        } catch (final InterruptedException e) {
                            throw new InterruptedRuntimeException(e);
                        }
                        return null;
                    }
                });
        schedulerTaskFuture.cancel(true);
        schedulerEventHandler.fireShutdownScheduler();
        log.log("DCHRONOS0015", null);
        executorServiceFactory.shutdown();
    }

    /*
     * (非 Javadoc)
     * 
     * @see org.seasar.chronos.core.Scheduler#start()
     */
    public void start() {
        initialize();
        log.log("DCHRONOS0011", null);
        schedulerEventHandler.fireRegisterTaskBeforeScheduler();
        registerTaskFromS2Container();
        setupHandler();
        schedulerEventHandler.fireRegisterTaskAfterScheduler();

        schedulerTaskFuture = executorService.submit(new Callable<Void>() {
            public Void call() throws InterruptedException {
                do {
                    for (final ScheduleExecuteHandler seh : scheduleExecuteHandlerList) {
                        seh.handleRequest();
                    }
                    Thread.sleep(configuration.getTaskScanIntervalTime());
                } while (!SchedulerImpl.this.getSchedulerFinish());
                return null;
            }
        });
        schedulerEventHandler.fireStartScheduler();
        log.log("DCHRONOS0012", null);
    }

    public void setExecutorServiceFactory(
            final ExecutorServiceFactory executorServiceFactory) {
        this.executorServiceFactory = executorServiceFactory;
    }

    public void process() {
        start();
        join();
    }

}
