package jp.sourceforge.freegantt.swing;

import java.awt.Color;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.List;

import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;

import jp.sourceforge.freegantt.data.Project;
import jp.sourceforge.freegantt.data.Task;
import jp.sourceforge.freegantt.util.CalendarUtil;
import jp.sourceforge.freegantt.util.Pair;
import jp.sourceforge.freegantt.util.StringUtil;

public class TaskLineDataPane extends JPanel implements MouseListener,
		MouseMotionListener, KeyListener {
	private static final long serialVersionUID = 4173201298740221449L;

	Application app;
	Project project;
	TaskLinePane taskLinePane;
	CursorState cursorState;
	PrintRange printRange;
	TaskUI taskUI;
	RestrictionUI restrictionUI;
	int dragOperation = OPERATION_NONE;
	int pressOperation = OPERATION_NONE;
	Point dragStartPoint;
	Task targetTask;

	final static int OPERATION_NONE = 0x00;
	final static int OPERATION_TASK_CREATE = 0x01;
	final static int OPERATION_TASK_SLIDE = 0x02;
	final static int OPERATION_TASK_PERIOD = 0x03;
	final static int OPERATION_TASK_COMPLETION = 0x04;
	final static int OPERATION_RESTRICTION_CREATE = 0x05;
	final static int OPERATION_RESTRICTION_DELETE = 0x06;
	final static int OPERATION_PRINT_RANGE_X = 0x07;
	final static int OPERATION_PRINT_RANGE_Y = 0x08;
	final static int OPERATION_PRINT_RANGE = 0x09;

	// chart colors
	Color weekLineColor = new Color(0xD0, 0xD0, 0xD0);
	Color bgColor = Color.white;
	Color holidayBgColor = new Color(0xF0, 0xF0, 0xF0);

	// task colors
	Color borderColor = new Color(0x75, 0x51, 0xe0);
	Color fillColor = new Color(0xda, 0xce, 0xfc);
	Color completionColor = borderColor;
	Color groupColor = new Color(0x30, 0x30, 0x30);
	Color fontColor = new Color(0x30, 0x30, 0x30);
	Color activeColor = new Color(0xE0, 0x30, 0x30);
	Color milestoneColor = groupColor;
	Color restrictionColor = milestoneColor;

	public PrintRange getPrintRange() {
		return printRange;
	}

	public boolean isPrintRangeMode() {
		return printRange.printRangeMode;
	}

	public void setPrintRangeMode(boolean printRangeMode) {
		this.printRange.printRangeMode = printRangeMode;
	}

	public int getCellWidth() {
		return taskLinePane.getCellWidth();
	}

	public int getCellHeight() {
		return taskLinePane.getCellHeight();
	}

	public TaskLineDataPane(Application app, Project project,
			TaskLinePane taskLinePane) {
		super();
		this.app = app;
		this.project = project;
		this.taskLinePane = taskLinePane;
		this.cursorState = new CursorState(this);
		this.printRange = new PrintRange();
		this.taskUI = new TaskUI();
		this.restrictionUI = new RestrictionUI();
		setBackground(bgColor);
		setFocusable(true);

		setPreferredSize(new Dimension(1, project.getTaskTableModel()
				.getRowCount() * getCellHeight()));

		addMouseListener(this);
		addMouseMotionListener(this);
		addKeyListener(this);
	}
	
	public void updateWidth() {
		Calendar fromDate = app.getProject().getChartFromDate();
		Calendar toDate = app.getProject().getChartToDate();
		setSize(CalendarUtil.subDate(toDate, fromDate) * getCellWidth(), getHeight());
	}

	@Override
	protected void paintComponent(Graphics g) {
		super.paintComponent(g);

		// 背景描画
		drawBackground(g);
		// タスク描画
		drawTasks(g);
		// 制約条件描画
		restrictionUI.drawRestrictions(g);
		// 印刷範囲描画
		printRange.drawPrintRange(g);
	}

	private void drawBackground(Graphics g) {
		int drawableHeight = project.getTaskTableModel().getRowCount()
				* getCellHeight();

		g.setColor(bgColor);
		g.fillRect(0, 0, getWidth(), getHeight());

		Calendar now = (Calendar) project.getChartFromDate().clone();
		Calendar to = project.getChartToDate();

		int offset = 0;
		while (now.getTimeInMillis() < to.getTimeInMillis()) {
			if (project.isHoliday(now)) {
				g.setColor(holidayBgColor);
				g.fillRect(offset, 0, getCellWidth(), drawableHeight - 1);
			}

			now.add(Calendar.DATE, 1);
			offset += getCellWidth();

			if (now.get(Calendar.DAY_OF_WEEK) == Calendar.MONDAY) {
				g.setColor(weekLineColor);
				g.drawLine(offset - 1, 0, offset - 1, drawableHeight - 1);
			}
		}

		g.setColor(weekLineColor);
		g.drawLine(0, drawableHeight - 1, getWidth(), drawableHeight - 1);
	}

	private void drawTasks(Graphics g) {
		for (Task task : project.getTasks()) {
			drawTask(g, task);
		}
	}

	private void drawTask(Graphics g, Task task) {
		if (task == null)
			return;

		Rectangle rect = getTaskRect(task);
		if (rect == null)
			return;
		if (task.isMilestone()) {
			drawMilestone(g, task, rect);
		} else if (task.isParent()) {
			drawParentTask(g, task, rect);
		} else {
			drawBasicTask(g, task, rect);
		}
	}

	/**
	 * マイルストーンを描画する
	 * 
	 * @param g
	 * @param task
	 * @param rect
	 */
	private void drawMilestone(Graphics g, Task task, Rectangle rect) {
		g.setColor(milestoneColor);
		int half = rect.height / 2;
		Polygon polygon = new Polygon(new int[] { rect.x, rect.x - half,
				rect.x, rect.x + half }, new int[] { rect.y, rect.y + half,
				rect.y + rect.height, rect.y + half }, 4);
		g.fillPolygon(polygon);
	}

	/**
	 * 親タスクを描画する
	 * 
	 * @param g
	 * @param task
	 * @param rect
	 */
	private void drawParentTask(Graphics g, Task task, Rectangle rect) {
		g.setColor(groupColor);
		int half = rect.height * 2 / 3;
		Polygon polygon = new Polygon(new int[] { rect.x, rect.x,
				rect.x + half, rect.x + rect.width - half, rect.x + rect.width,
				rect.x + rect.width, rect.x + rect.width }, new int[] { rect.y,
				rect.y + rect.height, rect.y + rect.height - half,
				rect.y + rect.height - half, rect.y + rect.height, rect.y }, 6);
		g.fillPolygon(polygon);
	}

	/**
	 * 子タスクを描画する
	 * 
	 * @param g
	 * @param task
	 * @param rect
	 */
	private void drawBasicTask(Graphics g, Task task, Rectangle rect) {
		g.setColor(fillColor);
		g.fillRect(rect.x, rect.y, rect.width, rect.height);
		if (task.getCompletion() > 0) {
			g.setColor(borderColor);
			g.fillRect(rect.x, rect.y, (rect.width) * task.getCompletion()
					/ 100, rect.height);
		}

		if (task.getCompletion() > 0) {
			MessageFormat format = new MessageFormat("{0}%");
			g.setColor(fontColor);
			g.drawString(format.format(new Object[] { task.getCompletion() }),
					rect.x + rect.width + getCellWidth(), rect.y + rect.height);
		}

		g.setColor(borderColor);
		g.drawRect(rect.x, rect.y, rect.width, rect.height);
	}

	/**
	 * タスクの描画範囲を返す
	 * 
	 * @param task
	 * @return
	 */
	private Rectangle getTaskRect(Task task) {
		if (task == null || !task.isDrawable())
			return null;

		int dateOffset = (int) ((task.getStartDate().getTimeInMillis() - project
				.getChartFromDate().getTimeInMillis()) / 86400000);
		int period = task.getRealPeriod().intValue();
		int offsetY = task.getIndex() * getCellHeight();

		return new Rectangle(dateOffset * getCellWidth(), offsetY
				+ getCellHeight() / 4, getCellWidth() * period,
				getCellHeight() / 2);
	}

	/**
	 * タスクのヒット判定範囲を返す
	 * 
	 * @param task
	 * @return
	 */
	private Rectangle getTaskHitRect(Task task) {
		if (task == null || task.getStartDate() == null)
			return null;

		int dateOffset = (int) ((task.getStartDate().getTimeInMillis() - project
				.getChartFromDate().getTimeInMillis()) / 86400000);
		int period = task.getRealPeriod().intValue();
		int offsetY = task.getIndex() * getCellHeight();

		return new Rectangle(dateOffset * getCellWidth() - getCellWidth() / 3,
				offsetY + getCellHeight() / 4, getCellWidth() * period
						+ (getCellWidth() / 3) * 2, getCellHeight() / 2);
	}

	private void repaintProject() {
		if (app.getTaskListPane() != null) {
			app.getTaskListPane().repaint();
		}
		if (app.getTaskLineHeaderPane() != null) {
			app.getTaskLineHeaderPane().repaint();
		}
		repaint();
	}

	@Override
	public void mouseDragged(MouseEvent e) {
		if (dragOperation == OPERATION_NONE)
			return;

		if (printRange.mouseDragged(e))
			return;
		if (taskUI.mouseDragged(e))
			return;

		if (dragOperation == OPERATION_RESTRICTION_CREATE) {
			// TODO 操作中の制約をわかりやすくしたい
		}
	}

	@Override
	public void mouseMoved(MouseEvent e) {
		if (pressOperation == OPERATION_NONE) {
			if (printRange.mouseMoved(e))
				return;
			if (taskUI.mouseMoved(e))
				return;
			cursorState.setDefaultCursor();
		}
	}

	@Override
	public void mouseClicked(MouseEvent e) {
	}

	@Override
	public void mousePressed(MouseEvent e) {
		if (e.isPopupTrigger()) {
			triggerPopup(e);
			return;
		}

		targetTask = null;
		dragOperation = OPERATION_NONE;
		dragStartPoint = e.getPoint();

		int index = getIndexAtPoint(e.getPoint());
		if (index >= project.getTaskTableModel().getRowCount())
			return;

		if (printRange.mousePressed(e))
			return;
		if (restrictionUI.mousePressed(e))
			return;
		if (taskUI.mousePressed(e))
			return;
	}

	@Override
	public void mouseReleased(MouseEvent e) {
		if (e.isPopupTrigger()) {
			triggerPopup(e);
		}

		if (restrictionUI.mouseReleased(e))
			return;
	}

	private void triggerPopup(MouseEvent e) {
		JPopupMenu menu = new JPopupMenu();
		JMenuItem item = new JMenuItem("タスクを削除");
		item.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				// TODO Auto-generated method stub

			}
		});
	}

	@Override
	public void mouseEntered(MouseEvent e) {
		requestFocus();
	}

	@Override
	public void mouseExited(MouseEvent e) {

	}

	private Calendar getCalendarAtPoint(Point p) {
		int offsetDate = p.x / getCellWidth();
		Calendar calendar = (Calendar)project.getChartFromDate().clone();
		calendar.add(Calendar.DATE, offsetDate);
		return calendar;
	}

	/**
	 * 指定した座標に近く、最も早い営業日を返す
	 * 
	 * @param x
	 * @return
	 */
	private Calendar getEarlyBusinessDayAtPoint(Point p) {
		Calendar calendar = getCalendarAtPoint(p);
		for (int i = 0; i < 365; i++) {
			if (app.getProject().isHoliday(calendar)) {
				calendar.add(Calendar.DATE, -1);
			} else {
				break;
			}
		}
		return calendar;
	}

	/**
	 * 指定した座標に近く、最も遅い営業日を返す
	 * 
	 * @param x
	 * @return
	 */
	private Calendar getLateBusinessDayAtPoint(Point p) {
		Calendar calendar = getCalendarAtPoint(p);
		for (int i = 0; i < 365; i++) {
			if (app.getProject().isHoliday(calendar)) {
				calendar.add(Calendar.DATE, 1);
			} else {
				break;
			}
		}
		return calendar;
	}

	/**
	 * 指定した期間内の営業日数を返す。（範囲に開始日は含み、終了日は含まない)
	 * 
	 * @param from
	 * @param to
	 * @return
	 */
	private int getBusinessDayCount(Calendar from, Calendar to) {
		Calendar calendar = (Calendar) from.clone();
		int businessDayCount = 0;
		while (calendar.before(to)) {
			if (!app.getProject().isHoliday(calendar))
				businessDayCount++;
			calendar.add(Calendar.DATE, 1);
		}
		return businessDayCount;
	}

	protected Task getTaskAtPoint(Point p) {
		if (p == null)
			return null;

		int index = getIndexAtPoint(p);
		Task task = project.findTask(index);
		if (task == null)
			return null;
		Rectangle rect = getTaskRect(task);
		if (rect == null)
			return null;
		return rect.contains(p) ? task : null;
	}

	private int getIndexAtPoint(Point point) {
		return point.y / getCellHeight();
	}

	@Override
	public void keyTyped(KeyEvent e) {
		// TODO Auto-generated method stub

	}

	@Override
	public void keyPressed(KeyEvent e) {
		if (e.getKeyCode() == KeyEvent.VK_CONTROL || e.getKeyCode() == KeyEvent.VK_SHIFT) {
			if (e.isShiftDown() && e.isControlDown()) {
				cursorState.setCursor(CursorState.RESTRICTION_DELETE_CURSOR);
				pressOperation = OPERATION_RESTRICTION_DELETE;
			} else if (e.isControlDown()) {
				cursorState.setCursor(CursorState.RESTRICTION_CREATE_CURSOR);
				pressOperation = OPERATION_RESTRICTION_CREATE;
			}
		}
	}

	@Override
	public void keyReleased(KeyEvent e) {
		if (e.getKeyCode() == KeyEvent.VK_CONTROL || e.getKeyCode() == KeyEvent.VK_SHIFT) {
			if (e.isControlDown()) {
				cursorState.setCursor(CursorState.RESTRICTION_CREATE_CURSOR);
				pressOperation = OPERATION_RESTRICTION_CREATE;
			} else {
				cursorState.setDefaultCursor();
				pressOperation = OPERATION_NONE;
			}
		}
	}

	public class TaskUI {

		public boolean mousePressed(MouseEvent e) {
			Task task = null;
			Pair<Integer, Task> operation = getOperation(e);
			if (operation.getFirst() == OPERATION_TASK_CREATE) {
				task = createTaskOperation(e);
				operation = new Pair<Integer, Task>(OPERATION_TASK_PERIOD, task);
			}
			targetTask = operation.getSecond();
			dragOperation = operation.getFirst();
			return true;
		}

		public boolean mouseMoved(MouseEvent e) {
			Pair<Integer, Task> operation = getOperation(e);
			switch (operation.getFirst()) {
			case OPERATION_TASK_SLIDE:
				cursorState.setCursor(Cursor.E_RESIZE_CURSOR);
				return true;
			case OPERATION_TASK_COMPLETION:
				cursorState.setCursor(CursorState.COMPLETION_CURSOR);
				return true;
			case OPERATION_TASK_PERIOD:
				cursorState.setCursor(Cursor.E_RESIZE_CURSOR);
				return true;
			}
			return false;
		}

		public boolean mouseDragged(MouseEvent e) {
			if (dragOperation == OPERATION_TASK_SLIDE) {
				targetTask
						.setStartDate(getLateBusinessDayAtPoint(e.getPoint()));
				project.update();
				repaintProject();
				return true;
			} else if (dragOperation == OPERATION_TASK_PERIOD) {
				Calendar toCalendar = getLateBusinessDayAtPoint(e.getPoint());
				int count = getBusinessDayCount(targetTask.getStartDate(),
						toCalendar);
				targetTask.setPeriod(count);
				project.update();
				repaintProject();
				return true;
			} else if (dragOperation == OPERATION_TASK_COMPLETION) {
				int offset = e.getX();
				Rectangle rect = getTaskRect(targetTask);
				int completion = ((offset - rect.x) * 100 / rect.width / 5) * 5;
				completion = Math.min(Math.max(completion, 0), 100);
				targetTask.setCompletion(completion);
				repaintProject();
				return true;
			}
			return false;
		}

		public Pair<Integer, Task> getOperation(MouseEvent e) {
			int index = getIndexAtPoint(e.getPoint());
			Task task = project.findTask(index);
			if (task == null || task.getStartDate() == null)
				return new Pair<Integer, Task>(OPERATION_TASK_CREATE, null);

			Rectangle rect = getTaskHitRect(task);
			int margin = (getCellWidth() / 3) * 2;
			if (task.isMilestone()) {
				rect.x -= margin / 2;
				rect.width += margin;
			}

			int left = rect.x;
			int right = rect.x + rect.width;
			if (rect.contains(e.getPoint())) {
				if (e.getPoint().x < left + margin) {
					return new Pair<Integer, Task>(OPERATION_TASK_SLIDE, task);
				} else if (e.getPoint().x >= right - margin) {
					return new Pair<Integer, Task>(OPERATION_TASK_PERIOD, task);
				} else {
					return new Pair<Integer, Task>(OPERATION_TASK_COMPLETION, task);
				}
			}

			return new Pair<Integer, Task>(OPERATION_NONE, task);
		}

		public Task createTaskOperation(MouseEvent e) {
			int index = getIndexAtPoint(e.getPoint());
			Task task = project.findTask(index);
			if (task == null) {
				task = new Task();
				task.setIndex(index);
				task.setName("新規タスク");
				project.getTasks().add(task);
			}
			targetTask = task;
			if (task.getStartDate() == null) {
				task.setStartDate(getEarlyBusinessDayAtPoint(e.getPoint()));
				project.update();
				app.getTaskListPane().getTaskListTableModel()
						.fireTableChanged();
				repaintProject();
			}
			return task;
		}
	}

	/**
	 * 制約条件操作の実装
	 */
	public class RestrictionUI {

		public boolean mousePressed(MouseEvent e) {
			if (pressOperation != OPERATION_NONE) {
				
				Task task = getTaskAtPoint(e.getPoint());
				if (task == null)
					return true;
	
				if (pressOperation == OPERATION_RESTRICTION_CREATE) {
					targetTask = task;
					dragOperation = OPERATION_RESTRICTION_CREATE;
					return true;
				} else if (pressOperation == OPERATION_RESTRICTION_DELETE) {
					removeRestriction(e);
					return true;
				}
				
				return true;
			}
			
			return false;
		}

		public boolean mouseReleased(MouseEvent e) {
			if (dragOperation != OPERATION_RESTRICTION_CREATE)
				return false;

			Task dropTask = getTaskAtPoint(e.getPoint());
			if (dropTask == null)
				return false;

			targetTask.addRestriction(dropTask);
			repaintProject();
			return true;
		}

		private void removeRestriction(MouseEvent e) {
			Task task = getTaskAtPoint(e.getPoint());
			if (task == null) return;
			
			Rectangle rect = getTaskRect(task);
			if (rect == null) return;
			
			if (e.getX() < rect.x + rect.width / 2) {
				for (Task src: project.getTasks()) {
					src.getRestrictions().remove(task);
				}
				repaint();
			} else {
				task.getRestrictions().clear();
				repaint();
			}
		}

		/**
		 * 制約条件をひとつ描画する
		 * 
		 * @param g
		 * @param from
		 * @param to
		 */
		private void drawRestriction(Graphics g, Task from, Task to) {
			int half = getCellHeight() / 2;
			int quarter = getCellHeight() / 4;
			Rectangle fromRect = getTaskRect(from);
			Rectangle toRect = getTaskRect(to);

			// 制約対象の開始日が制約元より右側にある場合は
			// 矢印の表示形式を変える
			if (from.getStartDate() == null || from.getRealPeriod() == null
					|| to.getStartDate() == null)
				return;
			long fromTime = from.getStartDate().getTimeInMillis()
					+ from.getRealPeriod().longValue() * 86400000;
			long toTime = to.getStartDate().getTimeInMillis();

			if (fromTime >= toTime) {
				if (from.getIndex() < to.getIndex()) {
					Point startingPoint = new Point(fromRect.x + fromRect.width,
							fromRect.y + fromRect.height / 2);
					Point endPoint = new Point(toRect.x + quarter, toRect.y);

					if (from.isMilestone())
						startingPoint.x -= fromRect.width;
					if (to.isMilestone())
						endPoint.x -= quarter;

					g.setColor(restrictionColor);
					g.drawPolyline(new int[] { startingPoint.x,
							startingPoint.x + quarter, startingPoint.x + quarter,
							endPoint.x, endPoint.x }, new int[] { startingPoint.y,
							startingPoint.y, startingPoint.y + half,
							startingPoint.y + half, endPoint.y }, 5);
					g.fillPolygon(new int[] { endPoint.x, endPoint.x - quarter,
							endPoint.x + quarter }, new int[] { endPoint.y,
							endPoint.y - quarter, endPoint.y - quarter }, 3);
				} else {
					Point startingPoint = new Point(fromRect.x + fromRect.width,
							fromRect.y + fromRect.height / 2);
					Point endPoint = new Point(toRect.x + quarter, toRect.y
							+ toRect.height);

					if (from.isMilestone())
						startingPoint.x -= fromRect.width;
					if (to.isMilestone())
						endPoint.x -= quarter;

					g.setColor(restrictionColor);
					g.drawPolyline(new int[] { startingPoint.x,
							startingPoint.x + quarter, startingPoint.x + quarter,
							endPoint.x, endPoint.x }, new int[] { startingPoint.y,
							startingPoint.y, startingPoint.y - half,
							startingPoint.y - half, endPoint.y }, 5);
					g.fillPolygon(new int[] { endPoint.x, endPoint.x - quarter - 1,
							endPoint.x + quarter + 1 }, new int[] { endPoint.y,
							endPoint.y + quarter + 1, endPoint.y + quarter + 1 }, 3);
				}
			} else {
				Point startingPoint = new Point(fromRect.x + fromRect.width,
						fromRect.y + fromRect.height / 2);
				Point endPoint = new Point(toRect.x, toRect.y + fromRect.height / 2);

				if (from.isMilestone())
					startingPoint.x -= fromRect.width;
				if (to.isMilestone())
					endPoint.x -= quarter;

				g.setColor(restrictionColor);
				g.drawPolyline(new int[] { startingPoint.x,
						startingPoint.x + quarter, startingPoint.x + quarter,
						endPoint.x }, new int[] { startingPoint.y, startingPoint.y,
						endPoint.y, endPoint.y }, 4);
				g.fillPolygon(new int[] { endPoint.x, endPoint.x - quarter,
						endPoint.x - quarter }, new int[] { endPoint.y,
						endPoint.y + quarter, endPoint.y - quarter }, 3);
			}
		}

		/**
		 * 制約条件を描画する
		 * 
		 * @param g
		 */
		private void drawRestrictions(Graphics g) {
			for (Task from : project.getTasks()) {
				for (Task to : from.getRestrictions()) {
					drawRestriction(g, from, to);
				}
			}
		}

	}

	/**
	 * 印刷範囲周りの実装
	 */
	public class PrintRange {
		Color printRangeFirstColor = new Color(0xF0, 0x50, 0x50);
		Color printRangeColor = new Color(0xF0, 0xA0, 0xA0);

		boolean printRangeMode = true;

		public void drawPrintRange(Graphics g) {
			if (!printRangeMode) return;
			
			List<Rectangle> ranges = getPrintRanges();

			int page = ranges.size();
			Collections.reverse(ranges);
			for (Rectangle r : ranges) {

				g.setColor(page == 1 ? printRangeFirstColor : printRangeColor);
				g.drawRect(r.x, r.y, r.width, r.height);
				g.drawRect(r.x + 1, r.y + 1, r.width - 2, r.height - 2);
				g.drawRect(r.x - 1, r.y - 1, r.width + 2, r.height + 2);

				Font oldFont = g.getFont();
				Font newFont = new Font(oldFont.getFontName(), Font.BOLD, 32);
				g.setFont(newFont);
				String str = "ページ" + page;
				Shape oldClip = g.getClip();
				g.clipRect(r.x, r.y, r.width, r.height);
				StringUtil
						.drawStringCenter(g, str, r.x, r.y, r.width, r.height);
				g.setClip(oldClip);
				g.setFont(oldFont);

				page--;
			}

		}

		public boolean mousePressed(MouseEvent e) {
			int printRangeOperation = getOperation(e);
			if (printRangeOperation == OPERATION_NONE)
				return false;

			dragOperation = printRangeOperation;
			return true;
		}

		public boolean mouseMoved(MouseEvent e) {
			int operation = getOperation(e);
			switch (operation) {
			case OPERATION_PRINT_RANGE:
				cursorState.setCursor(Cursor.SE_RESIZE_CURSOR);
				return true;
			case OPERATION_PRINT_RANGE_X:
				cursorState.setCursor(Cursor.E_RESIZE_CURSOR);
				return true;
			case OPERATION_PRINT_RANGE_Y:
				cursorState.setCursor(Cursor.S_RESIZE_CURSOR);
				return true;
			}
			return false;
		}

		public boolean mouseDragged(MouseEvent e) {
			if (dragOperation == OPERATION_PRINT_RANGE_X) {
				Rectangle rect = getFirstPrintRange();
				if (rect == null)
					return false;
				project.getPrintCellSize().width = Math.max(10,
						(e.getX() - rect.x) / getCellWidth());
				repaintProject();
				return true;
			} else if (dragOperation == OPERATION_PRINT_RANGE_Y) {
				Rectangle rect = getFirstPrintRange();
				if (rect == null)
					return false;
				project.getPrintCellSize().height = Math.max(10,
						(e.getY() - rect.y) / getCellHeight());
				repaintProject();
				return true;
			} else if (dragOperation == OPERATION_PRINT_RANGE) {
				Rectangle rect = getFirstPrintRange();
				if (rect == null)
					return false;
				project.getPrintCellSize().width = Math.max(10,
						(e.getX() - rect.x) / getCellWidth());
				project.getPrintCellSize().height = Math.max(10,
						(e.getY() - rect.y) / getCellHeight());
				repaintProject();
				return true;
			}
			return false;
		}

		public int getOperation(MouseEvent e) {
			if (!printRangeMode)
				return OPERATION_NONE;

			Rectangle rect = getFirstPrintRange();
			if (rect != null
					&& e.getX() >= rect.x + rect.width - getCellWidth() / 2
					&& e.getX() < rect.x + rect.width
					&& e.getY() >= rect.y + rect.height - getCellHeight() / 2
					&& e.getY() < rect.y + rect.height) {
				return OPERATION_PRINT_RANGE;
			} else if (rect != null
					&& e.getX() >= rect.x + rect.width - getCellWidth() / 2
					&& e.getX() < rect.x + rect.width && e.getY() >= rect.y
					&& e.getY() < rect.y + rect.height - getCellHeight() / 2) {
				return OPERATION_PRINT_RANGE_X;
			} else if (rect != null && e.getX() >= rect.x
					&& e.getX() < rect.x + rect.width - getCellWidth() / 2
					&& e.getY() >= rect.y + rect.height - getCellHeight() / 2
					&& e.getY() < rect.y + rect.height) {
				return OPERATION_PRINT_RANGE_Y;
			}

			return OPERATION_NONE;
		}

		/**
		 * 最初の印刷範囲を取得する
		 * 
		 * @return
		 */
		public Rectangle getFirstPrintRange() {
			Calendar dataRangeFirstDate = project.getFirstDate();
			int dataRangePeriod = project.getWholePeriod();
			int maxIndex = project.getMaxIndex();
			if (dataRangeFirstDate == null)
				return null;

			Calendar printRangeFirstDate = dataRangeFirstDate;
			printRangeFirstDate.add(Calendar.DATE, -1);
			int printRangePeriod = dataRangePeriod + 2;

			int offsetX = CalendarUtil.subDate(printRangeFirstDate,
					project.getChartFromDate())
					* getCellWidth();
			Dimension printCellSize = project.getPrintCellSize();

			return new Rectangle(offsetX, 0, printCellSize.width
					* getCellWidth(), printCellSize.height * getCellHeight());
		}

		/**
		 * 印刷範囲をページごとに取得する
		 * 
		 * @return
		 */
		public List<Rectangle> getPrintRanges() {
			List<Rectangle> result = new ArrayList<Rectangle>();

			Calendar dataRangeFirstDate = project.getFirstDate();
			int dataRangePeriod = project.getWholePeriod();
			int maxIndex = project.getMaxIndex();
			if (dataRangeFirstDate == null)
				return result;

			Calendar printRangeFirstDate = dataRangeFirstDate;
			printRangeFirstDate.add(Calendar.DATE, -1);
			int printRangePeriod = dataRangePeriod + 2;

			int page = 1;
			int offsetX = CalendarUtil.subDate(printRangeFirstDate,
					project.getChartFromDate())
					* getCellWidth();
			Dimension printCellSize = project.getPrintCellSize();

			for (int cursorY = 0; cursorY <= maxIndex; cursorY += printCellSize.height) {
				for (int cursorX = 0; cursorX < printRangePeriod; cursorX += printCellSize.width) {
					Rectangle r = new Rectangle(offsetX + cursorX
							* getCellWidth(), cursorY * getCellHeight(),
							printCellSize.width * getCellWidth(),
							printCellSize.height * getCellHeight());
					result.add(r);
					page++;
				}
			}

			return result;
		}

	}
}
