package jp.sourceforge.freegantt.swing;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.text.SimpleDateFormat;
import java.util.Calendar;

import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;

import jp.sourceforge.freegantt.locale.Resource;
import jp.sourceforge.freegantt.util.CalendarUtil;
import jp.sourceforge.freegantt.util.StringUtil;

public class TaskLineHeaderPane extends JPanel implements MouseListener {
	private static final long serialVersionUID = 4416176242747877014L;
	
	Application app;
	TaskLinePane taskLinePane;
	
	Color lineColor = new Color(0xC0, 0xC0, 0xC0);
	Color bgColor = Color.white;
	Color fontColor = Color.black;
	
	public TaskLineHeaderPane(Application app, TaskLinePane taskLinePane) {
		super();
		this.app = app;
		this.taskLinePane = taskLinePane;
		setBackground(bgColor);
		addMouseListener(this);
	}

	private int getCellHeight() {
		return taskLinePane.getCellHeight();
	}
	
	private int getCellWidth() {
		return taskLinePane.getCellWidth();
	}
	
	public void updateWidth() {
		Calendar fromDate = app.getProject().getChartFromDate();
		Calendar toDate = app.getProject().getChartToDate();
		setSize(CalendarUtil.subDate(toDate, fromDate) * getCellWidth(), getHeight());
	}

	@Override
	protected void paintComponent(Graphics g) {
		super.paintComponent(g);
		
		g.setColor(lineColor);
		g.drawLine(0, getCellHeight() - 1, getWidth(), getCellHeight() - 1);
		g.drawLine(0, getCellHeight() * 2 - 1, getWidth(), getCellHeight() * 2 - 1);
		
		Calendar now = (Calendar)app.getProject().getChartFromDate().clone();
		Calendar to = app.getProject().getChartToDate();

		int offset = 0;
		SimpleDateFormat format = new SimpleDateFormat(Resource.get("chartHeaderMonth"), Resource.getLocale());
		while (now.getTimeInMillis() < to.getTimeInMillis()) {
			g.setColor(fontColor);
			if (now.get(Calendar.DAY_OF_MONTH ) == 1) {
				String text = format.format(now.getTime());
				StringUtil.drawStringLeft(g, text, offset, 0, getCellWidth(), getCellHeight());
			}
			StringUtil.drawScaledDate(g, String.valueOf(now.get(Calendar.DAY_OF_MONTH)), offset, getCellHeight(), getCellWidth(), getCellHeight(), 0.7);
			
			now.add(Calendar.DATE, 1);
			offset += getCellWidth();
			
			// 縦線の描画（月末は上段まで線を引く）
			g.setColor(lineColor);
			if (now.get(Calendar.DAY_OF_MONTH ) == 1) {
				g.drawLine(offset - 1, 0, offset - 1, getHeight() - 1);
			} else {
				g.drawLine(offset - 1, getCellHeight() - 1, offset - 1, getCellHeight() * 2 - 1);
			}
		}
	}

	@Override
	public void mouseClicked(MouseEvent e) {
	}

	@Override
	public void mousePressed(MouseEvent e) {
		if (e.isPopupTrigger() && e.getY() >= getCellHeight()) {
			triggerPopup(e);
		}
	}

	@Override
	public void mouseReleased(MouseEvent e) {
		if (e.isPopupTrigger() && e.getY() >= getCellHeight()) {
			triggerPopup(e);
		}
	}
	

	@Override
	public void mouseEntered(MouseEvent e) {
	}
	

	@Override
	public void mouseExited(MouseEvent e) {
	}

	private Calendar getCalendarAtPoint(Point p) {
		long offsetDate = p.x / getCellWidth();
		Calendar calendar = Calendar.getInstance();
		calendar.setTimeInMillis(app.getProject().getChartFromDate().getTimeInMillis() + offsetDate * 86400000);
		return calendar;
	}
	
	private void triggerPopup(MouseEvent e) {
		final Calendar calendar = getCalendarAtPoint(e.getPoint());
		
		JPopupMenu menu = new JPopupMenu();
		JMenuItem item = new JMenuItem(Resource.get("chartHeaderAddHoliday"));
		item.setEnabled(!app.getProject().isHoliday(calendar));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				app.getProject().getAdditionalHolidays().add(calendar);
				app.getProject().update();
				app.getProject().getHolidayTableModel().fireTableChanged();
				app.getProject().getTaskTableModel().fireTableChanged();
			}
		});
		menu.add(item);
		item = new JMenuItem(Resource.get("chartHeaderRemoveHoliday"));
		item.setEnabled(app.getProject().isAdditionalHoliday(calendar));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				app.getProject().getAdditionalHolidays().remove(calendar);
				app.getProject().update();
				app.getProject().getHolidayTableModel().fireTableChanged();
				app.getProject().getTaskTableModel().fireTableChanged();
			}
		});
		menu.add(item);
		menu.show(this, e.getX(), e.getY());
	}

}
