package jp.sourceforge.freegantt.data.controller;

import java.awt.Dimension;
import java.awt.print.PageFormat;
import java.util.Calendar;
import java.util.List;

import javax.swing.undo.CompoundEdit;
import javax.swing.undo.UndoManager;

import jp.sourceforge.freegantt.data.Column;
import jp.sourceforge.freegantt.data.Member;
import jp.sourceforge.freegantt.data.Project;
import jp.sourceforge.freegantt.data.Task;
import jp.sourceforge.freegantt.data.undo.CreateAdditionalHolidayEdit;
import jp.sourceforge.freegantt.data.undo.CreateMemberEdit;
import jp.sourceforge.freegantt.data.undo.CreateRestrictionEdit;
import jp.sourceforge.freegantt.data.undo.CreateTaskEdit;
import jp.sourceforge.freegantt.data.undo.InsertTaskEdit;
import jp.sourceforge.freegantt.data.undo.RemoveAdditionalHolidayEdit;
import jp.sourceforge.freegantt.data.undo.RemoveMemberEdit;
import jp.sourceforge.freegantt.data.undo.RemoveRestrictioinEdit;
import jp.sourceforge.freegantt.data.undo.RemoveTaskEdit;
import jp.sourceforge.freegantt.data.undo.UpdateAdditionalHolidayEdit;
import jp.sourceforge.freegantt.data.undo.UpdateCalendarModeEdit;
import jp.sourceforge.freegantt.data.undo.UpdateCellSizeEdit;
import jp.sourceforge.freegantt.data.undo.UpdateFixedHolidayEdit;
import jp.sourceforge.freegantt.data.undo.UpdateMemberEdit;
import jp.sourceforge.freegantt.data.undo.UpdateModifiedEdit;
import jp.sourceforge.freegantt.data.undo.UpdatePrintCellSizeEdit;
import jp.sourceforge.freegantt.data.undo.UpdateProjectNameEdit;
import jp.sourceforge.freegantt.data.undo.UpdateProjectSummaryEdit;
import jp.sourceforge.freegantt.data.undo.UpdateTaskCriticalPathEdit;
import jp.sourceforge.freegantt.data.undo.UpdateTaskEdit;
import jp.sourceforge.freegantt.data.undo.UpdateTaskGroupEdit;
import jp.sourceforge.freegantt.data.undo.UpdateTaskTableWidthEdit;
import jp.sourceforge.freegantt.data.undo.UpdateTaskVisibleEdit;
import jp.sourceforge.freegantt.data.undo.UpdateViewTaskTableColumnsEdit;
import jp.sourceforge.freegantt.util.CalendarUtil;

public class ProjectController {

	Project project;
	
	public ProjectController(Project project) {
		this.project = project;
	}
	
	private UndoManager getUndoManager() {
		return project.getUndoManager();
	}
	
	/**
	 * プロジェクトの変更フラグを変更する。
	 * アンドゥ可能な操作に組み合わせて使う
	 * @param flag
	 */
	public void setModified(boolean flag) {
		if (project.isModified() == flag) return;
		
		project.getUndoManager().addEdit(new UpdateModifiedEdit(project, flag));
		project.setModified(flag);
		project.getProjectModifiedModel().fireProjectModifiedEment();
	}
	
	/**
	 * 行を削除する
	 * @param index
	 */
	public void removeIndex(int index) {
		CompoundEdit compound = new CompoundEdit();
		getUndoManager().addEdit(compound);
		try {
			setModified(true);

			Task targetTask = project.getTaskAtIndex(index);
			removeDestRestrictions(targetTask);
			compound.addEdit(new RemoveTaskEdit(project, index));
			if (targetTask != null) project.getTasks().remove(targetTask);
		} finally {
			compound.end();
			project.getTaskTableModel().fireTableChanged();
		}
	}
	
	/**
	 * 行にタスクを挿入する
	 * @param newTask
	 * @param row
	 */
	public void insertTaskAtIndex(Task newTask, int index) {
		CompoundEdit compound = new CompoundEdit();
		getUndoManager().addEdit(compound);
		try {
			setModified(true);

			getUndoManager().addEdit(new InsertTaskEdit(project, newTask, index));
			project.getTasks().add(index, newTask);
		} finally {
			compound.end();
			project.getTaskTableModel().fireTableChanged();
		}
	}
	
	/**
	 * 指定した行のタスクを差し替える
	 * 必要があれば指定行までのパディングを行う
	 * また、差し替え前のタスクが制約先として指定されていた場合は付け替える
	 * @param task
	 * @param row
	 */
	public void setTaskAtIndex(Task newTask, int index) {
		if (newTask == null) newTask = new Task();
		CompoundEdit compound = new CompoundEdit();
		getUndoManager().addEdit(compound);
		try {
			setModified(true);
			
			fillNewTaskAtIndex(index);
			
			// 被参照の付け替え
			Task oldTask = project.getTaskAtIndex(index);
			for (Task task: project.getTasks()) {
				if (task.getRestrictions().remove(oldTask)) {
					task.getRestrictions().add(newTask);
				}
			}
	
			compound.addEdit(new UpdateTaskEdit(project, 
					newTask, 
					index));
			project.getTasks().set(index, newTask);
		} finally {
			compound.end();
		}
	}

	/**
	 * 指定行まで空タスクで埋める
	 * @param row
	 * @return
	 */
	public Task fillNewTaskAtIndex(int index) {
		CompoundEdit compound = new CompoundEdit();
		getUndoManager().addEdit(compound);
		try {
			setModified(true);
			
			while (project.getTasks().size() <= index) {
				Task fillTask = new Task();
				compound.addEdit(new CreateTaskEdit(project, fillTask));
				project.getTasks().add(fillTask);
			}
		} finally {
			compound.end();
		}
		return project.getTaskAtIndex(index);
	}

	
	/**
	 * 制約を追加する
	 * @param src
	 * @param dst
	 */
	public void addRestriction(Task src, Task dst) {
		CompoundEdit compound = new CompoundEdit();
		project.getUndoManager().addEdit(compound);
		try {
			setModified(true);
	
			compound.addEdit(new CreateRestrictionEdit(project, 
					project.getIndexByTask(src), 
					project.getIndexByTask(dst)));
			src.addRestriction(dst);
			project.getTaskTableModel().fireTableChanged();
		} finally {
			compound.end();
		}
	}
	
	/**
	 * 制約を削除する
	 * @param src
	 * @param dst
	 */
	public void removeRestriction(Task src, Task dst) {
		CompoundEdit compound = new CompoundEdit();
		project.getUndoManager().addEdit(compound);
		try {
			setModified(true);
	
			compound.addEdit(new RemoveRestrictioinEdit(project, 
					project.getIndexByTask(src), 
					project.getIndexByTask(dst)));
			src.removeRestriction(dst);
		} finally {
			compound.end();
			project.getTaskTableModel().fireTableChanged();
		}
	}
	
	/**
	 * 全ての制約先から対象タスクを削除する
	 * @param task
	 */
	public void removeDestRestrictions(Task task) {
		CompoundEdit compound = new CompoundEdit();
		getUndoManager().addEdit(compound);
		try {
			setModified(true);
			
			List<Task> srcs = project.getRestrictionSrcTasks(task);
			for (Task src: srcs) {
				compound.addEdit(new RemoveRestrictioinEdit(project, 
						project.getIndexByTask(src), 
						project.getIndexByTask(task)));
				src.getRestrictions().remove(task);
			}
		} finally {
			compound.end();
		}
	}
	
	/**
	 * 全ての制約元から対象タスクを削除する
	 * @param task
	 */
	public void removeSrcRestrictions(Task task) {
		CompoundEdit compound = new CompoundEdit();
		getUndoManager().addEdit(compound);
		try {
			setModified(true);
			
			for (Task dst: task.getRestrictions()) {
				compound.addEdit(new RemoveRestrictioinEdit(project, 
						project.getIndexByTask(task), 
						project.getIndexByTask(dst)));
			}
			task.getRestrictions().clear();
		} finally {
			compound.end();
		}
	}
	
	/**
	 * タスクのレベルを設定する
	 * @param task
	 * @param level
	 */
	public void setTaskLevel(Task task, int level) {
		if (task == null) return;
		CompoundEdit compound = new CompoundEdit();
		getUndoManager().addEdit(compound);
		try {
			setModified(true);
			
			int index = project.getIndexByTask(task);
			Task newTask = task.clone();
			newTask.setLevel(level);

			setTaskAtIndex(newTask, index);
		} finally {
			compound.end();
		}
	}

	/**
	 * タスクのレベルを上げる
	 * @param task
	 */
	public void levelUpTask(Task task) {
		if (task == null) return;
		setTaskLevel(task, task.getLevel() + 1);
	}
	
	/**
	 * タスクのレベルを下げる
	 * @param task
	 */
	public void levelDownTask(Task task) {
		if (task == null) return;
		if (task.getLevel() <= 0) return;
		setTaskLevel(task, task.getLevel() - 1);
	}

	/**
	 * 印刷範囲を変更する
	 * @param newPrintCellSize
	 */
	public void setPrintCellSize(Dimension newPrintCellSize) {
		CompoundEdit compound = new CompoundEdit();
		getUndoManager().addEdit(compound);
		try {
			setModified(true);
	
			compound.addEdit(new UpdatePrintCellSizeEdit(project, newPrintCellSize));
			project.getPrint().setPrintCellSize(newPrintCellSize);
		} finally {
			compound.end();
		}
	}
	
	/**
	 * 固定休日を更新する
	 * @param week
	 */
	public void setFixedHoliday(int week, boolean value) {
		CompoundEdit compound = new CompoundEdit();
		getUndoManager().addEdit(compound);
		try {
			setModified(true);
	
			compound.addEdit(new UpdateFixedHolidayEdit(project, week, value));
			if (value) {
				project.getFixedHolidays().remove(new Integer(week));
				project.getFixedHolidays().add(new Integer(week));
			} else {
				project.getFixedHolidays().remove(new Integer(week));
			}
		} finally {
			compound.end();
			project.getTaskTableModel().fireTableChanged();
			project.getHolidayTableModel().fireTableChanged();
		}
	}
	
	/**
	 * 個別休日を追加する
	 * @param calendar
	 * @param row
	 */
	public void addAdditionalHoliday(Calendar calendar, int index) {
		CompoundEdit compound = new CompoundEdit();
		getUndoManager().addEdit(compound);
		try {
			setModified(true);
	
			compound.addEdit(new CreateAdditionalHolidayEdit(project, calendar, index));
			project.getAdditionalHolidays().add(index, calendar);
		} finally {
			compound.end();
			project.getTaskTableModel().fireTableChanged();
			project.getHolidayTableModel().fireTableChanged();
		}
	}
	
	public void addAdditionalHoliday(Calendar calendar) {
		addAdditionalHoliday(calendar, project.getAdditionalHolidays().size());
	}

	
	/**
	 * 個別休日を削除する
	 * @param row
	 */
	public void removeAdditionalHoliday(int index) {
		CompoundEdit compound = new CompoundEdit();
		getUndoManager().addEdit(compound);
		try {
			setModified(true);
	
			compound.addEdit(new RemoveAdditionalHolidayEdit(project, index));
			project.getAdditionalHolidays().remove(index);
		} finally {
			compound.end();
			project.getTaskTableModel().fireTableChanged();
			project.getHolidayTableModel().fireTableChanged();
		}
	}
	
	public void removeAdditionalHoliday(Calendar calendar) {
		List<Calendar> holidays = project.getAdditionalHolidays();
		for (int i=0; i<holidays.size(); i++) {
			if (CalendarUtil.dateEquals(calendar, holidays.get(i))) {
				removeAdditionalHoliday(i);
				return;
			}
		}
	}
	
	/**
	 * 個別休日を変更する
	 * @param calendar
	 * @param row
	 */
	public void setAdditionalHoliday(Calendar calendar, int index) {
		CompoundEdit compound = new CompoundEdit();
		getUndoManager().addEdit(compound);
		try {
			setModified(true);
	
			compound.addEdit(new UpdateAdditionalHolidayEdit(project, calendar, index));
			project.getAdditionalHolidays().get(index).setTime(calendar.getTime());
		} finally {
			compound.end();
			project.getTaskTableModel().fireTableChanged();
			project.getHolidayTableModel().fireTableChanged();
		}
	}
	
	/**
	 * リソースを追加する
	 * @param member
	 * @param row
	 */
	public void addMember(Member member, int index) {
		CompoundEdit compound = new CompoundEdit();
		getUndoManager().addEdit(compound);
		try {
			compound.addEdit(new CreateMemberEdit(project, member, index));
			project.getMembers().add(index, member);
		} finally {
			compound.end();
			project.getMemberTableModel().fireTableChanged();
			project.getTaskMemberComboBoxModel().fireListDataListener();
		}
	}
	
	/**
	 * リソースを削除する
	 * @param row
	 */
	public void removeMember(int index) {
		CompoundEdit compound = new CompoundEdit();
		getUndoManager().addEdit(compound);
		try {
			setModified(true);
			
			compound.addEdit(new RemoveMemberEdit(project, index));
			Member oldMember = project.getMembers().get(index);
			for (Task task: project.getTasks()) {
				if (task.getMember() == oldMember) task.setMember(null);
			}
			project.getMembers().remove(index);
		} finally {
			compound.end();
			project.getTaskTableModel().fireTableChanged();
			project.getMemberTableModel().fireTableChanged();
			project.getTaskMemberComboBoxModel().fireListDataListener();
		}
	}
	
	/**
	 * リソースを変更する
	 * @param member
	 * @param row
	 */
	public void setMember(Member member, int index) {
		CompoundEdit compound = new CompoundEdit();
		getUndoManager().addEdit(compound);
		try {
			setModified(true);
			
			compound.addEdit(new UpdateMemberEdit(project, member, index));
			Member oldMember = project.getMembers().get(index);
			for (Task task: project.getTasks()) {
				if (task.getMember() == oldMember) task.setMember(member);
			}
			project.getMembers().set(index, member);
		} finally {
			compound.end();
			project.getTaskTableModel().fireTableChanged();
			project.getMemberTableModel().fireTableChanged();
			project.getTaskMemberComboBoxModel().fireListDataListener();
		}
	}

	/**
	 * プロジェクト名を変更する
	 * @param text
	 */
	public void setName(String text) {
		CompoundEdit compound = new CompoundEdit();
		getUndoManager().addEdit(compound);
		try {
			setModified(true);
			
			compound.addEdit(new UpdateProjectNameEdit(project, text));
			project.setName(text);
		} finally {
			compound.end();
			project.getProjectInfoModel().fireProjectInfoChangedEvent();
		}
	}

	/**
	 * プロジェクト概要を変更する
	 * @param text
	 */
	public void setSummary(String text) {
		CompoundEdit compound = new CompoundEdit();
		getUndoManager().addEdit(compound);
		try {
			setModified(true);
			
			compound.addEdit(new UpdateProjectSummaryEdit(project, text));
			project.setSummary(text);
		} finally {
			compound.end();
			project.getProjectInfoModel().fireProjectInfoChangedEvent();
		}
	}

	/**
	 * ページフォーマットを変更する
	 * ページフォーマットの変更はUndo,Redoの対象外とする
	 * @param project2
	 * @param pageFormat
	 */
	public void setPageFormat(Project project, PageFormat pageFormat) {
		project.getPrint().setPageFormat(pageFormat);
		project.setModified(true);
	}

	/**
	 * タスクテーブル表示幅を変更する
	 * @param dividerMouseListener
	 * @param project2
	 * @param dividerLocation
	 */
	public void setTaskTableWidth(Project project, int oldDividerLocation, int dividerLocation) {
		if (oldDividerLocation == dividerLocation) return;
		
		CompoundEdit compound = new CompoundEdit();
		getUndoManager().addEdit(compound);
		try {
			setModified(true);
			
			compound.addEdit(new UpdateTaskTableWidthEdit(project, oldDividerLocation, dividerLocation));
			project.getView().setTaskTableWidth(dividerLocation);
		} finally {
			compound.end();
			project.getProjectViewModel().fireProjectViewChangedEvent();
		}
	}

	/**
	 * タスクテーブルのカラム表示を全て変更する
	 * @param project
	 * @param columns
	 */
	public void setViewTaskTableColumns(Project project, List<Column> oldColumns, List<Column> newColumns) {
		CompoundEdit compound = new CompoundEdit();
		getUndoManager().addEdit(compound);
		try {
			setModified(true);
			
			compound.addEdit(new UpdateViewTaskTableColumnsEdit(project, oldColumns, newColumns));
			project.getView().setColumns(newColumns);
		} finally {
			compound.end();
			project.getProjectViewModel().fireProjectViewChangedEvent();
		}
	}

	/**
	 * セルの描画サイズを設定する
	 * @param cellSize
	 */
	public void setCellSize(Dimension cellSize) {
		if (cellSize.width < 6) return;
		if (cellSize.width > 20) return;
		
		CompoundEdit compound = new CompoundEdit();
		getUndoManager().addEdit(compound);
		try {
			setModified(true);
			
			compound.addEdit(new UpdateCellSizeEdit(project, cellSize));
			project.setCellSize(cellSize);
		} finally {
			compound.end();
			project.getProjectViewModel().fireProjectViewChangedEvent();
		}
	}

	/**
	 * 日次/週次を設定する
	 * @param mode
	 */
	public void setCalendarMode(int mode) {
		CompoundEdit compound = new CompoundEdit();
		getUndoManager().addEdit(compound);
		try {
			setModified(true);
			
			Dimension dim = (Dimension)project.getCellSize().clone();
			if (mode == Project.CALENDAR_MODE_DATE) {
				dim.width += 4;
				setCellSize(dim);
			} else {
				dim.width -= 4;
				setCellSize(dim);
			}
			
			compound.addEdit(new UpdateCalendarModeEdit(project, mode));
			project.setCalendarMode(mode);
		} finally {
			compound.end();
			project.getProjectViewModel().fireProjectViewChangedEvent();
		}
	}

	/**
	 * 指定行のタスクにグループフラグを設定する
	 * @param b
	 * @param rowByTask
	 */
	public void setTaskGroup(boolean group, int index) {
		Task task = project.getTaskAtIndex(index);
		if (task == null) return;
		if (task.isGroup() == group) return;
		
		CompoundEdit compound = new CompoundEdit();
		getUndoManager().addEdit(compound);
		try {
			setModified(true);
			
			compound.addEdit(new UpdateTaskGroupEdit(project, group, index));
			task.setGroup(group);
		} finally {
			compound.end();
		}
	}

	public void setTaskCriticalPath(boolean criticalPath, int index) {
		Task task = project.getTaskAtIndex(index);
		if (task == null) return;
		if (task.isCriticalPath() == criticalPath) return;
		
		CompoundEdit compound = new CompoundEdit();
		getUndoManager().addEdit(compound);
		try {
			setModified(true);
			
			compound.addEdit(new UpdateTaskCriticalPathEdit(project, criticalPath, index));
			task.setCriticalPath(criticalPath);
		} finally {
			compound.end();
		}
	}

	public void setTaskVisible(boolean visible, int index) {
		Task task = project.getTaskAtIndex(index);
		if (task == null) return;
		if (task.isVisible() == visible) return;
		
		CompoundEdit compound = new CompoundEdit();
		getUndoManager().addEdit(compound);
		try {
			setModified(true);
			
			compound.addEdit(new UpdateTaskVisibleEdit(project, visible, index));
			task.setVisible(visible);
		} finally {
			compound.end();
		}
	}
}
