package jp.sourceforge.freegantt.swing;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.MouseInfo;
import java.awt.Point;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.Shape;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.geom.Path2D;
import java.awt.geom.Point2D;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;

import javax.swing.JPanel;
import javax.swing.JScrollBar;
import javax.swing.JViewport;
import javax.swing.SwingUtilities;
import javax.swing.undo.CompoundEdit;

import jp.sourceforge.freegantt.data.Project;
import jp.sourceforge.freegantt.data.Task;
import jp.sourceforge.freegantt.locale.Resource;
import jp.sourceforge.freegantt.util.CalendarUtil;
import jp.sourceforge.freegantt.util.GraphicsUtil;
import jp.sourceforge.freegantt.util.Pair;

public class TaskLineDataPane extends JPanel implements MouseListener,
		MouseMotionListener, KeyListener {
	private static final long serialVersionUID = 4173201298740221449L;

	Application app;
	Project project;
	TaskLinePane taskLinePane;
	/** カーソル状態保管理*/
	CursorState cursorState;
	/** 印刷範囲表示モード */
	boolean printRangeMode = false;
	/** 稲妻線表示モード */
	boolean progressLineMode = false;

	TaskUI taskUI;
	RestrictionUI restrictionUI;
	PrintRangeUI printRangeUI;
	ProgressLineUI progressLineUI;
	FreeScrollUI freeScrollUI;
	
	/** 操作種別(ドラッグ用) */
	int dragOperation = OPERATION_NONE;
	/** 操作種別(押下用) */
	int pressOperation = OPERATION_NONE;
	/** ドラッグ開始位置 */
	Point dragStartPoint;
	/** 操作対象タスク */
	int targetRow;
	/** 操作中タスク */
	Task floatingTask = null;
	/** 操作中制約 */
	Restriction floatingRestriction = null;
	/** 操作中印刷範囲 */
	Dimension floatingPrintCellSize = null;
	/** 自動スクロール用タイマー */
	Timer autoScrollTimer;

	final static int OPERATION_NONE = 0x00;
	final static int OPERATION_TASK_CREATE = 0x01;
	final static int OPERATION_TASK_SLIDE = 0x02;
	final static int OPERATION_TASK_PERIOD = 0x03;
	final static int OPERATION_TASK_COMPLETION = 0x04;
	final static int OPERATION_RESTRICTION_CREATE = 0x05;
	final static int OPERATION_RESTRICTION_DELETE = 0x06;
	final static int OPERATION_PRINT_RANGE_X = 0x07;
	final static int OPERATION_PRINT_RANGE_Y = 0x08;
	final static int OPERATION_PRINT_RANGE = 0x09;
	final static int OPERATION_FREE_SCROLL = 0x0A;

	// chart colors
	Color weekLineColor 		= new Color(0xD0, 0xD0, 0xD0);
	Color bgColor 				= Color.white;
	Color holidayBgColor 		= new Color(0xF0, 0xF0, 0xF0);
	Color progressLineColor		= new Color(0xF0, 0x50, 0x50);

	// task colors
	Color borderColor 			= new Color(0x85, 0x51, 0xe0);
	Color fillColor 			= new Color(0xda, 0xce, 0xfc);
	Color completionColor 		= borderColor;
	Color floatingBorderColor 	= Color.black;
	Color floatingFillColor 	= new Color(0x90, 0x90, 0x90);
	Color floatingCompletionColor = floatingBorderColor;
	
	Color groupColor 			= new Color(0x50, 0x50, 0x50);
	Color fontColor 			= new Color(0x30, 0x30, 0x30);
	Color milestoneColor 		= groupColor;
	Color floatingMilestoneColor = Color.black;
	
	// restriction colors
	Color restrictionColor 		= new Color(0x50, 0x50, 0x50);
	Color floatingRestrictionColor = Color.black;

	// critical path colors
	Color criticalPathBorderColor = new Color(0xE0, 0x50, 0x50);
	Color criticalPathFillColor = new Color(0xE8, 0xB8, 0xB8);
	Color criticalPathCompletionColor = criticalPathBorderColor;
	Color criticalPathRestrictionColor = criticalPathBorderColor;

	public PrintRangeUI getPrintRange() {
		return printRangeUI;
	}

	public boolean isProgressLineMode() {
		return progressLineMode;
	}

	public void setProgressLineMode(boolean progressLineMode) {
		this.progressLineMode = progressLineMode;
	}

	public boolean isPrintRangeMode() {
		return printRangeMode;
	}

	public void setPrintRangeMode(boolean printRangeMode) {
		this.printRangeMode = printRangeMode;
	}

	public int getCellWidth() {
		return taskLinePane.getCellWidth();
	}

	public int getCellHeight() {
		return taskLinePane.getCellHeight();
	}

	public TaskLineDataPane(Application app, Project project,
			TaskLinePane taskLinePane) {
		super();
		this.app = app;
		this.project = project;
		this.taskLinePane = taskLinePane;
		this.cursorState = new CursorState(this);
		this.printRangeUI = new PrintRangeUI();
		this.taskUI = new TaskUI();
		this.restrictionUI = new RestrictionUI();
		this.progressLineUI = new ProgressLineUI();
		this.freeScrollUI = new FreeScrollUI();
		setBackground(bgColor);
		setFocusable(true);

		setPreferredSize(new Dimension(1, project.getTaskTableModel()
				.getRowCount() * getCellHeight()));

		addMouseListener(this);
		addMouseMotionListener(this);
		addKeyListener(this);
	}
	
	public void updateWidth() {
		Calendar fromDate = taskLinePane.getChartFromDate();
		Calendar toDate = taskLinePane.getChartToDate();
		setSize(CalendarUtil.subDate(toDate, fromDate) * getCellWidth(), getHeight());
	}

	@Override
	protected void paintComponent(Graphics g) {
		super.paintComponent(g);

		// 背景描画
		drawBackground(g);
		// タスク描画
		taskUI.drawTasks(g);
		// 制約条件描画
		restrictionUI.drawRestrictions(g);
		// 稲妻線描画
		progressLineUI.drawProgressLine(g);
		// 印刷範囲描画
		printRangeUI.drawPrintRange(g);
	}

	private void drawBackground(Graphics g) {
		int drawableHeight = project.getTaskTableModel().getRowCount()
				* getCellHeight();

		g.setColor(bgColor);
		g.fillRect(0, 0, getWidth(), getHeight());

		Calendar now = (Calendar) taskLinePane.getChartFromDate().clone();
		Calendar to = taskLinePane.getChartToDate();

		int offset = 0;
		while (now.getTimeInMillis() < to.getTimeInMillis()) {
			if (project.isHoliday(now)) {
				g.setColor(holidayBgColor);
				g.fillRect(offset, 0, getCellWidth(), drawableHeight - 1);
			}

			now.add(Calendar.DATE, 1);
			offset += getCellWidth();

			if (now.get(Calendar.DAY_OF_WEEK) == Calendar.MONDAY) {
				g.setColor(weekLineColor);
				g.drawLine(offset - 1, 0, offset - 1, drawableHeight - 1);
			}
		}

		g.setColor(weekLineColor);
		g.drawLine(0, drawableHeight - 1, getWidth(), drawableHeight - 1);
	}


	/**
	 * タスクの描画範囲を返す
	 * 
	 * @param task
	 * @return
	 */
	private Rectangle getTaskRect(Task task) {
		if (task == null || !task.isDrawable())
			return null;
		
		// 描画範囲を求めるにはデータモデルとつき合わせて
		// 行数を知る必要がある。
		// ただし、操作中のタスクはデータモデルに含まれない一時オブジェクトなので
		// 捜査対象のタスクの行数を使用する
		int row = project.getRowByTask(task);
		if (task == floatingTask) {
			row = targetRow;
		}

		int dateOffset = (int) CalendarUtil.subDate(task.getStartDate(), taskLinePane
				.getChartFromDate());
		int period = task.getRealPeriod().intValue();
		int offsetY = row * getCellHeight();

		return new Rectangle(dateOffset * getCellWidth(), offsetY
				+ getCellHeight() / 4, getCellWidth() * period,
				getCellHeight() / 2);
	}

	/**
	 * タスクのヒット判定範囲を返す
	 * 
	 * @param task
	 * @return
	 */
	private Rectangle getTaskHitRect(Task task) {
		if (task == null || task.getStartDate() == null)
			return null;

		// 描画範囲を求めるにはデータモデルとつき合わせて
		// 行数を知る必要がある。
		// ただし、操作中のタスクはデータモデルに含まれない一時オブジェクトなので
		// 捜査対象のタスクの行数を使用する
		int row = project.getRowByTask(task);
		if (task == floatingTask) {
			row = targetRow;
		}

		int dateOffset = (int) ((task.getStartDate().getTimeInMillis() - taskLinePane
				.getChartFromDate().getTimeInMillis()) / 86400000);
		int period = task.getRealPeriod().intValue();
		int offsetY = row * getCellHeight();

		return new Rectangle(dateOffset * getCellWidth() - getCellWidth() / 3,
				offsetY + getCellHeight() / 4, getCellWidth() * period
						+ (getCellWidth() / 3) * 2, getCellHeight() / 2);
	}

	private void repaintProject() {
		if (app.getTaskListTable() != null) {
			app.getTaskListTable().repaint();
		}
		if (app.getTaskLineHeaderPane() != null) {
			app.getTaskLineHeaderPane().repaint();
		}
		repaint();
	}

	@Override
	public void mouseDragged(MouseEvent e) {
		if (dragOperation != OPERATION_FREE_SCROLL && autoScrollTimer == null) {
			autoScrollTimer = new Timer(true);
			autoScrollTimer.schedule(new AutoScrollHandler(), AutoScrollHandler.SCROLL_INTERVAL, AutoScrollHandler.SCROLL_INTERVAL);
		}

		if (dragOperation == OPERATION_NONE) return;

		if (freeScrollUI.mouseDragged(e)) return;
		if (printRangeUI.mouseDragged(e)) return;
		if (taskUI.mouseDragged(e)) return;
		if (restrictionUI.mouseDragged(e)) return;
	}

	@Override
	public void mouseMoved(MouseEvent e) {
		if (pressOperation == OPERATION_NONE) {
			if (printRangeUI.mouseMoved(e)) return;
			if (taskUI.mouseMoved(e)) return;
			cursorState.setDefaultCursor();
		} 
	}

	@Override
	public void mouseClicked(MouseEvent e) {
	}

	@Override
	public void mousePressed(MouseEvent e) {
		if (e.isPopupTrigger()) {
			triggerPopup(e);
			return;
		}

		targetRow = -1;
		dragOperation = OPERATION_NONE;
		dragStartPoint = e.getPoint();
	
		if (e.getButton() == MouseEvent.BUTTON1) {
			int row = getRowAtPoint(e.getPoint());
			if (row >= project.getTaskTableModel().getRowCount()) return;
	
			if (freeScrollUI.mousePressed(e)) return;
			if (printRangeUI.mousePressed(e)) return;
			if (restrictionUI.mousePressed(e)) return;
			if (taskUI.mousePressed(e)) return;
		}
	}

	@Override
	public void mouseReleased(MouseEvent e) {
		if (autoScrollTimer != null) {
			autoScrollTimer.cancel();
			autoScrollTimer = null;
		}

		if (e.isPopupTrigger()) {
			triggerPopup(e);
		}

		if (e.getButton() == MouseEvent.BUTTON1) {
			if (freeScrollUI.mouseReleased(e)) return;
			if (taskUI.mouseReleased(e)) return;
			if (restrictionUI.mouseReleased(e)) return;
			if (printRangeUI.mouseReleased(e)) return;
		}
	}

	private void triggerPopup(MouseEvent e) {
	}

	@Override
	public void mouseEntered(MouseEvent e) {
		requestFocus();
	}

	@Override
	public void mouseExited(MouseEvent e) {

	}

	private Calendar getCalendarAtPoint(Point p) {
		int offsetDate = p.x / getCellWidth();
		Calendar calendar = (Calendar)taskLinePane.getChartFromDate().clone();
		calendar.add(Calendar.DATE, offsetDate);
		return calendar;
	}

	/**
	 * 指定した座標に近く、最も早い営業日を返す
	 * 
	 * @param x
	 * @return
	 */
	private Calendar getEarlyBusinessDayAtPoint(Point p) {
		Calendar calendar = getCalendarAtPoint(p);
		for (int i = 0; i < 365; i++) {
			if (app.getProject().isHoliday(calendar)) {
				calendar.add(Calendar.DATE, -1);
			} else {
				break;
			}
		}
		return calendar;
	}

	/**
	 * 指定した座標に近く、最も遅い営業日を返す
	 * 
	 * @param x
	 * @return
	 */
	private Calendar getLateBusinessDayAtPoint(Point p) {
		Calendar calendar = getCalendarAtPoint(p);
		for (int i = 0; i < 365; i++) {
			if (app.getProject().isHoliday(calendar)) {
				calendar.add(Calendar.DATE, 1);
			} else {
				break;
			}
		}
		return calendar;
	}

	/**
	 * 指定した期間内の営業日数を返す。（範囲に開始日は含み、終了日は含まない)
	 * 
	 * @param from
	 * @param to
	 * @return
	 */
	private int getBusinessDayCount(Calendar from, Calendar to) {
		Calendar calendar = (Calendar) from.clone();
		int businessDayCount = 0;
		while (calendar.before(to)) {
			if (!app.getProject().isHoliday(calendar))
				businessDayCount++;
			calendar.add(Calendar.DATE, 1);
		}
		return businessDayCount;
	}

	protected Task getTaskAtPoint(Point p) {
		if (p == null)
			return null;

		int row = getRowAtPoint(p);
		Task task = project.getTaskAtRow(row);
		if (task == null)
			return null;
		Rectangle rect = getTaskHitRect(task);
		if (rect == null)
			return null;
		return rect.contains(p) ? task : null;
	}

	private int getRowAtPoint(Point point) {
		return point.y / getCellHeight();
	}

	@Override
	public void keyTyped(KeyEvent e) {
	}

	@Override
	public void keyPressed(KeyEvent e) {
		if (e.getKeyCode() == KeyEvent.VK_CONTROL || e.getKeyCode() == KeyEvent.VK_SHIFT) {
			if (e.isShiftDown() && e.isControlDown()) {
				cursorState.setCursor(CursorState.RESTRICTION_DELETE_CURSOR);
				pressOperation = OPERATION_RESTRICTION_DELETE;
			} else if (e.isControlDown()) {
				cursorState.setCursor(CursorState.RESTRICTION_CREATE_CURSOR);
				pressOperation = OPERATION_RESTRICTION_CREATE;
			}
		}
		freeScrollUI.keyPressed(e);
	}

	@Override
	public void keyReleased(KeyEvent e) {
		if (e.getKeyCode() == KeyEvent.VK_CONTROL || e.getKeyCode() == KeyEvent.VK_SHIFT) {
			if (e.isControlDown()) {
				cursorState.setCursor(CursorState.RESTRICTION_CREATE_CURSOR);
				pressOperation = OPERATION_RESTRICTION_CREATE;
			} else {
				cursorState.setDefaultCursor();
				pressOperation = OPERATION_NONE;
			}
		}
		freeScrollUI.keyReleased(e);
	}

	public class TaskUI {

		public boolean mousePressed(MouseEvent e) {
			Pair<Integer, Task> operation = getOperation(e);
			if (operation.getFirst() == OPERATION_NONE) {
				targetRow = getRowAtPoint(e.getPoint());
				dragOperation = operation.getFirst();
				return false;
			}
			
			if (operation.getFirst() == OPERATION_TASK_CREATE) {
				floatingTask = createTaskOperation(e);
				targetRow = getRowAtPoint(e.getPoint());
				dragOperation = OPERATION_TASK_PERIOD;
			} else {
				targetRow = getRowAtPoint(e.getPoint());
				dragOperation = operation.getFirst();
				floatingTask = project.getTaskAtRow(targetRow).clone();
			}
			return true;
		}

		public boolean mouseMoved(MouseEvent e) {
			Pair<Integer, Task> operation = getOperation(e);
			switch (operation.getFirst()) {
			case OPERATION_TASK_SLIDE:
				cursorState.setCursor(Cursor.E_RESIZE_CURSOR);
				return true;
			case OPERATION_TASK_COMPLETION:
				cursorState.setCursor(CursorState.COMPLETION_CURSOR);
				return true;
			case OPERATION_TASK_PERIOD:
				cursorState.setCursor(Cursor.E_RESIZE_CURSOR);
				return true;
			}
			return false;
		}

		public boolean mouseDragged(MouseEvent e) {
			if (dragOperation == OPERATION_TASK_SLIDE) {
				floatingTask
						.setStartDate(getLateBusinessDayAtPoint(e.getPoint()));
				project.updateChildrenPeriod(floatingTask, true);
				repaint();
				return true;
			} else if (dragOperation == OPERATION_TASK_PERIOD) {
				Calendar toCalendar = getLateBusinessDayAtPoint(e.getPoint());
				int count = getBusinessDayCount(floatingTask.getStartDate(),
						toCalendar);
				floatingTask.setPeriod(count);
				project.updateChildrenPeriod(floatingTask, true);
				repaint();
				return true;
			} else if (dragOperation == OPERATION_TASK_COMPLETION) {
				int offset = e.getX();
				Rectangle rect = getTaskRect(project.getTaskAtRow(targetRow));
				int completion = ((offset - rect.x) * 100 / rect.width / 5) * 5;
				completion = Math.min(Math.max(completion, 0), 100);
				floatingTask.setCompletion(completion);
				repaint();
				return true;
			}
			return false;
		}
		
		public boolean mouseReleased(MouseEvent e) {
			if (dragOperation != OPERATION_TASK_SLIDE &&
					dragOperation != OPERATION_TASK_PERIOD &&
					dragOperation != OPERATION_TASK_COMPLETION) return false;
			
			if (dragOperation == OPERATION_TASK_SLIDE) {
				floatingTask.setStartDate(getLateBusinessDayAtPoint(e.getPoint()));
			} else if (dragOperation == OPERATION_TASK_PERIOD) {
				Calendar toCalendar = getLateBusinessDayAtPoint(e.getPoint());
				int count = getBusinessDayCount(floatingTask.getStartDate(),
						toCalendar);
				floatingTask.setStartDate(floatingTask.getStartDate());
				floatingTask.setPeriod(count);
			} else if (dragOperation == OPERATION_TASK_COMPLETION) {
				int offset = e.getX();
				Rectangle rect = getTaskRect(project.getTaskAtRow(targetRow));
				int completion = ((offset - rect.x) * 100 / rect.width / 5) * 5;
				completion = Math.min(Math.max(completion, 0), 100);
				floatingTask.setCompletion(completion);
			}
			
			CompoundEdit compound = new CompoundEdit();
			project.getUndoManager().addEdit(compound);
			try {
				project.getController().setTaskAtIndex(floatingTask, project.getIndexFromRow(targetRow));
				project.update();
			} finally {
				compound.end();
				project.getTaskTableModel().fireTableChanged();
			}

			dragOperation = OPERATION_NONE;
			floatingTask = null;
			repaintProject();
			return true;
		}

		public Pair<Integer, Task> getOperation(MouseEvent e) {
			int row = getRowAtPoint(e.getPoint());
			Task task = project.getTaskAtRow(row);
			if (task != null && task.isGroup()) 
				return new Pair<Integer, Task>(OPERATION_NONE, null);
			if (task == null || task.getStartDate() == null)
				return new Pair<Integer, Task>(OPERATION_TASK_CREATE, null);

			Rectangle rect = getTaskHitRect(task);
			int margin = (getCellWidth() / 3) * 2;
			if (task.isMilestone()) {
				rect.x -= margin / 2;
				rect.width += margin;
			}

			int left = rect.x;
			int right = rect.x + rect.width;
			if (rect.contains(e.getPoint())) {
				if (e.getPoint().x < left + margin) {
					return new Pair<Integer, Task>(OPERATION_TASK_SLIDE, task);
				} else if (e.getPoint().x >= right - margin) {
					return new Pair<Integer, Task>(OPERATION_TASK_PERIOD, task);
				} else {
					return new Pair<Integer, Task>(OPERATION_TASK_COMPLETION, task);
				}
			}

			return new Pair<Integer, Task>(OPERATION_NONE, null);
		}

		public Task createTaskOperation(MouseEvent e) {
			int row = getRowAtPoint(e.getPoint());
			Task task = project.getTaskAtRow(row);
			if (task == null) {
				task = new Task();
				task.setName(Resource.get("projectNewTaskName"));
			}
			task.setStartDate(getEarlyBusinessDayAtPoint(e.getPoint()));
			task.setPeriod(0);
			task.setRealPeriod(0);
			return task;
		}
		
		private void drawTasks(Graphics g) {
			List<Task> visibleTasks = project.getVisibleTasks();
			for (Task task : visibleTasks) {
				drawTask(g, task);
			}
			
			if (floatingTask != null) {
				if (floatingTask.isMilestone()) {
					drawMilestone(g, floatingTask, true);
				} else {
					drawBasicTask(g, floatingTask, true);
				}
			}
		}

		private void drawTask(Graphics g, Task task) {
			if (task == null)
				return;
			
			if (task.isMilestone()) {
				drawMilestone(g, task, false);
			} else if (task.isGroup()) {
				drawParentTask(g, task);
			} else {
				drawBasicTask(g, task, false);
			}
		}

		/**
		 * マイルストーンを描画する
		 * 
		 * @param g
		 * @param task
		 * @param rect
		 */
		private void drawMilestone(Graphics g, Task task, boolean floating) {
			if (!(g instanceof Graphics2D)) return;
			
			if (task == null) return;
			Rectangle rect = getTaskRect(task);
			if (rect == null) return;
			
			Color groupColor = TaskLineDataPane.this.groupColor;
			if (floating) {
				groupColor = TaskLineDataPane.this.floatingMilestoneColor;
			}
			if (task.isCriticalPath()) {
				groupColor = TaskLineDataPane.this.criticalPathBorderColor;
			}
			
			g.setColor(groupColor);
			int half = rect.height / 2;
			Polygon polygon = new Polygon(new int[] { rect.x, rect.x - half,
					rect.x, rect.x + half }, new int[] { rect.y, rect.y + half,
					rect.y + rect.height, rect.y + half }, 4);
			g.fillPolygon(polygon);
		}

		/**
		 * 親タスクを描画する
		 * 
		 * @param g
		 * @param task
		 * @param rect
		 */
		private void drawParentTask(Graphics g, Task task) {
			if (task == null) return;
			Rectangle rect = getTaskRect(task);
			if (rect == null) return;

			g.setColor(groupColor);
			int half = rect.height * 2 / 3;
			Polygon polygon = new Polygon(new int[] { rect.x, rect.x,
					rect.x + half, rect.x + rect.width - half, rect.x + rect.width,
					rect.x + rect.width, rect.x + rect.width }, new int[] { rect.y,
					rect.y + rect.height, rect.y + rect.height - half,
					rect.y + rect.height - half, rect.y + rect.height, rect.y }, 6);
			g.fillPolygon(polygon);
		}

		/**
		 * 子タスクを描画する
		 * 
		 * @param g
		 * @param task
		 * @param rect
		 */
		private void drawBasicTask(Graphics g, Task task,  boolean floating) {
			if (task == null) return;
			Rectangle rect = getTaskRect(task);
			if (rect == null) return;

			Color borderColor = TaskLineDataPane.this.borderColor;
			Color fillColor = TaskLineDataPane.this.fillColor;
			Color completionColor = TaskLineDataPane.this.completionColor;
			if (task.isCriticalPath()) {
				borderColor = TaskLineDataPane.this.criticalPathBorderColor;
				fillColor = TaskLineDataPane.this.criticalPathFillColor;
				completionColor = TaskLineDataPane.this.criticalPathCompletionColor;
			}
			if (floating) {
				borderColor = TaskLineDataPane.this.floatingBorderColor;
				fillColor = TaskLineDataPane.this.floatingFillColor;
				completionColor = TaskLineDataPane.this.floatingCompletionColor;
			}

			g.setColor(fillColor);
			g.fillRect(rect.x, rect.y, rect.width, rect.height);
			if (task.getCompletion() > 0) {
				g.setColor(completionColor);
				g.fillRect(rect.x, rect.y, (rect.width) * task.getCompletion()
						/ 100, rect.height);
			}
			
			if (project.getRowByTask(task) != targetRow || dragOperation != OPERATION_TASK_COMPLETION) {
				StringBuilder builder = new StringBuilder();
	
				if (task.getMember() != null) {
					builder.append(task.getMember().getName() + " ");
				}
				if (task.getCompletion() > 0) {
					MessageFormat format = new MessageFormat("[{0}%]");
					builder.append(format.format(new Object[]{task.getCompletion()}));
				}
				if (builder.length() > 0) {
					g.setColor(fontColor);
					GraphicsUtil.drawStringLeft(g, builder.toString(), rect.x + rect.width + getCellWidth() / 2, rect.y - getCellHeight() / 4, 1, getCellHeight());
				}
			}

			g.setColor(borderColor);
			GraphicsUtil.drawRect(g, rect.x, rect.y, rect.width - 1, rect.height - 1);
		}

	}

	/**
	 * 制約条件操作の実装
	 */
	public class RestrictionUI {

		public boolean mousePressed(MouseEvent e) {
			if (pressOperation == OPERATION_RESTRICTION_CREATE ||
					pressOperation == OPERATION_RESTRICTION_DELETE) {
				
				Task task = getTaskAtPoint(e.getPoint());
				if (task == null)
					return true;
	
				if (pressOperation == OPERATION_RESTRICTION_CREATE) {
					targetRow = project.getRowByTask(task);
					floatingRestriction = new Restriction(task, null);
					dragOperation = OPERATION_RESTRICTION_CREATE;
					return true;
				} else if (pressOperation == OPERATION_RESTRICTION_DELETE) {
					removeRestriction(e);
					return true;
				}
				
				return true;
			}
			
			return false;
		}

		public boolean mouseDragged(MouseEvent e) {
			if (dragOperation != OPERATION_RESTRICTION_CREATE) return false;
				
			Task dest = getTaskAtPoint(e.getPoint());
			floatingRestriction.setDestTask(dest);
			repaint();
			
			return true;
		}

		public boolean mouseReleased(MouseEvent e) {
			if (dragOperation != OPERATION_RESTRICTION_CREATE)
				return false;

			addRestriction(e);
			dragOperation = OPERATION_NONE;
			return true;
		}
		
		private void addRestriction(MouseEvent e) {
			floatingRestriction = null;
			Task detTask = getTaskAtPoint(e.getPoint());
			if (detTask == null) return;
			
			CompoundEdit compound = new CompoundEdit();
			project.getUndoManager().addEdit(compound);
			try {
				project.getController().addRestriction(project.getTaskAtRow(targetRow), detTask);
				project.update();
			} finally {
				compound.end();
				repaintProject();
			}
		}

		private void removeRestriction(MouseEvent e) {
			Task task = getTaskAtPoint(e.getPoint());
			if (task == null) return;
			
			Rectangle rect = getTaskRect(task);
			if (rect == null) return;
	
			CompoundEdit compound = new CompoundEdit();
			project.getUndoManager().addEdit(compound);
			try {
				if (e.getX() < rect.x + rect.width / 2) {
					project.getController().removeDestRestrictions(task);
					repaintProject();
				} else {
					project.getController().removeSrcRestrictions(task);
					repaintProject();
				}
				project.update();
			} finally {
				compound.end();
			}
		}
		
		private void drawRestriction(Graphics g, Restriction restriction, boolean floating) {
			if (restriction == null || restriction.getSrcTask() == null || restriction.getDestTask() == null) return;
			
			drawRestriction(g, restriction.getSrcTask(), restriction.getDestTask(), floating);
		}

		/**
		 * 制約条件をひとつ描画する
		 * 
		 * @param g
		 * @param src
		 * @param dest
		 */
		private void drawRestriction(Graphics g, Task src, Task dest, boolean floating) {
			int half = getCellHeight() / 2;
			int quarter = getCellHeight() / 4;
			Rectangle fromRect = getTaskRect(src);
			Rectangle toRect = getTaskRect(dest);

			if (src == dest 
					|| src.getStartDate() == null 
					|| src.getRealPeriod() == null
					|| dest.getStartDate() == null)
				return;
			
			Color restrictionColor = TaskLineDataPane.this.restrictionColor;
			if (src.isCriticalPath() && dest.isCriticalPath()) {
				restrictionColor = TaskLineDataPane.this.criticalPathRestrictionColor;
			}
			if (floating) {
				restrictionColor = TaskLineDataPane.this.floatingRestrictionColor;
			}

			// 制約対象の開始日が制約元より右側にある場合は
			// 矢印の表示形式を変える
			long fromTime = src.getStartDate().getTimeInMillis()
					+ src.getRealPeriod().longValue() * 86400000;
			long toTime = dest.getStartDate().getTimeInMillis();

			if (fromTime >= toTime) {
				if (project.getRowByTask(src) < project.getRowByTask(dest)) {
					Point startingPoint = new Point(fromRect.x + fromRect.width,
							fromRect.y + fromRect.height / 2);
					Point endPoint = new Point(toRect.x + quarter, toRect.y);

					if (src.isMilestone())
						startingPoint.x -= fromRect.width;
					if (dest.isMilestone())
						endPoint.x -= quarter;

					g.setColor(restrictionColor);
					g.drawPolyline(new int[] { startingPoint.x,
							startingPoint.x + quarter, startingPoint.x + quarter,
							endPoint.x, endPoint.x }, new int[] { startingPoint.y,
							startingPoint.y, startingPoint.y + half,
							startingPoint.y + half, endPoint.y }, 5);
					g.fillPolygon(new int[] { endPoint.x, endPoint.x - quarter,
							endPoint.x + quarter }, new int[] { endPoint.y,
							endPoint.y - quarter, endPoint.y - quarter }, 3);
				} else {
					Point startingPoint = new Point(fromRect.x + fromRect.width,
							fromRect.y + fromRect.height / 2);
					Point endPoint = new Point(toRect.x + quarter, toRect.y
							+ toRect.height);

					if (src.isMilestone())
						startingPoint.x -= fromRect.width;
					if (dest.isMilestone())
						endPoint.x -= quarter;

					g.setColor(restrictionColor);
					g.drawPolyline(new int[] { startingPoint.x,
							startingPoint.x + quarter, startingPoint.x + quarter,
							endPoint.x, endPoint.x }, new int[] { startingPoint.y,
							startingPoint.y, startingPoint.y - half,
							startingPoint.y - half, endPoint.y }, 5);
					g.fillPolygon(new int[] { endPoint.x, endPoint.x - quarter - 1,
							endPoint.x + quarter + 1 }, new int[] { endPoint.y,
							endPoint.y + quarter + 1, endPoint.y + quarter + 1 }, 3);
				}
			} else {
				Point startingPoint = new Point(fromRect.x + fromRect.width,
						fromRect.y + fromRect.height / 2);
				Point endPoint = new Point(toRect.x, toRect.y + fromRect.height / 2);

				if (src.isMilestone())
					startingPoint.x -= fromRect.width;
				if (dest.isMilestone())
					endPoint.x -= quarter;

				g.setColor(restrictionColor);
				g.drawPolyline(new int[] { startingPoint.x,
						startingPoint.x + quarter, startingPoint.x + quarter,
						endPoint.x }, new int[] { startingPoint.y, startingPoint.y,
						endPoint.y, endPoint.y }, 4);
				g.fillPolygon(new int[] { endPoint.x, endPoint.x - quarter,
						endPoint.x - quarter }, new int[] { endPoint.y,
						endPoint.y + quarter, endPoint.y - quarter }, 3);
			}
		}

		/**
		 * 制約条件を描画する
		 * 
		 * @param g
		 */
		private void drawRestrictions(Graphics g) {
			List<Task> visibleTasks = project.getVisibleTasks();
			for (Task from : visibleTasks) {
				List<Task> visibleRestrictions = from.getVisibleRestrictions();
				for (Task to : visibleRestrictions) {
					drawRestriction(g, from, to, false);
				}
			}
			
			// 操作中の制約も描画する
			drawRestriction(g, floatingRestriction, true);
		}

	}

	/**
	 * 印刷範囲周りの実装
	 */
	public class PrintRangeUI {
		Color printRangeFirstColor 		= new Color(0xF0, 0x50, 0x50);
		Color printRangeColor 			= new Color(0xF0, 0xA0, 0xA0);
		Color floatingPrintRangeColor 	= Color.black;

		public void drawPrintRange(Graphics g) {
			if (!printRangeMode) return;
			
			List<Rectangle> ranges = getPrintRanges();

			int page = ranges.size();
			Collections.reverse(ranges);
			for (Rectangle r : ranges) {
				drawPrintRange(g, page, r, false);
				page--;
			}
			
			if (floatingPrintCellSize != null) {
				drawPrintRange(g, 1, getFirstPrintRange(floatingPrintCellSize), true);
			}
		}
		
		protected void drawPrintRange(Graphics g, int page, Rectangle r, boolean floating) {
			Color printRangeColor = this.printRangeColor;
			if (page == 1) printRangeColor = this.printRangeFirstColor;
			if (floating) printRangeColor = this.floatingPrintRangeColor;
			
			g.setColor(printRangeColor);
			g.drawRect(r.x, r.y, r.width, r.height);
			g.drawRect(r.x + 1, r.y + 1, r.width - 2, r.height - 2);
			g.drawRect(r.x - 1, r.y - 1, r.width + 2, r.height + 2);

			Font oldFont = g.getFont();
			Font newFont = new Font(oldFont.getFontName(), Font.BOLD, 32);
			g.setFont(newFont);
			String str = Resource.get("chartPage") + page;
			Shape oldClip = g.getClip();
			g.clipRect(r.x, r.y, r.width, r.height);
			GraphicsUtil.drawStringCenter(g, str, r.x, r.y, r.width, r.height);
			g.setClip(oldClip);
			g.setFont(oldFont);
		}

		public boolean mousePressed(MouseEvent e) {
			int printRangeOperation = getOperation(e);
			if (printRangeOperation == OPERATION_NONE)
				return false;

			floatingPrintCellSize = (Dimension)project.getPrint().getPrintCellSize().clone();
			dragOperation = printRangeOperation;
			return true;
		}

		public boolean mouseMoved(MouseEvent e) {
			int operation = getOperation(e);
			switch (operation) {
			case OPERATION_PRINT_RANGE:
				cursorState.setCursor(Cursor.SE_RESIZE_CURSOR);
				return true;
			case OPERATION_PRINT_RANGE_X:
				cursorState.setCursor(Cursor.E_RESIZE_CURSOR);
				return true;
			case OPERATION_PRINT_RANGE_Y:
				cursorState.setCursor(Cursor.S_RESIZE_CURSOR);
				return true;
			}
			return false;
		}

		public boolean mouseDragged(MouseEvent e) {
			if (dragOperation == OPERATION_PRINT_RANGE_X) {
				Rectangle rect = getFirstPrintRange(floatingPrintCellSize);
				if (rect == null)
					return false;
				floatingPrintCellSize.width = Math.max(10,
						(e.getX() - rect.x) / getCellWidth());
				repaintProject();
				return true;
			} else if (dragOperation == OPERATION_PRINT_RANGE_Y) {
				Rectangle rect = getFirstPrintRange(floatingPrintCellSize);
				if (rect == null)
					return false;
				int pointY = Math.min(e.getY(), project.getTaskTableModel().getRowCount() * getCellHeight());
				floatingPrintCellSize.height = Math.max(10,
						(pointY - rect.y) / getCellHeight());
				repaintProject();
				return true;
			} else if (dragOperation == OPERATION_PRINT_RANGE) {
				Rectangle rect = getFirstPrintRange(floatingPrintCellSize);
				if (rect == null)
					return false;
				int pointY = Math.min(e.getY(), project.getTaskTableModel().getRowCount() * getCellHeight());
				floatingPrintCellSize.width = Math.max(10,
						(e.getX() - rect.x) / getCellWidth());
				floatingPrintCellSize.height = Math.max(10,
						(pointY - rect.y) / getCellHeight());
				repaintProject();
				return true;
			}
			return false;
		}

		public boolean mouseReleased(MouseEvent e) {
			if (dragOperation == OPERATION_PRINT_RANGE_X ||
					dragOperation == OPERATION_PRINT_RANGE_Y ||
					dragOperation == OPERATION_PRINT_RANGE) {
				project.getController().setPrintCellSize(floatingPrintCellSize);
				floatingPrintCellSize = null;
				repaintProject();
				dragOperation = OPERATION_NONE;
				return true;
			}
			return false;
		}

		public int getOperation(MouseEvent e) {
			if (!printRangeMode)
				return OPERATION_NONE;

			Rectangle rect = getFirstPrintRange();
			if (rect != null
					&& e.getX() >= rect.x + rect.width - getCellWidth() / 2
					&& e.getX() < rect.x + rect.width
					&& e.getY() >= rect.y + rect.height - getCellHeight() / 2
					&& e.getY() < rect.y + rect.height) {
				return OPERATION_PRINT_RANGE;
			} else if (rect != null
					&& e.getX() >= rect.x + rect.width - getCellWidth() / 2
					&& e.getX() < rect.x + rect.width && e.getY() >= rect.y
					&& e.getY() < rect.y + rect.height - getCellHeight() / 2) {
				return OPERATION_PRINT_RANGE_X;
			} else if (rect != null && e.getX() >= rect.x
					&& e.getX() < rect.x + rect.width - getCellWidth() / 2
					&& e.getY() >= rect.y + rect.height - getCellHeight() / 2
					&& e.getY() < rect.y + rect.height) {
				return OPERATION_PRINT_RANGE_Y;
			}

			return OPERATION_NONE;
		}

		/**
		 * 最初の印刷範囲を取得する
		 * 
		 * @return
		 */
		public Rectangle getFirstPrintRange() {
			Calendar dataRangeFirstDate = project.getFirstDate();
			if (dataRangeFirstDate == null)
				return null;
			
			return getFirstPrintRange(project.getPrint().getPrintCellSize());
		}
		
		/**
		 * 範囲を指定して最初の印刷範囲を取得する
		 * @param printCellSize
		 * @return
		 */
		public Rectangle getFirstPrintRange(Dimension printCellSize) {
			Calendar dataRangeFirstDate = project.getFirstDate();
			if (dataRangeFirstDate == null)
				return null;

			Calendar printRangeFirstDate = dataRangeFirstDate;
			printRangeFirstDate.add(Calendar.DATE, -1);

			int offsetX = CalendarUtil.subDate(printRangeFirstDate,
					taskLinePane.getChartFromDate())
					* getCellWidth();

			return new Rectangle(offsetX, 0, printCellSize.width
					* getCellWidth(), printCellSize.height * getCellHeight());
		}

		/**
		 * 印刷範囲をページごとに取得する
		 * 
		 * @return
		 */
		public List<Rectangle> getPrintRanges() {
			List<Rectangle> result = new ArrayList<Rectangle>();

			Calendar dataRangeFirstDate = project.getFirstDate();
			int dataRangePeriod = project.getWholePeriod();
			int rowCount = project.getRowCount();
			if (dataRangeFirstDate == null)
				return result;

			Calendar printRangeFirstDate = dataRangeFirstDate;
			printRangeFirstDate.add(Calendar.DATE, -1);
			int printRangePeriod = dataRangePeriod + 2;

			int offsetX = CalendarUtil.subDate(printRangeFirstDate,
					taskLinePane.getChartFromDate())
					* getCellWidth();
			Dimension printCellSize = project.getPrint().getPrintCellSize();

			for (int cursorY = 0; cursorY < rowCount; cursorY += printCellSize.height) {
				for (int cursorX = 0; cursorX < printRangePeriod; cursorX += printCellSize.width) {
					Rectangle r = new Rectangle(offsetX + cursorX
							* getCellWidth(), cursorY * getCellHeight(),
							printCellSize.width * getCellWidth(),
							printCellSize.height * getCellHeight());
					result.add(r);
				}
			}

			return result;
		}

	}
	
	/**
	 * 制約クラス（ドラッグ中のフロート表示にのみ使用する）
	 */
	class Restriction {
		Task srcTask;
		Task destTask;
		
		public Restriction(Task src, Task dest) {
			this.srcTask = src;
			this.destTask = dest;
		}
		
		public Task getSrcTask() {
			return srcTask;
		}
		public void setSrcTask(Task srcTask) {
			this.srcTask = srcTask;
		}
		public Task getDestTask() {
			return destTask;
		}
		public void setDestTask(Task destTask) {
			this.destTask = destTask;
		}
	}
	
	/**
	 * 稲妻線UI
	 *
	 */
	class ProgressLineUI {
		public void drawProgressLine(Graphics g) {
			if (!progressLineMode) return;
			Calendar now = CalendarUtil.toDateCalendar(Calendar.getInstance());
			drawProgressLine(g, CalendarUtil.toDateCalendar(now));
		}
		
		public void drawProgressLine(Graphics g, Calendar now) {
			if (!progressLineMode) return;
			if (!(g instanceof Graphics)) return;
			Graphics2D g2d = (Graphics2D)g;
			
			int dateOffset = (int) CalendarUtil.subDate(now, taskLinePane
					.getChartFromDate());
			int drawableHeight = project.getTaskTableModel().getRowCount()
					* getCellHeight();
			
			Path2D.Float path = new Path2D.Float();
			path.moveTo(dateOffset * getCellWidth(), 0.75);
			
			for (int row = 0; row < project.getRowCount(); row ++) {
				path.lineTo(dateOffset * getCellWidth(), row * getCellHeight());
				
				Task task = project.getTaskAtRow(row);
				if (task.isGroup()) {
					// スルー
				} else if (task != null && task.getStartDate() != null && 
						CalendarUtil.subDate(task.getStartDate(), now) > 0 &&
						task.getCompletion() == 0) {
					// スルー
				} else if (task != null && task.getEndDate() != null &&
						CalendarUtil.subDate(now, task.getEndDate()) > 0 &&
						task.getCompletion() == 100) {
					// スルー
				} else {
					Point2D.Float progressPoint = getProgressPoint(task, row);
					if (progressPoint != null) {
						path.lineTo(progressPoint.x, progressPoint.y);
					}
				}
				path.lineTo(dateOffset * getCellWidth(), (row + 1) * getCellHeight());
			}
			path.lineTo(dateOffset * getCellWidth(), drawableHeight - 0.75);
			g2d.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
			g2d.setStroke(new BasicStroke(2.0f, BasicStroke.CAP_SQUARE, BasicStroke.JOIN_MITER));
			g2d.setColor(progressLineColor);
			g2d.draw(path);
			g2d.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_DEFAULT);
		}

		private Point2D.Float getProgressPoint(Task task, int row) {
			Rectangle rect = getTaskRect(task);
			if (rect == null) return null;
			int pointY = row * getCellHeight() + getCellHeight() / 2;
			int pointX = rect.x + rect.width * task.getCompletion() / 100;
			return new Point2D.Float((float)pointX, (float)pointY);
		}
	}
	
	class FreeScrollUI {

		/** フリースクロール開始位置 */
		Point freeScrollStartPoint;

		public void keyPressed(KeyEvent e) {
			if (e.getKeyCode() == KeyEvent.VK_SPACE && pressOperation == OPERATION_NONE) {
				cursorState.setCursor(Cursor.HAND_CURSOR);
				pressOperation = OPERATION_FREE_SCROLL;
			}
		}

		public void keyReleased(KeyEvent e) {
			if (e.getKeyCode() == KeyEvent.VK_SPACE) {
				cursorState.setDefaultCursor();
				pressOperation = OPERATION_NONE;
			}
		}

		public boolean mousePressed(MouseEvent e) {
			if (pressOperation == OPERATION_FREE_SCROLL) {
				dragStartPoint = e.getLocationOnScreen();
				freeScrollStartPoint = new Point(
						app.getTaskLineRootPane().getHorizontalScrollBar().getValue(),
						app.getTaskListRootPane().getVerticalScrollBar().getValue());
				dragOperation = OPERATION_FREE_SCROLL;
				return true;
			}
			return false;
		}

		public boolean mouseDragged(MouseEvent e) {
			if (dragOperation == OPERATION_FREE_SCROLL) {
				Point pressPoint = e.getLocationOnScreen();
				Dimension moveDistance = new Dimension(pressPoint.x - dragStartPoint.x, pressPoint.y - dragStartPoint.y);
				JScrollBar vbar = app.getTaskListRootPane().getVerticalScrollBar();
				JScrollBar hbar = app.getTaskLineRootPane().getHorizontalScrollBar();
				
				int vvalue = freeScrollStartPoint.y - moveDistance.height;
				int hvalue = freeScrollStartPoint.x - moveDistance.width;
				hvalue = Math.min(hbar.getMaximum() - hbar.getVisibleAmount() - 1, Math.max(1, hvalue));
				
				vbar.setValue(vvalue);
				hbar.setValue(hvalue);
				return true;
			}
			return false;
		}

		public boolean mouseReleased(MouseEvent e) {
			if (dragOperation == OPERATION_FREE_SCROLL) {
				dragOperation = OPERATION_NONE;
				return true;
			}
			return false;
		}

	}
	
	class AutoScrollHandler extends TimerTask {
		
		public final static int SCROLL_INTERVAL = 25;
		public final static int EDGE_LENGTH = 32;
		
		@Override
		public void run() {
			SwingUtilities.invokeLater(new Runnable(){
				@Override
				public void run() {
					JViewport viewport = app.getTaskLineRootPane().getViewport();
					
					Point absolutePoint = MouseInfo.getPointerInfo().getLocation();
					Point vpePoint = viewport.getLocationOnScreen();
					Point relativePoint = new Point(absolutePoint.x - vpePoint.x, absolutePoint.y - vpePoint.y);
//					System.out.println("relativePoint: " + relativePoint);
					
					if (relativePoint.x <= EDGE_LENGTH) {
						JScrollBar hbar = app.getTaskLineRootPane().getHorizontalScrollBar();
						int scrollLength = (EDGE_LENGTH - relativePoint.x) / 3;
						int hvalue = Math.max(0, hbar.getValue() - scrollLength);
						hbar.setValue(hvalue);
						fireMouseDragEvent();
					} else if (relativePoint.x >= viewport.getWidth() - EDGE_LENGTH) {
						JScrollBar hbar = app.getTaskLineRootPane().getHorizontalScrollBar();
						int scrollLength = (relativePoint.x - (viewport.getWidth() - EDGE_LENGTH)) / 3;
						int hvalue = Math.min(hbar.getMaximum() - hbar.getVisibleAmount(), hbar.getValue() + scrollLength);
						hbar.setValue(hvalue);
						fireMouseDragEvent();
					}
					if (relativePoint.y <= EDGE_LENGTH) {
						JScrollBar vbar = app.getTaskListRootPane().getVerticalScrollBar();
						int scrollLength = (EDGE_LENGTH - relativePoint.y) / 3;
						vbar.setValue(vbar.getValue() - scrollLength);
						fireMouseDragEvent();
					} else if (relativePoint.y >= viewport.getHeight() - EDGE_LENGTH) {
						JScrollBar vbar = app.getTaskListRootPane().getVerticalScrollBar();
						int scrollLength = (relativePoint.y - (viewport.getHeight() - EDGE_LENGTH)) / 3;
						vbar.setValue(vbar.getValue() + scrollLength);
						fireMouseDragEvent();
					}
				}
			});
		}
		
		public void fireMouseDragEvent() {
			Point point = MouseInfo.getPointerInfo().getLocation();
			point.x -= getLocationOnScreen().x;
			point.y -= getLocationOnScreen().y;
			MouseEvent event = new MouseEvent(TaskLineDataPane.this, MouseEvent.MOUSE_DRAGGED, 0, 0, point.x, point.y, 0, false);
			processMouseMotionEvent(event);
		}
	}
}
