﻿/*
 *	Fugashi Project
 *
 *	This software is distributed under a BSD-style license.
 *	See license.txt for more information.
 */

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

namespace Fugashi
{
    public partial class MainForm : Form
    {
        #region fields
        private BitSwitch[] bits;
        private int length = 4;
        #endregion

        #region constructor
        public MainForm()
        {
            InitializeComponent();

            bits = new BitSwitch[64];
            foreach (Control c in table.Controls)
            {
                if (c is BitSwitches4)
                {
                    BitSwitches4 bs = c as BitSwitches4;
                    int i = bs.StartBit;
                    bits[i + 0] = bs.B0;
                    bits[i + 1] = bs.B1;
                    bits[i + 2] = bs.B2;
                    bits[i + 3] = bs.B3;
                }
            }
            foreach (BitSwitch b in bits)
            {
                b.StateChanged += BitChanged;
            }
            Font mono = new Font(FontFamily.GenericMonospace, textHex.Font.Size);
            textHex.Font = mono;
            textValue.Font = mono;

            PerformAutoScale();
        }
        #endregion

        #region event handlers

        private void MainForm_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.Alt)
            {
                switch (e.KeyCode)
                {
                    case Keys.V:
                        textValue.Focus();
                        break;
                    case Keys.H:
                        textHex.Focus();
                        break;
                    default:
                        return;
                }
                e.SuppressKeyPress = true;
            }
        }

        private void radioUint_CheckedChanged(object sender, EventArgs e)
        {
            if (radioUint.Checked)
            {
                UpdateValue(GetBytes());
            }
        }

        private void radioInt_CheckedChanged(object sender, EventArgs e)
        {
            if (radioInt.Checked)
            {
                UpdateValue(GetBytes());
            }
        }

        private void radioFloat_CheckedChanged(object sender, EventArgs e)
        {
            if (radioFloat.Checked)
            {
                radio1B.Enabled = false;
                radio2B.Enabled = false;
                if (radio1B.Checked || radio2B.Checked)
                {
                    radio4B.Checked = true;
                }
                SetForFloatingPoint(radio8B.Checked);
                UpdateValue(GetBytes());
            }
            else
            {
                SetForInteger();
                radio1B.Enabled = true;
                radio2B.Enabled = true;
            }
        }

        private void MainForm_Load(object sender, EventArgs e)
        {
            radioUint.Checked = true;
            radio4B.Checked = true;
            BitChanged(null, null);
        }

        private void radio8B_CheckedChanged(object sender, EventArgs e)
        {
            if (radio8B.Checked)
            {
                SetLength(8);
                if (radioFloat.Checked)
                {
                    SetForFloatingPoint(true);
                    UpdateValue(GetBytes());
                }
            }
        }

        private void radio4B_CheckedChanged(object sender, EventArgs e)
        {
            if (radio4B.Checked)
            {
                SetLength(4);
                if (radioFloat.Checked)
                {
                    SetForFloatingPoint(false);
                    UpdateValue(GetBytes());
                }
            }
        }

        private void radio2B_CheckedChanged(object sender, EventArgs e)
        {
            if (radio2B.Checked)
            {
                SetLength(2);
            }
        }

        private void radio1B_CheckedChanged(object sender, EventArgs e)
        {
            if (radio1B.Checked)
            {
                SetLength(1);
                UpdateValue(GetBytes());
            }
        }

        
        private void buttonInc_Click(object sender, EventArgs e)
        {
            if (radioFloat.Checked)
            {
                if (length == 4)
                {
                    int value = BitConverter.ToInt32(GetBytes(), 0);
                    if (value == -0x80000000)
                    {
                        value = 0;
                    }
                    else
                    {
                        value += value >= 0 ? 1 : -1;
                    }
                    UpdateAll(BitConverter.GetBytes(value));
                }
                else
                {
                    long value = BitConverter.ToInt64(GetBytes(), 0);
                    if (value == (1L << 63))
                    {
                        value = 0;
                    }
                    else
                    {
                        value += value >= 0 ? 1 : -1;
                    }
                    UpdateAll(BitConverter.GetBytes(value));
                }
            }
            else
            {
                ulong value = BitConverter.ToUInt64(GetBytes(), 0);
                UpdateAll(BitConverter.GetBytes(++value));
            }
        }

        private void buttonDec_Click(object sender, EventArgs e)
        {
            if (radioFloat.Checked)
            {
                if (length == 4)
                {
                    int value = BitConverter.ToInt32(GetBytes(), 0);
                    if (value == 0)
                    {
                        value = 1 << 31;
                    }
                    else
                    {
                        value -= value >= 0 ? 1 : -1;
                    }
                    UpdateAll(BitConverter.GetBytes(value));
                }
                else
                {
                    long value = BitConverter.ToInt64(GetBytes(), 0);
                    if (value == 0)
                    {
                        value = 1L << 63;
                    }
                    else
                    {
                        value -= value >= 0 ? 1 : -1;
                    }
                    UpdateAll(BitConverter.GetBytes(value));
                }
            }
            else
            {
                ulong value = BitConverter.ToUInt64(GetBytes(), 0);
                UpdateAll(BitConverter.GetBytes(--value));
            }
        }

        private void buttonClr_Click(object sender, EventArgs e)
        {
            UpdateAll(BitConverter.GetBytes((ulong)0));
        }

        private void buttonNot_Click(object sender, EventArgs e)
        {
            ulong value = BitConverter.ToUInt64(GetBytes(), 0);
            UpdateAll(BitConverter.GetBytes(~value));
        }

        private void buttonLsh_Click(object sender, EventArgs e)
        {
            ulong value = BitConverter.ToUInt64(GetBytes(), 0);
            UpdateAll(BitConverter.GetBytes(value << 1));
        }

        private void buttonRsh_Click(object sender, EventArgs e)
        {
            if (radioInt.Checked)
            {
                long value = ToInt64(GetBytes());
                UpdateAll(BitConverter.GetBytes(value >> 1));
            }
            else
            {
                ulong value = BitConverter.ToUInt64(GetBytes(), 0);
                UpdateAll(BitConverter.GetBytes(value >> 1));
            }
        }

        private void textHex_TextChanged(object sender, EventArgs e)
        {
            if (!textHex.Focused)
            {
                return;
            }
            ulong value;
            UInt64.TryParse(textHex.Text,
                System.Globalization.NumberStyles.HexNumber, null, out value);
            byte[] b = BitConverter.GetBytes(value);
            UpdateBits(b);
            UpdateValue(b);
        }

        private void textValue_TextChanged(object sender, EventArgs e)
        {
            string text = textValue.Text;
            if (!textValue.Focused && sender != null)
            {
                return;
            }
            byte[] b;
            if (radioUint.Checked)
            {
                ulong value;
                UInt64.TryParse(text, out value);
                b = BitConverter.GetBytes(value);
            }
            else if (radioInt.Checked)
            {
                long value;
                Int64.TryParse(text, out value);
                b = BitConverter.GetBytes(value);
            }
            else if (radioFloat.Checked && length == 4)
            {
                float value;
                Single.TryParse(text, out value);
                b = BitConverter.GetBytes(value);
            }
            else if (radioFloat.Checked && length == 8)
            {
                double value;
                Double.TryParse(text, out value);
                b = BitConverter.GetBytes(value);
            }
            else
            {
                throw new Exception();
            }
            UpdateBits(b);
            UpdateHex(b);
        }

        private void textValue_Leave(object sender, EventArgs e)
        {
            UpdateValue(GetBytes());
        }

        private void textHex_Leave(object sender, EventArgs e)
        {
            UpdateHex(GetBytes());
        }

        private void BitChanged(object sender, EventArgs e)
        {
            byte[] b = GetBytes();
            UpdateValue(b);
            UpdateHex(b);
        }

        #endregion

        #region private methods

        private void SetForFloatingPoint(bool isDouble)
        {
            for (int i = 23; i < 31; ++i)
            {
                bits[i].IsExponential = !isDouble;
            }
            for (int i = 52; i < 63; ++i)
            {
                bits[i].IsExponential = isDouble;
            }
        }

        private void SetForInteger()
        {
            for (int i = 0; i < 64; ++i)
            {
                bits[i].IsExponential = false;
            }
        }

        private byte[] GetBytes()
        {
            byte[] b = new Byte[8];
            int i = 0;
            for (int j = 0; j < 8; ++j)
            {
                b[j] = 0;
                i += 8;
                for (int k = 0; k < 8; ++k)
                {
                    b[j] = (byte)(b[j] << 1 | (bits[--i].State ? 1 : 0));
                }
                i += 8;
            }
            return b;
        }

        private long ToInt64(byte[] bytes)
        {
            switch (length)
            {
                case 1:
                    return (sbyte)bytes[0];
                case 2:
                    return BitConverter.ToInt16(bytes, 0);
                case 4:
                    return BitConverter.ToInt32(bytes, 0);
                case 8:
                    return BitConverter.ToInt64(bytes, 0);
            }
            return 0;
        }

        private ulong ToUInt64(byte[] bytes)
        {
            switch (length)
            {
                case 1:
                    return bytes[0];
                case 2:
                    return BitConverter.ToUInt16(bytes, 0);
                case 4:
                    return BitConverter.ToUInt32(bytes, 0);
                case 8:
                    return BitConverter.ToUInt64(bytes, 0);
            }
            return 0;
        }

        private void SetLength(int length)
        {
            this.length = length;
            int b = length * 8;
            for (int i = 8; i < b; ++i)
            {
                bits[i].Enabled = true;
            }
            for (int i = b; i < 64; ++i)
            {
                bits[i].Enabled = false;
            }
            if (radioFloat.Checked)
            {
                textValue_TextChanged(null, null);
            }
            else
            {
                BitChanged(null, null);
            }
        }

        private void UpdateValue(byte[] bytes)
        {
            if (radioUint.Checked)
            {
                textValue.Text = ToUInt64(bytes).ToString();
            }
            else if (radioInt.Checked)
            {
                textValue.Text = ToInt64(bytes).ToString();
            }
            else if (radioFloat.Checked)
            {
                if (length == 4)
                {
                    float value = BitConverter.ToSingle(bytes, 0);
                    textValue.Text = value.ToString("g8");
                }
                else
                {
                    double value = BitConverter.ToDouble(bytes, 0);
                    textValue.Text = value.ToString("g17");
                }
            }
        }

        private void UpdateHex(byte[] bytes)
        {
            textHex.Text = "";
            for (int i = length - 1; i >= 0; --i)
            {
                textHex.Text += bytes[i].ToString("X2");
            }
        }

        private void UpdateBits(byte[] bytes)
        {
            int i = 0;
            for (int j = 0; j < length; ++j)
            {
                int mask = 1;
                for (int k = 0; k < 8; ++k)
                {
                    bits[i++].SetStateWithoutEvent((bytes[j] & mask) != 0);
                    mask <<= 1;
                }
            }
        }

        private void UpdateAll(byte[] bytes)
        {
            UpdateBits(bytes);
            UpdateValue(bytes);
            UpdateHex(bytes);
        }

        #endregion
    }
}
