/*
 * Galatea Dialog Studio:
 * (c)2009 Takuya NISHIMOTO
 * Based on Phoenix By Takuya NISHIMOTO and Mitsuhiro KIZU
 *
 * $Id: SubProcess.java,v 1.9 2009/01/22 08:09:14 nishimoto Exp $
 */
package galatea.util;


import galatea.logger.Logger;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.UnsupportedEncodingException;

public class SubProcess {
	private Logger logger = new Logger(this.getClass()); 
	private String module_command;   // command of submodule
	private Runtime rt;              // Runtime object of current application
	private Process child;           // Process object of submodule
	private InputStream out;         // stdout of submodule
	private InputStream err;         // stderr of submodule
	private OutputStream in;         // stdin  of submodule
	private PrintStream stdin;
	private BufferedReader stdout;
	private BufferedReader stderr;
	private boolean running = false;
	private String error_;
	private int maxCommandLength_ = 99999;
	private String dirPath_;
	
//	private SubProcess(String com) {
//		maxCommandLength_ = Property.getForPlatformAsInt( "SubProcess.MaxCommandLength", 99999);
//		error_ = "";
//		module_command = com;
//	}
	
	/**
	 * SubProcess Manager
	 * @param command
	 * @param dirPath : current directory path
	 */
	public SubProcess(String command, String dirPath) {
		maxCommandLength_ = Property.getForPlatformAsInt( "SubProcess.MaxCommandLength", 99999);
		error_ = "";
		module_command = command;
		dirPath_ = dirPath;
	}
	
	/**
	 * do Runtime.exec
	 * @return false if error
	 */
	public boolean run() {
		return run(dirPath_);
	}
	
	/**
	 * do Runtime.exec
	 * @param dirpath : current directory path
	 * @return false if error
	 */
	private boolean run(String dirpath) {
		if (module_command.length() == 0){
			String s ="SubProcess.run(): 'module_command' is undefined."; 
			logger.err(s);
			error_ += s;
			return false;
		}
		if (module_command.length() > maxCommandLength_){
			String s = "SubProcess.run(): command too long:" + module_command + "\n";
			logger.err(s);
			error_ += s;
			return false;
		}
		rt = Runtime.getRuntime();
		if (rt == null) {
			logger.err("SubProcess.run(): getRuntime() failed.");
			error_ += "SubProcess.run(): getRuntime() failed.";
			return false;
		}
		File dir = new File(dirpath);
		String[] com = module_command.split(" ");
		try {
			child = rt.exec(com, null, dir);
		} catch (IOException e) {
			error_ += "SubProcess.run(): Runtime.exec() failed.\n";
			error_ += "dir=" + dir.toString() + "\n";
			error_ += e.toString();
			logger.err("SubProcess.run(): Runtime.exec() failed.");
			return false;
		}
		in = child.getOutputStream();
		out = child.getInputStream();
		err = child.getErrorStream();
		try {
			stdin = new PrintStream(in, true, Util.getEncoding());
		} catch (UnsupportedEncodingException e) {
			logger.print(e.toString());
			stdin = new PrintStream(in, true);
		}
		try {
			stdout = new BufferedReader(new InputStreamReader(out, Util.getEncoding()));
		} catch (UnsupportedEncodingException e) {
			logger.print(e.toString());
			stdout = new BufferedReader(new InputStreamReader(out));
		}
		try {
			stderr = new BufferedReader(new InputStreamReader(err, Util.getEncoding()));
		} catch (UnsupportedEncodingException e) {
			logger.print(e.toString());
			stderr = new BufferedReader(new InputStreamReader(err));
		}
		running = true;
		return true;
	}
	
	public boolean isRunning() {
		return running;
	}
	
	public void sendToStdin(String str) {
		logger.print("ToStdin: "+str, 9);
		stdin.println(str);
		stdin.flush();
	}
	
	/**
	 * receive from stdout (non-blocking)
	 * @return string or null (if exception)
	 */
	public String receiveFromStdoutNB() {
		String s;
		try {
			if (stdout != null && stdout.ready()) {
				s = stdout.readLine();
				logger.print("FromStdoutNB: "+s, 9);
			} else {
				s = "";
			}
		} catch(IOException e) {
			error_ += e.toString();
			logger.err("SubProcess.FromStdout(): BufferedReader.readLine() failed.");
			return null;
		}
		return s;
	}
	
	/**
	 * receive from stdout (non-blocking)
	 * @return string or "" (if exception)
	 */
	public String receiveFromStdoutNB2() {
		String s = receiveFromStdoutNB();
		if (s == null) return "";
		return s;
	}
	
	public boolean isStdoutReadable() {
		try {
			return stdout.ready();
		} catch (IOException e) {
		}
		return true;
	}
	
	/**
	 * receive from stderr (non-blocking)
	 * @return string or null (if exception)
	 */
	public String receiveFromStderrNB() {
		String s;
		try {
			if (stderr != null && stderr.ready())
				s = stderr.readLine();
			else
				s = "";
		} catch (IOException e) {
			error_ += e.toString();
			logger.err("SubProcess.FromStderr(): BufferedReader.readLine() failed.");
			return null;
		}
		return s;
	}
	
	public void waitForTerminate() {
		try {
			child.waitFor();
		} catch (InterruptedException e) {
			logger.print(e.toString()); // e.printStackTrace();
		}
		running = false;
	}
	
	public void destroy() {
		child.destroy();
		running = false;
	}
	
	public int exitValue() throws IllegalThreadStateException{
		return child.exitValue();
	}

	public String getError() {
		return error_;
	}
	
	public void clearError() {
		error_ = "";
	}
	
}
