/*
 * Copyright (C) 2008-2011 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
 * Copyright (C) 2011 OgakiSoft
 * 
 * Supported multiple strokes, mesh(grid) visible on the gesture area.
 * Excluded the keyboard area from the gesture area, to use as IME.
 * Deleted some methods by comment out.
 */

package ogakisoft.android.gesture.reform;

import java.util.ArrayList;
import java.util.List;

import ogakisoft.android.gestureime.R;
import ogakisoft.android.util.LOG;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.Rect;
import android.graphics.RectF;
import android.os.SystemClock;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.view.animation.AccelerateDecelerateInterpolator;
import android.view.animation.AnimationUtils;
import android.widget.FrameLayout;

/**
 * View Class for showing gesture-data
 * 
 * @author The Android Open Source Project
 * @author noritoshi ogaki
 * @version 1.1
 */
public class GestureView extends FrameLayout {
	private class FadeOutRunnable implements Runnable {
		boolean fireActionPerformed;
		float interpolatedTime = 0;

		public void run() {
			if (mIsFadingOut) {
				final long now = AnimationUtils.currentAnimationTimeMillis();
				final long duration = now - mFadingStart;
				if (duration > mFadeDuration) {
					if (fireActionPerformed) {
						fireOnGesturePerformed();
					}
					mPreviousWasGesturing = false;
					mIsFadingOut = false;
					mFadingHasStarted = false;
					mPath.rewind();
					mCurrentGesture = null;
					setPaintAlpha(255);
				} else {
					mFadingHasStarted = true;
					interpolatedTime = Math.max(0.0f,
							Math.min(1.0f, duration / (float) mFadeDuration));
					mFadingAlpha = 1.0f - mInterpolator
							.getInterpolation(interpolatedTime);
					setPaintAlpha((int) (255 * mFadingAlpha));
					postDelayed(this, FADE_ANIMATION_RATE);
				}
			} else {
				fireOnGesturePerformed();
				mFadingHasStarted = false;
				mPath.rewind();
				mCurrentGesture = null;
				mPreviousWasGesturing = false;
				setPaintAlpha(255);
			}
			invalidate();
		}
	}

	public static interface OnGestureListener {
		void onGesture(GestureView v, MotionEvent e);

		void onGestureCancelled(GestureView v, MotionEvent e);

		void onGestureEnded(GestureView v, MotionEvent e);

		void onGestureStarted(GestureView v, MotionEvent e);
	}

	public static interface OnGesturePerformedListener {
		void onGesturePerformed(GestureView v, Gesture g);
	}

	public static interface OnGesturingListener {
		void onGesturingEnded(GestureView v);

		void onGesturingStarted(GestureView v);
	}

	public static interface OnSizeChangedListener {
		void onSizeChanged(int width, int height);
	}

	private class TouchUpRunnable implements Runnable {
		MotionEvent event;
		boolean immediate;
		GestureView view;

		private TouchUpRunnable(GestureView view) {
			this.view = view;
		}

		public void run() {
			final long now = AnimationUtils.currentAnimationTimeMillis();
			final long duration = now - mFadingStart;
			if (immediate || duration > mFadeDuration) {
				final List<OnGestureListener> listeners = mOnGestureListeners;
				final int count = listeners.size();
				for (int i = 0; i < count; i++) {
					listeners.get(i).onGestureEnded(view, event);
				}
				clear(false, true, false);
			} else if (!immediate) {
				postDelayed(this, mFadeOffset);
			}
		}
	}

	private static final boolean DITHER_FLAG = true;
	private static final int FADE_ANIMATION_RATE = 16;
	private static final boolean GESTURE_RENDERING_ANTIALIAS = true;
	public static final int GESTURE_STROKE_TYPE_MULTIPLE = 1;
	public static final int GESTURE_STROKE_TYPE_SINGLE = 0;
	private static final float MESH_SIZE = 8f;
	public static final int ORIENTATION_HORIZONTAL = 0;
	public static final int ORIENTATION_VERTICAL = 1;
	private static final String TAG = "GestureView";
	private static final float TOUCH_TOLERANCE = 8f;
	private int mCertainGestureColor = 0xFFFFFF00;
	private int mCurrentColor;
	private Gesture mCurrentGesture;
	private float mCurveEndX;
	private float mCurveEndY;
	private long mFadeDuration = 150;
	private boolean mFadeEnabled = false;
	private long mFadeOffset = 10000; // 420;
	private float mFadingAlpha = 1.0f;
	private boolean mFadingHasStarted;
	private final FadeOutRunnable mFadingOut = new FadeOutRunnable();
	private long mFadingStart;
	private final Paint mGesturePaint = new Paint();
	private float mGestureStrokeAngleThreshold = 40.0f;
	private float mGestureStrokeLengthThreshold = 50.0f;
	private float mGestureStrokeSquarenessTreshold = 0.275f;
	private int mGestureStrokeType = GESTURE_STROKE_TYPE_MULTIPLE; // GESTURE_STROKE_TYPE_SINGLE;
	private float mGestureStrokeWidth = 12.0f;
	private boolean mGestureVisible = true;
	private boolean mHandleGestureActions;
	private float mHeight;
	private boolean mInterceptEvents = true;
	private final AccelerateDecelerateInterpolator mInterpolator = new AccelerateDecelerateInterpolator();
	private int mInvalidateExtraBorder = 10;
	private final Rect mInvalidRect = new Rect();
	private boolean mIsFadingOut = false;
	private boolean mIsGesturing = false;
	private boolean mIsListeningForGestures;
	private boolean mMeshEnabled = true; // ogaki
	private final Paint mMeshPaint = new Paint(); // ogaki
	// TODO: Make this a list of WeakReferences
	private final List<OnGestureListener> mOnGestureListeners = new ArrayList<OnGestureListener>();
	// TODO: Make this a list of WeakReferences
	private final List<OnGesturePerformedListener> mOnGesturePerformedListeners = new ArrayList<OnGesturePerformedListener>();
	// TODO: Make this a list of WeakReferences
	private final List<OnGesturingListener> mOnGesturingListeners = new ArrayList<OnGesturingListener>();
	private final List<OnSizeChangedListener> mOnSizeChangedListeners = new ArrayList<OnSizeChangedListener>(); // ogaki
	private int mOrientation = ORIENTATION_VERTICAL;
	private final Path mPath = new Path();
	private boolean mPreviousWasGesturing = false;
	private boolean mResetGesture;
	private final List<GesturePoint> mStrokeBuffer = new ArrayList<GesturePoint>();
	private float mTotalLength;
	private final TouchUpRunnable mTouchUp = new TouchUpRunnable(this);
	private int mUncertainGestureColor = 0x48FFFF00;
	private float mWidth;
	private float mX;
	private float mY;

	public GestureView(Context context) {
		super(context);
		init();
	}

	public GestureView(Context context, AttributeSet attrs) {
		this(context, attrs, R.attr.gestureViewStyle);
	}

	public GestureView(Context context, AttributeSet attrs, int defStyle) {
		super(context, attrs, defStyle);

		final TypedArray typedArray = context.obtainStyledAttributes(attrs,
				R.styleable.GestureView, defStyle, 0);

		mGestureStrokeWidth = typedArray
				.getFloat(R.styleable.GestureView_gestureStrokeWidth,
						mGestureStrokeWidth);
		mInvalidateExtraBorder = Math.max(1, ((int) mGestureStrokeWidth) - 1);
		mCertainGestureColor = typedArray.getColor(
				R.styleable.GestureView_gestureColor, mCertainGestureColor);
		mUncertainGestureColor = typedArray.getColor(
				R.styleable.GestureView_uncertainGestureColor,
				mUncertainGestureColor);
		mFadeDuration = typedArray.getInt(R.styleable.GestureView_fadeDuration,
				(int) mFadeDuration);
		mFadeOffset = typedArray.getInt(R.styleable.GestureView_fadeOffset,
				(int) mFadeOffset);
		mGestureStrokeType = typedArray.getInt(
				R.styleable.GestureView_gestureStrokeType, mGestureStrokeType);
		mGestureStrokeLengthThreshold = typedArray.getFloat(
				R.styleable.GestureView_gestureStrokeLengthThreshold,
				mGestureStrokeLengthThreshold);
		mGestureStrokeAngleThreshold = typedArray.getFloat(
				R.styleable.GestureView_gestureStrokeAngleThreshold,
				mGestureStrokeAngleThreshold);
		mGestureStrokeSquarenessTreshold = typedArray.getFloat(
				R.styleable.GestureView_gestureStrokeSquarenessThreshold,
				mGestureStrokeSquarenessTreshold);
		mInterceptEvents = typedArray.getBoolean(
				R.styleable.GestureView_eventsInterceptionEnabled,
				mInterceptEvents);
		mFadeEnabled = typedArray.getBoolean(
				R.styleable.GestureView_fadeEnabled, mFadeEnabled);
		mOrientation = typedArray.getInt(R.styleable.GestureView_orientation,
				mOrientation);
		typedArray.recycle();
		init();
	}

	public void addOnGestureListener(OnGestureListener listener) {
		mOnGestureListeners.add(listener);
	}

	public void addOnGesturePerformedListener(
			OnGesturePerformedListener listener) {
		mOnGesturePerformedListeners.add(listener);
		if (mOnGesturePerformedListeners.size() > 0) {
			mHandleGestureActions = true;
		}
	}

	public void addOnGesturingListener(OnGesturingListener listener) {
		mOnGesturingListeners.add(listener);
	}

	public void addOnSizeChangedListener(OnSizeChangedListener listener) {
		mOnSizeChangedListeners.add(listener);
	}

	public void cancelClearAnimation() {
		setPaintAlpha(255);
		mIsFadingOut = false;
		mFadingHasStarted = false;
		removeCallbacks(mFadingOut);
		mPath.rewind();
		mCurrentGesture = null;
	}

	public void cancelGesture() {
		long now = 0L;
		MotionEvent event = null;
		List<OnGestureListener> listeners = null;
		List<OnGesturingListener> otherListeners = null;
		int count = 0;

		mIsListeningForGestures = false;
		mCurrentGesture.addStroke(new GestureStroke(mStrokeBuffer));
		now = SystemClock.uptimeMillis();
		event = MotionEvent.obtain(now, now, MotionEvent.ACTION_CANCEL, 0.0f,
				0.0f, 0);
		listeners = mOnGestureListeners;
		count = listeners.size();
		for (int i = 0; i < count; i++) {
			listeners.get(i).onGestureCancelled(this, event);
		}
		event.recycle();
		clear(false);
		mIsGesturing = false;
		mPreviousWasGesturing = false;
		mStrokeBuffer.clear();
		otherListeners = mOnGesturingListeners;
		count = otherListeners.size();
		for (int i = 0; i < count; i++) {
			otherListeners.get(i).onGesturingEnded(this);
		}
	}

	private void cancelGesture(MotionEvent event) {
		// pass the event to handlers
		final List<OnGestureListener> listeners = mOnGestureListeners;
		final int count = listeners.size();
		for (int i = 0; i < count; i++) {
			listeners.get(i).onGestureCancelled(this, event);
		}
		clear(false);
	}

	public void clear(boolean animated) {
		clear(animated, false, true);
	}

	private void clear(boolean animated, boolean fireActionPerformed,
			boolean immediate) {
		setPaintAlpha(255);
		removeCallbacks(mFadingOut);
		mResetGesture = false;
		mFadingOut.fireActionPerformed = fireActionPerformed;
		// mFadingOut.resetMultipleStrokes = false;

		if (animated && null != mCurrentGesture) {
			mFadingAlpha = 1.0f;
			mIsFadingOut = true;
			mFadingHasStarted = false;
			mFadingStart = AnimationUtils.currentAnimationTimeMillis()
					+ mFadeOffset;
			postDelayed(mFadingOut, mFadeOffset);
		} else {
			mFadingAlpha = 1.0f;
			mIsFadingOut = false;
			mFadingHasStarted = false;

			if (immediate) {
				mCurrentGesture = null;
				mPath.rewind();
				invalidate();
			} else if (fireActionPerformed) {
				postDelayed(mFadingOut, mFadeOffset);
			} else {
				mCurrentGesture = null;
				mPath.rewind();
				invalidate();
			}
		}
	}

	@Override
	public boolean dispatchTouchEvent(MotionEvent event) {
		if (isEnabled()) {
			// Skip if tap on keyboard in GestureKeyboard
			if (event.getX() > mWidth || event.getY() > mHeight) {
				LOG.d(TAG,
						"touchDown: over positon x={0,number,#}, y={1,number,#}",
						event.getX(), event.getY());
				mIsGesturing = false;
				return super.dispatchTouchEvent(event);
			}
			final boolean cancelDispatch = (mIsGesturing || (null != mCurrentGesture
					&& mCurrentGesture.getStrokesCount() > 0 && mPreviousWasGesturing))
					&& mInterceptEvents;
			processEvent(event);
			if (cancelDispatch) {
				event.setAction(MotionEvent.ACTION_CANCEL);
			}
			super.dispatchTouchEvent(event);
			return true;
		}
		return super.dispatchTouchEvent(event);
	}

	@Override
	public void draw(final Canvas canvas) {
		final float yCenter = mHeight * 0.5f;
		final float xCenter = mWidth * 0.5f;
		float step = mHeight / MESH_SIZE;

		super.draw(canvas);

		if (mMeshEnabled) {
			// Draw mesh
			for (float yy = step; yy < mHeight; yy += step) {
				if (0 == (yy - yCenter)) {
					mMeshPaint.setColor(Color.GREEN);
					canvas.drawLine(0, yy, mWidth, yy, mMeshPaint);
				} else {
					mMeshPaint.setColor(Color.LTGRAY);
					for (int x = 0; x < mWidth; x += 8) {
						canvas.drawLine(x, yy, x + 1, yy, mMeshPaint);
					}
				}
			}
			step = mWidth / MESH_SIZE;
			for (float xx = step; xx < mWidth; xx += step) {
				if (0 == (xx - xCenter)) {
					mMeshPaint.setColor(Color.GREEN);
					canvas.drawLine(xx, 0, xx, mHeight, mMeshPaint);
				} else {
					mMeshPaint.setColor(Color.LTGRAY);
					for (int y = 0; y < mHeight; y += 8) {
						canvas.drawLine(xx, y, xx, y + 1, mMeshPaint);
					}
				}
			}
		}
		if (null != mCurrentGesture && mGestureVisible) {
			canvas.drawPath(mPath, mGesturePaint);
		}
	}

	private void fireOnGesturePerformed() {
		final List<OnGesturePerformedListener> actionListeners = mOnGesturePerformedListeners;
		final int count = actionListeners.size();
		for (int i = 0; i < count; i++) {
			actionListeners.get(i).onGesturePerformed(GestureView.this,
					mCurrentGesture);
		}
	}

	public List<GesturePoint> getCurrentStroke() {
		return mStrokeBuffer;
	}

	public long getFadeOffset() {
		return mFadeOffset;
	}

	public Gesture getGesture() {
		return mCurrentGesture;
	}

	public int getGestureColor() {
		return mCertainGestureColor;
	}

	public Paint getGesturePaint() {
		return mGesturePaint;
	}

	public Path getGesturePath() {
		return mPath;
	}

	// public float getGestureStrokeAngleThreshold() {
	// return mGestureStrokeAngleThreshold;
	// }

	public Path getGesturePath(Path path) {
		path.set(mPath);
		return path;
	}

	// public float getGestureStrokeSquarenessTreshold() {
	// return mGestureStrokeSquarenessTreshold;
	// }
	//
	// public int getGestureStrokeType() {
	// return mGestureStrokeType;
	// }

	public float getGestureStrokeLengthThreshold() {
		return mGestureStrokeLengthThreshold;
	}

	public float getGestureStrokeWidth() {
		return mGestureStrokeWidth;
	}

	public int getOrientation() {
		return mOrientation;
	}
	
	public Path getPath() {
		return mPath;
	}

	public int getUncertainGestureColor() {
		return mUncertainGestureColor;
	}

	private void init() {
		final Paint gesturePaint = mGesturePaint;
		gesturePaint.setAntiAlias(GESTURE_RENDERING_ANTIALIAS);
		gesturePaint.setColor(mCertainGestureColor);
		gesturePaint.setStyle(Paint.Style.STROKE);
		gesturePaint.setStrokeJoin(Paint.Join.ROUND);
		gesturePaint.setStrokeCap(Paint.Cap.ROUND);
		gesturePaint.setStrokeWidth(mGestureStrokeWidth);
		gesturePaint.setDither(DITHER_FLAG);
		setWillNotDraw(false);
		mCurrentColor = mCertainGestureColor;
		setPaintAlpha(255);
	}

	public boolean isEventsInterceptionEnabled() {
		return mInterceptEvents;
	}

	public boolean isFadeEnabled() {
		return mFadeEnabled;
	}

	public boolean isGestureVisible() {
		return mGestureVisible;
	}

	public boolean isGesturing() {
		return mIsGesturing;
	}

	@Override
	protected void onDetachedFromWindow() {
		cancelClearAnimation();
	}

	@Override
	protected void onSizeChanged(int w, int h, int oldw, int oldh) {
		super.onSizeChanged(w, h, oldw, oldh);
		if (getId() == android.R.id.inputArea) { // in IME
			final int keyheight = getResources().getDimensionPixelSize(
					R.dimen.key_height);
			mWidth = w;
			mHeight = h - keyheight;
			LOG.d(TAG, "onSizeChanged: key-height={0,number,#}", keyheight);
		} else { // CreateGestureActivity or CheckGestureActivity
			mWidth = w;
			mHeight = h;
			LOG.d(TAG, "onSizeChanged: w={0,number,#}, h={1,number,#}", w, h);
		}
		for (OnSizeChangedListener ob : mOnSizeChangedListeners) {
			ob.onSizeChanged((int) mWidth, (int) mHeight);
		}
	}

	private boolean processEvent(MotionEvent event) {
		switch (event.getAction()) {
		case MotionEvent.ACTION_DOWN:
			touchDown(event);
			invalidate();
			return true;
		case MotionEvent.ACTION_MOVE:
			if (mIsListeningForGestures) {
				final Rect rect = touchMove(event);
				if (null != rect) {
					invalidate(rect);
				}
				return true;
			}
			break;
		case MotionEvent.ACTION_UP:
			if (mIsListeningForGestures) {
				touchUp(event, false);
				invalidate();
				return true;
			}
			break;
		case MotionEvent.ACTION_CANCEL:
			if (mIsListeningForGestures) {
				touchUp(event, true);
				invalidate();
				return true;
			}
			break;
		default:
			break;
		}
		return false;
	}

	public void removeAllOnGestureListeners() {
		mOnGestureListeners.clear();
	}

	public void removeAllOnGesturePerformedListeners() {
		mOnGesturePerformedListeners.clear();
		mHandleGestureActions = false;
	}

	public void removeAllOnGesturingListeners() {
		mOnGesturingListeners.clear();
	}

	public void removeOnGestureListener(OnGestureListener listener) {
		mOnGestureListeners.remove(listener);
	}

	public void removeOnGesturePerformedListener(
			OnGesturePerformedListener listener) {
		mOnGesturePerformedListeners.remove(listener);
		if (mOnGesturePerformedListeners.size() <= 0) {
			mHandleGestureActions = false;
		}
	}

	public void removeOnGesturingListener(OnGesturingListener listener) {
		mOnGesturingListeners.remove(listener);
	}

	public void removeOnSizeChangedListener(OnSizeChangedListener listener) {
		mOnSizeChangedListeners.remove(listener);
	}

	private void setCurrentColor(int color) {
		mCurrentColor = color;
		if (mFadingHasStarted) {
			setPaintAlpha((int) (255 * mFadingAlpha));
		} else {
			setPaintAlpha(255);
		}
		invalidate();
	}

	public void setEventsInterceptionEnabled(boolean enabled) {
		mInterceptEvents = enabled;
	}

	public void setFadeEnabled(boolean fadeEnabled) {
		mFadeEnabled = fadeEnabled;
	}

	public void setFadeOffset(long fadeOffset) {
		mFadeOffset = fadeOffset;
	}

	// public void setGestureStrokeAngleThreshold(float
	// gestureStrokeAngleThreshold) {
	// mGestureStrokeAngleThreshold = gestureStrokeAngleThreshold;
	// }

	// ogaki
	public void setGesture(Gesture gesture) {
		if (null != mCurrentGesture) {
			cancelClearAnimation();
			clear(false);
		}
		final Path path = gesture.toPath();
		final RectF bounds = new RectF();
		path.computeBounds(bounds, true);
		mPath.reset(); // .rewind();
		mPath.addPath(path, getWidth(), getHeight());
		mCurrentGesture = new Gesture();
		mCurrentGesture.setId(gesture.getId());
		mCurrentGesture.setStrokes(gesture.getStrokes());
		invalidate();
		// mIsListeningForGestures = false;
		// mPreviousWasGesturing = mIsGesturing;
		// mIsGesturing = true; //false;
	}

	// public void setGestureStrokeSquarenessTreshold(
	// float gestureStrokeSquarenessTreshold) {
	// mGestureStrokeSquarenessTreshold = gestureStrokeSquarenessTreshold;
	// }
	//
	// public void setGestureStrokeType(int gestureStrokeType) {
	// mGestureStrokeType = gestureStrokeType;
	// }

	public void setGestureColor(int color) {
		mCertainGestureColor = color;
	}

	public void setGestureStrokeLengthThreshold(
			float gestureStrokeLengthThreshold) {
		mGestureStrokeLengthThreshold = gestureStrokeLengthThreshold;
	}

	public void setGestureStrokeWidth(float gestureStrokeWidth) {
		mGestureStrokeWidth = gestureStrokeWidth;
		mInvalidateExtraBorder = Math.max(1, ((int) gestureStrokeWidth) - 1);
		mGesturePaint.setStrokeWidth(gestureStrokeWidth);
	}

	public void setGestureVisible(boolean visible) {
		mGestureVisible = visible;
	}

	public void setMeshEnabled(boolean meshEnabled) {
		mMeshEnabled = meshEnabled;
	}

	public void setOrientation(int orientation) {
		mOrientation = orientation;
	}

	private void setPaintAlpha(int alpha) {
		alpha += alpha >> 7;
		final int baseAlpha = mCurrentColor >>> 24;
		final int useAlpha = baseAlpha * alpha >> 8;
		mGesturePaint.setColor((mCurrentColor << 8 >>> 8) | (useAlpha << 24));
	}

	public void setUncertainGestureColor(int color) {
		mUncertainGestureColor = color;
	}

	private void touchDown(MotionEvent event) {
		List<OnGestureListener> listeners = null;
		final float x = event.getX();
		final float y = event.getY();
		int border = 0;
		int count = 0;

		mIsListeningForGestures = true;
		mX = x;
		mY = y;
		mTotalLength = 0;
		mIsGesturing = false;

		if (mGestureStrokeType == GESTURE_STROKE_TYPE_SINGLE || mResetGesture) {
			// LOG.d(TAG, "touchDown; type_single or mResetGesture is {0}",
			// mResetGesture);
			if (mHandleGestureActions) {
				setCurrentColor(mUncertainGestureColor);
			}
			mResetGesture = false;
			mCurrentGesture = null;
			mPath.rewind();
		} else if (null == mCurrentGesture
				|| 0 == mCurrentGesture.getStrokesCount()) {
			// LOG.d(TAG, "touchDown; mCurrentGesture is null");
			if (mHandleGestureActions) {
				setCurrentColor(mUncertainGestureColor);
			}
		}

		// if there is fading out going on, stop it.
		if (mFadingHasStarted) {
			cancelClearAnimation();
		} else if (mIsFadingOut) {
			setPaintAlpha(255);
			mIsFadingOut = false;
			mFadingHasStarted = false;
			removeCallbacks(mFadingOut);
		}

		if (null == mCurrentGesture) {
			mCurrentGesture = new Gesture();
		}

		// Skip if tap on keyboard in GestureKeyboard
		if (x > mWidth || y > mHeight) {
			LOG.d(TAG,
					"touchDown: over positon x={0,number,#}, y={1,number,#}",
					event.getX(), event.getY());
		} else {
			mStrokeBuffer.add(new GesturePoint(x, y, event.getEventTime()));
			mPath.moveTo(x, y);
			border = mInvalidateExtraBorder;
			mInvalidRect.set((int) x - border, (int) y - border, (int) x
					+ border, (int) y + border);
			mCurveEndX = x;
			mCurveEndY = y;
		}

		// pass the event to handlers
		listeners = mOnGestureListeners;
		count = listeners.size();
		for (int i = 0; i < count; i++) {
			listeners.get(i).onGestureStarted(this, event);
		}
	}

	private Rect touchMove(MotionEvent event) {
		final float x = event.getX();
		final float y = event.getY();
		// Skip if tap on keyboard in GestureKeyboard
		if (x > mWidth || y > mHeight) {
			LOG.d(TAG,
					"touchMove: over positon x={0,number,#}, y={1,number,#}",
					event.getX(), event.getY());
			return null;
		}
		Rect areaToRefresh = null;
		final float previousX = mX;
		final float previousY = mY;
		final float dx = Math.abs(x - previousX);
		final float dy = Math.abs(y - previousY);
		// final float mCurveEndX = (x + previousX) / 2;
		// final float mCurveEndY = (y + previousY) / 2;
		final float cX = mCurveEndX;
		final float cY = mCurveEndY;
		int border = 0;
		List<OnGesturingListener> listeners = null;
		List<OnGestureListener> glisteners = null;
		int count = 0;

		mCurveEndX = (x + previousX) / 2;
		mCurveEndY = (y + previousY) / 2;

		if (dx >= TOUCH_TOLERANCE || dy >= TOUCH_TOLERANCE) {
			areaToRefresh = mInvalidRect;

			// start with the curve end
			border = mInvalidateExtraBorder;
			areaToRefresh.set((int) mCurveEndX - border, (int) mCurveEndY
					- border, (int) mCurveEndX + border, (int) mCurveEndY
					+ border);
			mPath.quadTo(previousX, previousY, cX, cY);

			// union with the control point of the new curve
			areaToRefresh.union((int) previousX - border, (int) previousY
					- border, (int) previousX + border, (int) previousY
					+ border);

			// union with the end point of the new curve
			areaToRefresh.union((int) cX - border, (int) cY - border, (int) cX
					+ border, (int) cY + border);

			mX = x;
			mY = y;

			mStrokeBuffer.add(new GesturePoint(x, y, event.getEventTime()));

			if (mHandleGestureActions && !mIsGesturing) {
				mTotalLength += (float) Math.sqrt(dx * dx + dy * dy);

				if (mTotalLength > mGestureStrokeLengthThreshold) {
					final OrientedBoundingBox box = GestureUtilities
							.computeOrientedBoundingBox(mStrokeBuffer);

					float angle = Math.abs(box.orientation);
					if (angle > 90) {
						angle = 180 - angle;
					}

					if (box.squareness > mGestureStrokeSquarenessTreshold
							|| ((mOrientation == ORIENTATION_VERTICAL) ? angle < mGestureStrokeAngleThreshold
									: angle > mGestureStrokeAngleThreshold)) {
						mIsGesturing = true;
						setCurrentColor(mCertainGestureColor);
						listeners = mOnGesturingListeners;
						count = listeners.size();
						for (int i = 0; i < count; i++) {
							listeners.get(i).onGesturingStarted(this);
						}
					}
				}
			}

			// pass the event to handlers
			glisteners = mOnGestureListeners;
			count = glisteners.size();
			for (int i = 0; i < count; i++) {
				glisteners.get(i).onGesture(this, event);
			}
		}
		return areaToRefresh;
	}

	private void touchUp(MotionEvent event, boolean cancel) {
		// Skip if tap on keyboard in GestureKeyboard
		if (event.getX() > mWidth || event.getY() > mHeight) {
			LOG.d(TAG, "touchUp: over positon x={0,number,#}, y={1,number,#}",
					event.getX(), event.getY());
			return;
		}
		mIsListeningForGestures = false;

		// A gesture wasn't started or was canceled
		if (mCurrentGesture != null) {
			// add the stroke to the current gesture
			mCurrentGesture.addStroke(new GestureStroke(mStrokeBuffer));
			if (!cancel) {
				if (mGestureStrokeType == GESTURE_STROKE_TYPE_SINGLE) {
					// pass the event to handlers
					final List<OnGestureListener> listeners = mOnGestureListeners;
					int count = listeners.size();
					for (int i = 0; i < count; i++) {
						listeners.get(i).onGestureEnded(this, event);
					}
					clear(mHandleGestureActions && mFadeEnabled,
							mHandleGestureActions && mIsGesturing, false);
				}
			} else {
				cancelGesture(event);
			}
		} else {
			cancelGesture(event);
		}

		mStrokeBuffer.clear();
		if (mGestureStrokeType == GESTURE_STROKE_TYPE_MULTIPLE) {
			mIsListeningForGestures = true;
			// timer call onGestureEnded() and clear()
			if (mFadeEnabled) {
				mTouchUp.event = event;
				mTouchUp.immediate = false;
				postDelayed(mTouchUp, mFadeOffset);
			}
		} else {
			mPreviousWasGesturing = mIsGesturing;
			mIsGesturing = false;
		}
		final List<OnGesturingListener> listeners = mOnGesturingListeners;
		final int count = listeners.size();
		for (int i = 0; i < count; i++) {
			listeners.get(i).onGesturingEnded(this);
		}
	}
}
