/*
 * Copyright (C) 2011 OgakiSoft
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package ogakisoft.android.svm;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import ogakisoft.android.gesture.reform.GestureLibrary;
import ogakisoft.android.gesture.reform.Learner;
import ogakisoft.android.util.LOG;
import ogakisoft.android.util.Utils;

/**
 * Wrapper class for native method of libsvm
 * 
 * @version 1.1
 * @author noritoshi ogaki
 */
public class SvmWrapper {
	static {
		System.loadLibrary("svm");
	}

	/**
	 * native method scale.
	 * 
	 * @param args
	 *            String[]
	 */
	public static native void scale(String[] args);

	/**
	 * native method train.
	 * 
	 * @param args
	 *            String[]
	 */
	public static native void train(String[] args);

	/**
	 * native method predict.
	 * 
	 * @param args
	 *            String[]
	 */
	public static native void predict(String[] args);

	/** data directory */
	public static final File DataDir = GestureLibrary.DataDir;
	/** converted gesture data */
	public static final File[] LibsvmData = {
			new File(DataDir, "libsvm_data.0"),
			new File(DataDir, "libsvm_data.1"),
			new File(DataDir, "libsvm_data.2"),
			new File(DataDir, "libsvm_data.3"),
			new File(DataDir, "libsvm_data.4"),
			new File(DataDir, "libsvm_data.5") };
	/** output from scale, input to train */
	private static final File[] LibsvmScaledData = {
			new File(DataDir, "libsvm_scaled_data.0"),
			new File(DataDir, "libsvm_scaled_data.1"),
			new File(DataDir, "libsvm_scaled_data.2"),
			new File(DataDir, "libsvm_scaled_data.3"),
			new File(DataDir, "libsvm_scaled_data.4"),
			new File(DataDir, "libsvm_scaled_data.5") };
	/** output from scale, input to train */
	private static final File[] LibsvmScaleRestore = {
			new File(DataDir, "libsvm_scale.0"),
			new File(DataDir, "libsvm_scale.1"),
			new File(DataDir, "libsvm_scale.2"),
			new File(DataDir, "libsvm_scale.3"),
			new File(DataDir, "libsvm_scale.4"),
			new File(DataDir, "libsvm_scale.5") };
	/** output from train, input to predict */
	private static final File[] LibsvmModel = {
			new File(DataDir, "libsvm_model.0"),
			new File(DataDir, "libsvm_model.1"),
			new File(DataDir, "libsvm_model.2"),
			new File(DataDir, "libsvm_model.3"),
			new File(DataDir, "libsvm_model.4"),
			new File(DataDir, "libsvm_model.5"), };
	/** output from predict, result-file */
	private static final File LibsvmOut = new File(DataDir, "libsvm_out");
	/** input to scale */
	private static final File LibsvmTest = new File(DataDir, "libsvm_test");
	/** output from scale, input to predict */
	private static final File LibsvmScaledTest = new File(DataDir,
			"libsvm_scaled_test");

	/** LOG tag */
	private static final String TAG = "SvmWrapper";
	/** instance of this class */
	private static SvmWrapper Me;
	/** for nothing result */
	private static final String[] EMPTY_STRING_ARRAY = new String[0];
	/** train mode for scale */
	private static final int SCALE_FOR_TRAIN = 1;
	/** predict mode for scale */
	private static final int SCALE_FOR_PREDICT = 2;
	/** file buffer size */
	private static final int IO_BUFFER_SIZE = 32 * 1024; // 32K
	/** line-feed string */
	private static final String LINE_FEED = "\n";

	private SvmWrapper() {
	}

	public static SvmWrapper getInstance() {
		if (null == Me) {
			Me = new SvmWrapper();
		}
		return Me;
	}

	public int[] classify(int fileno, float[] vector, Learner learner) {
		if (fileno < 0 || GestureLibrary.MAX_NUM_OF_STORE <= fileno) {
			return new int[0];
		}
		int count = 0;
		String[] result = null;
//		List<Integer> list = new ArrayList<Integer>();
		File file = LibsvmModel[fileno];
		int[] result_array = new int[0];
		// remove output-file
		if (LibsvmOut.exists()) {
			if (!LibsvmOut.delete()) {
				LOG.e(TAG, "classify: can not delete {0}",
						LibsvmOut.getAbsolutePath());
			}
		}
		// output test-file
		save_test(vector);
		// output scaled-test file
		svmScale(SCALE_FOR_PREDICT, fileno);
		// output result file
		svmPredict(file);
		result = load_result();
		if (null != result) {
			count = result.length;
			int id = 0;
			result_array = new int[count];
			for (int i = 0; i < count; i++) {
				id = Integer.parseInt(result[i]);
				result_array[i] = id;
//				if (learner.containsKey(id)) {
//					final Instance ins = learner.getInstance(id);
//					if (null != ins) {
//						list.add(id);
//					}
//				}
			}
		}
		return result_array;
	}

	public void removeAllFiles() {
		final List<File> files = new ArrayList<File>();
		files.addAll(Arrays.asList(LibsvmScaledData));
		files.addAll(Arrays.asList(LibsvmScaleRestore));
		files.addAll(Arrays.asList(LibsvmModel));
		files.add(LibsvmTest);
		files.add(LibsvmScaledTest);
		files.add(LibsvmOut);
		final int count = files.size();
		for (int i = 0; i < count; i++) {
			if (files.get(i).exists()) {
				if (!files.get(i).delete()) {
					LOG.e(TAG, "removeAllFiles: can not delete file: {0}",
							files.get(i).getAbsoluteFile());
				}
			}
		}
	}

	private void svmScale(int mode, int num) {
		if (num < 0 || GestureLibrary.MAX_NUM_OF_STORE <= num) {
			return;
		}
		final String[] args = new String[4];
		long lastmodified;
		switch (mode) {
		case SCALE_FOR_TRAIN:
			if (!existFile("svmScale", LibsvmData[num])) {
				return;
			}
			// file update check
			lastmodified = LibsvmData[num].lastModified();
			if (LibsvmScaledData[num].exists()
					&& lastmodified <= LibsvmScaledData[num].lastModified()
					&& LibsvmScaleRestore[num].exists()
					&& lastmodified <= LibsvmScaleRestore[num].lastModified()) {
				return;
			}
			// scale-save (output) file
			args[0] = "-s";
			args[1] = LibsvmScaleRestore[num].getAbsolutePath();
			// input-data file
			args[2] = LibsvmData[num].getAbsolutePath();
			// output file
			args[3] = LibsvmScaledData[num].getAbsolutePath();
			break;
		case SCALE_FOR_PREDICT:
			if (!existFile("svmScale", LibsvmScaleRestore[num])) {
				return;
			}
			// scale-restore (input) file
			args[0] = "-r";
			args[1] = LibsvmScaleRestore[num].getAbsolutePath();
			// input-test file
			args[2] = LibsvmTest.getAbsolutePath();
			// output file
			args[3] = LibsvmScaledTest.getAbsolutePath();
			break;
		}
		// call native method
		scale(args);
	}

	private void svmPredict(File modelFile) {
		if (!existFile("svmPredict", modelFile, LibsvmScaledTest)) {
			return;
		}
		final String[] args = new String[3];
		// test-data (input) file
		args[0] = LibsvmScaledTest.getAbsolutePath();
		// model (input) file
		args[1] = modelFile.getAbsolutePath();
		// result (output) file
		args[2] = LibsvmOut.getAbsolutePath();
		// call native method
		predict(args);
	}

	private String[] load_result() {
		BufferedReader in = null;
		final StringBuffer sb = new StringBuffer(0);
		if (!LibsvmOut.exists()) {
			return EMPTY_STRING_ARRAY;
		}
		try {
			in = new BufferedReader(new FileReader(LibsvmOut), IO_BUFFER_SIZE);
			for (String str = in.readLine(); null != str; str = in.readLine()) {
				sb.append(str).append(LINE_FEED);
				// LOG.d(TAG, "load_result: {0}", str);
			}
		} catch (IOException e) {
			LOG.e(TAG, "load_result: {0}", e.getMessage());
		} finally {
			try {
				if (null != in) {
					in.close();
				}
			} catch (IOException e) {
				LOG.e(TAG, "load_result: {0}", e.getMessage());
			}
		}
		if (sb.length() == 0) {
			return EMPTY_STRING_ARRAY;
		}
		return sb.toString().split(LINE_FEED);
	}

	private void save_test(float[] vector) {
		FileOutputStream out = null;
		int count = 0;
		try {
			out = new FileOutputStream(LibsvmTest);
		} catch (IOException e) {
			LOG.e(TAG, "save_test: {0}", e.getMessage());
			return;
		}
		try {
			out.write(String.valueOf("0 ").getBytes());
			count = vector.length;
			for (int i = 0; i < count; i++) {
				out.write(Utils.concat(String.valueOf(i + 1), ":",
						String.valueOf(vector[i]), " ").getBytes());
			}
			out.write(String.valueOf(LINE_FEED).getBytes());
			out.flush();
		} catch (IOException e) {
			LOG.e(TAG, "save_test: {0}", e.getMessage());
		} finally {
			if (null != out) {
				try {
					out.close();
				} catch (IOException e) {
					LOG.e(TAG, "save_test: {0}", e.getMessage());
				}
			}
		}
	}

	public void svmTrain(int num, Learner classifier) {
		if (num < 0 || GestureLibrary.MAX_NUM_OF_STORE <= num) {
			return;
		}
		final String[] args = new String[2];
		// file update check
		if (!LibsvmData[num].exists()
				|| LibsvmData[num].lastModified() < GestureLibrary.DataFile[num]
						.lastModified()) {
			// write LIBSVM_DATA_FILE
			classifier.saveSvmData(LibsvmData[num]);
		}
		// output LIBSVM_SCALED_DATA_FILE
		svmScale(SCALE_FOR_TRAIN, num);
		// file update check
		if (LibsvmScaledData[num].exists()
				&& LibsvmModel[num].exists()
				&& LibsvmScaledData[num].lastModified() <= LibsvmModel[num]
						.lastModified()) {
			return;
		}
		// scaled-data (input) file
		args[0] = LibsvmScaledData[num].getAbsolutePath();
		// model (output) file
		args[1] = LibsvmModel[num].getAbsolutePath();
		// call native method
		train(args);
	}

	private boolean existFile(String tag, File... files) {
		final int count = files.length;
		for (int i = 0; i < count; i++) {
			if (!files[i].exists()) {
				LOG.e(TAG, "{0}: file not found {1}", tag, files[i]
						.getAbsoluteFile().toString());
				return false;
			}
		}
		return true;
	}

}
