/*
 * Copyright (C) 2008-2011 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
 * Copyright (C) 2011 OgakiSoft
 * 
 * Moved a variable definition in the method top by refactoring.
 */

package ogakisoft.android.gesture.reform;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.RectF;

/**
 * A gesture stroke started on a touch down and ended on a touch up.
 * 
 * @author The Android Open Source Project
 * @author noritoshi ogaki
 * @version 1.0
 */
public class GestureStroke {
	static final float TOUCH_TOLERANCE = 8;

	public final RectF boundingBox;

	public final float length;
	public final float[] points;

	public final long[] timestamps;
	public Path mCachedPath;

	/**
	 * Construct a gesture stroke from a list of gesture points
	 * 
	 * @param points
	 */
	public GestureStroke(List<GesturePoint> points) {
		final int count = points.size();
		final float[] tmpPoints = new float[count * 2];
		final long[] times = new long[count];

		RectF bx = null;
		float len = 0;
		int index = 0;
		GesturePoint gesturePoint = null;
		for (int i = 0; i < count; i++) {
			gesturePoint = points.get(i);
			tmpPoints[i * 2] = gesturePoint.x;
			tmpPoints[i * 2 + 1] = gesturePoint.y;
			times[index] = gesturePoint.timestamp;

			if (null == bx) {
				bx = new RectF();
				bx.top = gesturePoint.y;
				bx.left = gesturePoint.x;
				bx.right = gesturePoint.x;
				bx.bottom = gesturePoint.y;
				len = 0;
			} else {
				len += Math.sqrt(Math.pow(gesturePoint.x - tmpPoints[(i - 1) * 2], 2)
						+ Math.pow(gesturePoint.y - tmpPoints[(i - 1) * 2 + 1], 2));
				bx.union(gesturePoint.x, gesturePoint.y);
			}
			index++;
		}

		timestamps = times;
		this.points = tmpPoints;
		boundingBox = bx;
		length = len;
	}

	/**
	 * Draw the gesture with a given canvas and paint
	 * 
	 * @param canvas
	 */
	void draw(Canvas canvas, Paint paint) {
		if (null == mCachedPath) {
			makePath();
		}

		canvas.drawPath(mCachedPath, paint);
	}

	public Path getPath() {
		if (null == mCachedPath) {
			makePath();
		}

		return mCachedPath;
	}

	private void makePath() {
		final float[] localPoints = points;
		final int count = localPoints.length;
		float mX = 0f;
		float mY = 0f;
		float x = 0f;
		float y = 0f;
		float dx = 0f;
		float dy = 0f;
		Path path = null;
		
		for (int i = 0; i < count; i += 2) {
			x = localPoints[i];
			y = localPoints[i + 1];
			if (null == path) {
				path = new Path();
				path.moveTo(x, y);
				mX = x;
				mY = y;
			} else {
				dx = Math.abs(x - mX);
				dy = Math.abs(y - mY);
				if (dx >= TOUCH_TOLERANCE || dy >= TOUCH_TOLERANCE) {
					path.quadTo(mX, mY, (x + mX) / 2, (y + mY) / 2);
					mX = x;
					mY = y;
				}
			}
		}

		mCachedPath = path;
	}

	/**
	 * Convert the stroke to a Path based on the number of points
	 * 
	 * @param width
	 *            the width of the bounding box of the target path
	 * @param height
	 *            the height of the bounding box of the target path
	 * @param numSample
	 *            the number of points needed
	 * 
	 * @return the path
	 */
	public Path toPath(float width, float height, int numSample) {
		final float[] pts = GestureUtilities.temporalSampling(this, numSample);
		final RectF rect = boundingBox;
		final float sx = width / rect.width();
		final float sy = height / rect.height();
		final float scale = (sx > sy) ? sy : sx;
		float mX = 0f;
		float mY = 0f;
		float x = 0f;
		float y = 0f;
		float dx = 0f;
		float dy = 0f;
		final int count = pts.length;
		Path path = null;
		
		GestureUtilities.translate(pts, -rect.left, -rect.top);
		GestureUtilities.scale(pts, scale, scale);

		for (int i = 0; i < count; i += 2) {
			x = pts[i];
			y = pts[i + 1];
			if (null == path) {
				path = new Path();
				path.moveTo(x, y);
				mX = x;
				mY = y;
			} else {
				dx = Math.abs(x - mX);
				dy = Math.abs(y - mY);
				if (dx >= TOUCH_TOLERANCE || dy >= TOUCH_TOLERANCE) {
					path.quadTo(mX, mY, (x + mX) / 2, (y + mY) / 2);
					mX = x;
					mY = y;
				}
			}
		}

		return path;
	}

	public void serialize(DataOutputStream out) throws IOException {
		final float[] pts = points;
		final long[] times = timestamps;
		final int count = points.length;

		// Write number of points
		out.writeInt(count / 2);

		for (int i = 0; i < count; i += 2) {
			// Write X
			out.writeFloat(pts[i]);
			// Write Y
			out.writeFloat(pts[i + 1]);
			// Write timestamp
			out.writeLong(times[i / 2]);
		}
	}

	public static GestureStroke deserialize(DataInputStream in)
			throws IOException {
		float x = 0f;
		float y = 0f;
		long timeStamp = 0L;
		
		// Number of points
		final int count = in.readInt();
		final List<GesturePoint> points = new ArrayList<GesturePoint>(
				count);
		for (int i = 0; i < count; i++) {
			x = in.readFloat();
			y = in.readFloat();
			timeStamp = in.readLong();
			points.add(new GesturePoint(x, y, timeStamp));
		}
		return new GestureStroke(points);
	}

	/**
	 * Invalidate the cached path that is used to render the stroke
	 */
	public void clearPath() {
		if (null != mCachedPath) {
			mCachedPath.rewind();
		}
	}

	/**
	 * Compute an oriented bounding box of the stroke
	 * 
	 * @return OrientedBoundingBox
	 */
	public OrientedBoundingBox computeOrientedBoundingBox() {
		return GestureUtilities.computeOrientedBoundingBox(points);
	}
	
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((boundingBox == null) ? 0 : boundingBox.hashCode());
		result = prime * result + Float.floatToIntBits(length);
		result = prime * result
				+ ((mCachedPath == null) ? 0 : mCachedPath.hashCode());
		result = prime * result + Arrays.hashCode(points);
		result = prime * result + Arrays.hashCode(timestamps);
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		GestureStroke other = (GestureStroke) obj;
		if (boundingBox == null) {
			if (other.boundingBox != null)
				return false;
		} else if (!boundingBox.equals(other.boundingBox))
			return false;
		if (Float.floatToIntBits(length) != Float.floatToIntBits(other.length))
			return false;
		if (mCachedPath == null) {
			if (other.mCachedPath != null)
				return false;
		} else if (!mCachedPath.equals(other.mCachedPath))
			return false;
		if (!Arrays.equals(points, other.points))
			return false;
		if (!Arrays.equals(timestamps, other.timestamps))
			return false;
		return true;
	}
}
