/*
 * Copyright (C) 2011 OgakiSoft
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ogakisoft.android.gestureime;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import ogakisoft.android.util.LOG;
import ogakisoft.android.util.Utils;
import android.content.Context;
import android.content.res.Resources;
import android.database.DataSetObserver;
import android.graphics.Bitmap;
import android.util.DisplayMetrics;
import android.view.Gravity;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.View.OnLongClickListener;
import android.view.ViewGroup;
import android.view.Window;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.ListAdapter;
import android.widget.PopupWindow;
import android.widget.TextView;
import android.widget.Toast;

/**
 * GridView Adapter for showing gesture assigned string
 * 
 * @author noritoshi ogaki
 * @version 1.1
 */
public class GridViewAdapter implements ListAdapter {
	private final static String TAG = "GridViewAdapter";
	private final Resources mResources;
	private final int mPathColor;
	private final int mThumbnailInset;
	private final int mThumbnailSize;
	private final Context mContext;
	private final Window mWindow;
	private final PopupWindow mPopup;
	private boolean mGuideEnabled = false;
	private NamedGesture mCurrentNamedGesture;

	private final List<DataSetObserver> mObserver = Collections
			.synchronizedList(new ArrayList<DataSetObserver>());

	private final List<NamedGesture> mList = Collections
			.synchronizedList(new ArrayList<NamedGesture>());

	private final Comparator<NamedGesture> mSorter = new Comparator<NamedGesture>() {
		public int compare(NamedGesture object1, NamedGesture object2) {
			return object1.getName().compareTo(object2.getName());
		}
	};

	private class GestureOnLongClickListener implements OnLongClickListener {
		@Override
		public boolean onLongClick(View v) {
			// for showing rename/delete dialog
			return false;
		}
	}

	private class GestureOnClickListener implements OnClickListener {
		@Override
		public void onClick(View v) {
			if (mPopup.isShowing()) {
				mPopup.dismiss();
				mCurrentNamedGesture = null;
			} else {
				final LinearLayout view = (LinearLayout) v;
				final NamedGesture gesture = (NamedGesture) view.getTag();
				mCurrentNamedGesture = gesture;
				// LOG.d(TAG, "GestureOnClickListener: gesture id={0,number,#}",
				// gesture.getId());
				final Bitmap bitmap = gesture.getGesture().toBitmap(
						mThumbnailSize, mThumbnailSize, mThumbnailInset,
						mPathColor);
				final LinearLayout layout = (LinearLayout) mPopup
						.getContentView();
				final ImageView popupImage = (ImageView) layout
						.findViewById(R.id.popup_image);
				final TextView popupText = (TextView) layout
						.findViewById(R.id.popup_text);
				popupImage.setMinimumHeight(bitmap.getHeight());
				popupImage.setMinimumWidth(bitmap.getWidth());
				popupImage.setImageBitmap(bitmap);
				popupText.setText(gesture.getName());
				mPopup.showAsDropDown(v);
				if (mGuideEnabled) {
					Toast.makeText(mContext,
							R.string.message_gesture_longclick,
							Toast.LENGTH_LONG).show();
				}
			}
		}
	}

	public GridViewAdapter(Context context, Resources resources, Window window) {
		mContext = context;
		mResources = resources;
		mWindow = window;

		mPathColor = mResources.getColor(R.color.gesture_color);
		final LinearLayout layout = (LinearLayout) mWindow.getLayoutInflater()
				.inflate(R.layout.popup, null);
		final DisplayMetrics m = mResources.getDisplayMetrics();
		final int popupWidth = (int) (mResources
				.getDimension(R.dimen.gesture_popup_width));
		final int popupHeight = (int) (mResources
				.getDimension(R.dimen.gesture_popup_height));
		int thumbInset = (int) (mResources
				.getDimension(R.dimen.gesture_thumbnail_inset));
		int thumbSize = (int) (mResources
				.getDimension(R.dimen.gesture_thumbnail_size));
		if (m.widthPixels < popupWidth || m.heightPixels < popupHeight) {
			LOG.d(TAG, "widthPixels={0,number,#}, heightPixels={1,number,#}", m.widthPixels, m.heightPixels);
			LOG.d(TAG, "popup width={0,number,#}, height={1,number,#}", popupWidth, popupHeight);
			LOG.e(TAG, "window size too small, thumbnail size={0,number,#}", thumbSize);
		}
		mThumbnailInset = thumbInset;
		mThumbnailSize = thumbSize;
		mPopup = new PopupWindow(mContext);
		mPopup.setContentView(layout);
		mPopup.setWidth(popupWidth);
		mPopup.setHeight(popupHeight);
	}

	public void add(NamedGesture gesture) {
		mList.add(gesture);
	}

	@Override
	public boolean areAllItemsEnabled() {
		return false;
	}

	public void clear() {
		mList.clear();
	}

	@Override
	public int getCount() {
		return mList.size();
	}

	@Override
	public NamedGesture getItem(int position) {
		return mList.get(position);
	}

	// public NamedGesture getItemById(long gesture_id) {
	// NamedGesture result = null;
	// synchronized (mList) {
	// final List<NamedGesture> list = mList;
	// for (NamedGesture g : list) {
	// if (g.getId() == gesture_id) {
	// result = g;
	// break;
	// }
	// }
	// }
	// return result;
	// }

	@Override
	public long getItemId(int position) {
		return position;
	}

	@Override
	public int getItemViewType(int position) {
		return android.widget.Adapter.IGNORE_ITEM_VIEW_TYPE;
	}

	@Override
	public View getView(int position, View convertView, ViewGroup parent) {
		final NamedGesture gesture = getItem(position);
		LinearLayout layout = null;
		if (null == convertView) {
			final TextView view = new TextView(mContext);
			String text = gesture.getName();
			if (text.length() > 1) {
				text = Utils.concat(text.substring(0, 1), "...");
			}
			view.setText(text);
			view.setTextSize(24f);
			view.setGravity(Gravity.CENTER_HORIZONTAL);
			layout = new LinearLayout(mContext);
			layout.setGravity(Gravity.CENTER_HORIZONTAL);
			layout.addView(view);
			layout.setTag(gesture);
			layout.setOnLongClickListener(new GestureOnLongClickListener());
			layout.setOnClickListener(new GestureOnClickListener());
		} else {
			layout = (LinearLayout) convertView;
		}
		return layout;
	}

	@Override
	public int getViewTypeCount() {
		return 1;
	}

	@Override
	public boolean hasStableIds() {
		return false;
	}

	@Override
	public boolean isEmpty() {
		return false;
	}

	@Override
	public boolean isEnabled(int position) {
		return false;
	}

	public void remove(NamedGesture gesture) {
		mList.remove(gesture);
	}

	public void sort() {
		Collections.sort(mList, mSorter);
	}

	@Override
	public void registerDataSetObserver(DataSetObserver observer) {
		mObserver.add(observer);
	}

	@Override
	public void unregisterDataSetObserver(DataSetObserver observer) {
		mObserver.remove(observer);
	}

	public void notifyDataSetChanged() {
		for (DataSetObserver ob : mObserver) {
			ob.onChanged();
		}
	}

	public NamedGesture getSelectedItem() {
		return mCurrentNamedGesture;
	}

	public void closePopup() {
		if (mPopup.isShowing()) {
			mPopup.dismiss();
		}
	}
}
